/*
 * OProfile User Interface
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Author: Robert Bradford <rob@openedhand.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


#include <stdlib.h>
#include <glib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <wait.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <netdb.h>
#include <assert.h>

#include <gtk/gtk.h>
#include <glade/glade.h>
#include <glib.h>
#include <libgnomevfs/gnome-vfs.h>

#include "response.h"
#include "command.h"

#include "archive.h"
#include "archive_save_window.h"
#include "archive_window.h"

#include "view.h"
#include "report.h"
#include "client.h"
#include "main.h"

#include "config.h"

static GladeXML *xml;

static gboolean group_by_application = TRUE;
static gboolean group_by_module = TRUE;

GtkWidget *main_window;

#define SOURCE_GLADE "data/oprofile-viewer.glade"
#define INSTALLED_GLADE PKG_DATA_DIR"/oprofile-viewer.glade"

void
on_stop_clicked (GtkWidget *widget, gpointer user_data)
{
  GtkWidget *toolbar_stop;
  GtkWidget *toolbar_disconnect;
  GtkWidget *toolbar_download;
  GtkWidget *toolbar_reset;

  GtkWidget *menu_stop;
  GtkWidget *menu_disconnect;
  GtkWidget *menu_download;
  GtkWidget *menu_reset;

  /* Turn off stop button */
  toolbar_stop = glade_xml_get_widget (xml, "toolbar_stop");
  gtk_widget_set_sensitive (toolbar_stop, FALSE);

  /* Turn off stop menu item */
  menu_stop = glade_xml_get_widget (xml, "menu_stop");
  gtk_widget_set_sensitive (menu_stop, FALSE);

  /* Turn off download button */
  toolbar_download = glade_xml_get_widget (xml, "toolbar_download");
  gtk_widget_set_sensitive (toolbar_download, FALSE);

  /* Turn off download menu item*/
  menu_download = glade_xml_get_widget (xml, "menu_download");
  gtk_widget_set_sensitive (menu_download, FALSE);

  /* Turn off disconnect button */
  toolbar_disconnect = glade_xml_get_widget (xml, "toolbar_disconnect");
  gtk_widget_set_sensitive (toolbar_disconnect, FALSE);

  /* Turn off disconnect menu item */
  menu_disconnect = glade_xml_get_widget (xml, "menu_disconnect");
  gtk_widget_set_sensitive (menu_disconnect, FALSE);

  /* Turn off reset button */
  toolbar_reset = glade_xml_get_widget (xml, "toolbar_reset");
  gtk_widget_set_sensitive (toolbar_reset, FALSE);

  /* Turn off reset menu item*/
  menu_reset = glade_xml_get_widget (xml, "menu_reset");
  gtk_widget_set_sensitive (menu_reset, FALSE);

  client_send_command_stop ();
}

void
on_reset_clicked (GtkWidget *widget, gpointer user_data)
{
  GtkWidget *toolbar_stop;
  GtkWidget *toolbar_start;
  GtkWidget *toolbar_disconnect;
  GtkWidget *toolbar_download;
  GtkWidget *toolbar_reset;

  GtkWidget *menu_stop;
  GtkWidget *menu_start;
  GtkWidget *menu_disconnect;
  GtkWidget *menu_download;
  GtkWidget *menu_reset;

  /* Turn off stop button */
  toolbar_stop = glade_xml_get_widget (xml, "toolbar_stop");
  gtk_widget_set_sensitive (toolbar_stop, FALSE);

  /* Turn off stop menu item */
  menu_stop = glade_xml_get_widget (xml, "menu_stop");
  gtk_widget_set_sensitive (menu_stop, FALSE);

  /* Turn off start button */
  toolbar_start = glade_xml_get_widget (xml, "toolbar_start");
  gtk_widget_set_sensitive (toolbar_start, FALSE);

  /* Turn off start menu item */
  menu_start = glade_xml_get_widget (xml, "menu_start");
  gtk_widget_set_sensitive (menu_start, FALSE);

  /* Turn off download button */
  toolbar_download = glade_xml_get_widget (xml, "toolbar_download");
  gtk_widget_set_sensitive (toolbar_download, FALSE);

  /* Turn off download menu item*/
  menu_download = glade_xml_get_widget (xml, "menu_download");
  gtk_widget_set_sensitive (menu_download, FALSE);

  /* Turn off disconnect button */
  toolbar_disconnect = glade_xml_get_widget (xml, "toolbar_disconnect");
  gtk_widget_set_sensitive (toolbar_disconnect, FALSE);

  /* Turn off disconnect menu item */
  menu_disconnect = glade_xml_get_widget (xml, "menu_disconnect");
  gtk_widget_set_sensitive (menu_disconnect, FALSE);

  /* Turn off reset button */
  toolbar_reset = glade_xml_get_widget (xml, "toolbar_reset");
  gtk_widget_set_sensitive (toolbar_reset, FALSE);

  /* Turn off reset menu item*/
  menu_reset = glade_xml_get_widget (xml, "menu_reset");
  gtk_widget_set_sensitive (menu_reset, FALSE);

  client_send_command_reset ();
}

void
on_start_clicked (GtkWidget *widget, gpointer user_data)
{
  GtkWidget *toolbar_start;
  GtkWidget *toolbar_disconnect;
  GtkWidget *toolbar_download;
  GtkWidget *toolbar_reset;

  GtkWidget *menu_start;
  GtkWidget *menu_disconnect;
  GtkWidget *menu_download;
  GtkWidget *menu_reset;

  /* Turn off start button */
  toolbar_start = glade_xml_get_widget (xml, "toolbar_start");
  gtk_widget_set_sensitive (toolbar_start, FALSE);

  /* Turn off start menu item */
  menu_start = glade_xml_get_widget (xml, "menu_start");
  gtk_widget_set_sensitive (menu_start, FALSE);

  /* Turn off download button */
  toolbar_download = glade_xml_get_widget (xml, "toolbar_download");
  gtk_widget_set_sensitive (toolbar_download, FALSE);

  /* Turn off download menu item*/
  menu_download = glade_xml_get_widget (xml, "menu_download");
  gtk_widget_set_sensitive (menu_download, FALSE);

  /* Turn off disconnect button */
  toolbar_disconnect = glade_xml_get_widget (xml, "toolbar_disconnect");
  gtk_widget_set_sensitive (toolbar_disconnect, FALSE);

  /* Turn off disconnect menu item */
  menu_disconnect = glade_xml_get_widget (xml, "menu_disconnect");
  gtk_widget_set_sensitive (menu_disconnect, FALSE);

  /* Turn off reset button */
  toolbar_reset = glade_xml_get_widget (xml, "toolbar_reset");
  gtk_widget_set_sensitive (toolbar_reset, FALSE);

  /* Turn off reset menu item*/
  menu_reset = glade_xml_get_widget (xml, "menu_reset");
  gtk_widget_set_sensitive (menu_reset, FALSE);

  client_send_command_start ();
}

void
on_download_clicked (GtkWidget *widget, gpointer user_data)
{
  GtkWidget *toolbar_start;
  GtkWidget *toolbar_disconnect;
  GtkWidget *toolbar_download;
  GtkWidget *toolbar_reset;

  GtkWidget *menu_start;
  GtkWidget *menu_disconnect;
  GtkWidget *menu_download;
  GtkWidget *menu_reset;

  /* Turn off start button */
  toolbar_start = glade_xml_get_widget (xml, "toolbar_start");
  gtk_widget_set_sensitive (toolbar_start, FALSE);

  /* Turn off start menu item */
  menu_start = glade_xml_get_widget (xml, "menu_start");
  gtk_widget_set_sensitive (menu_start, FALSE);

  /* Turn off download button */
  toolbar_download = glade_xml_get_widget (xml, "toolbar_download");
  gtk_widget_set_sensitive (toolbar_download, FALSE);

  /* Turn off download menu item*/
  menu_download = glade_xml_get_widget (xml, "menu_download");
  gtk_widget_set_sensitive (menu_download, FALSE);
  
  /* Turn off disconnect button */
  toolbar_disconnect = glade_xml_get_widget (xml, "toolbar_disconnect");
  gtk_widget_set_sensitive (toolbar_disconnect, FALSE);

  /* Turn off disconnect menu item */
  menu_disconnect = glade_xml_get_widget (xml, "menu_disconnect");
  gtk_widget_set_sensitive (menu_disconnect, FALSE);

  /* Turn off reset button */
  toolbar_reset = glade_xml_get_widget (xml, "toolbar_reset");
  gtk_widget_set_sensitive (toolbar_reset, FALSE);

  /* Turn off reset menu item*/
  menu_reset = glade_xml_get_widget (xml, "menu_reset");
  gtk_widget_set_sensitive (menu_reset, FALSE);

  client_send_command_archive ();
}

void
main_generic_error (gchar *err_message)
{
  GtkWidget *dialog;
  guint res;

  g_assert (err_message != NULL);

  dialog = gtk_message_dialog_new (GTK_WINDOW(main_window), GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, 
      GTK_BUTTONS_CLOSE, "%s", err_message);

  res = gtk_dialog_run (GTK_DIALOG(dialog));

  gtk_widget_destroy (dialog);
}

/*
 * This is fired by the client when a response from the start command has been
 * processed. It allows the updating of the state/display of error messages.
 */
void
main_start_response_cb (gboolean success, gchar *err_message)
{
  if (!success)
    main_generic_error (err_message);

  client_send_command_status ();
}

void
main_stop_response_cb (gboolean success, gchar *err_message)
{
  if (!success)
    {
      main_generic_error (err_message);
    } else {

      /* Start the archive retrieval process */
      archive_window_show (xml);
      client_send_command_archive ();
    }
}

void
main_reset_response_cb (gboolean success, gchar *err_message)
{
  if (!success)
    main_generic_error (err_message);

  client_send_command_status ();
}

void
main_archive_response_cb (gboolean success, gchar *err_message)
{
  if (!success)
  {
    main_generic_error (err_message);
    client_send_command_status ();
  } else {
    /* Start the archive retrieval process */
    archive_window_show (xml);
  }
}

void
main_archive_finished_cb (gboolean success, gchar *err_message)
{
  GtkWidget *toolbar_save;
  GtkWidget *menu_save;

  GtkWidget *menu_group_by_application;
  GtkWidget *menu_group_by_module;

  GtkWidget *filter_hbox;

  gchar *archive_path;

  if (!success)
    main_generic_error (err_message);

  /* Generate the report and display it */

  archive_path = archive_get_archive_path ();
  report_generate (archive_path);
  view_set_display (group_by_application, group_by_module);

  client_send_command_status ();

  /* Turn on the save toolbar button */
  toolbar_save = glade_xml_get_widget (xml, "toolbar_save");
  gtk_widget_set_sensitive (toolbar_save, TRUE);

  /* Turn on the save menu item */
  menu_save = glade_xml_get_widget (xml, "menu_save");
  gtk_widget_set_sensitive (menu_save, TRUE);

  /* Turn on view menu options */
  menu_group_by_application = glade_xml_get_widget (xml, "menu_group_by_application");
  menu_group_by_module = glade_xml_get_widget (xml, "menu_group_by_module");

  gtk_widget_set_sensitive (menu_group_by_application, TRUE);
  gtk_widget_set_sensitive (menu_group_by_module, TRUE);

  /* Turn on the filter box */
  filter_hbox = glade_xml_get_widget (xml, "filter_hbox");
  gtk_widget_set_sensitive (filter_hbox, TRUE);
}

void
main_connect_response_cb (gboolean success, gchar *err_message)
{
  GtkWidget *toolbar_disconnect;
  GtkWidget *toolbar_connect;

  GtkWidget *menu_disconnect;
  GtkWidget *menu_connect;

  if (!success)
    {
      main_generic_error (err_message);

      /* Turn on the connect button */
      toolbar_connect = glade_xml_get_widget (xml, "toolbar_connect");
      gtk_widget_set_sensitive (toolbar_connect, TRUE);

      /* Turn on the connect menu item */
      menu_connect = glade_xml_get_widget (xml, "menu_connect");
      gtk_widget_set_sensitive (menu_connect, TRUE);

    } else {
      client_send_command_status ();

      /* Turn on the disconnect button */
      toolbar_disconnect = glade_xml_get_widget (xml, "toolbar_disconnect");
      gtk_widget_set_sensitive (toolbar_disconnect, TRUE);

      /* Turn on the disconnect menu item */
      menu_disconnect = glade_xml_get_widget (xml, "menu_disconnect");
      gtk_widget_set_sensitive (menu_disconnect, TRUE);
    }
}

void
main_disconnect_response_cb (gboolean success, gchar *err_message)
{
  GtkWidget *toolbar_connect;
  GtkWidget *menu_connect;

  if (!success)
    main_generic_error (err_message);

  /* Trun on the connect button */
  toolbar_connect = glade_xml_get_widget (xml, "toolbar_connect");
  gtk_widget_set_sensitive (toolbar_connect, TRUE);

  /* Trun on the connect menu item */
  menu_connect = glade_xml_get_widget (xml, "menu_connect");
  gtk_widget_set_sensitive (menu_connect, TRUE);
}

void
main_status_response_cb (gboolean success, gchar *err_message, gchar status)
{
  GtkWidget *toolbar_stop;
  GtkWidget *toolbar_start;
  GtkWidget *toolbar_disconnect;
  GtkWidget *toolbar_download;
  GtkWidget *toolbar_reset;

  GtkWidget *menu_stop;
  GtkWidget *menu_start;
  GtkWidget *menu_disconnect;
  GtkWidget *menu_download;
  GtkWidget *menu_reset;

  if (!success)
    {
      main_generic_error (err_message);
    } else {
      
      toolbar_download = glade_xml_get_widget (xml, "toolbar_download");
      gtk_widget_set_sensitive (toolbar_download, TRUE);

      menu_download = glade_xml_get_widget (xml, "menu_download");
      gtk_widget_set_sensitive (menu_download, TRUE);

      toolbar_reset = glade_xml_get_widget (xml, "toolbar_reset");
      gtk_widget_set_sensitive (toolbar_reset, TRUE);

      menu_reset = glade_xml_get_widget (xml, "menu_reset");
      gtk_widget_set_sensitive (menu_reset, TRUE);

      switch (status)
        {
          case RESPONSE_STATUS_RUNNING:
            toolbar_stop = glade_xml_get_widget (xml, "toolbar_stop");
            menu_stop = glade_xml_get_widget (xml, "menu_stop");

            gtk_widget_set_sensitive (toolbar_stop, TRUE);
            gtk_widget_set_sensitive (menu_stop, TRUE);
            break;
          case RESPONSE_STATUS_STOPPED:
            toolbar_start = glade_xml_get_widget (xml, "toolbar_start");
            menu_start = glade_xml_get_widget (xml, "menu_start");

            gtk_widget_set_sensitive (toolbar_start, TRUE);
            gtk_widget_set_sensitive (menu_start, TRUE);
            break;
          default:
            g_warning ("Invalid payload value.");
        }
    }

  /* Turn the disconnect button on */
  toolbar_disconnect = glade_xml_get_widget (xml, "toolbar_disconnect");
  gtk_widget_set_sensitive (toolbar_disconnect, TRUE);

  /* Turn the disconnect menu on */
  menu_disconnect = glade_xml_get_widget (xml, "menu_disconnect");
  gtk_widget_set_sensitive (menu_disconnect, TRUE);
}

void
main_connection_lost_cb ()
{
  GtkWidget *toolbar_connect;
  GtkWidget *toolbar_disconnect;
  GtkWidget *toolbar_start;
  GtkWidget *toolbar_stop;
  GtkWidget *toolbar_download;
  GtkWidget *toolbar_reset;

  GtkWidget *menu_connect;
  GtkWidget *menu_disconnect;
  GtkWidget *menu_start;
  GtkWidget *menu_stop;
  GtkWidget *menu_download;
  GtkWidget *menu_reset;

  /* Turn the connect button on */
  toolbar_connect = glade_xml_get_widget (xml, "toolbar_connect");
  gtk_widget_set_sensitive (toolbar_connect, TRUE);

  /* Turn the connect menu item on */
  menu_connect = glade_xml_get_widget (xml, "menu_connect");
  gtk_widget_set_sensitive (menu_connect, TRUE);

  /* Turn the disconnect button off */
  toolbar_disconnect = glade_xml_get_widget (xml, "toolbar_disconnect");
  gtk_widget_set_sensitive (toolbar_disconnect, FALSE);

  /* Turn the disconnect menu item off */
  menu_disconnect = glade_xml_get_widget (xml, "menu_disconnect");
  gtk_widget_set_sensitive (menu_disconnect, FALSE);

  /* Turn the start button off */
  toolbar_start = glade_xml_get_widget (xml, "toolbar_start");
  gtk_widget_set_sensitive (toolbar_start, FALSE);

  /* Turn the start menu item off */
  menu_start = glade_xml_get_widget (xml, "menu_start");
  gtk_widget_set_sensitive (menu_start, FALSE);

  /* Turn the stop button off */
  toolbar_stop = glade_xml_get_widget (xml, "toolbar_stop");
  gtk_widget_set_sensitive (toolbar_stop, FALSE);

  /* Turn the stop menu item off */
  menu_stop = glade_xml_get_widget (xml, "menu_stop");
  gtk_widget_set_sensitive (menu_stop, FALSE);

  /* Turn off download button */
  toolbar_download = glade_xml_get_widget (xml, "toolbar_download");
  gtk_widget_set_sensitive (toolbar_download, FALSE);

  /* Turn off download menu item*/
  menu_download = glade_xml_get_widget (xml, "menu_download");
  gtk_widget_set_sensitive (menu_download, FALSE);

  /* Turn off reset button */
  toolbar_reset = glade_xml_get_widget (xml, "toolbar_reset");
  gtk_widget_set_sensitive (toolbar_reset, FALSE);

  /* Turn off reset menu item*/
  menu_reset = glade_xml_get_widget (xml, "menu_reset");
  gtk_widget_set_sensitive (menu_reset, FALSE);

  main_generic_error ("Connection to remote server lost.");
}

void 
on_save_clicked (GtkWidget *widget, gpointer user_data)
{
  GtkWidget *dialog;

  dialog = gtk_file_chooser_dialog_new ("Choose location to save archive",
      GTK_WINDOW(main_window),
      GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER,
      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
      GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
      NULL);

  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      char *filename;
      
      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      
      archive_save_window_setup (xml);

      /* Yield to gtk. */
      while (gtk_events_pending ())
        gtk_main_iteration ();

      gtk_widget_destroy (dialog);

      archive_save (filename);
      g_free (filename);
    }
}

void 
on_open_clicked (GtkWidget *widget, gpointer user_data)
{
  GtkWidget *dialog;
  gchar *archive_path;

  dialog = gtk_file_chooser_dialog_new ("Choose archive",
      GTK_WINDOW(main_window),
      GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
      NULL);

  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      char *filename;
      GtkWidget *menu_group_by_application;
      GtkWidget *menu_group_by_module;
      GtkWidget *filter_hbox;
      
      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      archive_load (filename);
      g_free (filename);

      /* Generate the report and display it */
      archive_path = archive_get_archive_path ();
      report_generate (archive_path);
      view_set_display (group_by_application, group_by_module);

      /* Turn on view menu options */
      menu_group_by_application = glade_xml_get_widget (xml, "menu_group_by_application");
      menu_group_by_module = glade_xml_get_widget (xml, "menu_group_by_module");

      gtk_widget_set_sensitive (menu_group_by_application, TRUE);
      gtk_widget_set_sensitive (menu_group_by_module, TRUE);

      /* Turn on the filter box */
      filter_hbox = glade_xml_get_widget (xml, "filter_hbox");
      gtk_widget_set_sensitive (filter_hbox, TRUE);
    }

  gtk_widget_destroy (dialog);
}



void
on_connect_clicked (GtkWidget *widget, gpointer user_data)
{
  GtkWidget *dialog;
  GtkWidget *toolbar_connect;
  GtkWidget *menu_connect;
  GtkWidget *connect_dialog_host;
  GtkWidget *connect_dialog_port;
  guint response;

  dialog = glade_xml_get_widget (xml, "connect_dialog");

  connect_dialog_host = glade_xml_get_widget (xml, "connect_dialog_host");
  connect_dialog_port = glade_xml_get_widget (xml, "connect_dialog_port");

  /* For some reason these properties are not set by glade on subsequent runs of
   * the dialog, so we have to set them manually here */
  gtk_entry_set_activates_default (GTK_ENTRY (connect_dialog_host), TRUE);
  gtk_entry_set_activates_default (GTK_ENTRY (connect_dialog_port), TRUE);

  response = gtk_dialog_run (GTK_DIALOG(dialog));

  if (response == GTK_RESPONSE_OK)
    {

      gchar *hostname;
      guint port;

      /* Must g_strdup here since the string returned is internal */
      hostname = g_strdup (gtk_entry_get_text (GTK_ENTRY (connect_dialog_host)));
      port = atoi (gtk_entry_get_text (GTK_ENTRY (connect_dialog_port)));

      /* Turn off the connect button */
      toolbar_connect = glade_xml_get_widget (xml, "toolbar_connect");
      gtk_widget_set_sensitive (toolbar_connect, FALSE);

      /* Turn off the connect menu item */
      menu_connect = glade_xml_get_widget (xml, "menu_connect");
      gtk_widget_set_sensitive (menu_connect, FALSE);

      client_connect (hostname, port);
    }
  
  gtk_widget_hide (dialog);

}

void
on_disconnect_clicked (GtkWidget *widget, gpointer user_data)
{
  GtkWidget *toolbar_disconnect;
  GtkWidget *toolbar_start;
  GtkWidget *toolbar_stop;
  GtkWidget *toolbar_download;
  GtkWidget *toolbar_reset;

  GtkWidget *menu_disconnect;
  GtkWidget *menu_start;
  GtkWidget *menu_stop;
  GtkWidget *menu_download;
  GtkWidget *menu_reset;

  /* Turn off disconnect button */
  toolbar_disconnect = glade_xml_get_widget (xml, "toolbar_disconnect");
  gtk_widget_set_sensitive (toolbar_disconnect, FALSE);

  /* Turn off disconnect menu item */
  menu_disconnect = glade_xml_get_widget (xml, "menu_disconnect");
  gtk_widget_set_sensitive (menu_disconnect, FALSE);

  /* Turn off stop button */
  toolbar_stop = glade_xml_get_widget (xml, "toolbar_stop");
  gtk_widget_set_sensitive (toolbar_stop, FALSE);

  /* Turn off stop menu item */
  menu_stop = glade_xml_get_widget (xml, "menu_stop");
  gtk_widget_set_sensitive (menu_stop, FALSE);

  /* Turn off start button */
  toolbar_start = glade_xml_get_widget (xml, "toolbar_start");
  gtk_widget_set_sensitive (toolbar_start, FALSE);

  /* Turn off start menu item */
  menu_start = glade_xml_get_widget (xml, "menu_start");
  gtk_widget_set_sensitive (menu_start, FALSE);

  /* Turn off download button */
  toolbar_download = glade_xml_get_widget (xml, "toolbar_download");
  gtk_widget_set_sensitive (toolbar_download, FALSE);

  /* Turn off download menu item */
  menu_download = glade_xml_get_widget (xml, "menu_download");
  gtk_widget_set_sensitive (menu_download, FALSE);

  /* Turn off reset button */
  toolbar_reset = glade_xml_get_widget (xml, "toolbar_reset");
  gtk_widget_set_sensitive (toolbar_reset, FALSE);

  /* Turn off reset menu item */
  menu_reset = glade_xml_get_widget (xml, "menu_reset");
  gtk_widget_set_sensitive (menu_reset, FALSE);

  client_disconnect ();
}

static gboolean
on_main_window_delete_event (GtkWidget *widget, GdkEvent  *event, gpointer data)
{
    return FALSE;
}

static void
on_main_window_destroy (GtkWidget *widget, gpointer data)
{
    gtk_main_quit ();
}

static void
on_menu_group_by_application_toggled (GtkCheckMenuItem *menuitem, gpointer user_data)
{
  gboolean new_val = gtk_check_menu_item_get_active (menuitem);

  if (new_val != group_by_application)
    {
      group_by_application = new_val;
      view_set_display (group_by_application, group_by_module);
    }
}

static void
on_menu_group_by_module_toggled (GtkCheckMenuItem *menuitem, gpointer user_data)
{
  gboolean new_val = gtk_check_menu_item_get_active (menuitem);

  if (new_val != group_by_module)
    {
      group_by_module = new_val;
      view_set_display (group_by_application, group_by_module);
    }
}

static void
on_menu_quit_activate (GtkMenuItem *menuitem, gpointer user_data)
{
  gtk_widget_destroy (GTK_WIDGET (main_window));
}

static void
on_menu_about_activate (GtkMenuItem *menuitem, gpointer user_data)
{
  const gchar *authors[] = {"Robert Bradford <rob@openedhand.com>", NULL};

  gtk_show_about_dialog (GTK_WINDOW (main_window), 
      "name", "Oprofile Viewer", 
      "version", PACKAGE_VERSION, 
      "authors", authors,
      "comments", "A user interface for OProfile",
      "copyright", "(C) 2007 Nokia Corporation",
      NULL);

}

static void
on_filter_entry_activate (GtkEntry *entry, gpointer user_data)
{
  gchar *filter_str = NULL;
  filter_str = (gchar *)gtk_entry_get_text (entry);

  if (filter_str != NULL)
    {
      g_strstrip (filter_str);

      if (strlen (filter_str) > 0)
        {
          view_update_filter (filter_str);
        } else {
          printf ("foo");
          view_update_filter (NULL);
        }
    }
}

static void
on_filter_clear_button_clicked (GtkButton *button, gpointer user_data)
{
  GtkWidget *entry;

  entry = glade_xml_get_widget (xml, "filter_entry");
  gtk_entry_set_text (GTK_ENTRY (entry), "");
  view_update_filter (NULL);
}

int
main(int argc, char *argv[])
{
  gtk_init(&argc, &argv);

  /* Set the window icon */
  gtk_window_set_default_icon_name ("oprofile-viewer");

  /* load the interface */
  if (g_file_test (SOURCE_GLADE, G_FILE_TEST_EXISTS) != FALSE) {
    xml = glade_xml_new (SOURCE_GLADE, NULL, NULL);
  } else {
    xml = glade_xml_new (INSTALLED_GLADE, NULL, NULL);
  }

  if (xml == NULL)
    {
      g_error ("Glade file not found.\n");
    }

  view_setup (xml);

  main_window = glade_xml_get_widget (xml, "main_window");

  glade_xml_signal_connect_data (xml, "on_toolbar_start_clicked",
      G_CALLBACK(on_start_clicked), NULL);
  glade_xml_signal_connect_data (xml, "on_toolbar_stop_clicked",
      G_CALLBACK(on_stop_clicked), NULL);
  
  glade_xml_signal_connect_data (xml, "on_menu_start_activate",
      G_CALLBACK(on_start_clicked), NULL);
  glade_xml_signal_connect_data (xml, "on_menu_stop_activate",
      G_CALLBACK(on_stop_clicked), NULL);

  glade_xml_signal_connect_data (xml, "on_toolbar_save_clicked",
      G_CALLBACK(on_save_clicked), NULL);
  glade_xml_signal_connect_data (xml, "on_toolbar_open_clicked",
      G_CALLBACK(on_open_clicked), NULL);

  glade_xml_signal_connect_data (xml, "on_menu_save_activate",
      G_CALLBACK(on_save_clicked), NULL);
  glade_xml_signal_connect_data (xml, "on_menu_open_activate",
      G_CALLBACK(on_open_clicked), NULL);

  glade_xml_signal_connect_data (xml, "on_toolbar_connect_clicked",
      G_CALLBACK(on_connect_clicked), NULL);
  glade_xml_signal_connect_data (xml, "on_toolbar_disconnect_clicked",
      G_CALLBACK(on_disconnect_clicked), NULL);

  glade_xml_signal_connect_data (xml, "on_menu_connect_activate",
      G_CALLBACK(on_connect_clicked), NULL);
  glade_xml_signal_connect_data (xml, "on_menu_disconnect_activate",
      G_CALLBACK(on_disconnect_clicked), NULL);

  glade_xml_signal_connect_data (xml, "on_main_window_delete_event",
      G_CALLBACK(on_main_window_delete_event), NULL);
  glade_xml_signal_connect_data (xml, "on_main_window_destroy",
      G_CALLBACK(on_main_window_destroy), NULL);

  glade_xml_signal_connect_data (xml, "on_menu_group_by_application_toggled",
      G_CALLBACK(on_menu_group_by_application_toggled), NULL);
  glade_xml_signal_connect_data (xml, "on_menu_group_by_module_toggled",
      G_CALLBACK(on_menu_group_by_module_toggled), NULL);

  glade_xml_signal_connect_data (xml, "on_menu_quit_activate",
      G_CALLBACK(on_menu_quit_activate), NULL);

  glade_xml_signal_connect_data (xml, "on_menu_about_activate",
      G_CALLBACK(on_menu_about_activate), NULL);

  glade_xml_signal_connect_data (xml, "on_toolbar_download_clicked",
      G_CALLBACK(on_download_clicked), NULL);
  glade_xml_signal_connect_data (xml, "on_menu_download_activate",
      G_CALLBACK(on_download_clicked), NULL);

  glade_xml_signal_connect_data (xml, "on_toolbar_reset_clicked",
      G_CALLBACK(on_reset_clicked), NULL);
  glade_xml_signal_connect_data (xml, "on_menu_reset_activate",
      G_CALLBACK(on_reset_clicked), NULL);

  glade_xml_signal_connect_data (xml, "on_filter_entry_activate",
      G_CALLBACK(on_filter_entry_activate), NULL);

  glade_xml_signal_connect_data (xml, "on_filter_clear_button_clicked",
      G_CALLBACK(on_filter_clear_button_clicked), NULL);

  /* start the event loop */
  gtk_main();

  /* Cleanup the old archive */
  archive_cleanup ();

  if (gnome_vfs_initialized())
    gnome_vfs_shutdown ();

  return 0;
}
