/* opkg_hash.c - the opkg package management system

   Steven M. Ayer
   
   Copyright (C) 2002 Compaq Computer Corporation

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
*/

#include "includes.h"
#include <errno.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include "hash_table.h"
#include "pkg.h"
#include "opkg_message.h"
#include "pkg_vec.h"
#include "pkg_hash.h"
#include "pkg_parse.h"
#include "opkg_utils.h"
#include "libbb/libbb.h"

void
pkg_hash_init(const char *name, hash_table_t *hash, int len)
{
	hash_table_init(name, hash, len);
}

static void
free_pkgs(const char *key, void *entry, void *data)
{
	int i;
	abstract_pkg_t *ab_pkg;

	/* Each entry in the hash table is an abstract package, which contains
	 * a list of packages that provide the abstract package.
	 */
  
	ab_pkg = (abstract_pkg_t*) entry;

	if (ab_pkg->pkgs) {
		for (i = 0; i < ab_pkg->pkgs->len; i++) {
			pkg_deinit (ab_pkg->pkgs->pkgs[i]);
			free (ab_pkg->pkgs->pkgs[i]);
		}
	}

	abstract_pkg_vec_free (ab_pkg->provided_by);
	abstract_pkg_vec_free (ab_pkg->replaced_by);
	pkg_vec_free (ab_pkg->pkgs);
	free (ab_pkg->depended_upon_by);
	free (ab_pkg->name);
	free (ab_pkg);
}

void
pkg_hash_deinit(hash_table_t *hash)
{
	hash_table_foreach(hash, free_pkgs, NULL);
	hash_table_deinit(hash);
}


/* Find the default arch for a given package status file if none is given. */
static char *
pkg_get_default_arch(opkg_conf_t *conf)
{
     nv_pair_list_elt_t *l;
     char *def_arch = HOST_CPU_STR;		/* Default arch */
     int def_prio = 0;				/* Other archs override this */

     list_for_each_entry(l , &conf->arch_list.head, node) {
	  nv_pair_t *nv = (nv_pair_t *)l->data;
	  int priority = strtol(nv->value, NULL, 0);

	  /* Check if this arch has higher priority, and is valid */
	  if ((priority > def_prio) &&
	      (strcmp(nv->name, "all")) && (strcmp(nv->name, "noarch"))) {
	       /* Our new default */
	       def_prio = priority;
	       def_arch = nv->name;
	  }
     }

     return xstrdup(def_arch);
}

int
pkg_hash_add_from_file(opkg_conf_t *conf, const char *file_name,
			pkg_src_t *src, pkg_dest_t *dest, int is_status_file)
{
	hash_table_t *hash = &conf->pkg_hash;
	pkg_t *pkg;
	FILE *fp;
	char *buf;
	const size_t len = 4096;
	int ret = 0;

	fp = fopen(file_name, "r");
	if (fp == NULL) {
		fprintf(stderr, "%s: fopen(%s): %s\n",
			     __FUNCTION__, file_name, strerror(errno));
		return -1;
	}

	buf = xmalloc(len);

	do {
		pkg = pkg_new();
		pkg->src = src;
		pkg->dest = dest;

		ret = pkg_parse_from_stream_nomalloc(pkg, fp, PFM_ALL,
				&buf, len);
		if (ret) {
			pkg_deinit (pkg);
			free(pkg);
			if (ret == -1)
				break;
			if (ret == EINVAL)
				/* Probably a blank line, continue parsing. */
				ret = 0;
			continue;
		}

		if (!pkg->architecture) {
			char *version_str = pkg_version_str_alloc(pkg);
			pkg->architecture = pkg_get_default_arch(conf);
			opkg_message(conf, OPKG_ERROR, "Package %s version %s has no architecture specified, defaulting to %s.\n",
			pkg->name, version_str, pkg->architecture);
			free(version_str);
		}

		hash_insert_pkg(hash, pkg, is_status_file, conf);

	} while (!feof(fp));

	free(buf);
	fclose(fp);

	return ret;
}

static abstract_pkg_t *
abstract_pkg_fetch_by_name(hash_table_t * hash, const char * pkg_name)
{
	return (abstract_pkg_t *)hash_table_get(hash, pkg_name);
}

pkg_t *
pkg_hash_fetch_best_installation_candidate(opkg_conf_t *conf,
		abstract_pkg_t *apkg,
		int (*constraint_fcn)(pkg_t *pkg, void *cdata),
		void *cdata, int quiet)
{
     int i;
     int nprovides = 0;
     int nmatching = 0;
     int wrong_arch_found = 0;
     pkg_vec_t *matching_pkgs;
     abstract_pkg_vec_t *matching_apkgs;
     abstract_pkg_vec_t *provided_apkg_vec;
     abstract_pkg_t **provided_apkgs;
     abstract_pkg_vec_t *providers;
     pkg_t *latest_installed_parent = NULL;
     pkg_t *latest_matching = NULL;
     pkg_t *priorized_matching = NULL;
     pkg_t *held_pkg = NULL;
     pkg_t *good_pkg_by_name = NULL;

     if (apkg == NULL || apkg->provided_by == NULL || (apkg->provided_by->len == 0))
	  return NULL;

     matching_pkgs = pkg_vec_alloc();
     matching_apkgs = abstract_pkg_vec_alloc();
     providers = abstract_pkg_vec_alloc();

     opkg_message(conf, OPKG_DEBUG, "best installation candidate for %s\n", apkg->name);

     provided_apkg_vec = apkg->provided_by;
     nprovides = provided_apkg_vec->len;
     provided_apkgs = provided_apkg_vec->pkgs;
     if (nprovides > 1)
	  opkg_message(conf, OPKG_DEBUG, " apkg=%s nprovides=%d\n", apkg->name, nprovides);

     /* accumulate all the providers */
     for (i = 0; i < nprovides; i++) {
	  abstract_pkg_t *provider_apkg = provided_apkgs[i];
	  opkg_message(conf, OPKG_DEBUG, " adding %s to providers\n", provider_apkg->name);
	  abstract_pkg_vec_insert(providers, provider_apkg);
     }
     nprovides = providers->len;

     for (i = 0; i < nprovides; i++) {
	  abstract_pkg_t *provider_apkg = abstract_pkg_vec_get(providers, i);
	  abstract_pkg_t *replacement_apkg = NULL;
	  pkg_vec_t *vec;

	  if (provider_apkg->replaced_by && provider_apkg->replaced_by->len) {
	       replacement_apkg = provider_apkg->replaced_by->pkgs[0];
	       if (provider_apkg->replaced_by->len > 1) {
		    opkg_message(conf, OPKG_NOTICE, "Multiple replacers for %s, using first one (%s)\n", 
				 provider_apkg->name, replacement_apkg->name);
	       }
	  }

	  if (replacement_apkg)
	       opkg_message(conf, OPKG_DEBUG, "   replacement_apkg=%s for provider_apkg=%s\n", 
			    replacement_apkg->name, provider_apkg->name);

	  if (replacement_apkg && (replacement_apkg != provider_apkg)) {
	       if (abstract_pkg_vec_contains(providers, replacement_apkg))
		    continue;
	       else
		    provider_apkg = replacement_apkg;
	  }

	  if (!(vec = provider_apkg->pkgs)) {
	       opkg_message(conf, OPKG_DEBUG, "   no pkgs for provider_apkg %s\n", provider_apkg->name);
	       continue;
	  }
    

	  /* now check for supported architecture */
	  {
	       int max_count = 0;
	       int i;

	       /* count packages matching max arch priority and keep track of last one */
	       for (i = 0; i < vec->len; i++) {
		    pkg_t *maybe = vec->pkgs[i];
		    opkg_message(conf, OPKG_DEBUG, "  %s arch=%s arch_priority=%d version=%s  \n",
				 maybe->name, maybe->architecture, maybe->arch_priority, maybe->version);
                    /* We make sure not to add the same package twice. Need to search for the reason why 
                       they show up twice sometimes. */
		    if ((maybe->arch_priority > 0) && (! pkg_vec_contains(matching_pkgs, maybe))) {
			 max_count++;
			 abstract_pkg_vec_insert(matching_apkgs, maybe->parent);
			 pkg_vec_insert(matching_pkgs, maybe);
		    }
	       }

		if (vec->len > 0 && matching_pkgs->len < 1)
			wrong_arch_found = 1;
	  }
     }

     if (matching_pkgs->len < 1) {
	  if (wrong_arch_found)
	        opkg_message (conf, OPKG_ERROR, "Packages for %s found, but"
			" incompatible with the architectures configured\n",
			apkg->name);
          pkg_vec_free(matching_pkgs);
          abstract_pkg_vec_free(matching_apkgs);
          abstract_pkg_vec_free(providers);
	  return NULL;
     }


     if (matching_pkgs->len > 1)
	  pkg_vec_sort(matching_pkgs, pkg_name_version_and_architecture_compare);
     if (matching_apkgs->len > 1)
	  abstract_pkg_vec_sort(matching_pkgs, abstract_pkg_name_compare);

/* Here it is usefull, if ( matching_apkgs->len > 1 ), to test if one of this matching packages has the same name of the
   needed package. In this case, I would return it for install, otherwise I will continue with the procedure */
/* The problem is what to do when there are more than a mathing package, with the same name and several version ?
   Until now I always got the latest, but that breaks the downgrade option.
   If I stop at the first one, I would probably miss the new ones 
   Maybe the way is to have some kind of flag somewhere, to see if the package been asked to install is from a file,
   or from a Packages feed.
   It it is from a file it always need to be checked whatever version I have in feeds or everywhere, according to force-down or whatever options*/
/*Pigi*/

     for (i = 0; i < matching_pkgs->len; i++) {
	  pkg_t *matching = matching_pkgs->pkgs[i];
          if (constraint_fcn(matching, cdata)) {  /* We found it */
             opkg_message(conf, OPKG_DEBUG, " Found a valid candidate for the install: %s %s  \n", matching->name, matching->version) ;
             good_pkg_by_name = matching;
             if ( matching->provided_by_hand == 1 )    /* It has been provided by hand, so it is what user want */
                break;                                 
          }
     }


     for (i = 0; i < matching_pkgs->len; i++) {
	  pkg_t *matching = matching_pkgs->pkgs[i];
	  latest_matching = matching;
	  if (matching->parent->state_status == SS_INSTALLED || matching->parent->state_status == SS_UNPACKED)
	       latest_installed_parent = matching;
	  if (matching->state_flag & (SF_HOLD|SF_PREFER)) {
	       if (held_pkg)
		    opkg_message(conf, OPKG_NOTICE, "Multiple packages (%s and %s) providing same name marked HOLD or PREFER.  Using latest.\n",
				 held_pkg->name, matching->name);
	       held_pkg = matching;
	  }
     }

     if (!good_pkg_by_name && !held_pkg && !latest_installed_parent && matching_apkgs->len > 1 && !quiet) {
          int prio = 0;
          for (i = 0; i < matching_pkgs->len; i++) {
              pkg_t *matching = matching_pkgs->pkgs[i];
                  if (matching->arch_priority > prio) {
                      priorized_matching = matching;
                      prio = matching->arch_priority;
                      opkg_message(conf, OPKG_DEBUG, "Match with priority %i    %s\n", prio, matching->name);
                  }
              }
          
          }

     if (conf->verbosity >= OPKG_INFO && matching_apkgs->len > 1) {
	  opkg_message(conf, OPKG_INFO, "%s: for apkg=%s, %d matching pkgs\n",
		       __FUNCTION__, apkg->name, matching_pkgs->len);
	  for (i = 0; i < matching_pkgs->len; i++) {
	       pkg_t *matching = matching_pkgs->pkgs[i];
	       opkg_message(conf, OPKG_INFO, "    %s %s %s\n",
			    matching->name, matching->version, matching->architecture);
	  }
     }

     nmatching = matching_apkgs->len;

     pkg_vec_free(matching_pkgs);
     abstract_pkg_vec_free(matching_apkgs);
     abstract_pkg_vec_free(providers);

     if (good_pkg_by_name) {   /* We found a good candidate, we will install it */ 
	  return good_pkg_by_name;
     }
     if (held_pkg) {
	  opkg_message(conf, OPKG_INFO, "  using held package %s\n", held_pkg->name);
	  return held_pkg;
     }
     if (latest_installed_parent) {
	  opkg_message(conf, OPKG_INFO, "  using latest version of installed package %s\n", latest_installed_parent->name);
	  return latest_installed_parent;
     }
     if (priorized_matching) {
	  opkg_message(conf, OPKG_INFO, "  using priorized matching %s %s %s\n",
		       priorized_matching->name, priorized_matching->version, priorized_matching->architecture);
	  return priorized_matching;
     }
     if (nmatching > 1) {
	  opkg_message(conf, OPKG_INFO, "  no matching pkg out of matching_apkgs=%d\n", nmatching);
	  return NULL;
     }
     if (latest_matching) {
	  opkg_message(conf, OPKG_INFO, "  using latest matching %s %s %s\n",
		       latest_matching->name, latest_matching->version, latest_matching->architecture);
	  return latest_matching;
     }
     return NULL;
}

static int
pkg_name_constraint_fcn(pkg_t *pkg, void *cdata)
{
	const char *name = (const char *)cdata;

	if (strcmp(pkg->name, name) == 0)
		return 1;
	else
		return 0;   
}

static pkg_vec_t *
pkg_vec_fetch_by_name(hash_table_t *hash, const char *pkg_name)
{
	abstract_pkg_t * ab_pkg;

	if(!(ab_pkg = abstract_pkg_fetch_by_name(hash, pkg_name)))
		return NULL;

	if (ab_pkg->pkgs)
		return ab_pkg->pkgs;

	if (ab_pkg->provided_by) {
		abstract_pkg_t *abpkg =  abstract_pkg_vec_get(ab_pkg->provided_by, 0);
		if (abpkg != NULL)
			return abpkg->pkgs;
		else
			return ab_pkg->pkgs;
	}

	return NULL;
}


pkg_t *
pkg_hash_fetch_best_installation_candidate_by_name(opkg_conf_t *conf,
		const char *name)
{
	hash_table_t *hash = &conf->pkg_hash;
	abstract_pkg_t *apkg = NULL;

	if (!(apkg = abstract_pkg_fetch_by_name(hash, name)))
		return NULL;

	return pkg_hash_fetch_best_installation_candidate(conf, apkg,
				pkg_name_constraint_fcn, apkg->name, 0);
}


pkg_t *
pkg_hash_fetch_by_name_version(hash_table_t *hash, 
			const char *pkg_name, const char * version)
{
	pkg_vec_t * vec;
	int i;
	char *version_str = NULL;
    
	if(!(vec = pkg_vec_fetch_by_name(hash, pkg_name)))
		return NULL;
    
	for(i = 0; i < vec->len; i++) {
		version_str = pkg_version_str_alloc(vec->pkgs[i]);
		if(!strcmp(version_str, version)) {
			free(version_str);
			break;
		}
		free(version_str);
	}

	if(i == vec->len)
		return NULL;
    
	return vec->pkgs[i];
}

pkg_t *
pkg_hash_fetch_installed_by_name_dest(hash_table_t *hash,
			const char *pkg_name, pkg_dest_t *dest)
{
	pkg_vec_t * vec;
	int i;

	if (!(vec = pkg_vec_fetch_by_name(hash, pkg_name))) {
		return NULL;
	}

	for (i = 0; i < vec->len; i++)
		if((vec->pkgs[i]->state_status == SS_INSTALLED
				|| vec->pkgs[i]->state_status == SS_UNPACKED)
				&& vec->pkgs[i]->dest == dest) {
			return vec->pkgs[i];
	}

	return NULL;
}

pkg_t *
pkg_hash_fetch_installed_by_name(hash_table_t *hash,
					const char *pkg_name)
{
	pkg_vec_t * vec;
	int i;

	if (!(vec = pkg_vec_fetch_by_name(hash, pkg_name))) {
		return NULL;
	}

	for (i = 0; i < vec->len; i++) {
		if (vec->pkgs[i]->state_status == SS_INSTALLED
				|| vec->pkgs[i]->state_status == SS_UNPACKED) {
			return vec->pkgs[i];
        	}
	}

	return NULL;
}


static void
pkg_hash_fetch_available_helper(const char *pkg_name, void *entry, void *data)
{
	int j;
	abstract_pkg_t *ab_pkg = (abstract_pkg_t *)entry;
	pkg_vec_t *all = (pkg_vec_t *)data;
	pkg_vec_t *pkg_vec = ab_pkg->pkgs;

	if (!pkg_vec)
		return;

	for (j = 0; j < pkg_vec->len; j++) {
		pkg_t *pkg = pkg_vec->pkgs[j];
		pkg_vec_insert(all, pkg);
	}
}

void
pkg_hash_fetch_available(hash_table_t *hash, pkg_vec_t *all)
{
	hash_table_foreach(hash, pkg_hash_fetch_available_helper, all);
}

static void
pkg_hash_fetch_all_installed_helper(const char *pkg_name, void *entry, void *data)
{
	abstract_pkg_t *ab_pkg = (abstract_pkg_t *)entry;
	pkg_vec_t *all = (pkg_vec_t *)data;
	pkg_vec_t *pkg_vec = ab_pkg->pkgs;
	int j;

	if (!pkg_vec)
		return;

	for (j = 0; j < pkg_vec->len; j++) {
		pkg_t *pkg = pkg_vec->pkgs[j];
		if (pkg->state_status == SS_INSTALLED
				|| pkg->state_status == SS_UNPACKED)
			pkg_vec_insert(all, pkg);
	}
}

void
pkg_hash_fetch_all_installed(hash_table_t *hash, pkg_vec_t *all)
{
	hash_table_foreach(hash, pkg_hash_fetch_all_installed_helper, all);
}

/*
 * This assumes that the abstract pkg doesn't exist.
 */
static abstract_pkg_t *
add_new_abstract_pkg_by_name(hash_table_t *hash, const char *pkg_name)
{
	abstract_pkg_t *ab_pkg;

	ab_pkg = abstract_pkg_new();

	ab_pkg->name = xstrdup(pkg_name);
	hash_table_insert(hash, pkg_name, ab_pkg);

	return ab_pkg;
}


abstract_pkg_t *
ensure_abstract_pkg_by_name(hash_table_t * hash, const char * pkg_name)
{
	abstract_pkg_t * ab_pkg;

	if (!(ab_pkg = abstract_pkg_fetch_by_name(hash, pkg_name)))
		ab_pkg = add_new_abstract_pkg_by_name(hash, pkg_name);

	return ab_pkg;
}

pkg_t *
hash_insert_pkg(hash_table_t *hash, pkg_t *pkg, int set_status,
		opkg_conf_t *conf)
{
	abstract_pkg_t * ab_pkg;

	if(!pkg)
		return pkg;

	buildDepends(hash, pkg);

	ab_pkg = ensure_abstract_pkg_by_name(hash, pkg->name);

	if (set_status) {
		if (pkg->state_status == SS_INSTALLED) {
			ab_pkg->state_status = SS_INSTALLED;
		} else if (pkg->state_status == SS_UNPACKED) {
			ab_pkg->state_status = SS_UNPACKED;
		}
	}

	if(!ab_pkg->pkgs)
		ab_pkg->pkgs = pkg_vec_alloc();

	buildProvides(hash, ab_pkg, pkg);

	/* Need to build the conflicts graph before replaces for correct
	 * calculation of replaced_by relation.
	 */
	buildConflicts(hash, ab_pkg, pkg);

	buildReplaces(hash, ab_pkg, pkg);

	buildDependedUponBy(pkg, ab_pkg);

	/* pkg_vec_insert_merge might munge package, but it returns an
	 * unmunged pkg.
	 */
	pkg = pkg_vec_insert_merge(ab_pkg->pkgs, pkg, set_status,conf );
	pkg->parent = ab_pkg;

	return pkg;
}


pkg_t *
file_hash_get_file_owner(opkg_conf_t *conf, const char *file_name)
{
	return hash_table_get(&conf->file_hash, file_name); 
}

void
file_hash_set_file_owner(opkg_conf_t *conf, const char *file_name,
		pkg_t *owning_pkg)
{
	pkg_t *old_owning_pkg = hash_table_get(&conf->file_hash, file_name);
	int file_name_len = strlen(file_name);

	if (file_name[file_name_len -1] == '/')
		return;

	if (conf->offline_root) {
		int len = strlen(conf->offline_root);
		if (strncmp(file_name, conf->offline_root, len) == 0) {
			file_name += len;
		}
	}

	hash_table_insert(&conf->file_hash, file_name, owning_pkg); 

	if (old_owning_pkg) {
		pkg_get_installed_files(conf, old_owning_pkg);
		str_list_remove_elt(old_owning_pkg->installed_files, file_name);
		pkg_free_installed_files(old_owning_pkg);

		/* mark this package to have its filelist written */
		old_owning_pkg->state_flag |= SF_FILELIST_CHANGED;
		owning_pkg->state_flag |= SF_FILELIST_CHANGED;
	}
}
