/* Map graphics for the Mac interface to Xconq.
   Copyright (C) 1992-1998 Stanley T. Shebs.

Xconq is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.  See the file COPYING.  */

#include "conq.h"
extern int any_thickness;
#include "macconq.h"
extern void xform_fractional(Map *map, int x, int y, int xf, int yf, int *sxp, int *syp);

extern int contour_interval;

static void draw_meridians(Map *map);
static void draw_borders(Map *map, int x, int y, int b);
static void draw_connections(Map *map, int x, int y, int c);
static void draw_units(Map *map, int x, int y);
static void draw_unit_and_occs(Map *map, Unit *unit, int sx, int sy, int sw, int sh);
static void draw_people_row(Map *map, int x0, int y0, int len);
static void draw_feature_boundary(Map *map, int x, int y);
static void draw_materials(Map *map, int x, int y);
static void draw_ai_region(Map *map, int x, int y);
static void draw_info_text(Map *map, int x, int y, int len, char *buf);
extern void draw_gridlines(Map *map, int x0, int y, int len);
extern void draw_simple_feature_borders(Map *map, int x0, int y0, int len);
extern void draw_shorelines(Map *map, int x0, int y0, int len);
extern void draw_unit_names_only(Map *map, int x, int y);

/* Adjust viewport for offscreen drawing and go offscreen.
   Then call draw_other_maps to erase other-map boxes */

#define GO_OFFSCREEN(m) {  \
	(m)->vp->sx -= (m)->offsetx + (m)->bufx - (m)->conw; \
	(m)->vp->sy -= (m)->offsety + (m)->bufy - (m)->toph; \
	LockPixels(GetGWorldPixMap((m)->gworldPortPtr)); \
	SetPort((GrafPtr)(m)->gworldPortPtr); \
	if ((m)->vp->draw_other_maps) draw_other_maps(m);\
}

/* Call draw_other_maps to redraw other-map boxes, then
   reset viewport for onscreen drawing and return onscreen. */

#define RETURN_ONSCREEN(m) {  \
	if ((m)->vp->draw_other_maps) draw_other_maps(m);\
	(m)->vp->sx += (m)->offsetx + (m)->bufx - (m)->conw; \
	(m)->vp->sy += (m)->offsety + (m)->bufy - (m)->toph; \
	UnlockPixels(GetGWorldPixMap((m)->gworldPortPtr)); \
	SetPort((GrafPtr)(m)->windowPortPtr); \
  }

/* These three macros are used to handle cases where an item stretches across the */
/* edge of a wrapped world. It is then necessary to draw it twice, once at each end. */
/* The macros should be applied to all drawing functions that take an sx value, rect */
/* or poly as a parameter. */

/* Repeat (function) once using sx shifted one area width to the right. */
#define WRAP_SX(map, sx, function) { \
	(function); \
	if (area.xwrap && (sx) < 0) { \
		(sx) += (map)->vp->sxmax; \
		(function); \
		(sx) -= (map)->vp->sxmax; \
	} \
}

/* Repeat (function) once using rect offset one area width to the right. */
#define WRAP_RECT(map, rect, function) { \
	(function); \
	if (area.xwrap && (rect).left < 0) { \
		OffsetRect(&(rect), (map)->vp->sxmax, 0); \
		(function); \
		OffsetRect(&(rect), -(map)->vp->sxmax, 0); \
	} \
}

/* Repeat (function) once using poly offset one area width to the right. */
#define WRAP_POLY(map, poly, function) { \
	(function); \
	if (area.xwrap && (*(*(poly))).polyBBox.left < 0) { \
		OffsetPoly((poly), (map)->vp->sxmax, 0); \
		(function); \
		OffsetPoly((poly), -(map)->vp->sxmax, 0); \
	} \
}

/* For these accessors, x must already be wrapped. */

#define cell_overlay(x, y)  \
  ((dside->designer && !map->see_all) \
   ? ((terrain_view(dside, x, y) == UNSEEN) ? -2 : 0)  \
   : (tmpdrawlighting  \
      ? (night_at(x, y)  \
         ? -2  \
         : (tmpdrawcoverage ? (cover(dside, x, y) == 0 ? -1 : 0) : 0))  \
      : (tmpdrawcoverage  \
          ? (cover(dside, x, y) == 0 ? -1 : 0)  \
          : 0)))

#define cell_terrain(map, x, y)  \
  ((map->see_all || dside->designer) \
   ? terrain_at(x, y)  \
   : (terrain_visible(dside, x, y)  \
	  ? vterrain(terrain_view(dside, x, y))  \
	  : NONTTYPE))

#define cell_style(map, x, y, power)  \
  ((map->see_all || terrain_visible(dside, x, y) || unseen_image != NULL)  \
   ? (power >= 4 ? usepolygons : useblocks)  \
   : dontdraw);

static void draw_cliffs(Map *map, int x, int y);
static void draw_contours(Map *map, int x, int y);

int tmpdrawlighting;

int tmpdrawcoverage;

/* The width of the left-side control panel. */

int conwid = 32;

/* The height of the top line. */

int tophgt;

int mapnum = 1;

int nummaps = 0;

int lastmaph = -1, lastmapv = -1;

/* Handles of the pictures that display all the map control buttons. */

PicHandle tlcontrols = nil;
PicHandle blcontrols = nil;

char *mouseover = "something";

int animation_pattern_state;

Map *frontmap;

/* This tests whether the given cell might possibly be visible on the given map.
   Not as precise as a real cliprect calculation. */

int
at_all_visible(Map *map, int x, int y)
{
	return cell_is_visible(map->vp, x, y);
}

/* Decide whether given location is away from the edge of the map's window. */

int
in_middle(Map *map, int x, int y)
{
	return cell_is_in_middle(map->vp, x, y);
}

/* Draw an individual detailed cell, as a row of one, on all maps. */

void  
update_cell_display(Side *side, int x, int y, int rightnow)
{
	int xw, leftcells = 0, cells = 1, name = FALSE, sx1, sy1, i, j;		
	Rect destRect;
	GrafPtr	oldport;
	Unit *unit;
	Map *map;

	if (!active_display(side))
	  return;
	xw = wrapx(x);
	GetPort(&oldport);
	for_all_maps(map) {

		/* If update was only to report changes in undrawn info, maybe don't bother. */
		if ((rightnow == 34 && !map->vp->draw_temperature) 
		    || (rightnow == 37 && !map->vp->draw_clouds)
		    || (rightnow == 35 && !map->vp->draw_winds)
		    || (rightnow == 36 && !map->vp->draw_cover))
		  continue;

		/* Dont update cells outside the current gworld */
		if (!cell_is_in_gworld(map, x, y))
		  continue;

		/* Find the max number of cells a unit name can occupy at this mag */
		if (map->vp->draw_names) {
			switch(map->vp->power) {
				case 7: 	cells = 2; break; 	/* 128 x 128 hexes */
				case 6: 	cells = 3; break;		/*      64 x 64 hexes */	
				case 5: 	cells = 4; break;		/*      32 x 32 hexes */
				case 4: 	cells = 4; break;		/*      16 x 16 hexes */
				case 3: 	cells = 8; break;		/*         8 x  8  hexes */
				default: 	cells = 1; break;
			}
		}
		/* Compute leftcells now in case we skip the erasing row */
		leftcells = cells - 1;

		/* Skip erasing row if name erasing was not called for */
		if (!map->erase_names)
		  cells = 1;

		/* Check if erasing row really is needed, but only if we have one */
		if (cells > 1) {
			/* First check if there is a named unit at (x, y)*/
			for_all_stack(xw, y, unit) 
				name = (unit->name != NULL);
			/* Check adjacent hexes. Stop as soon as a named unit is found */
			if (!name && inside_area(x, y + 1))
				for_all_stack(xw, y + 1, unit) 
				  name = (unit->name != NULL);	/* NE hex */
			if (!name && inside_area(x - 1, y + 1))	
				for_all_stack(wrapx(x - 1), y + 1, unit) 
				  name = (unit->name != NULL);	/* NW hex */
			if (!name && inside_area(x - 1, y))	
				for_all_stack(wrapx(x - 1), y, unit) 
				  name = (unit->name != NULL);	/* W hex */
			if (!name && inside_area(x + 1, y))	
				for_all_stack(wrapx(x + 1), y, unit) 
				  name = (unit->name != NULL);	/* E hex */
			if (!name && inside_area(x, y - 1))	
				for_all_stack(xw, y - 1, unit) 
				  name = (unit->name != NULL);	/* SW hex */
			if (!name && inside_area(x + 1, y - 1))	
				for_all_stack(wrapx(x + 1), y - 1, unit) 
				  name = (unit->name != NULL);	/* SE hex */
			/* No named units in adjacent hexes. A named unit could not */
			/* have left (x, y) in this step, thus we skip the erasing row */
			if (!name)
			  cells = 1;
		}

		GO_OFFSCREEN(map);

		/* First draw a row of cells starting at (x, y) in order to
		   erase the name of any unit that moved out of that cell */
		draw_row(map, x, y, cells, TRUE);

		/* Re-draw gridlines & shorelines for the (cells + 1) hexes
		   below the updated row that were erased by the update */				
		if (map->vp->draw_grid &! grid_matches_unseen)
		  draw_gridlines(map, x, y - 1, cells + 1);
		if (map->shorelines)
		  draw_shorelines(map, x, y - 1, cells + 1);

		/* Re-draw any named units in the (leftcells) hexes to the */
		/* left of (x, y) whose names were erased by the update */
		/* Note: this is done even if erase_names is not set! */
		if (map->vp->draw_names) {
			for (i = x - leftcells; i <  x; i++) {
				for_all_stack(wrapx(i), y, unit) {
					if (unit->name)	{							
						draw_unit_names_only(map, i, y);
						break;
					}
				}
			}
		}					

		/* Re-draw any selected units within the updated row */
		for (i = 0; i < map->numselections; ++i) {
			unit = map->selections[i];
			for (j = 0; j < cells; j++) {
				if (unit && unit->x ==  wrapx(x + j) && unit->y == y) {
					draw_selected_unit(map, unit);
				}						
			}
		}

		RETURN_ONSCREEN(map);

		/* Find the pixel coordinates of (x, y) */
		xform(map, x, y, &sx1, &sy1);

		/* Set destRect to enclose the redrawn row of cells */
		SetRect(&destRect, 
				sx1 - 2, sy1 - 2,			/* Add 2 for W & N shorelines */ 	
				sx1 + cells * map->vp->hw, 
				sy1 + map->vp->hh + 4);		/* Add 4 for SE & SW shorelines */

		/* Now copy offscreen rect to window */
		copy_from_gworld(map, destRect);			

		/* Update unit info line */
		if (map->numselections > 0) {
			unit = map->selections[0];
			if (in_play(unit) && unit->x == xw && unit->y == y) {
				draw_unit_info(map);
			}
		}
	}
	SetPort(oldport);
}

/* Create a new map window and all its paraphernalia. */

Map *
create_map(int power)
{
	int m, t, i, x, y, mainwidth, mainheight;
	Rect vscrollrect, hscrollrect;
	WindowPtr win;
	Map *map;

	build_optional_terrain_type_menu();
	DGprintf("Creating map, mag power %d\n", power);
	map = (Map *) xmalloc(sizeof(Map));
	map->id = mapnum++;
	map->vp = new_vp();
	set_map_power(map, power);
	/* Pick an appropriate focus of the view. */
	pick_a_focus(dside, &x, &y);
	set_view_focus(map->vp, x, y);
	/* Set default values for the display controls. */
	map->conw = conwid;
	map->toplineh = (default_draw_topline ? tophgt : 0);
	map->topunith = (default_draw_topunit ? topunithgt : 0);
	map->toph = map->toplineh + map->topunith;
	/* Mac viewport needs different xform style, let the common code know about this. */
	map->vp->wide_viewport = TRUE;
	map->vp->draw_terrain = TRUE;
	/* Display all types of borders and connections normally. */
	/* (aux_terrain_types[t] == 0 implies not optional type - i can't overflow
	   drawauxterrain because at least one ttype must be cell terrain.) */
	/* (Addressing of map->vp->aux_terrain_types is 1-based instead of 0-based
	    because it corresponds to menu items.) */
	i = 1;
	for_all_terrain_types(t) {
		if (!t_is_cell(t)) {
			map->vp->aux_terrain_types[i++] = t;
			map->vp->draw_aux_terrain[t] = TRUE;
		}
	}
	map->vp->draw_cell_pats = TRUE;  /* should get from prefs */
	map->vp->draw_feature_boundaries = TRUE;
	map->vp->draw_units = TRUE;
	map->vp->draw_names = default_draw_names;
	map->vp->draw_people = FALSE;
	map->vp->draw_control = FALSE;
	map->vp->draw_elevations = FALSE;
	for_all_material_types(m) {
		map->vp->draw_materials[m] = FALSE;
	}
	map->vp->num_materials_to_draw = 0;
	map->vp->draw_lighting = TRUE;
	map->vp->draw_temperature = FALSE;
	map->vp->draw_winds = FALSE;
	map->vp->draw_clouds = FALSE;
	map->vp->draw_storms = TRUE;
	map->vp->draw_grid = default_draw_grid;
	
	/* Display AI info by default if there is an AI present. */
	map->vp->draw_ai = side_has_ai(dside);

	map->sidecolors = default_sidecolors;
	map->iconmasks = default_iconmasks;
	map->boxmasks = default_boxmasks;
	map->textmasks = default_textmasks;
	map->featureborders = default_featureborders;
	map->simple_borders = default_simple_borders;
	map->featurenames = default_featurenames;
	map->shorelines = default_shorelines;
	map->optimize_fonts = default_optimize_fonts;
	map->max_bufx = default_max_bufx;
	map->max_bufy = default_max_bufy;	
	map->vp->draw_latlong = default_draw_latlong;
	map->solid_color_terrain = pref_solid_color_terrain;
	map->draw_topline = default_draw_topline;
	map->draw_topunit = default_draw_topunit;
	map->erase_names = default_erase_names;
	map->vp->draw_other_maps = default_drawothermaps;
	map->vp->latlong_interval = default_latlong_interval;
	map->autoselect = defaultautoselect;
	map->moveonclick = defaultmoveonclick;
	map->numselections = 0;
	map->maxselections = max(100, numunits + numunits / 2);
	map->selections = (Unit **) xmalloc(map->maxselections * sizeof(Unit *));
	/* Newest map goes on the front of the list. */
	map->next = maplist;
	maplist = map;
	if (hasColorQD) {
		win = GetNewCWindow(wMap, nil, (WindowPtr) -1L);
	} else {
		win = GetNewWindow(wMap, nil, (WindowPtr) -1L);
	}
	map->window = win;
	stagger_window(win, &lastmaph, &lastmapv);
	if (first_windows) {
		get_main_screen_size(&mainwidth, &mainheight);
		/* Use the main screen size if full-sized map desired, else
		   leave room for game window at side and notices at bottom. */
		SizeWindow(win,
				   mainwidth - (fullsize_map ? 0 : gamewinw) - 10,
				   mainheight - GetMBarHeight() - 16  - (fullsize_map ? 0 : 150) - 7,
				   FALSE);
	}
	ShowWindow(win);
	SetPort(win);
	sprintf(spbuf, "Map %d", map->id);
	add_window_menu_item(spbuf, win);
	set_content_rect(map);
	m_center_on_focus(map);
	/* Make the scrollbars. */
	vscrollrect = map->window->portRect;
	vscrollrect.top -= 1;
	vscrollrect.bottom -= sbarwid - 1;
	vscrollrect.left = vscrollrect.right - sbarwid;
	vscrollrect.right += 1;
	map->vscrollbar =
		NewControl(win, &vscrollrect, "\p", 1,
			 map->vp->sy, map->vp->symin, map->vp->symax, scrollBarProc, 0L);
	hscrollrect = win->portRect;
	hscrollrect.top = hscrollrect.bottom - sbarwid;
	hscrollrect.bottom += 1;
	hscrollrect.left += map->conw;
	hscrollrect.right -= sbarwid - 1;
	map->hscrollbar =
		NewControl(win, &hscrollrect, "\p", 1,
			 map->vp->sx, map->vp->sxmin, map->vp->sxmax, scrollBarProc, 0L);
	set_map_scrollbars(map);
	++nummaps;

	/* Save the onscreen gworld */
	GetGWorld(&map->windowPortPtr,&map->deviceHdl);

	/* Create an empty gworld of minimal size to be resized and filled in below */
	map->qdErr = NewGWorld(&map->gworldPortPtr,0, &map->contentrect,0, 0, 0);

	/* Alert and close the map window if we ran out of memory */
	if (map->gworldPortPtr == NULL || map->qdErr != noErr ) {
		/* close_window assumes other-map boxes exist */
		draw_related_maps(map);	
		close_window(win);
		StopAlert(aNoMem, NULL);
		/* Return zero to indicate no map was created */
		return NULL;			
	}
	
	/* Optimize gworld size and position and also draw the map */
	update_gworld(map);

	/* Finally draw the new other-map boxes */
	draw_related_maps(map);	

	return map;
}

/* Compute the content part of the map window. */

void
set_content_rect(Map *map)
{
	map->contentrect = map->window->portRect;
	map->contentrect.left += map->conw;  map->contentrect.top += map->toph;
	map->contentrect.right -= sbarwid;  map->contentrect.bottom -= sbarwid;
	set_view_size(map->vp,
				  map->contentrect.right - map->contentrect.left,
				  map->contentrect.bottom - map->contentrect.top);
}

void
m_focus_on_center(Map *map)
{
	focus_on_center(map->vp);
}

/* Put vcx,vcy in the middle of the map. */

void
m_center_on_focus(Map *map)
{
	center_on_focus(map->vp);
	set_content_rect(map);
}

/* Adjust the appearance and thumb of the scroll bars to reflect the map.  This is
   needed whenever the map is adjusted under program control, such as when magnifying
   or scrolling to a specified location. */

void
set_map_scrollbars(Map *map)
{
	int hilite;
	
	/* Scale the scrollbars to circumvent the 32K limit on control values. */

	/* This hack solves the scrolling problem for big maps at high magnification */
	/* Note that all direct access of the scrollbar values from now on first must */
	/* simulate the scaling that is applied below in order to get correct values! */

	int	scaled_sxmin, scaled_sx, scaled_sxmax, scaled_symin, scaled_sy, scaled_symax;

	scaled_sxmin = map->vp->sxmin;
	scaled_sxmax = map->vp->sxmax;
	scaled_sx = map->vp->sx;

	while (scaled_sxmax > 32000) {	/* Keep dividing by 2 as long as necessary */
		scaled_sxmax /= 2;
		scaled_sxmin /= 2;	
		scaled_sx /= 2;
	}

	/* Apply the scaled values */
	SetCtlMin(map->hscrollbar, scaled_sxmin);
	SetCtlMax(map->hscrollbar, scaled_sxmax);
	SetCtlValue(map->hscrollbar, scaled_sx);

	hilite = (map->vp->sxmin != map->vp->sxmax);
	/* Adjust the hiliting of the scrollbar, but only if the window is in front,
	   otherwise the scrollbar should remain unhilited. */
	if (map->window == FrontWindow())
	  HiliteControl(map->hscrollbar, (hilite ? 0 : 255));
	DGprintf("Hscroll (%shilite) is %d -- %d -- %d\n",
			 (hilite ? "" : "no "), GetCtlMin(map->hscrollbar),
	         GetCtlValue(map->hscrollbar), GetCtlMax(map->hscrollbar));

	/* Also scale the vertical scrollbar even though no current games are that high. */
	scaled_symin = map->vp->symin;
	scaled_symax = map->vp->symax;
	scaled_sy = map->vp->sy;

	while (scaled_symax > 32000) {	/* Keep dividing by 2 as long as necessary */
		scaled_symax /= 2;
		scaled_symin /= 2;	
		scaled_sy /= 2;
	}

	/* Apply the scaled values */
	SetCtlMin(map->vscrollbar, scaled_symin);
	SetCtlMax(map->vscrollbar, scaled_symax);
	SetCtlValue(map->vscrollbar, scaled_sy);

	hilite = (map->vp->symin != map->vp->symax);
	if (map->window == FrontWindow())
	  HiliteControl(map->vscrollbar, (hilite ? 0 : 255));
	set_content_rect(map);
	DGprintf("Vscroll (%shilite) is %d -- %d -- %d\n",
			 (hilite ? "" : "no "), GetCtlMin(map->vscrollbar),
	         GetCtlValue(map->vscrollbar), GetCtlMax(map->vscrollbar));
}

/* Given a magnification power, look up and/or calculate the sizes of everything,
   in pixels. */

void
set_map_power(Map *map, int power)
{
	set_view_power(map->vp, power);
	if (power >= 4 && cellrgns[power] == nil)
	  make_cell_clip(power);
}

/* Given a magnification power, compute the clipping regions to be used at
   that power. */

void
make_cell_clip(int power)
{
	int hw = hws[power], hh = hhs[power], delt = (hhs[power] - hcs[power]);
	PolyHandle poly;
	RgnHandle tmprgn;

	/* Make a hexagon region by drawing a polygon and then framing it while
	   a region is open. */
	poly = OpenPoly();
	MoveTo(hw / 2, 0);
	LineTo(hw, delt);
	LineTo(hw, hh - delt);
	LineTo(hw / 2, hh);
	LineTo(0, hh - delt);
	LineTo(0, delt);
	LineTo(hw / 2, 0);
	ClosePoly();
	cellrgns[power] = NewRgn();
	OpenRgn();
	FramePoly(poly);
	CloseRgn(cellrgns[power]);
	/* Make the grid-displaying version of the hexagon. */
	gridcellrgns[power] = NewRgn();
	CopyRgn(cellrgns[power], gridcellrgns[power]);
	/* Cut off a one-pixel line on the side. */
	tmprgn = NewRgn();
	SetRectRgn(tmprgn, hw - 1, 0, hw + 1, hh); 
	DiffRgn(gridcellrgns[power], tmprgn, gridcellrgns[power]);
	/* Now intersect with a region shifted upwards by one, which makes
	   the grid line along the bottom of the hex. */
	tmprgn = NewRgn();
	CopyRgn(cellrgns[power], tmprgn);
	OffsetRgn(tmprgn, 0, -1);
	SectRgn(gridcellrgns[power], tmprgn, gridcellrgns[power]);

	/* Similarly, but for cells at an angle. */
	/* (should only calc when angle view first requested) */
	/* First make a region as viewed at a 30-degree angle. */
	hh /= 2;  delt /= 2;
	poly = OpenPoly();
	MoveTo(hw / 2, 0);
	LineTo(hw, delt);
	LineTo(hw, hh - delt);
	LineTo(hw / 2, hh);
	LineTo(0, hh - delt);
	LineTo(0, delt);
	LineTo(hw / 2, 0);
	ClosePoly();
	cellrgns30[power] = NewRgn();
	OpenRgn();
	FramePoly(poly);
	CloseRgn(cellrgns30[power]);
	gridcellrgns30[power] = NewRgn();
	CopyRgn(cellrgns30[power], gridcellrgns30[power]);
	/* Cut off a one-pixel line on the side. */
	tmprgn = NewRgn();
	SetRectRgn(tmprgn, hw - 1, 0, hw + 1, hh/2); 
	DiffRgn(gridcellrgns30[power], tmprgn, gridcellrgns30[power]);
	/* Now intersect with a region shifted upwards by one, which makes
	   the grid line along the bottom of the hex. */
	tmprgn = NewRgn();
	CopyRgn(cellrgns30[power], tmprgn);
	OffsetRgn(tmprgn, 0, -1);
	SectRgn(gridcellrgns30[power], tmprgn, gridcellrgns30[power]);

	/* Now make a region as viewed at a 15-degree angle. */
	hh = hhs[power] / 4;
	delt = (hhs[power] - hcs[power]) / 4;
	poly = OpenPoly();
	MoveTo(hw / 2, 0);
	LineTo(hw, delt);
	LineTo(hw, hh - delt);
	LineTo(hw / 2, hh);
	LineTo(0, hh - delt);
	LineTo(0, delt);
	LineTo(hw / 2, 0);
	ClosePoly();
	cellrgns15[power] = NewRgn();
	OpenRgn();
	FramePoly(poly);
	CloseRgn(cellrgns15[power]);
	gridcellrgns15[power] = NewRgn();
	CopyRgn(cellrgns15[power], gridcellrgns15[power]);
	/* Cut off a one-pixel line on the side. */
	tmprgn = NewRgn();
	SetRectRgn(tmprgn, hw - 1, 0, hw + 1, hh/2); 
	DiffRgn(gridcellrgns15[power], tmprgn, gridcellrgns15[power]);
	/* Now intersect with a region shifted upwards by one, which makes
	   the grid line along the bottom of the hex. */
	tmprgn = NewRgn();
	CopyRgn(cellrgns15[power], tmprgn);
	OffsetRgn(tmprgn, 0, -1);
	SectRgn(gridcellrgns15[power], tmprgn, gridcellrgns15[power]);
}

/* Given a window, find the map that it belongs to. */

Map *
map_from_window(WindowPtr window)
{
	Map *map;
	
	if (dside == NULL)
	  return NULL;
	for_all_maps(map) {
		if (map->window == window)
		  return map;
	}
	return NULL;
}

void
grow_map(Map *map, int w, int h)	/* Manual resizing by dragging the corner */
{
	Map *map2;

	/* Erase other-map boxes in other windows. */
	draw_related_maps(map);

	/* Set the window to the new size. */
	SizeWindow(map->window, w, h, FALSE);

	/* Resize scrollbars. */
	adjust_map_decor(map);

	/* Dont forget this little guy. */
	DrawGrowIcon(map->window);

	/* Update maps in background immediately. */
	for_all_maps(map2) {
		if (map2 != map)
		  update_window(map2->window);
	}

	/* Update map content from gworld and then redraw gworld. */
	update_resized_map(map);

	/* Redraw other-map boxes in other windows. */
	draw_related_maps(map);
}

void
zoom_map(Map *map, int part)	/* Resizing by clicking in the zoom box */
{
	Map *map2;

	if (part == inZoomOut) {
	      set_standard_state(map->window,
		      area.width  * map->vp->hw  + map->conw + sbarwid + 1,
		      area.height * map->vp->hch + map->vp->hh - map->vp->hch + sbarwid + map->toph + 1);
	}
	
	/* Erase other-map boxes in other windows. */
	draw_related_maps(map);

	/* Zoom the window. */
	ZoomWindow(map->window, part, true);

	/* Resize scrollbars. */
	adjust_map_decor(map);

	/* Dont forget this little guy. */
	DrawGrowIcon(map->window);

	/* Update maps in background immediately. */
	for_all_maps(map2) {
		if (map2 != map)
		  update_window(map2->window);
	}

	/* Update map content from gworld and then redraw gworld. */
	update_resized_map(map);

	/* Redraw other-map boxes in other windows. */
	draw_related_maps(map);
}

/* Move and size the controls to be correct for the map. */

void
adjust_map_decor(Map *map)
{				
	int w, h;
	WindowPtr mapwin = map->window;

	w = mapwin->portRect.right - mapwin->portRect.left;
	h = mapwin->portRect.bottom - mapwin->portRect.top;
	MoveControl(map->hscrollbar, map->conw - 1, h - sbarwid);
	SizeControl(map->hscrollbar, w - map->conw - sbarwid + 2, sbarwid + 1);
	MoveControl(map->vscrollbar, w - sbarwid, -1);
	SizeControl(map->vscrollbar, sbarwid + 1, h - sbarwid + 1 + 1);
}

/* Given a map and a cell, compute the pixel coords of the cell's UL corner.
   This is the core routine that relates cells and pixels. */

void
xform(map, x, y, sxp, syp)
Map *map;
int x, y, *sxp, *syp;
{
	xform_cell(map->vp, x, y, sxp, syp);
	/* Shift the basic viewport result to account for both origin-shift and map decor. */
	*sxp += map->osx + map->conw;  *syp += map->osy + map->toph;
}

void
xform_fractional(map, x, y, xf, yf, sxp, syp)
Map *map;
int x, y, xf, yf, *sxp, *syp;
{
	xform_cell_fractional(map->vp, x, y, xf, yf, sxp, syp);
	/* Shift the basic viewport result to account for both origin-shift and map decor. */
	*sxp += map->osx + map->conw;  *syp += map->osy + map->toph;
}

void
m_xform_unit(map, unit, sxp, syp, swp, shp)
Map *map;
Unit *unit;
int *sxp, *syp, *swp, *shp;
{
	xform_unit(map->vp, unit, sxp, syp, swp, shp);
	/* Shift the basic viewport result to account for both origin-shift and map decor. */
	*sxp += map->osx + map->conw;  *syp += map->osy + map->toph;
}

void
m_xform_unit_self(map, unit, sxp, syp, swp, shp)
Map *map;
Unit *unit;
int *sxp, *syp, *swp, *shp;
{
	xform_unit_self(map->vp, unit, sxp, syp, swp, shp);
	/* Shift the basic viewport result to account for both origin-shift and map decor. */
	*sxp += map->osx + map->conw;  *syp += map->osy + map->toph;
}

void
m_xform_occupant(map, transport, unit, sx, sy, sw, sh, sxp, syp, swp, shp)
Map *map;
Unit *transport, *unit;
int sx, sy, sw, sh, *sxp, *syp, *swp, *shp;
{
	/* Transform the coordinates back to relative values. */
	sx -= map->osx + map->conw;  sy -= map->osy + map->toph;
	xform_occupant(map->vp, transport, unit, sx, sy, sw, sh, sxp, syp, swp, shp);
	/* Shift the basic viewport result to account for both origin-shift and map decor. */
	*sxp += map->osx + map->conw;  *syp += map->osy + map->toph;
}

/* Un-transform screen coordinates (as supplied by mouse perhaps) into
   map coordinates.  */

/* Note that only drawing is affected by SetOrigin - mouse locations are
   always window-relative, and thus only need to be adjusted by map decor
   before going to the generic routines. */

int
m_nearest_cell(Map *map, int sx, int sy, int *xp, int *yp)
{
	return nearest_cell(map->vp, sx - map->conw, sy - map->toph, xp, yp, NULL, NULL);
}

/* Find the closest direction of the closest boundary */

int
m_nearest_boundary(Map *map, int sx, int sy, int *xp, int *yp, int *dirp)
{
	return nearest_boundary(map->vp, sx - map->conw, sy - map->toph, xp, yp, dirp);
}

int
m_nearest_unit(Map *map, int sx, int sy, Unit **unitp)
{
	return nearest_unit(map->vp, sx - map->conw, sy - map->toph, unitp);
}

/* Display a map and all of its paraphernalia. */

void
draw_map(Map *map)
{
	GrafPtr	oldport;
	
	GetPort(&oldport);

	set_map_scrollbars(map);		
	set_content_rect(map);

	/* Draw map decor and info boxes */
	if (map->conw > 0)
	  draw_control_panel(map);					
	if (map->toplineh > 0)
	  draw_top_line(map);
	if (map->topunith > 0)
	  draw_unit_info(map);

	GO_OFFSCREEN(map);

	/* Draw grey area outside the world. */
	draw_window_background(map);
	/* Draw the grid/unseen color. */
	draw_area_background(map);
	/* Draw the actual content. */
	draw_map_content(map);

	RETURN_ONSCREEN(map);

	/* Copy the whole map content to the window. */
	copy_from_gworld(map, map->contentrect);				
	
	/* Draw selected units.  This handles offscreen drawing on
	   its own. */	
	draw_selections(map);

	SetPort(oldport);
}

/*
This draws the background color under both the grid/unseen color and the map cells
It is normally visible only outside the map area, where grid/unseen was never drawn
Note: map->contenrect replaced by portRect everywhere in draw_window_background!
*/

void
draw_window_background(Map *map)
{
	/* Replaces contentrect everywhere */
	Rect portRect = map->gworldPortPtr->portRect;

	/* If part of the window is entirely outside the world, we draw its shape on
	   top of gray, otherwise window starts out all white. */
/*
	This is the normal option where a gray background is drawn. It assumes that 
	the grid/unseencolor later is drawn on top of it by draw_area_background.
*/
	if (area.width * map->vp->hw < 32000) {
		switch (bggray) {
			case blackgray:
				FillRect(&portRect, QDPat(black));  break;
			case darkgray:
				FillRect(&portRect, QDPat(dkGray));  break;
			case mediumgray:
				FillRect(&portRect, QDPat(gray));  break;
			case lightgray:
				FillRect(&portRect, QDPat(ltGray));  break;
			case whitegray:
				FillRect(&portRect, QDPat(white));  break;
		}
/*
	This option is a hack for large maps at max magnification, where draw_area_background fails 
	to draw the grid/unseen color. The latter is therefore drawn as window background instead. 
*/
	} else {
		if (hasColorQD) {
			/* Use gridcolor as background if everything is visible. */
			if (g_terrain_seen() || (dside)->see_all) {
				 RGBForeColor(&gridcolor);
				 grid_matches_unseen = TRUE;	/* Enables quick grid drawing */ 
			} else
			  RGBForeColor(&unseencolor);
			PaintRect(&portRect);
			RGBForeColor(&blackcolor);
		} else {
			switch ((grid_matches_unseen ? gridgray : unseengray)) {
				case blackgray:
					FillRect(&portRect, QDPat(black));  break;
				case darkgray:
					FillRect(&portRect, QDPat(dkGray));  break;
				case mediumgray:
					FillRect(&portRect, QDPat(gray));  break;
				case lightgray:
					FillRect(&portRect, QDPat(ltGray));  break;
				case whitegray:
					FillRect(&portRect, QDPat(white));  break;
			}
		}
	}
}

/* Draw the actual map data.  This is basically a matter of drawing n rows of terrain,
   each of an appropriate length, but it's a bit of trouble to get the number and
   lengths right.  Actually, it's easy to get approximate sizes, but it's important
   to try to draw as few cells as humanly possible. */

/* Note: In addition to the minor modification described below, draw_map_content has also been 
extensively trimmed down by deleting a lot of code for drawn-area-minimization which was 
incompatible with the offscreen graphics. */ 

void
draw_map_content(map)
Map *map;
{
	int	y1, y2, y, x1, x2, vx, vy, vw, vh;

	/* Compute the size of the region to render as the basic size of the viewport
	   plus the scrolling buffer region. */ 
	vw = (map->vp->pxw + 2 * map->bufx) / map->vp->hw + 2;
	vh = (map->vp->pxh + 2 * map->bufy) / map->vp->hch + 4;

	/* Compute the bottom row to render. */
	vy = ((map->vp->totsh - map->vp->sy + map->toph) / map->vp->hch) - vh + 2;
	/* Now adjust the bottom row so it doesn't go outside the area. */
	if (vy < 0)
	  vy = 0;
	if (vy > area.height - vh)
	  vy = area.height - vh;
	/* Compute the leftmost "column". */
	vx = (map->vp->sx - map->conw) / map->vp->hw - vy / 2 - 1;
	DGprintf("Set %dx%d viewport at %d,%d\n", vw, vh, vx, vy);

	/* Compute top and bottom rows to be displayed. */
	y1 = min(vy + vh, area.height - 1);
	y2 = vy;

	/* Draw the rows, going from top to bottom. */
	for (y = y1; y >= y2; --y) {
		/* Adjust the right and left bounds to fill the viewport as
		   much as possible, without going too far (the drawing code
		   will clip, but clipped drawing is still expensive). */
		/* could test by drawing viewport rect as lines... */
		x1 = vx - (y - vy) / 2;
		x2 = x1 + vw + 2;
		/* If the area doesn't wrap, then we might have to stop
		   drawing before we reach the edge of the viewport. */
		if (area.xwrap) {
			/* (should clip to visrgn, but tricky to avoid wrapping bugs) */
		} else {
			/* Truncate x's to stay within the area. */
			x1 = max(0, min(x1, area.width-1));
			x2 = max(0, min(x2, area.width));
			/* If this row is entirely in the NE corner, don't draw anything. */
			if (x1 + y > area.width + area.halfheight)
			  continue;
			/* If this row is entirely in the SW corner, don't draw anything. */
			if (x2 + y < area.halfheight)
			  continue;
			/* If the row ends up in the NE corner, shorten it. */
			if (x2 + y > area.width + area.halfheight)
			  x2 = area.width + area.halfheight - y;
			/* If the row starts out in the SW corner, shorten it. */
			if (x1 + y < area.halfheight)
			  x1 = area.halfheight - y;
		}
		draw_row(map, x1, y, x2 - x1, FALSE);
	}
	/* Now draw the lat-long grid if asked to do so. */
	if (map->vp->draw_latlong && map->vp->latlong_interval > 0)
	  draw_meridians(map);
}

/* This draws a hexagon or rectangle that covers the totality of the area, whether
   discovered or not. */

void 
draw_area_background(Map *map)
{
	int sx, sy, llx, lly, lrx, lry, rx, ry, urx, ury, ulx, uly, lx, ly;
	PolyHandle poly;
	Rect arearect;

	/* Don't bother if area magnified greatly. */
	/* (should fix to not try to draw giant rects, but still draw something reasonable.
	   note that otherwise grid color may be wrong) */
/*
	LineTo uses shorts and does not work over 32K, so a giant hex cannot be drawn.
	If the area.wrap option is used instead with a large map, xform will freak out!
	The grid/unseen color for large maps is therfore drawn as window background
	instead (see draw_window_background).
*/
	if (area.width * map->vp->hw > 32000)
	  return;
	if (area.xwrap) {
		/* Area is cylinder; draw a rectangle. */
		xform(map, 0, area.height-1, &sx, &sy);
		/* Make sure the whole gworld gets a background. */
	 	arearect.left = map->gworldPortPtr->portRect.left;
        arearect.top = sy;
		xform(map, 0, 0, &sx, &sy);
 		/* Make sure the whole gworld gets a background. */
		arearect.right = map->gworldPortPtr->portRect.right;
		arearect.bottom = sy;
		/* If there's scrolling, then draw background under the extra scrolling
		   area. */
		if (map->vp->sxmin != map->vp->sxmax)
		  arearect.right += map->vp->pxw;
		/* Adjust so that edges of the rect pass through the middles of top/bottom edge cells. */
		OffsetRect(&arearect, 0, map->vp->hh / 2);
		if (hasColorQD) {
			/* If everything is visible, use gridcolor for background. */
			if (g_terrain_seen() || (dside)->see_all) {
				RGBForeColor(&gridcolor);
				grid_matches_unseen = TRUE;
			} else {
				RGBForeColor(&unseencolor);
			}
			PaintRect(&arearect);
			RGBForeColor(&blackcolor);
		} else {
			/* Try to use gray patterns if b/w. */
			switch (gridgray) {
				case blackgray:
					FillRect(&arearect, QDPat(black));
					break;
				case darkgray:
					FillRect(&arearect, QDPat(dkGray));
					break;
				case mediumgray:
					FillRect(&arearect, QDPat(gray));
					break;
				case lightgray:
					FillRect(&arearect, QDPat(ltGray));
					break;
				case whitegray:
					FillRect(&arearect, QDPat(white));
					break;
			}
		}
	} else {
		/* Area is hexagon; draw a hexagon. */
		/* (should make once and save?) */
		poly = OpenPoly();		
		xform(map, 0 + area.halfheight, 0, &llx, &lly);
		MoveTo(llx, lly);
		xform(map, area.width-1, 0, &lrx, &lry);
	 	LineTo(lrx, lry);
		xform(map, area.width-1, area.halfheight, &rx, &ry);
		LineTo(rx, ry);
 		xform(map, area.width-1 - area.halfheight, area.height-1, &urx, &ury);
		LineTo(urx, ury);
 		xform(map, 0, area.height-1, &ulx, &uly);
		LineTo(ulx, uly);
 		xform(map, 0, area.halfheight, &lx, &ly);
		LineTo(lx, ly);
		LineTo(llx, lly);
		ClosePoly();
		/* Adjust so that edges of the polygon pass through the middles of edge cells. */
		OffsetPoly(poly, map->vp->hw/2, map->vp->hh/2);
		if (hasColorQD) {
			if (g_terrain_seen() || (dside)->see_all) {
				 RGBForeColor(&gridcolor);
				 grid_matches_unseen = TRUE; 
			} else {
				RGBForeColor(&unseencolor);
			}
			PaintPoly(poly);
			RGBForeColor(&blackcolor);
		} else {
			switch (gridgray) {
				case blackgray:
					FillPoly(poly, QDPat(black));
					break;
				case darkgray:
					FillPoly(poly, QDPat(dkGray));
					break;
				case mediumgray:
					FillPoly(poly, QDPat(gray));
					break;
				case lightgray:
					FillPoly(poly, QDPat(ltGray));
					break;
				case whitegray:
					FillPoly(poly, QDPat(white));
					break;
			}
		}
		/* Free up the space for this hexagon. */
		KillPoly(poly);
	}
#if 0  /* The idea of a shaded border seems nice, but it doesn't look very good in practice. */
	for (x = 0; x < area.width; ++x) {
		xform(map, x, 0, &sx, &sy);
		draw_border_line(sx, sy, SW, map->vp->power, -1);
		draw_border_line(sx, sy, SE, map->vp->power, -2);
	}
	PenNormal();
#endif
}

/* Draw the map control panel as a pair of PICTs. */

void
draw_control_panel(map)
Map *map;
{
	int winhgt, basev;
	Rect tmprect;
	GrafPtr oldport;

	GetPort(&oldport);
	SetPort(map->window);
	winhgt = (map->window->portRect).bottom - (map->window->portRect).top;
	SetRect(&tmprect, 0, 0, map->conw, winhgt);
	FillRect(&tmprect, QDPat(white));
	MoveTo(map->conw - 1, 0);  Line(0, winhgt);
	if (tlcontrols == nil) {
		tlcontrols = (PicHandle) GetResource('PICT', pMapControlsTL);
	}
	if (tlcontrols != nil) {
		SetRect(&tmprect, 0, 0,
				picture_width(tlcontrols), picture_height(tlcontrols));
		DrawPicture(tlcontrols, &tmprect);
	}
	if (blcontrols == nil) {
		blcontrols = (PicHandle) GetResource('PICT', pMapControlsBL);
	}
	if (blcontrols != nil) {
		SetRect(&tmprect, 0, winhgt - picture_height(blcontrols) + 1,
				picture_width(blcontrols), winhgt);
		DrawPicture(blcontrols, &tmprect);
	}
	if (map->moveonclick && map->autoselect) {
		SetRect(&tmprect, 4, 5, 26, 26);
		InvertRect(&tmprect);
	}
	/* (should modify appearance of top left arrow buttons to reflect abilities) */
	basev = 32 + 5*15 + 2 + 5/*why?*/ + 1;
	SetRect(&tmprect, 0, basev, 30, basev + 10);
	if (map->vp->draw_grid) {
		InvertRect(&tmprect);
	}
	OffsetRect(&tmprect, 0, 11);
	if (map->vp->draw_names) {
		InvertRect(&tmprect);
	}
	OffsetRect(&tmprect, 0, 11);
	if (map->vp->draw_people) {
		InvertRect(&tmprect);
	} else if (!people_sides_defined()) {
		gray_out_rect(&tmprect);
	}
	OffsetRect(&tmprect, 0, 11);
	if (map->vp->draw_control) {
		InvertRect(&tmprect);
	} else if (!control_sides_defined()) {
		gray_out_rect(&tmprect);
	}
	OffsetRect(&tmprect, 0, 11);
	if (map->vp->draw_plans) {
		InvertRect(&tmprect);
	}
	OffsetRect(&tmprect, 0, 11);
	if (map->vp->draw_ai) {
		InvertRect(&tmprect);
	} else if (!side_has_ai(dside)) {
		/* (should ensure that this is updated when side gets an AI) */
		gray_out_rect(&tmprect);
	}
	OffsetRect(&tmprect, 0, 11);
	if (map->see_all) {
		InvertRect(&tmprect);
	} else if (!dside->may_set_see_all) {
		gray_out_rect(&tmprect);
	}
	OffsetRect(&tmprect, 0, 11);
	if (map->sidecolors) {
		InvertRect(&tmprect);
	}
	OffsetRect(&tmprect, 0, 11);
	if (map->iconmasks) {
		InvertRect(&tmprect);
	}
	/* Draw state of bottom left control buttons. */
	if (map->vp->power <= 0) {
		SetRect(&tmprect, 0, winhgt - 15, 15, winhgt);
		gray_out_rect(&tmprect);
	}
	if (map->vp->power >= NUMPOWERS - 1) {
		SetRect(&tmprect, 16, winhgt - 15, 30, winhgt);
		gray_out_rect(&tmprect);
	}
	SetPort(oldport);
}

/* Draw a single line of the most important info - the current date or turn,
   and a textual description of what the mouse is over. */

void
draw_top_line(Map *map)
{
	int numchars, strwid;
	Rect tmprect;

	/* It would be safest to set the grafport explicitly here, but in
	   the interests of efficiency, don't bother until somebody can
	   prove it's necessary. */
	/* Set up to draw the current date. */
	TextSize(small_font_size);
	TextFont(small_font_id);
	numchars = strlen(curdatestr);
	strwid = TextWidth(curdatestr, 0, numchars);
	/* Clear the whole topline area. */
	SetRect(&tmprect, map->conw, 0, map->conw + map->vp->pxw, map->toplineh);
	FillRect(&tmprect, QDPat(white));
	/* Now draw the date. */
	MoveTo(tmprect.right - strwid - 3, tmprect.top + 11);
	DrawText(curdatestr, 0, numchars);
	if (mouseover != NULL) {
		/* Draw description of what the mouse is over. */
		/* (should clip to avail space) */
		numchars = strlen(mouseover);
		MoveTo(map->conw + 3, tmprect.top + 11);
		DrawText(mouseover, 0, numchars);
	}
	/* Draw a line dividing this from other map content. */
	MoveTo(tmprect.left, tmprect.bottom - 1);
	Line(tmprect.right - tmprect.left, 0);
}

void
draw_unit_info(Map *map)
{
	int mrow, u, m;
	Rect tmprect;
	Unit *unit;
	char infobuf[100];
	GrafPtr oldport;
	RgnHandle tmprgn;

	if (map->topunith <= 0)
	  return;
 	GetPort(&oldport);
	SetPort(map->window);
	/* Save the current clipping region. */
	tmprgn = NewRgn();
	GetClip(tmprgn);
	SetRect(&tmprect, map->conw, map->toplineh, map->conw + map->vp->pxw, map->toph);
	ClipRect(&tmprect);
	/* Clear the whole area. */
	FillRect(&tmprect, QDPat(white));
	if (map->numselections == 1) {
		unit = map->selections[0];
		if (in_play(unit)) {
			u = unit->type;
			sprintf(infobuf, "%s", unit_handle(dside, unit));
			draw_info_text(map, 0, 0, 0, infobuf);
			/* Describe its current location. */
			location_desc(infobuf, dside, unit, unit->type, unit->x, unit->y);
			draw_info_text(map, 0, 1, 0, infobuf);
			/* Very briefly list the numbers and types of the occupants. */
			occupants_desc(infobuf, unit);
			draw_info_text(map, 0, 2, 0, infobuf);
			/* Describe the "important" parameters. */
    		hp_desc(infobuf, unit, TRUE);
    		strcat(infobuf, "   ");
    		acp_desc(infobuf + strlen(infobuf), unit, TRUE);
			draw_info_text(map, 50, 0, 0, infobuf);
			/* List other stack members that are here. */
			others_here_desc(infobuf, unit);
			draw_info_text(map, 50, 1, 0, infobuf);
			mrow = 0;
			while (supply_desc(infobuf, unit, mrow)) {
				draw_info_text(map, 50, mrow + 2, 0, infobuf);
				++mrow;
			}
		    /* Describe the current plan and task agenda. */
		    if (unit->plan) {
				int row = 3, n;
				Task *task;

				plan_desc(infobuf, unit);
				draw_info_text(map, 0, row++, 0, infobuf);
				n = 0;
				infobuf[0] = '\0';
				for_all_tasks(unit->plan, task) {
					if (n > 0 && n % 2 == 0) {
						draw_info_text(map, 0, row, 0, infobuf);
						infobuf[0] = '\0';
						row++;
					}
					task_desc(infobuf+strlen(infobuf), unit->side, task);
					if (task->next != NULL)
					  strcat(infobuf, "; ");
					++n;
				}
				if (strlen(infobuf) > 0)
				  draw_info_text(map, 0, row, 0, infobuf);
		    }
		}
	} else if (map->numselections > 1) {
		draw_info_text(map, 0, 0, 0, "(multiple units)");
	}
	/* Draw a line dividing this from the map content. */
	MoveTo(tmprect.left, tmprect.bottom - 1);
	Line(tmprect.right - tmprect.left, 0);
	/* Restore clipping region. */
	SetClip(tmprgn);
	DisposeRgn(tmprgn);
	/* Restore grafport. */
	SetPort(oldport);
}

static void
draw_info_text(map, x, y, len, buf)
Map *map;
int x, y, len;
char *buf;
{
	int sx, sy;
	Rect tmprect;

    if (empty_string(buf))
      return;
	SetRect(&tmprect, map->conw, map->toplineh, map->conw + map->vp->pxw, map->toph);
    /* Translate a 0-100 value for x to pixels. */
    if (x == 50)
      sx = tmprect.left + (tmprect.right - tmprect.left) / 2;
    else
      sx = tmprect.left + 3;
    sy = tmprect.top + small_line_spacing + y * (small_line_spacing + 1);
    if (len > 0 && strlen(buf) > len)
      buf[len-1] = '\0';
	MoveTo(sx, sy);
	TextFont(small_font_id);
	TextSize(small_font_size);
	DrawText(buf, 0, strlen(buf));
}

/* This function is now only called when the current port is an offscreen gworld. It it called by  GO_OFFSCREEN, to 
   erase the other-map boxes, and then again by RETURN_ONSCREEN, to redraw the other-map boxes. This makes 
   sure that the other-map boxes always are preserved during offscreen drawing. */

void
draw_other_maps(map)
Map *map;
{
	Map *map2;

	for_all_maps(map2) {
		if (map != map2 /* && reasonable to show? */) {
			draw_other_map(map, map2);
		}
	}
}

/* Draws both offscreen and on screen! */

/* This and the selection animation are the only two cases where things are drawn directly
   to the screen. draw_related_maps now also clips drawing correctly to the map content. */

void
draw_related_maps(Map *map)
{
	Map *map2;
	RgnHandle tmprgn;
	Rect tmpRect;
	GrafPtr oldport;

	/* Very important here (unlike most other drawing functions) */
	/* because of the SetPort(map2->window) call below! */
	GetPort(&oldport);

	for_all_maps(map2) {
		if (map != map2 && map2->vp->draw_other_maps) {

			SetPort(map2->window);

			/* Clip related map box to content of other window */
			tmprgn = NewRgn();
			GetClip(tmprgn);
			ClipRect(&map2->contentrect);

			/* Draw related map box onscreen */
			draw_other_map(map2, map);

			GO_OFFSCREEN(map2);

			/* Always draw it offscreen as well! */
			draw_other_map(map2, map);

			RETURN_ONSCREEN(map2);

			SetClip(tmprgn);
			DisposeRgn(tmprgn);
		}
	}
	SetPort(oldport);
}

void
draw_other_map(Map *map, Map *map2)
{
	int sx, sy, sw, sh, wsx;
	Rect tmpRect;

	/* Scale the viewports */
	scale_vp(map->vp, map2->vp, &sx, &sy, &sw, &sh);
	/* Handle non-xwrapped world */
	wsx = sx;
	/* Now draw the other map's box */
	SetRect(&tmpRect, wsx, sy, wsx + sw, sy + sh);
	OffsetRect(&tmpRect, map->conw, map->toph);
	if (map->vp->hw < 8)
	  PenSize(2, 2);
	PenMode(patXor);
	FrameRect(&tmpRect);
	if (area.xwrap) {
		/* Draw one extra copy to the left if necessary. */
		if (tmpRect.right > map->vp->sxmax) {
			OffsetRect(&tmpRect, -map->vp->sxmax, 0);
			FrameRect(&tmpRect);
			OffsetRect(&tmpRect, map->vp->sxmax, 0);
		}		
		/* Draw one extra copy to the right if necessary. */
		if (tmpRect.left < 0) {
			OffsetRect(&tmpRect, map->vp->sxmax, 0);
			FrameRect(&tmpRect);
		}		
		/* Draw more copies to the right ON SCREEN if necessary. */
		if (QD(thePort) == map->window) {
			int rest = map->vp->pxw - tmpRect.left;
			while (rest > 0) {
				OffsetRect(&tmpRect, map->vp->sxmax, 0);
				FrameRect(&tmpRect);
				rest -= map->vp->sxmax;
			} 
		}
	}
	PenNormal();
}

static void meridian_line_callback(int x1, int y1, int x1f, int y1f, int x2, int y2, int x2f, int y2f);
static void meridian_text_callback(int x1, int y1, int x1f, int y1f, char *str);

static Map *tmpmap;

static void
draw_meridians(Map *map)
{
	Rect portrect = map->gworldPortPtr->portRect;
	int real_pxw = map->vp->pxw, real_pxh = map->vp->pxh;
	
	tmpmap = map;

	/* Necessary to fool the kernel into believing that the vp is bigger than it is. */
	map->vp->pxw = portrect.right - portrect.left;		
	map->vp->pxh = portrect.bottom - portrect.top;		

	RGBForeColor(&meridiancolor);

	plot_meridians(map->vp, meridian_line_callback, meridian_text_callback);

	ForeColor(blackColor);

	/* Restore real vp size. */
	map->vp->pxw = real_pxw;  map->vp->pxh = real_pxh;
}

static void
meridian_line_callback(int x1, int y1, int x1f, int y1f, int x2, int y2, int x2f, int y2f)
{
	int sx1, sy1, sx2, sy2;

	xform_fractional(tmpmap, x1, y1, x1f, y1f, &sx1, &sy1);
	xform_fractional(tmpmap, x2, y2, x2f, y2f, &sx2, &sy2);
	MoveTo(sx1, sy1);
	LineTo(sx2, sy2);
}

static void
meridian_text_callback(int x1, int y1, int x1f, int y1f, char *str)
{
	int sx1, sy1;

	xform_fractional(tmpmap, x1, y1, x1f, y1f, &sx1, &sy1);
	draw_legend_text(sx1, sy1 - 12, 12, tmpbuf, 0, 0, 0);
}

/* x0 may be negative here. */

/* (should add a "draw interior only" version of this routine?) */

void
draw_row(map, x0, y0, len, clearit)
Map *map;
int x0, y0, len, clearit;
{
	int empty = FALSE;
	int i = 0, x, xw, sx, sy, t;

	if (!between(0, y0, area.height - 1))
	  return;
	/* (should make generic) */
	if (!map->see_all && !map->vp->draw_ai && unseen_image == NULL) {
		empty = TRUE;
		/* Examine row to see if we can skip it entirely. */
		for (x = x0; x < x0 + len; ++x) {
			xw = wrapx(x);
			if (map->see_all
				|| terrain_visible(dside, xw, y0)
				|| (numbordtypes > 0 && terrain_visible(dside, xw, y0 + 1))
				) {
				empty = FALSE;
				break;
			}
		}
	}
	if (empty && !clearit)
	  return;
	/* The terrain always comes first. */
	if (map->vp->draw_terrain) {
		draw_terrain_row(map, x0, y0, len);
		  draw_shorelines(map, x0, y0, len + 1);
		/* Maybe draw cliffs.  Note that we always want to draw cliffs if the view is
		   at an angle, since otherwise there's no point in having the view be at an
		   angle. */
		if ((elevations_defined() || any_thickness) && map->vp->angle != 90) {
			for (x = x0; x < x0 + len; ++x) {
				if (map->see_all || terrain_visible(dside, x, y0)) {
					draw_cliffs(map, x, y0);
				}
			}
		}
		if (elevations_defined() && map->vp->draw_elevations && map->vp->angle == 90) {
			for (x = x0; x < x0 + len; ++x) {
				if (map->see_all || terrain_visible(dside, x, y0)) {
					draw_contours(map, x, y0);
				}
			}
		}
		/* Note: the check for erase_names is because names will leak into the grid
		   if the quick method is used. It is therefore always necessary to use explicit
		   drawing of the grid also in this case. */
		if (map->vp->draw_grid
			&& (map->erase_names
				|| !grid_matches_unseen 
				|| map->vp->draw_plans
				|| map->vp->draw_ai
				|| (elevations_defined() 
					&& map->vp->draw_elevations && map->vp->angle == 90)))
			draw_gridlines(map, x0, y0, len + 1);
		/* Maybe draw shorelines. */
		if (map->shorelines)
		  draw_shorelines(map, x0, y0, len + 1);
		if (any_aux_terrain_defined()) {
			/* The relative ordering of these is quite important - connections
			   should always be drawn on top of borders. */
			if (bords_to_draw(map)) {
				for_all_terrain_types(t) {
					if (map->vp->draw_aux_terrain[t] && t_is_border(t) && aux_terrain_defined(t)) {
						for (x = x0; x < x0 + len; ++x) {
							draw_borders(map, x, y0, t);
						}
					}
				}
			}
			if (conns_to_draw(map)) {
				for_all_terrain_types(t) {
					if (map->vp->draw_aux_terrain[t] && t_is_connection(t) && aux_terrain_defined(t)) {
						for (x = x0; x < x0 + len; ++x) {
							draw_connections(map, x, y0, t);
						}
					}
				}
			}
		}
	}
	if (clouds_defined() && map->vp->draw_clouds) {
		for (x = x0; x < x0 + len; ++x) {
			if (draw_clouds_here(x, y0)) {
				xform(map, x, y0, &sx, &sy);
				if (map->vp->hh <= 16) {

				  WRAP_SX(map, sx, 
					draw_cloud_block(sx, sy, 1, map->vp->power, cloud_view(dside, x, y0), 
															map->vp->angle));

				} else { 

				  WRAP_SX(map, sx, 
					draw_clouds(sx, sy, map->vp->power, cloud_view(dside, x, y0)));
				}
			}
		}
	}
	if (winds_defined() && map->vp->draw_winds && map->vp->hh > 10) {
		for (x = x0; x < x0 + len; ++x) {
			if (draw_winds_here(dside, x, y0)) {
				xform(map, x, y0, &sx, &sy);
				WRAP_SX(map, sx,
					draw_winds(sx, sy, map->vp->power, wind_view(dside, x, y0)));
			}
		}
	}
	if (temperatures_defined() && map->vp->draw_temperature && map->vp->hh > 10) {
		for (x = x0; x < x0 + len; ++x) {
			if (draw_temperature_here(dside, x, y0)) {
				xform(map, x, y0, &sx, &sy);
				WRAP_SX(map, sx,
					draw_temperature(sx, sy, map->vp->power, temperature_view(dside, x, y0)));
			}
		}
	}
	/* Although we had to draw the terrain on the edge, we can skip everything else,
	   since edge cells have no units, features, etc. */
	/* Skip the top and bottom edge rows. */
	if (!between(1, y0, area.height - 2)) 
	  return;
	/* Shorten the row by one cell on each side, if those are edge cells. */
	if (!inside_area(x0 + len - 1, y0))
	  --len;
	if (!inside_area(x0, y0)) {
		++x0;
		--len;
	}
	if (len <= 0)
	  return;
	if (features_defined() && map->featureborders && map->vp->hw > 4) {
		if (map->simple_borders)
		  draw_simple_feature_borders(map, x0, y0, len + 1);
		else for (x = x0; x < x0 + len; ++x)
		  draw_feature_boundary(map, x, y0);
	}
	if (any_cell_materials_defined() && map->vp->num_materials_to_draw > 0 && map->vp->hh > 20) {
		for (x = x0; x < x0 + len; ++x) {
			draw_materials(map, x, y0);
		}
	}
	/* (should be global to entire map drawing somehow?) */
	/* Draw feature names even at low power! */
	if (map->featurenames && map->vp->hh > 3) {
		for (x = x0; x < x0 + len; ++x) {
			draw_legend(map, x, y0);
		}
	}
	if ((people_sides_defined() || control_sides_defined()) && bwid2[map->vp->power] > 0) {
		draw_people_row(map, x0, y0, len);
	}
	if (map->vp->draw_units && map->vp->hw > 2) {
		for (x = x0; x < x0 + len; ++x) {
			draw_units(map, x, y0);
		}
	}
	if (map->vp->draw_ai) {
		for (x = x0; x < x0 + len; ++x) {
			draw_ai_region(map, x, y0);
		}
	}
	/* If debugging, draw coverage on top of everything else. */
	if (DebugG && !all_see_all && map->vp->hw >= 16) {
		for (x = x0; x < x0 + len; ++x) {
			xform(map, x, y0, &sx, &sy);
			WRAP_SX(map, sx, 
				draw_coverage(sx, sy, map->vp->power, cover(dside, x, y0), alt_cover(dside, x, y0)));
		}
	}
}

/* Is this just to see if a cell is visible? */

int
cell_update(map, x, y)
Map *map;
int x, y;
{
#if 0
	int sx, sy;
	Rect tmprect;

	xform(map, x, y, &sx, &sy);
	SetRect(&tmprect, sx, sy, sx + map->vp->hw, sy + map->vp->hh);
	return (RectInRgn(&tmprect, map->window->visRgn));
#else
	return TRUE;
#endif
}

/* Draw an entire row of terrain, possibly with a single rectangle fill. */
 
/* x0 may be negative. */

void
draw_terrain_row(map, x0, y0, len)
Map *map;
int x0, y0, len;
{
	int x0w, x, xw, x1, x1w, sx, sy, i;
	int pwr = map->vp->power, ang = map->vp->angle;
	int dogrid = map->vp->draw_grid & grid_matches_unseen; 
	int dofill = map->vp->draw_cell_pats;
	int dosolid = map->solid_color_terrain;
	int inarea, seginarea, style, segstyle, terr, segterr, over, segover, update, segupdate;

	tmpdrawlighting = map->vp->draw_lighting;
	tmpdrawcoverage = (!dside->see_all && map->vp->draw_cover);
	x0w = wrapx(x0);
	i = 0;
	x1 = x0;
	x1w = wrapx(x1);
	seginarea = in_area(x0w, y0);
	segstyle = cell_style(map, x0w, y0, pwr);
	segterr = cell_terrain(map, x0w, y0);
	segover = cell_overlay(x0w, y0);
	segupdate = (seginarea ? cell_update(map, x0w, y0) : FALSE);
	/* Note that in this loop we do NOT wrap x for drawing, just for
	   looking at layer data. */
	for (x = x0; x < x0 + len + 1; ++x) {
		xw = wrapx(x);
		inarea = in_area(xw, y0);
		style = cell_style(map, xw, y0, pwr);
		terr = cell_terrain(map, xw, y0);
		over = cell_overlay(xw, y0);
		update = (inarea ? cell_update(map, xw, y0) : FALSE);
		/* Decide if the run is over and we need to dump some output. */
		if (x == x0 + len
			|| x == area.width
			|| inarea != seginarea
			|| style != segstyle
			|| terr != segterr
			|| over != segover
			|| update != segupdate
/*			|| segstyle == useblocks  fixes, but poor performance */
			|| segstyle == usepictures
			|| segstyle == usepolygons
			|| ang != 90) {
			/* don't draw anything that would match the window's bg */
#if 0 /* use this for heavy-duty debugging only */
			DGprintf("Seg is %d,%d len=%d inarea=%d style=%d terr=%d over=%d update=%d\n",
					 x1, y0, i, seginarea, segstyle, segterr, segover, segupdate);
#endif
			if (seginarea && segupdate && segstyle != dontdraw) {
				xform(map, x1, y0, &sx, &sy);
				switch (segstyle) {
					case useblocks:
						WRAP_SX(map, sx, 
							draw_cell_block(sx, sy, i, pwr, segterr, segover, ang, dosolid));													
						break;
					case usepictures:
						break;
					case usepolygons:
						WRAP_SX(map, sx, 
							draw_hex_region(sx, sy, pwr, segterr, segover, ang, dogrid, dosolid));
				}
			}
			/* Set up for the next segment. */
			i = 0;
			x1 = x;
			x1w = wrapx(x1);
			seginarea = inarea;
			segstyle = style;
			segterr = terr;
			segover = over;
			segupdate = update;
		}
		++i;
	}
}

static void
draw_cliffs(Map *map, int x, int y)
{
	int xw, elev, x1, y1, drop, t, t1, drawit, elev1, offset;
	int sx, sy, sx1, sy1;
	PolyHandle poly;
	RGBColor hexcolor, oldcolor;

	xw = wrapx(x);
	t = cell_terrain(map, xw, y);
	elev = (elevations_defined() ? elev_at(xw, y) : 0) + t_thickness(t);
	drawit = FALSE;
	if (point_in_dir(xw, y, SOUTHWEST, &x1, &y1)) {
		t1 = cell_terrain(map, x1, y1);
		if (t1 != NONTTYPE) {
			drop = elev - ((elevations_defined() ? elev_at(x1, y1) : 0) + t_thickness(t1));
			if (drop > 0) {
				drawit = TRUE;
				xform(map, x, y, &sx, &sy);
				xform(map, x1, y1, &sx1, &sy1);
			}
		}
	} else {
		drawit = TRUE;
		xform(map, x, y, &sx, &sy);
	    elev1 = elev - (elevations_defined() ? area.minelev : 0);
	    elev1 *= map->vp->vertscale;
	    offset = (elev1 * map->vp->hh) / map->vp->cellwidth;
	    sy1 = sy + map->vp->hch + offset; 
	}
	if (drawit) {
		if (sy1 > sy + map->vp->hch + (map->vp->draw_grid ? 1 : 0)) {
			poly = OpenPoly();
			MoveTo(sx, sy + map->vp->hch);
			LineTo(sx + map->vp->hw / 2, sy + map->vp->hh);
			LineTo(sx + map->vp->hw / 2, sy1 + (map->vp->hh - map->vp->hch));
			LineTo(sx, sy1);
			LineTo(sx, sy + map->vp->hch);
			ClosePoly();
			/* if (timg->colrpat != nil
			    && (minscreendepth > 1 || !timg->patdefined)) {
				FillCPoly(poly, timg->colrpat);
			} else */ if (tcolors[t] != NULL && maxscreendepth > 1) {
				hexcolor.red   = tcolors[t]->color.red / 2;
				hexcolor.green = tcolors[t]->color.green / 2;
				hexcolor.blue  = tcolors[t]->color.blue / 2;
				RGBForeColor(&hexcolor);
				WRAP_POLY(map, poly, 
					PaintPoly(poly));
				/* Restore the previous color. */
				RGBForeColor(&blackcolor);
			} else {
				hexcolor.red   = (128 << 8) / 2;
				hexcolor.green = (128 << 8) / 2;
				hexcolor.blue  = (128 << 8) / 2;
				RGBForeColor(&hexcolor);
				WRAP_POLY(map, poly, 
					PaintPoly(poly));
				/* Restore the previous color. */
				RGBForeColor(&blackcolor);
			}
			KillPoly(poly);
		}
	}
	drawit = FALSE;
	if (point_in_dir(xw, y, SOUTHEAST, &x1, &y1)) {
		t1 = cell_terrain(map, x1, y1);
		if (t1 != NONTTYPE) {
			drop = elev - ((elevations_defined() ? elev_at(x1, y1) : 0) + t_thickness(t1));
			if (drop > 0) {
				drawit = TRUE;
				xform(map, x, y, &sx, &sy);
				xform(map, x1, y1, &sx1, &sy1);
			}
		}
	} else {
		drawit = TRUE;
		xform(map, x, y, &sx, &sy);
	    elev1 = elev - (elevations_defined() ? area.minelev : 0);
	    elev1 *= map->vp->vertscale;
	    offset = (elev1 * map->vp->hh) / map->vp->cellwidth;
	    sy1 = sy + map->vp->hch + offset; 
	}
	if (drawit) {
		if (sy1 > sy + map->vp->hch + (map->vp->draw_grid ? 1 : 0)) {
			poly = OpenPoly();
			MoveTo(sx + map->vp->hw - (map->vp->draw_grid ? 1 : 0),     sy + map->vp->hch);
			LineTo(sx + map->vp->hw - (map->vp->draw_grid ? 1 : 0),     sy1);
			LineTo(sx + map->vp->hw / 2, sy1 + (map->vp->hh - map->vp->hch));
			LineTo(sx + map->vp->hw / 2, sy + map->vp->hh);
			LineTo(sx + map->vp->hw - (map->vp->draw_grid ? 1 : 0),     sy + map->vp->hch);
			ClosePoly();
			/* if (timg->colrpat != nil
			    && (minscreendepth > 1 || !timg->patdefined)) {
				FillCPoly(poly, timg->colrpat);
			} else */ if (tcolors[t] != NULL && maxscreendepth > 1) {
				hexcolor.red   = (3 * tcolors[t]->color.red) / 4;
				hexcolor.green = (3 * tcolors[t]->color.green) / 4;
				hexcolor.blue  = (3 * tcolors[t]->color.blue) / 4;
				RGBForeColor(&hexcolor);
				WRAP_POLY(map, poly, 
					PaintPoly(poly));
				/* Restore the previous color. */
				RGBForeColor(&blackcolor);
			} else {
				hexcolor.red   = (3 * 128 << 8) / 4;
				hexcolor.green = (3 * 128 << 8) / 4;
				hexcolor.blue  = (3 * 128 << 8) / 4;
				RGBForeColor(&hexcolor);
				WRAP_POLY(map, poly, 
					PaintPoly(poly));
				/* Restore the previous color. */
				RGBForeColor(&blackcolor);
			}
			KillPoly(poly);
		}
	}
}

static void
draw_contours(Map *map, int x, int y)
{
	int i, sx, sy, numlines;
	LineSegment *lines;

	init_contour_lines();
	if (contour_interval < 1)
	  return;
	xform(map, x, y, &sx, &sy);
	contour_lines_at(map->vp, wrapx(x), y, sx, sy, &lines, &numlines);
	RGBForeColor(&contourcolor);
	for (i = 0; i < numlines; ++i) {
		MoveTo(lines[i].sx1, lines[i].sy1);
		LineTo(lines[i].sx2, lines[i].sy2);
	}
	ForeColor(blackColor);
}

static void
draw_borders(map, x, y, b)
Map *map;
int x, y, b;
{
	int xw = wrapx(x), dir, sx, sy, bitmask = 0, over;
	
 	if (!(map->see_all || terrain_visible(dside, xw, y)) || !any_borders_at(xw, y, b))
 	  return;
	for_all_directions(dir) {
		if (border_at(xw, y, dir, b) && borders_visible(dside, xw, y, dir)) {
			bitmask |= 1 << dir;
		}
	}
	if (bitmask != 0) {
		tmpdrawlighting = map->vp->draw_lighting;
		tmpdrawcoverage = (!all_see_all && map->vp->draw_cover);
		over = cell_overlay(xw, y);
		xform(map, x, y, &sx, &sy);
		/* (should compute and pass in overlay) */
		WRAP_SX(map, sx, 
			draw_border_line_multiple(map->window, sx, sy, bitmask, map->vp->power, b, 
								map->vp->angle, over, map->solid_color_terrain));
		
	}
}

/* Draw all the connections of the given cell. */

static void
draw_connections(map, x, y, c)
Map *map;
int x, y, c;
{
	int xw = wrapx(x), dir, sx, sy, bitmask = 0, over;
	
	if (!(map->see_all || terrain_visible(dside, xw, y)) || !any_connections_at(xw, y, c))
	  return;
	for_all_directions(dir) {
		if (connection_at(xw, y, dir, c)) {
			bitmask |= 1 << dir;
		}
	}
	if (bitmask != 0) {
		tmpdrawlighting = map->vp->draw_lighting;
		tmpdrawcoverage = (!all_see_all && map->vp->draw_cover);
		over = cell_overlay(xw, y);
		xform(map, x, y, &sx, &sy);
		/* (should compute and pass in overlay) */
		WRAP_SX(map, sx, 
			draw_connection_line_multiple(map->window, sx, sy, bitmask, map->vp->power, c, 
									map->vp->angle, over, map->solid_color_terrain));
	}
}

/* This function draws units at a small magnification */
/* Draw all the units visible in the given cell. */
/* (x is not wrapped) */

static void
draw_units(map, x, y)
Map *map;
int x, y;
{
	int xw = wrapx(x), sx, sy, sw, sh, uview, u, s;
	Unit *unit;
	Rect tmprect;
	extern PicHandle dotdotdotpicture;

	if (map->see_all || units_visible(dside, xw, y)) {
		unit = unit_at(xw, y);
		if (unit != NULL) {
			xform(map, x, y, &sx, &sy);
			if (map->vp->uw <= 16) {
				/* Adjust to unit part of cell. */
				sw = map->vp->uw;  sh = map->vp->uh;
				sx += (map->vp->hw - sw) / 2;  sy += (map->vp->hh - sw) / 2;
				if (unit->occupant != NULL
					&& sw >= 8
	    			&& (unit->side == dside || all_see_all || u_see_occupants(unit->type))) {
					/* Draw a "grouping box", in white, but with no occs drawn. */
					SetRect(&tmprect, sx, sy, sx + sw, sy + sh);
					s = side_number(unit->side);
					if (map->boxmasks) {
						if (map->sidecolors && icon_mask_color[s]) {
							RGBBackColor(&(get_sideColor(s, icon_mask_color[s])));
						} else {
							RGBBackColor(&maskcolor);
						}
						WRAP_RECT(map, tmprect, 
							FillRect(&tmprect, QDPat(white)));
						BackColor(whiteColor);
					}
					WRAP_RECT(map, tmprect, 
						FrameRect(&tmprect));
	    		}
				WRAP_SX(map, sx, 
					draw_unit_image(map->window, sx, sy, sw, sh, unit->type, side_number(unit->side), 
																!completed(unit), TRUE));
				/* Indicate if more than one stacked here.  Since this may cover up the unit's
				   picture, adjust it down as far as possible while remaining inside the
				   "cell minus grid" area. */
				if (unit->nexthere != NULL && sw > 8) {
					SetRect(&tmprect, sx + sw/2 - 6, sy + sh - 3,
									  sx + sw/2 + 6, sy + sh + 1);
					/* (should clip to fit in cell) */
					/* (should do a copybits) */
					WRAP_RECT(map, tmprect, 
						DrawPicture(dotdotdotpicture, &tmprect));
				}
				if (map->vp->draw_names) {
					/* Draw city names in uniform size and position */
					if (!mobile(unit->type)) {
					WRAP_SX(map, sx, 
						draw_unit_name(unit, sx, sy, map->vp->uw, map->vp->uh, map->textmasks, 
																map->optimize_fonts)); 
					} else
					WRAP_SX(map, sx, 
						draw_unit_name(unit, sx, sy, sw, sh, map->textmasks, map->optimize_fonts));
				}
			} else {
				for_all_stack(xw, y, unit) {
					if (1 /* doesnt work right? - side_sees_unit(dside, unit) */) {
						m_xform_unit(map, unit, &sx, &sy, &sw, &sh);
						draw_unit_and_occs(map, unit, sx, sy, sw, sh);
					}
				}
			}
		}
	} else {
		uview = unit_view(dside, xw, y);
		if (uview != EMPTY) {
			u = vtype(uview);  s = vside(uview);
			xform(map, x, y, &sx, &sy);
			sw = map->vp->uw;  sh = map->vp->uh;
			/* Adjust to unit part of cell. */
			sx += (map->vp->hw - sw) / 2;  sy += (map->vp->hh - sh) / 2;
			WRAP_SX(map, sx, 
				draw_unit_image(map->window, sx, sy, sw, sh, u, s, 0, TRUE));
		}
	}
}

/* This function draws units at a normal or high magnification */

static void
draw_unit_and_occs(map, unit, sx, sy, sw, sh)
Map *map;
Unit *unit;
int sx, sy, sw, sh;
{
	int u = unit->type, s = side_number(unit->side), sx2, sy2, sw2, sh2;
	int emblem = TRUE;
	Unit *occ;
	Rect tmprect;
	
	/* If an occupant's side is the same as its transport's, then there's
	   really no need to draw its side emblem, since the transport's emblem
	   will also be visible. */
	if (unit->transport && unit->side == unit->transport->side)
	  emblem = FALSE;
	if (unit->occupant != NULL
		&& sw > 8
	    && (dside->see_all
		    || unit->side == dside
		    || u_see_occupants(unit->type)
		    || side_owns_occupant(dside, unit))) {

		/* Draw a sort of "grouping box", in white. */
		/* Make box 1 pixel smaller to avoid leakage of its lower left corner into the grid. */
		SetRect(&tmprect, sx, sy, sx + sw - 1, sy + sh - 1);
		if (map->boxmasks) {
			if (map->sidecolors && icon_mask_color[s]) {
				RGBBackColor(&(get_sideColor(s, icon_mask_color[s])));
			} else {
				RGBBackColor(&maskcolor);
			}
			WRAP_RECT(map, tmprect, 
				FillRect(&tmprect, QDPat(white)));
			BackColor(whiteColor);
		}
			WRAP_RECT(map, tmprect, 
				FrameRect(&tmprect));
		/* Draw the transport in the UL quarter of the box. */
		m_xform_occupant(map, unit, unit, sx, sy, sw, sh, &sx2, &sy2, &sw2, &sh2);
		WRAP_SX(map, sx2, 
			draw_unit_image(map->window, sx2, sy2, sw2, sh2, u, s, !completed(unit), TRUE));
		if (map->vp->draw_names) {
			/* Draw city names in uniform size and position */
			if (!mobile(unit->type)) {
			  WRAP_SX(map, sx, 
				draw_unit_name(unit, sx, sy, map->vp->uw, map->vp->uh, map->textmasks, 
															map->optimize_fonts));  
			/* sw2 doubled moves the name out of the box so that it can be read more easily */
			} else {

			  WRAP_SX(map, sx2, 
				draw_unit_name(unit, sx2, sy2, 2 * sw2, sh2, map->textmasks, map->optimize_fonts)); 

			}
		}				
		/* Draw all the occupants, in the bottom half of the box. */
		for_all_occupants(unit, occ) {
			m_xform_occupant(map, unit, occ, sx, sy, sw, sh, &sx2, &sy2, &sw2, &sh2);
			/* Accomodate the smaller box by moving occs 2 pixels up. */
			draw_unit_and_occs(map, occ, sx2, sy2 - 2, sw2, sh2);	/* RECURSION! */
		}
	} else {
		/* Just draw the one unit. */
	  WRAP_SX(map, sx, 
		draw_unit_image(map->window, sx, sy, sw, sh, u, s, !completed(unit), emblem));
		if (map->vp->draw_names) {
			/* Draw city names in uniform size and position */
			if (!mobile(unit->type)) {
			  WRAP_SX(map, sx, 
				draw_unit_name(unit, sx, sy, map->vp->uw, map->vp->uh, map->textmasks, 
															map->optimize_fonts));  
			} else {
		
			  WRAP_SX(map, sx, 
				draw_unit_name(unit, sx, sy, sw, sh, map->textmasks, map->optimize_fonts));
			}
		}
	}
}

/* Indicate what kind of people are living in the given row. */

/* (should optimize by sharing border drawing for max efficiency) */
/* (bulk of compute should go to ui.c) */
static void
draw_people_row(map, x0, y, len)
Map *map;
int x0, y, len;
{
	int pop, xx, x, sx, sy, sw, sh, ex, ey, ew, eh, dir, x1, y1, pop1;
	int con, con1, bitmask1, bitmask2, bitmask3, drawemblemhere;

	for (xx = x0; xx < x0 + len; ++xx) {
		x = wrapx(xx);
		if (!terrain_visible(dside, x, y))
		  continue;
		pop = (people_sides_defined() ? people_side_at(x, y) : NOBODY);
		con = (control_sides_defined() ? control_side_at(x, y) : NOCONTROL);
		bitmask1 = bitmask2 = bitmask3 = 0;
		drawemblemhere = FALSE;
		/* Decide which edges are borders of the country. */
		for_all_directions(dir) {
			/* Don't do anything about edge cells. */
			if (interior_point_in_dir(x, y, dir, &x1, &y1)) {
				if (terrain_visible(dside, x1, y1)) {
					pop1 = (people_sides_defined() ? people_side_at(x1, y1) : NOBODY);
					con1 = (control_sides_defined() ? control_side_at(x1, y1) : NOCONTROL);
					if (con != con1) {
						if (con == NOCONTROL || con1 == NOCONTROL) {
							bitmask2 |= 1 << dir;
						} else {
							bitmask1 |= 1 << dir;
						}
					} else if (pop != pop1) {
						/* Borders with uninhabitated regions are drawn differently. */
						if (pop == NOBODY || pop1 == NOBODY) {
							bitmask2 |= 1 << dir;
						} else if (control_sides_defined()) {
							bitmask3 |= 1 << dir;
						} else {
							bitmask1 |= 1 << dir;
						}
					}
				} else {
					/* Draw just people in the cells right at the edge of the known world. */
					drawemblemhere = TRUE;
				}
			}
		}
		/* Now draw both the edges and an emblem for the cell. */
		if ((bitmask1 | bitmask2 | bitmask3) != 0
			|| ((map->vp->draw_people || map->vp->draw_control) && drawemblemhere)) {
			/* Important to use unwrapped xx! */
			xform(map, xx, y, &sx, &sy);
			RGBForeColor(&frontcolor);		/* Use customized frontline color */
			if (bitmask1 != 0) {
			  WRAP_SX(map, sx, 
				draw_country_borders(map->window, sx, sy, bitmask1, map->vp->power, 0, 
														map->vp->angle));
			}
			/* Do not use shaded border to sea and empty land if simple_borders is on. */
			if (bitmask2 != 0 && !map->simple_borders) {
			  WRAP_SX(map, sx, 
				draw_country_borders(map->window, sx, sy, bitmask2, map->vp->power, 2, 
														map->vp->angle));
			}
			if (bitmask3 != 0 && !map->simple_borders) {
			  WRAP_SX(map, sx, 
				draw_country_borders(map->window, sx, sy, bitmask3, map->vp->power, 1, 
														map->vp->angle));
			}
			ForeColor(blackColor);					/* Restore Fore Color */

			/* Draw an emblem for the people in the cell. */
			if (map->vp->draw_people && pop != NOBODY) {
				sw = map->vp->uw;  sh = map->vp->uh;
				ew = min(sw, max(8, sw / 2));  eh = min(sh, max(8, sh / 2));
				ex = sx + (map->vp->hw - map->vp->uw) / 2 + sw / 2 - ew / 2;  
				ey = sy + (map->vp->hh - map->vp->uh) / 2 + sh / 2 - eh / 2;
				/* Maybe draw emblem so it will be visible underneath control emblem. */
				if (map->vp->draw_control && con != pop) {
					ex += 2;  ey += 2;
				}
				WRAP_SX(map, ex, 
					draw_side_emblem(map->window, ex, ey, ew, eh, pop, plain_emblem));

			}
			if (map->vp->draw_control && con != NOCONTROL && (bitmask1 | bitmask2) != 0) {
				sw = map->vp->uw;  sh = map->vp->uh;
				ew = min(sw, max(8, sw / 2));  eh = min(sh, max(8, sh / 2));
				ex = sx + (map->vp->hw - map->vp->uw) / 2 + sw / 2 - ew / 2;  
				ey = sy + (map->vp->hh - map->vp->uh) / 2 + sh / 2 - eh / 2;
				WRAP_SX(map, ex, 
					draw_side_emblem(map->window, ex, ey, ew, eh, con, plain_emblem));
			}
		}
	}
}

static void
draw_feature_boundary(map, x, y)
Map *map;
int x, y;
{
	int xw, fid, fid1, dir, x1, y1, bitmask, sx, sy;
	Feature *feature;

    xw = wrapx(x);
	if (terrain_visible(dside, xw, y)) {
		fid = raw_feature_at(xw, y);
		if (fid == 0)
		  return;
		bitmask = 0;
		/* Decide which edges are borders of the feature. */
		for_all_directions(dir) {
			/* Don't do anything about edge cells. */
			if (interior_point_in_dir(xw, y, dir, &x1, &y1)) {
				fid1 = raw_feature_at(x1, y1);
				if (fid != fid1) {
					bitmask |= 1 << dir;
				}
			}
		}
		if (bitmask != 0) {
			xform(map, x, y, &sx, &sy);
			if (bitmask != 0) {
			  WRAP_SX(map, sx, 
				draw_feature_borders(map->window, sx, sy, bitmask, map->vp->power));
			}
		}
	}
}

/* This draws a small set of bar charts, one for each material type. */

static void
draw_materials(map, x, y)
Map *map;
int x, y;
{
	int m, t, sx, sy, mx, my, mw, mh, amt, maxamt, h;
	Rect graphrect;
	
	if (nummtypes == 0)
	  return;
	mw = map->vp->uw / nummtypes /* should be count of displayable materials... */;  mh = map->vp->uh;
	if (mw <= 2 || mh <= 2)
	  return;
	/* (should this be apparent instead of real terrain?) */
	t = cell_terrain(map, wrapx(x), y);
	xform(map, x, y, &sx, &sy);
	mx = sx + (map->vp->hw - map->vp->uw) / 2;  my = sy + (map->vp->hh - map->vp->uh) / 2;
	for_all_material_types(m) {
		if (map->vp->draw_materials[m] && (maxamt = tm_storage_x(t, m)) > 0) {
			SetRect(&graphrect, mx + m * mw, my, mx + (m + 1) * mw, my + map->vp->uh);
			WRAP_RECT(map, graphrect, 
				FrameRect(&graphrect));
			amt = material_view(dside, x, y, m);
			h = (amt * mh) / maxamt;
			graphrect.top -= (mh - h);
			/* (should use a per-material color/pattern if available) */
			WRAP_RECT(map, graphrect, 
				FillRect(&graphrect, QDPat(black)));
		}
	}
}

static void
draw_ai_region(map, x, y)
Map *map;
int x, y;
{
	int xw, thid, sx, sy, dir, x1, y1, thid1, bitmask = 0;

	xw = wrapx(x);
	thid = ai_region_at(dside, xw, y);
	/* Decide which edges are borders of the theater. */
	for_all_directions(dir) {
		/* Don't do anything about edge cells. */
		if (interior_point_in_dir(xw, y, dir, &x1, &y1)) {
			thid1 = ai_region_at(dside, x1, y1);
			if (thid != thid1) {
				bitmask |= 1 << dir;
			}
		}
	}
	if (bitmask != 0) {
		xform(map, x, y, &sx, &sy);
		if (bitmask != 0) {
		  WRAP_SX(map, sx, 
			draw_ai_region_borders(map->window, sx, sy, bitmask, map->vp->power));
		}
	}
}

/* Draw any text that should be associated with this cell. */

/* (could precompute what the string will lap over and move or truncate str),
   should be deterministic for each mag, so redraw doesn't scramble */

void
draw_legend(map, x, y)
Map *map;
int x, y;
{
	int xw, sx, sy;
	char *str, buf[BUFSIZE];
	Feature *feature;

    xw = wrapx(x);
	/* Draw the name of a terrain feature. */
	/* (should limit to one cell of feature, preferably centering on quasi-centroid) */	
	if (terrain_visible(dside, x, y)) {
		feature = feature_at(x, y);
		if (feature != NULL) {
			if (feature->size > 0) {
				if ((feature->x == x && feature->y == y)
					|| (feature->x == 0 && feature->y == 0)
					|| 0 /* center far away */) {
					str = feature_desc(feature, buf);
					if (str != NULL) {
						xform(map, x, y, &sx, &sy);
						if (map->featurenames) {
							RGBForeColor(&featurecolor);
							/* use sy instead of sy + map->vp->hh/2 to prevent drawing on top of unit names */
							WRAP_SX(map, sx, 
								draw_legend_text(sx + map->vp->hw/2, sy, 2 * map->vp->uh, str, 0, 
													map->textmasks, map->optimize_fonts));
							ForeColor(blackColor);
						}
					}
				}
			}
		}
	}
}

void
draw_unit_blast(Map *map, Unit *unit, int blast)
{
	int sx, sy, sw, sh;
	Rect destRect;
	GrafPtr	oldport;
	
	GetPort(&oldport);

	GO_OFFSCREEN(map);

	/* Find offscreen coordinates of the unit rect */
	m_xform_unit(map, unit, &sx, &sy, &sw, &sh);

	draw_blast_image(map->window, sx, sy, sw, sh, blast);

	RETURN_ONSCREEN(map);
	
	/* Dont try to copy a blasted unit outside the map! */
	if (!in_area(unit->x, unit->y))
	  return;
	/* Find onscreen coordinates of the unit rect */
	m_xform_unit(map, unit, &sx, &sy, &sw, &sh);
	/* Set copy destination to the unit rect */
	SetRect(&destRect, sx, sy, sx + sw, sy + sh);
	/* Now copy offscreen map to window */
	copy_from_gworld(map, destRect);			

	SetPort(oldport);
}

void
clear_unit_blast(Map *map, Unit *unit, int blast)
{
	int x = unit->x, y = unit->y, sx, sy, i;
	Unit *unit2;
	Rect destRect;
	GrafPtr	oldport;

	GetPort(&oldport);

	GO_OFFSCREEN(map);

	/* Re-draw the hex containing the blast */
	draw_row(map, x, y, 1, TRUE);

	/* Draw any selections that are here. */
	for (i = 0; i < map->numselections; ++i) {
		unit2 = map->selections[i];
		if (unit2 && unit2->x == x && unit2->y == y) {
			draw_selected_unit(map, unit2);
		}
	}

	/* Re-draw gridlines & shorelines for the two hexes below */
	if (map->vp->draw_grid & !grid_matches_unseen)
	  draw_gridlines(map, x, y - 1, 2);
	if (map->shorelines)
	  draw_shorelines(map, x, y - 1, 2);
	
	RETURN_ONSCREEN(map);

	/* Find the pixel coordinates of (x, y) */
	xform(map, x, y, &sx, &sy);
	/* Set destRect to enclose the redrawn hex */
	SetRect(&destRect, 
			sx - 2, sy - 2,			/* Add 2 for redrawn W & N shorelines */ 	
			sx + map->vp->hw, 
			sy + map->vp->hh + 4);	/* Add 4 for redrawn SE & SW shorelines */

	/* Now copy offscreen map to window */
	copy_from_gworld(map, destRect);			

	SetPort(oldport);
}

void
invert_map(Map *map, int x, int y, int r)
{
	int sx, sy, sw, sh;
	Rect tmprect;
	GrafPtr	oldport;
	
	GetPort(&oldport);

	GO_OFFSCREEN(map);

	xform(map, x, y, &sx, &sy);
	SetRect(&tmprect, sx - (r * map->vp->hw) / 2, 
					  sy - (r * map->vp->hch) / 2,
					  sx + (r * map->vp->hw) / 2, 
					  sy + (r * map->vp->hch) / 2);

	InvertRect(&tmprect);

	RETURN_ONSCREEN(map);

	/* Copy tmprect to the screen */
	copy_from_gworld(map, tmprect);

	SetPort(oldport);
}
