/* Copyright (c) 1987, 1988  Stanley T. Shebs. */
/* This program may be used, copied, modified, and redistributed freely */
/* for noncommercial purposes, so long as this notice remains intact. */

/* Handling of unit orders. */

#include "config.h"
#include "misc.h"
#include "dir.h"
#include "period.h"
#include "side.h"
#include "unit.h"
#include "mplay.h"
#include "global.h"

char *ordernames[] = ORDERNAMES;   /* full names of orders */
char *dirnames[] = DIRNAMES;		/* short names of directions */
char orderbuf[BUFSIZE];            /* buffer for printed form of an order */
char oargbuf[BUFSIZE];             /* buffer for order's arguments */

int orderargs[] = ORDERARGS;       /* types of parameters for each order */

/* Display orders in some coherent fashion.  Use the information about the */
/* types of order parameters to decide how to display them. */

char *order_desig(Order *orders) {

    switch (orderargs[orders->type]) {
    case NOARG:
        sprintf(oargbuf, "");
        break;
    case DIR:
        sprintf(oargbuf, "%s ", dirnames[orders->p.dir]);
        break;
    case POS:
        sprintf(oargbuf, "%d,%d ", orders->p.pt[0].x, orders->p.pt[0].y);
        break;
    case EDGE_A:
        sprintf(oargbuf, "%d/%d ", orders->p.edge.forward, orders->p.edge.ccw);
        break;
    case LEADER:
        sprintf(oargbuf, "%s ", unit_handle((Side *) NULL,
                                            find_unit(orders->p.leader_id)));
        break;
    case WAYPOINTS:
        sprintf(oargbuf, "%d,%d %d,%d ",
                orders->p.pt[0].x, orders->p.pt[0].y,
                orders->p.pt[1].x, orders->p.pt[1].y);
        break;
    default:
        case_panic("order arg type", orderargs[orders->type]);
    }
    if (orders->rept > 1)
        sprintf(orderbuf, "%s %s(%d)",
                ordernames[orders->type], oargbuf, orders->rept);
    else
        sprintf(orderbuf, "%s %s",
                ordernames[orders->type], oargbuf);
    return orderbuf;
}

/* Yeah yeah, assignment statements supposedly copy structures. */

void copy_orders(Order *dst, Order *src) {

  /* the first two options should work fine since the Order
     is (theoretically) contiguous with no pointers. */

  memcpy(dst, src, sizeof(*dst));
}

/* Switch from standing order teaching mode back to normal, and confirm */
/* that the standing order has been set. */

void finish_teach(Side *side) {

    Unit *occ;

    side->teach = FALSE;
    if (side->sounit->standing->orders[side->soutype] != NULL)
      free(side->sounit->standing->orders[side->soutype]);
    side->sounit->standing->orders[side->soutype] = side->tmporder;
    if (side->tmporder==NULL) {
      notify(side, "standing orders for %s deleted",
             utypes[side->soutype].name);
    } else {
      notify(side, "%s has orders for %s to %s%0s.",
             unit_handle(side, side->sounit), utypes[side->soutype].name,
             order_desig(side->tmporder),
             ((side->tmporder->morder) ? " and auto" : ""));
      for_all_occupants(side->sounit, occ) {
        if (occ->type == side->soutype) {
          get_standing_orders(occ, side->sounit);
        }
      }
    }
    show_mode(side);
}

/* General routine to wake a unit up (and maybe all its cargo). */

wake_unit(Unit *unit, bool wakeocc, int reason, Unit *other) {

    Unit *occ;

/* make sure that a side always gets to move a unit that is woken up
late in the turn */

    if (unit->side != NULL && unit->movesleft > 0 &&
        alive(unit) && !unit->side->more_units) {
      unit->side->more_units = TRUE;
      show_all_sides(unit->side);
      move_mode(unit->side);
      unit->side->movunit = unit;
/*      cancel_request(unit->side); */
    }

    unit->orders.type = AWAKE;
    unit->orders.rept = 0;
    unit->orders.flags = NORMAL;
    unit->wakeup_reason = reason;
    if (reason == WAKEFULL) {
      unit->orders.morder = FALSE;
    }
    if (reason == WAKEOWNER)
      unit->area = area_index(unit->x, unit->y);
    if (reason == WAKEENEMY && other!=NULL) {
      unit->waking_side = other->side;
      unit->waking_type = other-> type;
      unit->area = area_index(unit->x, unit->y);
    }
    if (wakeocc) {
        for_all_occupants(unit, occ) wake_unit(occ, wakeocc, reason, other);
    }
}

/* Stash a "wakeup call" - will only be for main unit, not occupants. */

void cache_awake(Side *side) {

    side->tmporder->type = AWAKE;
    side->tmporder->rept = 0;
    side->tmporder->morder = FALSE;
    finish_teach(side);
}

/* Give a unit sentry orders. */

void order_sentry(Unit *unit, int n) {

    unit->orders.type = SENTRY;
    unit->orders.rept = n;
}

/* Stash sentry orders. */

void cache_sentry(Side *side, int n) {

    side->tmporder->type = SENTRY;
    side->tmporder->rept = n;
    finish_teach(side);
}

/* Give the unit orders to move to a given place - it only needs to do this */
/* once, repetition is nonsensical. */

void order_moveto(Unit *unit, int x, int y) {

    unit->orders.type = MOVETO;
    unit->orders.rept = 1;
    unit->orders.p.pt[0].x = x;
    unit->orders.p.pt[0].y = y;
}

void cache_moveto(Side *side, int x, int y) {

    side->tmporder->type = MOVETO;
    side->tmporder->rept = 1;
    side->tmporder->p.pt[0].x = x;
    side->tmporder->p.pt[0].y = y;
    finish_teach(side);
}

/* order a unit to return to base */

void order_return(Unit *unit, int n) {

  unit->orders.type = RETURN;
  unit->orders.rept = n;
}

void cache_return(Side *side, int n) {

  side->tmporder->type = RETURN;
  side->tmporder->rept = n;
  finish_teach(side);
}

/* Order to follow a unit just needs the unit to follow. */

void order_follow(Unit *unit, Unit *leader, int n) {

    unit->orders.type = FOLLOW;
    unit->orders.rept = n;
    unit->orders.p.leader_id = leader->id;;
}

void cache_follow(Side *side, Unit *leader, int n) {

    side->tmporder->type = FOLLOW;
    side->tmporder->rept = n;
    side->tmporder->p.leader_id = leader->id;
    finish_teach(side);
}

/* Order to follow an edge needs to know the direction and which side the
   obstacle is on. */

void order_edge(Unit *unit, int d, int ccw, int n) {

    unit->orders.type = EDGE;
    unit->orders.rept = n;
    unit->orders.p.edge.forward = d;
    unit->orders.p.edge.ccw = ccw;
}

void cache_edge(Side *side, int d, int ccw, int n) {

    side->tmporder->type = EDGE;
    side->tmporder->rept = n;
    side->tmporder->p.edge.forward = d;
    side->tmporder->p.edge.ccw = ccw;
    finish_teach(side);
}

/* A two-waypoint patrol suffices for many purposes. */
/* Should have a more general patrol routine eventually (> 2 waypoints). */

void order_patrol(Unit *unit, int x0, int y0, int x1, int y1, int n) {

    unit->orders.type = PATROL;
    unit->orders.rept = n;
    unit->orders.p.pt[0].x = x0;
    unit->orders.p.pt[0].y = y0;
    unit->orders.p.pt[1].x = x1;
    unit->orders.p.pt[1].y = y1;
}

void cache_patrol(Side *side, int x0, int y0, int x1, int y1, int n) {

    side->tmporder->type = PATROL;
    side->tmporder->rept = n;
    side->tmporder->p.pt[0].x = x0;
    side->tmporder->p.pt[0].y = y0;
    side->tmporder->p.pt[1].x = x1;
    side->tmporder->p.pt[1].y = y1;
    finish_teach(side);
}

/* Wait around to embark on some unit when it comes to our location. */

void order_embark(Unit *unit, int n) {

    unit->orders.type = EMBARK;
    unit->orders.rept = n;
}

void cache_embark(Side *side, int n) {

    side->tmporder->type = EMBARK;
    side->tmporder->rept = n;
    finish_teach(side);
}

/* Fill in the given unit with direction-moving orders. */

void order_movedir(Unit *unit, int dir, int n) {

    unit->orders.type = MOVEDIR;
    unit->orders.p.dir = dir;
    unit->orders.rept = n;
}

/* order a unit to move toward another unit */

void order_movetounit(Unit *unit, Unit *dest, int n) {

  unit->orders.type = MOVETOUNIT;
  unit->orders.rept = n;
  unit->orders.p.leader_id = dest->id;
}

void cache_movetounit(Side *side, Unit *dest, int n) {

  side->tmporder->type = MOVETOUNIT;
  side->tmporder->rept = n;
  side->tmporder->p.leader_id = dest->id;
  finish_teach(side);
}

void order_fill(Unit *unit, int n) {

    unit->orders.type = FILL;
    unit->orders.rept = n;
}

void cache_fill(Side *side, int n) {

    side->tmporder->type = FILL;
    side->tmporder->rept = n;
    finish_teach(side);
}

/* Order to move to the nearest filling transport */

void order_movetotransport(Unit *unit, int n) {

  unit->orders.type = MOVETOTRANSPORT;
  unit->orders.rept = n;
}

void cache_movetotransport(Side *side, int n) {

  side->tmporder->type = MOVETOTRANSPORT;
  side->tmporder->rept = n;
  finish_teach(side);
}

#ifndef NO_DISEMBARK
void cache_disembark(Side *side, int n) {

    side->tmporder->type = DISEMBARK;
    side->tmporder->rept = n;
    finish_teach(side);
}
#endif

/* Note that this leaves all other orders the same. */

void cache_auto(Side *side) {

  Order *oldorder;

  if ((oldorder = side->sounit->standing->orders[side->soutype]) != NULL)
    copy_orders(side->tmporder, oldorder);
  else {
    side->tmporder->type = AWAKE;
    side->tmporder->rept = 0;
  }
  side->tmporder->morder = TRUE;
  finish_teach(side);
}
