/* Copyright (c) 1987, 1988, 1991  Stanley T. Shebs. */
/* This program may be used, copied, modified, and redistributed freely */
/* for noncommercial purposes, so long as this notice remains intact. */

/* Geometry is somewhat tricky because our viewports are supposed */
/* to wrap around a cylinder transparently.  The general idea is that */
/* if modulo opns map x coordinates onto a cylinder, adding and subtracting */
/* the diameter of the cylinder "un-modulos" things.  If this doesn't make */
/* any sense to you, then be careful about fiddling with the code! */

/* If that wasn't bad enough, the hexes constitute an oblique coordinate */
/* where the axes form a 60 degree angle to each other.  Fortunately, no */
/* trig is necessary - to convert to/from rectangular, add/subtract 1/2 of */
/* the y coordinate to x, and leave the y coordinate alone. */

/* The graphical code uses mostly text drawing functions, which are more */
/* likely to be efficient than is random bitblting.  (The interface may */
/* implement the operations as random blitting, but that's OK.) */

#include "config.h"
#include "misc.h"
#include "period.h"
#include "side.h"
#include "unit.h"
#include "map.h"
#include "X11.h"
#include "Motif.h"
#include "global.h"

/* Show a list of all sides in action, shading any that have lost out. */
/* This has to be called for each side if everybody's list is */
/* to be updated. */

/* Highlight the side whose turn it is, using white for current */
/* player's own display (to wake she/he/it up) background colore */
/* for everybody else.  Also add a star for the benefit of monochrome. */

void show_all_sides(Side *side) {

    update_sides(side);
}

void show_turn_clock(Side *side, char *clockbuf) {

    draw_turn_clock(side, clockbuf);
}

void show_side_clock(Side *side, char *clockbuf) {

    draw_side_clock(side, clockbuf);
}

/* Update the turn number and game mode display.  The mode is inverted */
/* so will stand out (it governs what player input will be accepted, so */
/* quite important). */

void show_time(Side *side) {

    if (active_display(side)) {
        draw_time(side);
    }
}

/* Update the turn number and game mode display.  The mode is inverted */
/* so will stand out (it governs what player input will be accepted, so */
/* quite important). */

void show_mode(Side *side) {

    if (active_display(side)) {
        draw_mode(side);
    }
}

/* Would be faster to stash these, but enough difference to care? */

int total_gain(Side *side, int u) {

    int i, total = 0;

    for (i = 0; i < DUMMYREAS; ++i) total += side->balance[u][i];
    return total;
}

int total_loss(Side *side, int u) {

    int i, total = 0;

    for (i = DUMMYREAS+1; i < NUMREASONS; ++i) total += side->balance[u][i];
    return total;
}

/* Alter the numbers for a single type of unit.  Should be called right */
/* after any changes.  Formatted to look nice, but kind of messy to set */
/* up correctly - Cobol isn't all bad! */

void update_state(Side *side, int u) {

    static char _15space[] = "               ";

    if (active_display(side)) {
        sprintf(spbuf, "");
        if (side->unitslist[u].units > 0)
            sprintf(tmpbuf, "%3d", side->unitslist[u].units);
        else
            sprintf(tmpbuf, "   ");
        strcat(spbuf, tmpbuf);
        if (side->unitslist[u].building > 0) {
            sprintf(tmpbuf, "(%d)", side->unitslist[u].building);
            strcat(spbuf, tmpbuf);
        }
        strcat(spbuf, _15space);
        if (total_gain(side, u) > 0) {
            sprintf(spbuf+8, "%4d", total_gain(side, u));
        }
        strcat(spbuf, _15space);
        if (total_loss(side, u) > 0) {
            sprintf(tmpbuf, "- %d", total_loss(side, u));
            sprintf(spbuf+13, "%s", tmpbuf);
        }
        draw_state(side, u, spbuf);
    }
}

/* The state display summarizes all the units and any other global info. */

void show_state(Side *side) {

    int u;

    if (active_display(side)) {
        for_all_unit_types(u) update_state(side, u);
    }
}

/* Draw an outline box on the world map.  Since we adopt the dubious trick */
/* of inverting through all planes, must be careful to undo before moving; */
/* also, draw/undraw shiftedly so both boxes appear on both sides of world. */

void draw_box(Side *side) {

    invert_box(side, side->vcx, side->vcy);
    invert_box(side, side->vcx - world.width, side->vcy);
    side->lastvcx = side->vcx;  side->lastvcy = side->vcy;
}

void undraw_box(Side *side) {

    invert_box(side, side->lastvcx, side->lastvcy);
    invert_box(side, side->lastvcx - world.width, side->lastvcy);
}

/* Compute the color representing a hex from the given side's point of view. */

world_color(Side *side, int x, int y) {

    viewdata view = side_view(side, x, y);
    Side *side2;

    if (side->monochrome) {
        return ((view == EMPTY) ? side->appdata.bgcolor : side->appdata.fgcolor);
    } else {
        if (view == UNSEEN) {
            return (side->appdata.bgcolor);
        } else if (view == EMPTY) {
            return (side->hexcolor[terrain_at(x, y)]);
        } else {
            side2 = side_n(vside(view));
            return ((side2 == NULL) ? side->appdata.neutcolor :
                    (allied_side(side2, side) ? side->appdata.altcolor :
                     side->appdata.enemycolor));
        }
    }
}

/* Redraw the map of the whole world.  We use square blobs instead of icons, */
/* since individual "hexes" may be as little as 1x1 pixels in size, and */
/* there are lots of them.  Algorithm uses run-length encoding to find and */
/* draw bars of constant color.  Monochrome just draws units - no good */
/* choices for terrain display. */

void show_world(Side *side) {

    int x, y, x1;
    Pixel color, barcolor;

    if (active_display(side)) {
        clear_window(side, XtWindow(side->world));
        for (y = world.height-1; y >= 0; --y) {
            x1 = 0;
            barcolor = world_color(side, x1, y);
            for (x = 0; x < world.width; ++x) {
                color = world_color(side, x, y);
                if (color != barcolor) {
                    draw_bar(side, x1, y, x - x1, barcolor);
                    x1 = x;
                    barcolor = color;
                }
            }
            draw_bar(side, x1, y, world.width - x1, barcolor);
        }
        if (side->vcy >= 0) draw_box(side);
    }
}

/* Undo the wrapping effect, relative to viewport location. */
/* Note that both conditions cannot both be true at the same time, */
/* since viewport is smaller than map. */

/* This function has been updated to calculate the exact locations of */
/* the hex in the viewport, even if the viewport is small. */

int unwrap(Side *side, int x, int y) {

    int vcx = side->vcx;
    int vcy = side->vcy;
    int left_hex, right_hex;

    left_hex = vcx - world.w2 - ((y - vcy) / 2) - (y > vcy);
    right_hex = vcx + world.w2 + world.w_odd - 
                ((y - vcy) / 2) + ((y < vcy) && world.w_odd)
                - ((y > vcy) && !world.w_odd);

    if ((left_hex < 0) && (x > right_hex)) x -= world.width;
    else if ((right_hex >= world.width) && (x < left_hex)) x += world.width;
    return x;
}

/* Transform coordinates in the world display.  Simpler, since no moving */
/* viewport nonsense. */

void w_xform(Side *side, int x, int y, int *sxp, int *syp) {

    *sxp = side->mm * x + (side->mm * y) / 2;
    *syp = side->mm * (world.height - 1 - y);
}

/* Transform map coordinates into screen coordinates, relative to the given */
/* side.  Allow for cylindricalness and number of pixels in a hex. */

void xform(Side *side, int x, int y, int *sxp, int *syp) {
  
    *sxp = ((side->hw * (x - (side->vcx - world.w2))) +
	    (side->hw * (y - side->vcy)) / 2);
    *syp = side->hch * ((side->vcy + world.h2) - y);
}

/* Un-transform screen coordinates (as supplied by mouse perhaps) into */
/* map coordinates.  This doesn't actually account for the details of */
/* hexagonal boundaries, and actually discriminates box-shaped areas. */

void deform(Side *side, int sx, int sy, int *xp, int *yp) {

    int vcx = side->vcx, vcy = side->vcy, adjust;

    *yp = (vcy + world.h2) - (sy / side->hch);
    adjust = (((*yp - vcy) & 1) ? ((side->hw/2) * (*yp >= vcy ? 1 : -1)) : 0);
    *xp = wrap(((sx - adjust) / side->hw) - (*yp - vcy) / 2 +
               (vcx - world.w2));
}

/* Return the color of a terrain icon overlaying a colored hex. */

Pixel terricon_color(Side *side, int x, int y) {

    return ((side_view(side, wrap(x), y) == UNSEEN) ? side->appdata.bgcolor :
	    (ttypes[terrain_at(wrap(x), y)].dark ? side->appdata.fgcolor :
	     side->appdata.bgcolor));
}

/* Return the color of the hex. (for color displays only) */

Pixel hex_color(Side *side, int x, int y) {

  Pixel color;

  color = ((side_view(side, wrap(x), y) == UNSEEN) ? side->appdata.bgcolor :
            side->hexcolor[terrain_at(wrap(x), y)]);
  return (color);
}

/* Draw a single unit icon as appropriate.  This *also* has a bunch of */
/* details to worry about: centering of icon in hex, clearing a rectangular */
/* area for the icon, picking a color for the unit, using either a bitmap */
/* or font char, and adding a side number for many-player games. */
/* Must also be careful not to draw black-on-black for units in space. */
/* This routine has also been drafted into drawing populace side numbers */
/* for otherwise empty hexes. */

draw_unit(Side *side, int x, int y) {

    viewdata view = side_view(side, wrap(x), y);
    int sx, sy, ucolor, hcolor, n;
    int terr = terrain_at(wrap(x), y);
    Side *side2;

    if (view != UNSEEN) {
        if (view != EMPTY) {
            xform(side, x, y, &sx, &sy);
            side2 = side_n(vside(view));
            ucolor = (allied_side(side, side2) ? side->appdata.owncolor :
                      (enemy_side(side, side2) ? side->appdata.enemycolor :
                       side->appdata.neutcolor));
            if (ucolor == side->appdata.owncolor &&
                (ttypes[terr].dark ||
                 side->monochrome ||
                 (side->appdata.showmode == TERRICONS)))
                ucolor = side->appdata.fgcolor;
            if (side->monochrome && side != side2)
                ucolor = side->appdata.bgcolor;
            hcolor = (side == side2 ? side->appdata.bgcolor : side->appdata.fgcolor);
            if (side->monochrome) {
                /* erasing background */
                draw_hex_icon(side, sx, sy, HEX, hcolor);
            } else if (side->appdata.showmode != TERRICONS) {
                draw_hex_icon(side, sx, sy, ((side->appdata.showmode == BORDERHEX) ? OHEX : HEX),
                    hex_color(side, x, y));
            }
            draw_unit_icon(side, sx, sy, vtype(view), ucolor);
            n = side_number(side2);
            if ((numsides > 2 || side->monochrome) && n != side_number(side)) {
                draw_side_number(side, sx, sy, n, ucolor);
            }
        }
    }
}

/* The basic map drawing routine does an entire row at a time, which yields */
/* order-of-magnitude speedups (!).  This routine is complicated by several */
/* tricks:  1) in monochrome, the entire line can be drawn at once; 2) in */
/* color, run-length encoding maximizes the length of constant-color strings */
/* and 3) anything which is in the background color need not be drawn. */
/* In general, this routine dominates the map viewing process, so efficiency */
/* here is very important. */

char rowbuf[BUFSIZE];           /* buffer for terrain row drawing */

void draw_row(Side *side, int x0, int y0, int len) {

    bool empty = TRUE;
    char ch;
    int i = 0, x, x1, sx, sy;
    long color, segcolor;

    if (side->monochrome) {
	xform(side, x0, y0, &sx, &sy);
	for (x = x0; x < x0 + len; ++x) {
	    if (side_view(side, wrap(x), y0) == EMPTY) {
		rowbuf[i++] = ttypes[terrain_at(wrap(x), y0)].tchar;
		empty = FALSE;
	    } else {
		rowbuf[i++] = ' ';
	    }
	}
	if (!empty) draw_terrain_row(side, sx, sy, rowbuf, i, side->appdata.fgcolor);
    } else {
	x1 = x0;
	segcolor = hex_color(side, x0, y0);
	for (x = x0; x < x0 + len; ++x) {
	    color = hex_color(side, x, y0);
	    sx = side->appdata.bgcolor;
	    if (color != segcolor) {
		if (segcolor != side->appdata.bgcolor) {
		    xform(side, x1, y0, &sx, &sy);
		    draw_terrain_row(side, sx, sy, rowbuf, i, segcolor);
		}
		i = 0;
		x1 = x;
		segcolor = color;
	    }
	    switch(side->appdata.showmode) {
	    case FULLHEX:
	    case BOTHICONS:
		ch = HEX;
		break;
	    case BORDERHEX:
		ch = OHEX;
		break;
	    case TERRICONS:
		ch = ttypes[terrain_at(wrap(x), y0)].tchar;
		break;
	    }
	    rowbuf[i++] = ch;
	}
	if (len == 1) i = 1;
	xform(side, x1, y0, &sx, &sy);
	draw_terrain_row(side, sx, sy, rowbuf, i, segcolor);
	if (side->appdata.showmode == BOTHICONS) {
	    i = 0;
	    x1 = x0;
	    segcolor = terricon_color(side, x0, y0);
	    for (x = x0; x < x0 + len; ++x) {
		color = terricon_color(side, x, y0);
		if (color != segcolor) {
		    xform(side, x1, y0, &sx, &sy);
		    draw_terrain_row(side, sx, sy, rowbuf, i, segcolor);
		    i = 0;
		    x1 = x;
		    segcolor = color;
		}
		rowbuf[i++] = ttypes[terrain_at(wrap(x), y0)].tchar;
	    }
	    if (len == 1) i = 1;
	    xform(side, x1, y0, &sx, &sy);
	    draw_terrain_row(side, sx, sy, rowbuf, i, segcolor);
	}
    }
    /* Units are much harder to optimize - fortunately they're sparse */
    for (x = x0; x < x0 + len; ++x) {
        draw_unit(side, x, y0);
    }
}

/* Draw an individual detailed hex, as a row of one. */
/* This routine may be called in cases where the hex is not on the main */
/* screen;  if so, then the world map but not the local map is drawn on. */
/* (should the display be shifted to make visible?) */

void draw_hex(Side *side, int x, int y, bool flushit) {

    int sx, sy;

    if (active_display(side)) {
        if (side->monochrome || side->appdata.showmode == TERRICONS) {
            xform(side, unwrap(side, x, y), y, &sx, &sy);
            draw_hex_icon(side, sx, sy, HEX, side->appdata.bgcolor);
        }
        draw_row(side, unwrap(side, x, y), y, 1);
        draw_bar(side, x, y, 1, world_color(side, x, y));
        if (flushit) flush_output(side);
    }
}

/* Cursor drawing also draws the unit in some other color if it's not the */
/* "top-level" unit in a hex, as well as getting the player's attention */
/* if the new location is sufficiently far from the last. */

void draw_cursor(Side *side) {

    int sx, sy;

    if (active_display(side)) {
        /* ugly hack to prevent extra cursor draw */
        if (side->cury == 0) return;
        xform(side, unwrap(side, side->curx, side->cury), side->cury, &sx, &sy);
        if (side->curunit != NULL && side->curunit->transport != NULL) {
            if (side->monochrome) {
                draw_hex_icon(side, sx, sy, HEX, side->appdata.bgcolor);
                draw_unit_icon(side, sx, sy,
                    side->curunit->type, side->appdata.fgcolor);
            } else {
                draw_unit_icon(side, sx, sy,
                    side->curunit->type, side->appdata.diffcolor);
            }
        }

        if (active_display(side)) {
            if ((distance(side->curx, side->cury, side->lastx, side->lasty) > 3) &&
                 (side->directorder))
                /* Flash something to draw the eye a long ways */
                flash_position(side, sx, sy, 100);
            draw_cursor_icon(side, sx, sy);
            if (side->lastx != side->curx || side->lasty != side->cury) {
                erase_cursor(side);
                side->lastx = side->curx;
                side->lasty = side->cury;
            }
        }
    }
}

/* Get rid of cursor by redrawing the hex. */

void erase_cursor(Side *side) {

    if (side->lasty > 0) draw_hex(side, side->lastx, side->lasty, TRUE);
}

/* Draw a splat visible to both sides at a given location.  Several splats */
/* available, depending on the seriousness of the hit.  Make an extra-flashy */
/* display when The Bomb goes off.  Because of the time delays involved, we */
/* have to update both sides' displays more or less simultaneously. Would be */
/* better to exhibit to all sides maybe, but I'm not going to bother! */

void draw_blast(Unit *unit, Side *es, int hit) {

    char ch;
    int ux = unit->x, uy = unit->y, sx, sy, i;
    Side *us = unit->side;

    if (hit >= period.nukehit) {
        if (active_display(us)) invert_whole_map(us);
        if (active_display(es)) invert_whole_map(es);
        /* may need a time delay if X is too speedy */
        if (active_display(us)) invert_whole_map(us);
        if (active_display(es)) invert_whole_map(es);
        for (i = 0; i < 4; ++i) {
            if (active_display(us)) draw_mushroom(us, ux, uy, i);
            if (active_display(es)) draw_mushroom(es, ux, uy, i);
            if (i != 2 && (active_display(us) || active_display(es))) sleep(1);
        }
        if (active_display(us) || active_display(es)) sleep(1);
    } 
    else {

        ch = ((hit >= unit->hp) ? 'd' : ((hit > 0) ? 'c' : 'b'));
        if (active_display(us)) {
            xform(us, unwrap(us, ux, uy), uy, &sx, &sy);
            draw_blast_icon(us, sx, sy, ch, us->appdata.enemycolor);
        }
        if (active_display(es)) {
            xform(es, unwrap(es, ux, uy), uy, &sx, &sy);
            draw_blast_icon(es, sx, sy, ch, es->appdata.owncolor);
        }
    }
}

/* Draw immediate area in more detail.  We make a little effort to avoid */
/* drawing hexes off the visible part of the screen, but are still somewhat */
/* conservative, so as not to get holes in the display.  Implication is that */
/* some lower level of routines has to be able to clip the map window. */

void show_map(Side *side) {

    int y1, y2, y, x1, x2, adj;

    if (active_display(side)) {
        if (side->flashtimer >= 0) {
            XtRemoveTimeOut(side->flashtimer);
            side->flashtimer = -1;
        }
	clear_window(side, XtWindow(side->map_area));
        clear_map_pix(side);
	y1 = side->vcy + world.h2;
	y2 = side->vcy - world.h2 + 1 - (world.height & 1);
	for (y = y1; y >= y2; --y) {
	    adj = (y - side->vcy) / 2;
	    x1 = side->vcx - world.w2 - adj - 1;
	    x2 = side->vcx + world.w2 - adj + 1 + (world.width & 1);
	    draw_row(side, x1, y, x2 - x1);
	}
        draw_cursor(side);
    }
}

/* Put the point x, y in the center of the screen.  Limit to make sure */
/* it keeps the screen properly aligned.  */

void recenter(Side *side, int x, int y) {

  int oldx, oldy;

  oldx = side->vcx;
  oldy = side->vcy;
  side->vcx = wrap(x);
  side->vcy = min(max(world.h2-(1-(world.height&1)), y),
                  (world.height-1)-world.h2);
  if ((side->vcx == oldx) && (side->vcy == oldy)) ;
  else {
    if (side->lastvcx >= 0) undraw_box(side);
    draw_box(side);
    show_map(side);
  }
}

/* Decide whether given location is not too close to edge of screen. */
/* We do this because it's a pain to move units when half the adjacent */
/* places aren't even visible.  This routine effectively places a lower */
/* limit of 5x5 for the map window. (I think) */

bool in_middle(Side *side, int x, int y) {

    int vcx = side->vcx, vw2 = world.w2, odd = world.w_odd;
    int vcy = side->vcy, vh2 = world.h2;
    int left_hex, right_hex;

    if(Debug>1) printf("In_middle: %d, %d, %d, %d\n", x, y, vcx, vcy);

    /* changed so that units on top and bottom lines do the right thing. */
    if (!between(vcy-vh2+2, y, vcy+vh2-1) && between(1, y, world.height-2))
        return FALSE;

    left_hex = vcx - vw2 - ((y - vcy) / 2) - (y > vcy) + 3;
    right_hex = vcx + vw2 + odd - ((y - vcy) / 2) + ((y < vcy) && odd)
      - ((y > vcy) && !odd) - 3;

    if (left_hex < 0) return !between(right_hex, x, left_hex + world.width);
    else if (right_hex >= world.width)
      return !between(right_hex - world.width, x, left_hex);
    return between(left_hex, x, right_hex);
}

/* Ensure that given location is visible.  We also flush the input because */
/* any input relating to a different screen is probably worthless. */

void put_on_screen(Side *side, int x, int y) {

    if (active_display(side))
        if (!in_middle(side, x, y))
          recenter(side, x, y);
}

/* Display unit status */

void show_status(Side *side, Unit *unit) {

    char tmp2[BUFSIZE];

    /* If we woke up, add an explanation. */
    tmp2[0] = '\0';
    if (unit->wakeup_reason) { /* and test that unit is awake? */
        if (unit->wakeup_reason == WAKEENEMY)
          sprintf(tmp2, " (Saw %s %s)",
                  (unit->waking_side == NULL ? "neutral" :
                   unit->waking_side->name),
                  utypes[unit->waking_type].name);
        if (unit->wakeup_reason == WAKERESOURCE)
          sprintf(tmp2, " (Low on fuel)");
        /*      if (unit->wakeup_reason == WAKETIME)
                sprintf(tmp2, " (Last order done)"); */
        if (unit->wakeup_reason == WAKELEADERDEAD)
          sprintf(tmp2, " (Leader died)");
        if (unit->wakeup_reason == WAKELOST)
          sprintf(tmp2, " (Lost & confused)");
    }
    sprintf(spbuf, "%s%s %s Moves %d",
            order_desig(&(unit->orders)), tmp2,
            (unit->standing != NULL ? "*" : ""),
            unit->movesleft);
    draw_info_text(side->info_status, spbuf);
}

/* Display unit HP */

void show_hp(Side *side, Unit *unit) {

    sprintf(spbuf, "HP %d/%d", unit->hp, utypes[unit->type].hp);
    draw_info_text(side->info_hp, spbuf);
}

/* Display unit production */

void show_product(Side *side, Unit *unit) {

    sprintf(spbuf, "");
    if (producing(unit)) {
        sprintf(spbuf, "New %s in %d turns",
                utypes[unit->product].name, unit->schedule);
    }
    draw_info_text(side->info_production, spbuf);

    sprintf(spbuf, "");
    if (producing(unit) && unit->product != unit->next_product) {
        sprintf(spbuf, "Next product is %s",
                utypes[unit->next_product].name);
    }
    draw_info_text(side->info_next_product, spbuf);
}

/* Display unit resources */

void show_resources(Side *side, Unit *unit) {

    int r;

    sprintf(spbuf, "");
    for_all_resource_types(r) {
        if (utypes[unit->type].storage[r] > 0) {
            sprintf(tmpbuf, "%s %d/%d  ",
                    rtypes[r].name, unit->supply[r], utypes[unit->type].storage[r]);
            strcat(spbuf, tmpbuf);
        }
    }
    draw_info_text(side->info_resources, spbuf);
}

/* Display all the details that only the owner or God (== debugger) sees. */

void show_intimate_details(Side *side, Unit *unit) {

    int i, nums[MAXUTYPES];
    Unit *occ;

    sprintf(spbuf, "In %s at %d,%d",
            (unit->transport != NULL ? short_unit_handle(unit->transport) :
             ttypes[terrain_at(unit->x, unit->y)].name),
             unit->x, unit->y);
    draw_info_text(side->info_terrain, spbuf);

    sprintf(spbuf, "");
    if (unit->occupant != NULL) {
        strcpy(spbuf, "Occ ");
        for_all_unit_types(i) nums[i] = 0;
        for_all_occupants(unit, occ) nums[occ->type]++;
        for_all_unit_types(i) {
            if (nums[i] > 0) {
                sprintf(tmpbuf, "%d %c  ", nums[i], utypes[i].uchar);
                strcat(spbuf, tmpbuf);
            }
        }
    }
    draw_info_text(side->info_misc1, spbuf);

    show_status(side, unit);

    show_hp(side, unit);

    show_product(side, unit);

    show_resources(side, unit);
}

/* Info about a unit is complicated by the overriding requirement that it */
/* be quickly graspable.  Graphics is helpful, so is fixed format (trains */
/* eyes to get a value of interest from same place each time). */

/* Verbal description of what you can see in your view.  Can't always show */
/* names because the view doesn't store them, and the unit at that spot may */
/* not exist anymore.  Units that are "always seen" (like cities) can be */
/* described in more detail, however. */

void show_info(Side *side) {

    bool more = (Debug || Build);
    int u, terr, age;
    char *filler = "";
    viewdata view, prevview;
    Unit *unit = side->curunit, *realunit;
    Side *side2;

    if (active_display(side)) {    
      if (side->curx >= 0 && side->cury >= 0) {
        view = side_view(side, side->curx, side->cury);
        terr = terrain_at(side->curx, side->cury);
        realunit = unit_at(side->curx, side->cury);
        put_on_screen(side, side->curx, side->cury);
        if (unit != NULL) {
          sprintf(tmpbuf, "%s", unit_handle(side, unit));
          draw_info_text(side->info_handle, tmpbuf);
          more = TRUE;
        } else {
          /* Describe any unit image that might be present. */
          if (view == UNSEEN || view == EMPTY) {
              sprintf(tmpbuf, "");
          } else {
              side2 = side_n(vside(view));
              u = vtype(view);
              if ((utypes[u].seealways ||
                   cover(side, side->curx, side->cury) > 0
                   || more) && realunit != NULL) {
                  sprintf(tmpbuf, "%s", unit_handle(side, realunit));
              } else {
                  sprintf(tmpbuf, "%s %s",
                          (side2 == NULL ? "neutral" : side2->name),
                          utypes[u].name);
              }
              filler = "In ";
          }
          draw_info_text(side->info_handle, tmpbuf);
          /* Now describe terrain and position. */
          sprintf(tmpbuf, "%s%s at %d,%d",
                  filler, (view == UNSEEN ? "(unknown)" : ttypes[terr].name),
                  side->curx, side->cury);
          draw_info_text(side->info_terrain, tmpbuf);
        }
        /* put here so overwrites any spillover from name writing */
        if (more) {
          if (unit == NULL) unit = realunit;
          if (unit != NULL) show_intimate_details(side, unit);
        } else {
#ifdef PREVVIEW
          /* Compose and display view history of this hex. */
          if (Debug) printf("Drawing previous view info\n");
          age = side_view_age(side, side->curx, side->cury);
          prevview = side_prevview(side, side->curx, side->cury);
          if (age == 0) {
              if (prevview != view) {
                  if (prevview == EMPTY) {
                      /* misleading if prevview was set during init. */
                      sprintf(tmpbuf, "Up to date; had been empty.");
                  } else if (prevview == UNSEEN) {
                      sprintf(tmpbuf, "Up to date; had been unexplored.");
                  } else {
                      side2 = side_n(vside(prevview));
                      u = vtype(prevview);
                      if (side2 != side) {
                          sprintf(tmpbuf, "Up to date; had seen %s %s.",
                                  (side2 == NULL ? "neutral" : side2->name),
                                  utypes[u].name);
                      } else {
                          sprintf(tmpbuf,
                                  "Up to date; had been occupied by your %s.",
                                  utypes[u].name);
                      }
                  }
              } else {
                  sprintf(tmpbuf, "Up to date.");
              }
          } else {
              if (prevview == EMPTY) {
                  sprintf(tmpbuf, "Was empty %d turns ago.", age);
              } else if (prevview == UNSEEN) {
                  sprintf(tmpbuf, "Terrain first seen %d turns ago.", age);
              } else {
                  side2 = side_n(vside(prevview));
                  u = vtype(prevview);
                  if (side2 != side) {
                      sprintf(tmpbuf, "Saw %s %s, %d turns ago.",
                              (side2 == NULL ? "neutral" : side2->name),
                              utypes[u].name, age);
                  } else {
                      sprintf(tmpbuf, "Was occupied by your %s %d turns ago.",
                              utypes[u].name, age);
                  }
              }
          }
          draw_info_text(side->info_misc1, tmpbuf);
#endif
          sprintf(tmpbuf, "");
          draw_info_text(side->info_hp, tmpbuf);
          draw_info_text(side->info_production, tmpbuf);
          draw_info_text(side->info_next_product, tmpbuf);
          draw_info_text(side->info_resources, tmpbuf);

          if (side->curdeadunit != NULL &&
              side->curdeadunit->prevx == side->curx &&
              side->curdeadunit->prevy == side->cury) {
            sprintf(tmpbuf, "%s died here",
                    unit_handle(side, side->curdeadunit));
          }
          draw_info_text(side->info_status, tmpbuf);
        }
      }
    }
}
