
/*--------------------------------------------------------------------*/
/*--- Low-level syscall code for PowerPC.         vg_ppc_syscall.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Valgrind, an extensible
   emulator for monitoring program execution on Unixes.

   Copyright (C) 2000-2004 Julian Seward 
      jseward@acm.org
   Copyright (C) 2004 Paul Mackerras
      paulus@samba.org

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#include "core.h"

extern void do_thread_syscall(Int arg1, Int syscallno, Int *gprs, Int *cr,
			      enum PXState *statep, enum PXState poststate);

asm(
".text\n"
"	.type do_thread_syscall,@function\n"

"do_thread_syscall:\n"
"	stwu	1,-32(1)\n"
"	mflr	0\n"
"	stw	0,36(1)\n"
"	stw	31,28(1)\n"
"	mr	31,5\n"		/* gprs */
"	stw	30,24(1)\n"
"	mr	30,6\n"		/* cr */
"	stw	29,20(1)\n"
"	mr	29,7\n"		/* statep */
"	stw	28,16(1)\n"
"	mr	28,8\n"		/* poststate */
".vga_sys_before:\n"
"	mr	0,4\n"		/* syscall */
/* arg1 is in r3 already */
"	lwz	4,16(31)\n"	/* arg2 */
"	lwz	5,20(31)\n"	/* arg3 */
"	lwz	6,24(31)\n"	/* arg4 */
"	lwz	7,28(31)\n"	/* arg5 */
"	lwz	8,32(31)\n"	/* arg6 */
".vga_sys_restarted:\n"
"	sc\n"
".vga_sys_after:\n"
"	stw	3,12(31)\n"	/* save result */
"	lwz	3,0(30)\n"	/* update cr */
"	mfcr	4\n"
"	rlwimi	3,4,0,3,3\n"	/* transfer cr0.so */
"	stw	3,0(30)\n"
"	cmpwi	29,0\n"
"	beq	.vga_sys_done\n"
"	stw	28,0(29)\n"
".vga_sys_done:\n"
"	lwz	28,16(1)\n"
"	lwz	29,20(1)\n"
"	lwz	30,24(1)\n"
"	lwz	31,28(1)\n"
"	addi	1,1,32\n"
"	blr\n"
"	.size do_thread_syscall,.-do_thread_syscall\n"
".previous\n"

".section .rodata\n"
"	.globl	vga_sys_before\n"
"vga_sys_before:	.long	.vga_sys_before\n"
"	.globl	vga_sys_restarted\n"
"vga_sys_restarted:	.long	.vga_sys_restarted\n"
"	.globl	vga_sys_after\n"
"vga_sys_after:	.long	.vga_sys_after\n"
"	.globl	vga_sys_done\n"
"vga_sys_done:	.long	.vga_sys_done\n"
".previous\n"
);

/* Run a syscall for a particular thread */
void VGA_(thread_syscall)(Int syscallno, arch_thread_t *tst, 
			  enum PXState *state , enum PXState poststate)
{
   do_thread_syscall(tst->m_orig_gpr3, syscallno,
		     tst->m_gpr, &tst->m_cr,
		     state,	  /* state to update */
		     poststate);  /* state when syscall has finished */
   tst->m_result = (tst->m_cr & 0x10000000)? - tst->m_gpr[3]: tst->m_gpr[3];
}

void VGA_(do_syscall_direct)(Int syscallno, arch_thread_t *tst)
{
   do_thread_syscall(tst->m_orig_gpr3, syscallno,
		     tst->m_gpr, &tst->m_cr, NULL, 0);
   tst->m_result = (tst->m_cr & 0x10000000)? - tst->m_gpr[3]: tst->m_gpr[3];
}

/* Back up to restart a system call. */
void VGA_(restart_syscall)(arch_thread_t *tst)
{
   tst->m_eip -= 4;		/* sizeof(sc) */
   {
      UInt *p = (UInt *)tst->m_eip;
      
      if (p[0] != 0x44000002)
	 VG_(message)(Vg_DebugMsg, 
		      "?! restarting over syscall at %p %08x\n",
		      tst->m_eip, p[0]);

      vg_assert(p[0] == 0x44000002);
   }
}

/* ---------------------------------------------------------------------
   PRE/POST wrappers for ppc/Linux-specific syscalls
   ------------------------------------------------------------------ */

// Nb: See the comment above the generic PRE/POST wrappers in
// coregrind/vg_syscalls.c for notes about how they work.

#define PRE(x,f) \
   static UInt ppc_linux_##x##_flags = f; \
   static void ppc_linux_##x##_before(ThreadId tid, ThreadState *tst)
#define POST(x) \
   static void ppc_linux_##x##_after (ThreadId tid, ThreadState *tst)

#define SYSNO	PLATFORM_SYSCALL_NUM(tst->arch)    // in PRE(x)
#define res	PLATFORM_SYSCALL_RET(tst->arch)	   // in POST(x)
#define arg1	PLATFORM_SYSCALL_ARG1(tst->arch)
#define arg2	PLATFORM_SYSCALL_ARG2(tst->arch)
#define arg3	PLATFORM_SYSCALL_ARG3(tst->arch)
#define arg4	PLATFORM_SYSCALL_ARG4(tst->arch)
#define arg5	PLATFORM_SYSCALL_ARG5(tst->arch)
#define arg6	PLATFORM_SYSCALL_ARG6(tst->arch)

#define set_result(val) PLATFORM_SET_SYSCALL_RESULT(tst->arch, (val))

#define PRINT(format, args...)  \
   if (VG_(clo_trace_syscalls))        \
      VG_(printf)(format, ## args)

PRE(old_select, MayBlock)
{
   /* struct sel_arg_struct {
      unsigned long n;
      fd_set *inp, *outp, *exp;
      struct timeval *tvp;
      };
   */
   PRE_REG_READ1(long, "old_select", struct sel_arg_struct *, args);
   PRE_MEM_READ( "old_select(args)", arg1, 5*sizeof(UWord) );

   {
      UInt* arg_struct = (UInt*)arg1;
      UInt a1, a2, a3, a4, a5;

      a1 = arg_struct[0];
      a2 = arg_struct[1];
      a3 = arg_struct[2];
      a4 = arg_struct[3];
      a5 = arg_struct[4];

      PRINT("old_select ( %d, %p, %p, %p, %p )", a1,a2,a3,a4,a5);
      if (a2 != (Addr)NULL)
	 PRE_MEM_READ( "old_select(readfds)",   a2, a1/8 /* __FD_SETSIZE/8 */ );
      if (a3 != (Addr)NULL)
	 PRE_MEM_READ( "old_select(writefds)",  a3, a1/8 /* __FD_SETSIZE/8 */ );
      if (a4 != (Addr)NULL)
	 PRE_MEM_READ( "old_select(exceptfds)", a4, a1/8 /* __FD_SETSIZE/8 */ );
      if (a5 != (Addr)NULL)
	 PRE_MEM_READ( "old_select(timeout)", a5, sizeof(struct vki_timeval) );
   }
}

PRE(sys_clone, Special)
{
   PRINT("sys_clone ( %d, %p, %p, %p, %p )",arg1,arg2,arg3,arg4,arg5);
   // XXX: really not sure about the last two args... if they are really
   // there, we should do PRE_MEM_READs for both of them...
   PRE_REG_READ4(int, "clone",
                 unsigned long, flags, void *, child_stack,
                 int *, parent_tidptr, int *, child_tidptr);

   if (arg2 == 0 &&
       (arg1 == (VKI_CLONE_CHILD_CLEARTID|VKI_CLONE_CHILD_SETTID|VKI_SIGCHLD)
     || arg1 == (VKI_CLONE_PARENT_SETTID|VKI_SIGCHLD))) 
   {
      VGA_(gen_sys_fork_before)(tid, tst);
      set_result( VG_(do_syscall)(SYSNO, arg1, arg2, arg3, arg4, arg5) );
      VGA_(gen_sys_fork_after) (tid, tst);
   } else {
      VG_(unimplemented)
         ("clone(): not supported by Valgrind.\n   "
          "We do support programs linked against\n   "
          "libpthread.so, though.  Re-run with -v and ensure that\n   "
          "you are picking up Valgrind's implementation of libpthread.so.");
   }
}

PRE(sys_mmap, 0)
{
   PRINT("sys_mmap ( %p, %llu, %d, %d, %d, %d )",
         arg1, (ULong)arg2, arg3, arg4, arg5, arg6 );
   PRE_REG_READ6(long, "mmap",
                 unsigned long, start, unsigned long, length,
                 unsigned long, prot,  unsigned long, flags,
                 unsigned long, fd,    unsigned long, offset);

   if (arg4 & VKI_MAP_FIXED) {
      if (!VG_(valid_client_addr)(arg1, arg2, tid, "mmap2"))
	 set_result( -VKI_ENOMEM );
   } else {
      arg1 = VG_(find_map_space)(arg1, arg2, True);
      if (arg1 == 0)
	 set_result( -VKI_ENOMEM );
      else 
         arg4 |= VKI_MAP_FIXED;
   }
}

POST(sys_mmap)
{
   vg_assert(VG_(valid_client_addr)(res, arg2, tid, "mmap2"));
   VG_(mmap_segment)( (Addr)res, arg2, arg3, arg4, arg5,
                 arg6 * (ULong)VKI_PAGE_SIZE );
}

PRE(sys_sigreturn, Special)
{
   /* no arguments, expects a non-RT signal frame on the stack */
   PRINT("sys_sigreturn");
   VG_(handle_signal_return)(tid, False);
}

PRE(sys_rt_sigreturn, Special)
{
   /* no arguments, expects an RT signal frame on the stack */
   PRINT("sys_rt_sigreturn");
   VG_(handle_signal_return)(tid, True);
}

#undef PRE
#undef POST

/* ---------------------------------------------------------------------
   The ppc/Linux syscall table
   ------------------------------------------------------------------ */

#define GENX_(const, name) \
   [const] = { &VGA_(gen_##name##_flags), VGA_(gen_##name##_before), NULL }
#define GENXY(const, name) \
   [const] = { &VGA_(gen_##name##_flags), VGA_(gen_##name##_before), \
                                          VGA_(gen_##name##_after) }

#define LINX_(const, name) \
   [const] = { &VGA_(linux_##name##_flags), VGA_(linux_##name##_before), NULL }
#define LINXY(const, name) \
   [const] = { &VGA_(linux_##name##_flags), VGA_(linux_##name##_before), \
                                            VGA_(linux_##name##_after) }
#define PLAX_(const, name) \
   [const] = { &ppc_linux_##name##_flags, ppc_linux_##name##_before, NULL }
#define PLAXY(const, name) \
   [const] = { &ppc_linux_##name##_flags, ppc_linux_##name##_before, \
                                          ppc_linux_##name##_after }

// This table maps from __NR_xxx syscall numbers to the appropriate
// PRE/POST sys_foo() wrappers on ppc.
const struct SyscallTableEntry VGA_(syscall_table)[] = {
   // 0 is restart_syscall
   GENX_(__NR_exit,             sys_exit),         // 1
   GENXY(__NR_fork,             sys_fork),         // 2
   GENXY(__NR_read,             sys_read),         // 3
   GENX_(__NR_write,            sys_write),        // 4

   GENXY(__NR_open,             sys_open),         // 5
   GENXY(__NR_close,            sys_close),        // 6
   GENXY(__NR_waitpid,          sys_waitpid),      // 7
   GENXY(__NR_creat,            sys_creat),        // 8
   GENX_(__NR_link,             sys_link),         // 9

   GENX_(__NR_unlink,           sys_unlink),       // 10
   GENX_(__NR_execve,           sys_execve),       // 11
   GENX_(__NR_chdir,            sys_chdir),        // 12
   GENXY(__NR_time,             sys_time),         // 13
   GENX_(__NR_mknod,            sys_mknod),        // 14

   GENX_(__NR_chmod,            sys_chmod),        // 15
   GENX_(__NR_lchown,           sys_lchown),       // 16
   GENX_(__NR_break,            sys_ni_syscall),   // 17
   //   (__NR_oldstat,          sys_stat),         // 18
   GENX_(__NR_lseek,            sys_lseek),        // 19

   GENX_(__NR_getpid,           sys_getpid),       // 20
   LINX_(__NR_mount,            sys_mount),        // 21
   LINX_(__NR_umount,           sys_oldumount),    // 22
   GENX_(__NR_setuid,           sys_setuid),       // 23
   GENX_(__NR_getuid,           sys_getuid),       // 24

   //   (__NR_stime,            sys_stime),        // 25 * (SVr4,SVID,X/OPEN)
   GENXY(__NR_ptrace,           sys_ptrace),       // 26
   GENX_(__NR_alarm,            sys_alarm),        // 27
   //   (__NR_oldfstat,         sys_fstat),        // 28 * L -- obsolete
   GENX_(__NR_pause,            sys_pause),        // 29

   GENX_(__NR_utime,            sys_utime),        // 30
   GENX_(__NR_stty,             sys_ni_syscall),   // 31
   GENX_(__NR_gtty,             sys_ni_syscall),   // 32
   GENX_(__NR_access,           sys_access),       // 33
   GENX_(__NR_nice,             sys_nice),         // 34

   GENX_(__NR_ftime,            sys_ni_syscall),   // 35
   GENX_(__NR_sync,             sys_sync),         // 36
   GENXY(__NR_kill,             sys_kill),         // 37
   GENX_(__NR_rename,           sys_rename),       // 38
   GENX_(__NR_mkdir,            sys_mkdir),        // 39

   GENX_(__NR_rmdir,            sys_rmdir),        // 40
   GENXY(__NR_dup,              sys_dup),          // 41
   GENXY(__NR_pipe,             sys_pipe),         // 42
   GENXY(__NR_times,            sys_times),        // 43
   GENX_(__NR_prof,             sys_ni_syscall),   // 44

   GENX_(__NR_brk,              sys_brk),          // 45
   GENX_(__NR_setgid,           sys_setgid),       // 46
   GENX_(__NR_getgid,           sys_getgid),       // 47
   //   (__NR_signal,           sys_signal),       // 48
   GENX_(__NR_geteuid,          sys_geteuid),      // 49

   GENX_(__NR_getegid,          sys_getegid),      // 50
   GENX_(__NR_acct,             sys_acct),         // 51
   LINX_(__NR_umount2,          sys_umount),       // 52
   GENX_(__NR_lock,             sys_ni_syscall),   // 53
   GENXY(__NR_ioctl,            sys_ioctl),        // 54

   GENXY(__NR_fcntl,            sys_fcntl),        // 55
   GENX_(__NR_mpx,              sys_ni_syscall),   // 56
   GENXY(__NR_setpgid,          sys_setpgid),      // 57
   GENX_(__NR_ulimit,           sys_ni_syscall),   // 58
   //   (__NR_oldolduname,      sys_olduname),     // 59

   GENX_(__NR_umask,            sys_umask),        // 60
   GENX_(__NR_chroot,           sys_chroot),       // 61
   //   (__NR_ustat,            sys_ustat)         // 62
   GENXY(__NR_dup2,             sys_dup2),         // 63
   GENX_(__NR_getppid,          sys_getppid),      // 64

   GENX_(__NR_getpgrp,          sys_getpgrp),      // 65
   GENX_(__NR_setsid,           sys_setsid),       // 66
   GENXY(__NR_sigaction,        sys_sigaction),    // 67
   //   (__NR_sgetmask,         sys_sgetmask),     // 68
   //   (__NR_ssetmask,         sys_ssetmask),     // 69

   GENX_(__NR_setreuid,         sys_setreuid),     // 70
   GENX_(__NR_setregid,         sys_setregid),     // 71
   GENX_(__NR_sigsuspend,       sys_sigsuspend),   // 72
   GENXY(__NR_sigpending,       sys_sigpending),   // 73
   //   (__NR_sethostname,      sys_sethostname),  // 74

   GENX_(__NR_setrlimit,        sys_setrlimit),    // 75
   GENXY(__NR_getrlimit,        sys_old_getrlimit),// 76
   GENXY(__NR_getrusage,        sys_getrusage),    // 77
   GENXY(__NR_gettimeofday,     sys_gettimeofday), // 78
   GENX_(__NR_settimeofday,     sys_settimeofday), // 79

   GENXY(__NR_getgroups,        sys_getgroups),    // 80
   GENX_(__NR_setgroups,        sys_setgroups),    // 81
   PLAX_(__NR_select,           old_select),       // 82
   GENX_(__NR_symlink,          sys_symlink),      // 83
   //   (__NR_oldlstat,         sys_lstat),        // 84

   GENX_(__NR_readlink,         sys_readlink),     // 85
   //   (__NR_uselib,           sys_uselib),       // 86
   //   (__NR_swapon,           sys_swapon),       // 87
   //   (__NR_reboot,           sys_reboot),       // 88
   //   (__NR_readdir,          old_readdir),      // 89

   PLAXY(__NR_mmap,             sys_mmap),         // 90
   GENXY(__NR_munmap,           sys_munmap),       // 91
   GENX_(__NR_truncate,         sys_truncate),     // 92
   GENX_(__NR_ftruncate,        sys_ftruncate),    // 93
   GENX_(__NR_fchmod,           sys_fchmod),       // 94

   GENX_(__NR_fchown,           sys_fchown),       // 95
   GENX_(__NR_getpriority,      sys_getpriority),  // 96
   GENX_(__NR_setpriority,      sys_setpriority),  // 97
   GENX_(__NR_profil,           sys_ni_syscall),   // 98
   GENXY(__NR_statfs,           sys_statfs),       // 99

   GENXY(__NR_fstatfs,          sys_fstatfs),      // 100
   GENX_(__NR_ioperm,           sys_ni_syscall),   // 101
   GENXY(__NR_socketcall,       sys_socketcall),   // 102
   LINXY(__NR_syslog,           sys_syslog),       // 103
   GENXY(__NR_setitimer,        sys_setitimer),    // 104

   GENXY(__NR_getitimer,        sys_getitimer),    // 105
   GENXY(__NR_stat,             sys_newstat),      // 106
   GENXY(__NR_lstat,            sys_newlstat),     // 107
   GENXY(__NR_fstat,            sys_newfstat),     // 108
   //   (__NR_olduname,         sys_uname),        // 109

   GENX_(__NR_iopl,             sys_ni_syscall),   // 110
   LINX_(__NR_vhangup,          sys_vhangup),      // 111
   GENX_(__NR_idle,             sys_ni_syscall),   // 112
   //   (__NR_vm86old,          sys_vm86old),      // 113
   GENXY(__NR_wait4,            sys_wait4),        // 114

   //   (__NR_swapoff,          sys_swapoff),      // 115
   LINXY(__NR_sysinfo,          sys_sysinfo),      // 116
   GENXY(__NR_ipc,              sys_ipc),          // 117
   GENX_(__NR_fsync,            sys_fsync),        // 118
   PLAX_(__NR_sigreturn,        sys_sigreturn),    // 119 ppc/Linux

   PLAX_(__NR_clone,            sys_clone),        // 120
   //   (__NR_setdomainname,    sys_setdomainname),// 121
   GENXY(__NR_uname,            sys_newuname),     // 122
   GENX_(__NR_modify_ldt,       sys_ni_syscall),   // 123
   LINXY(__NR_adjtimex,         sys_adjtimex),     // 124

   GENXY(__NR_mprotect,         sys_mprotect),     // 125
   GENXY(__NR_sigprocmask,      sys_sigprocmask),  // 126
   // Nb: create_module() was removed 2.4-->2.6
   GENX_(__NR_create_module,    sys_ni_syscall),   // 127
   GENX_(__NR_init_module,      sys_init_module),  // 128
   //   (__NR_delete_module,    sys_delete_module),// 129

   // Nb: get_kernel_syms() was removed 2.4-->2.6
   GENX_(__NR_get_kernel_syms,  sys_ni_syscall),   // 130
   GENX_(__NR_quotactl,         sys_quotactl),     // 131
   GENX_(__NR_getpgid,          sys_getpgid),      // 132
   GENX_(__NR_fchdir,           sys_fchdir),       // 133
   //   (__NR_bdflush,          sys_bdflush),      // 134

   //   (__NR_sysfs,            sys_sysfs),        // 135
   LINX_(__NR_personality,      sys_personality),  // 136
   GENX_(__NR_afs_syscall,      sys_ni_syscall),   // 137
   LINX_(__NR_setfsuid,         sys_setfsuid),     // 138
   LINX_(__NR_setfsgid,         sys_setfsgid),     // 139

   LINXY(__NR__llseek,          sys_llseek),       // 140
   GENXY(__NR_getdents,         sys_getdents),     // 141
   GENX_(__NR__newselect,       sys_select),       // 142
   GENX_(__NR_flock,            sys_flock),        // 143
   GENX_(__NR_msync,            sys_msync),        // 144

   GENXY(__NR_readv,            sys_readv),        // 145
   GENX_(__NR_writev,           sys_writev),       // 146
   GENX_(__NR_getsid,           sys_getsid),       // 147
   GENX_(__NR_fdatasync,        sys_fdatasync),    // 148
   LINXY(__NR__sysctl,          sys_sysctl),       // 149

   GENX_(__NR_mlock,            sys_mlock),           // 150
   GENX_(__NR_munlock,          sys_munlock),         // 151
   GENX_(__NR_mlockall,         sys_mlockall),        // 152
   GENX_(__NR_munlockall,       sys_munlockall),      // 153
   GENXY(__NR_sched_setparam,   sys_sched_setparam),  // 154

   GENXY(__NR_sched_getparam,        sys_sched_getparam),         // 155
   GENX_(__NR_sched_setscheduler,    sys_sched_setscheduler),     // 156
   GENX_(__NR_sched_getscheduler,    sys_sched_getscheduler),     // 157
   GENX_(__NR_sched_yield,           sys_sched_yield),            // 158
   GENX_(__NR_sched_get_priority_max,sys_sched_get_priority_max), // 159

   GENX_(__NR_sched_get_priority_min,sys_sched_get_priority_min), // 160
   //   (__NR_sched_rr_get_interval, sys_sched_rr_get_interval),  // 161
   GENXY(__NR_nanosleep,        sys_nanosleep),       // 162
   GENX_(__NR_mremap,           sys_mremap),          // 163
   LINX_(__NR_setresuid,        sys_setresuid),       // 164

   LINXY(__NR_getresuid,        sys_getresuid),       // 165
   GENX_(__NR_query_module,     sys_ni_syscall),      // 166
   GENXY(__NR_poll,             sys_poll),            // 167
   //   (__NR_nfsservctl,       sys_nfsservctl),      // 168
   LINX_(__NR_setresgid,        sys_setresgid),       // 169

   LINXY(__NR_getresgid,        sys_getresgid),       // 170
   LINX_(__NR_prctl,            sys_prctl),           // 171
   PLAX_(__NR_rt_sigreturn,     sys_rt_sigreturn),    // 172 ppc/Linux
   GENXY(__NR_rt_sigaction,     sys_rt_sigaction),    // 173
   GENXY(__NR_rt_sigprocmask,   sys_rt_sigprocmask),  // 174

   GENXY(__NR_rt_sigpending,    sys_rt_sigpending),   // 175
   GENXY(__NR_rt_sigtimedwait,  sys_rt_sigtimedwait), // 176
   GENXY(__NR_rt_sigqueueinfo,  sys_rt_sigqueueinfo), // 177
   GENX_(__NR_rt_sigsuspend,    sys_rt_sigsuspend),   // 178
   GENXY(__NR_pread64,          sys_pread64),         // 179

   GENX_(__NR_pwrite64,         sys_pwrite64),     // 180
   GENX_(__NR_chown,            sys_chown),        // 181
   GENXY(__NR_getcwd,           sys_getcwd),       // 182
   GENXY(__NR_capget,           sys_capget),       // 183
   GENX_(__NR_capset,           sys_capset),       // 184

   GENXY(__NR_sigaltstack,      sys_sigaltstack),  // 185
   LINXY(__NR_sendfile,         sys_sendfile),     // 186
   GENXY(__NR_getpmsg,          sys_getpmsg),      // 187
   GENX_(__NR_putpmsg,          sys_putpmsg),      // 188
   // Nb: we convert vfork() to fork() in VG_(pre_syscall)().
   //   (__NR_vfork,            sys_vfork),        // 189 -- Valgrind avoids

   GENXY(__NR_ugetrlimit,       sys_getrlimit),    // 190
   //   (__NR_readahead,        sys_readahead),    // 191
   GENXY(__NR_mmap2,            sys_mmap2),        // 192
   GENX_(__NR_truncate64,       sys_truncate64),   // 193
   GENX_(__NR_ftruncate64,      sys_ftruncate64),  // 194
   
   GENXY(__NR_stat64,           sys_stat64),       // 195
   GENXY(__NR_lstat64,          sys_lstat64),      // 196
   GENXY(__NR_fstat64,          sys_fstat64),      // 197
   GENX_(__NR_pciconfig_read,	sys_ni_syscall),   // 198 ppc/Linux
   GENX_(__NR_pciconfig_write,	sys_ni_syscall),   // 199 ppc/Linux

   GENX_(__NR_pciconfig_iobase,	sys_ni_syscall),   // 200 ppc/Linux
   GENX_(__NR_multiplexer,	sys_ni_syscall),   // 201 ppc/Linux
   GENXY(__NR_getdents64,       sys_getdents64),   // 202
   //   (__NR_pivot_root,       sys_pivot_root),   // 203
   GENXY(__NR_fcntl64,          sys_fcntl64),      // 204

   GENX_(__NR_madvise,          sys_madvise),      // 205
   GENXY(__NR_mincore,          sys_mincore),      // 206
   //   (__NR_gettid,           sys_gettid),       // 207
   //   (__NR_tkill,            sys_tkill),        // 208
   GENX_(__NR_setxattr,         sys_setxattr),     // 209

   GENX_(__NR_lsetxattr,        sys_lsetxattr),    // 210
   GENX_(__NR_fsetxattr,        sys_fsetxattr),    // 211
   GENXY(__NR_getxattr,         sys_getxattr),     // 212
   GENXY(__NR_lgetxattr,        sys_lgetxattr),    // 213
   GENXY(__NR_fgetxattr,        sys_fgetxattr),    // 214

   GENXY(__NR_listxattr,        sys_listxattr),    // 215
   GENXY(__NR_llistxattr,       sys_llistxattr),   // 216
   GENXY(__NR_flistxattr,       sys_flistxattr),   // 217
   GENX_(__NR_removexattr,      sys_removexattr),  // 218
   GENX_(__NR_lremovexattr,     sys_lremovexattr), // 219

   GENX_(__NR_fremovexattr,     sys_fremovexattr), // 220
   LINXY(__NR_futex,            sys_futex),        // 221
   GENX_(__NR_sched_setaffinity,sys_sched_setaffinity),  // 222
   GENXY(__NR_sched_getaffinity,sys_sched_getaffinity),  // 223
   GENX_(224,                   sys_ni_syscall),   // 224

   // Nb: 225 is reserved for the Tux webserver
   GENX_(__NR_tuxcall,          sys_ni_syscall),      // 225
   LINXY(__NR_sendfile64,       sys_sendfile64),      // 226
   LINX_(__NR_io_setup,         sys_io_setup),        // 227
   LINX_(__NR_io_destroy,       sys_io_destroy),      // 228
   LINXY(__NR_io_getevents,     sys_io_getevents),    // 229

   LINX_(__NR_io_submit,        sys_io_submit),       // 230
   LINXY(__NR_io_cancel,        sys_io_cancel),       // 231
   GENX_(__NR_set_tid_address,  sys_set_tid_address), // 232
   //   (__NR_fadvise64,        sys_fadvise64),       // 233
   GENX_(__NR_exit_group,       sys_exit_group),      // 234

   GENXY(__NR_lookup_dcookie,   sys_lookup_dcookie),  // 235
   LINXY(__NR_epoll_create,     sys_epoll_create),    // 236
   LINX_(__NR_epoll_ctl,        sys_epoll_ctl),       // 237
   LINXY(__NR_epoll_wait,       sys_epoll_wait),      // 238
   //   (__NR_remap_file_pages, sys_remap_file_pages),// 239

   GENXY(__NR_timer_create,     sys_timer_create),    // 240
   GENXY(__NR_timer_settime,    sys_timer_settime),   // (timer_create+1)
   GENXY(__NR_timer_gettime,    sys_timer_gettime),   // (timer_create+2)
   GENX_(__NR_timer_getoverrun, sys_timer_getoverrun),// (timer_create+3)
   GENX_(__NR_timer_delete,     sys_timer_delete),    // (timer_create+4)

   GENX_(__NR_clock_settime,    sys_clock_settime),   // (timer_create+5)
   GENXY(__NR_clock_gettime,    sys_clock_gettime),   // (timer_create+6)
   GENXY(__NR_clock_getres,     sys_clock_getres),    // (timer_create+7)
   //   (__NR_clock_nanosleep,  sys_clock_nanosleep), // (timer_create+8)
   GENX_(__NR_swapcontext,	sys_ni_syscall),      // 249

   //   (__NR_tgkill,           sys_tgkill),          // 250
   GENX_(__NR_utimes,           sys_utimes),          // 251
   GENXY(__NR_statfs64,         sys_statfs64),        // 252
   GENXY(__NR_fstatfs64,        sys_fstatfs64),       // 253
   //   (__NR_fadvise64_64,     sys_fadvise64_64),    // 254

   //   (__NR_rtas,		sys_ni_syscall),      // 255 ppc/Linux (64b)
   //   (__NR_debug_setcontext, sys_ni_syscall),      // 256 ppc/Linux
   //   (__NR_vserver,		sys_ni_syscall),      // 257
   //   (__NR_remap_file_pages,	sys_ni_syscall),      // 258
   //   (__NR_mbind,            sys_mbind),           // 259

   //   (__NR_get_mempolicy,    sys_get_mempolicy),   // 260
   //   (__NR_set_mempolicy,    sys_set_mempolicy),   // 261
   GENXY(__NR_mq_open,          sys_mq_open),         // 262
   GENX_(__NR_mq_unlink,        sys_mq_unlink),       // (mq_open+1)
   GENX_(__NR_mq_timedsend,     sys_mq_timedsend),    // (mq_open+2)

   GENXY(__NR_mq_timedreceive,  sys_mq_timedreceive), // (mq_open+3)
   GENX_(__NR_mq_notify,        sys_mq_notify),       // (mq_open+4)
   GENXY(__NR_mq_getsetattr,    sys_mq_getsetattr),   // (mq_open+5)
   //   (__NR_sys_kexec_load,   sys_ni_syscall),      // 268
};

const UInt VGA_(syscall_table_size) = 
            sizeof(VGA_(syscall_table)) / sizeof(VGA_(syscall_table)[0]);

#undef GENX_
#undef GENXY
