/*
 * tools/vgcfgrestore.c
 *
 * Copyright (C) 1997 - 1999  Heinz Mauelshagen, Germany
 *
 * June,October 1997
 * May 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    12/10/1997  - removed vg_check_exist
 *                - added pv_read() to check againts overwrite of
 *                  physical volume belonging to a different volume group
 *                - added check for physical volume size
 *    08/11/1997  - added message for malloc errors
 *                - moved restore code to lib/vg_cfgrestore.c to use
 *                  it with lvm_tab code
 *                - added -n option to support test runs
 *    02/05/1998 - enhanced error checking
 *    02/05/1998 - used pv_check_new() instead of str-function
 *    16/05/1998 - added lvmtab checking
 *    21/05/1998 - implemented option -o
 *
 */

#include <lvm_user.h>

char *cmd = NULL;

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv)
{
   int c = 0;
   int opt_f = 0;
   int opt_l = 0;
   int opt_n = 0;
   int opt_o = 0;
   int opt_t = 0;
   int opt_v = 0;
   int index = -1;
   int pv_index = -1;
   int pv_index_old = -1;
   int ret = 0;
#ifdef DEBUG
   char *options = "df:h?ln:o:tv";
#else
   char *options = "f:h?ln:o:tv";
#endif
   char *pv_name = NULL;
   char *pv_name_old = NULL;
   char *vg_name = NULL;
   char *vg_conf_path = NULL;
   char vg_backup_path[NAME_LEN] = { 0, };
   pv_t *pv = NULL;
   vg_t vg;
   vg_t vg_write;

   cmd = basename ( argv[0]);

   memset ( &vg, 0, sizeof ( vg));
   memset ( &vg_write, 0, sizeof ( vg_write));

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_d++;
            break;
#endif

         case 'f':
            if ( opt_f> 0) {
               fprintf ( stderr, "%s -- f option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            vg_conf_path = optarg;
            opt_f++;
            break;

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Volume Group Configuration Restore\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-f VGConfPath]\n"
                     "\t[-l[l]]\n"
                     "\t[-n VolumeGroupName]\n"
                     "\t[-h/-?]\n"
                     "\t[-o OldPhysicalVolumePath]\n"
                     "\t[-t]\n"
                     "\t[-v]\n"
                     "\t[PhysicalVolumePath]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'l':
            if ( opt_l > 1) {
               fprintf ( stderr, "%s -- l option already given two times\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_l++;
            break;

         case 'n':
            if ( opt_n> 0) {
               fprintf ( stderr, "%s -- n option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            vg_name = optarg;
            if ( vg_check_name ( vg_name) < 0) {
               fprintf ( stderr, "%s -- invalid volume group name \"%s\"\n\n",
                                 cmd, vg_name);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_n++;
            break;

         case 'o':
            if ( opt_o> 0) {
               fprintf ( stderr, "%s -- o option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            pv_name_old = optarg;
            if ( pv_check_name ( pv_name_old) < 0) {
               fprintf ( stderr, "%s -- invalid physical volume path "
                                 "\"%s\"\n\n",
                                 cmd, pv_name_old);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_o++;
            break;

         case 't':
            if ( opt_t > 0) {
               fprintf ( stderr, "%s -- t option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_t++;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n\n",
                      cmd, c);
            return LVM_EINVALID_CMD_LINE;
      }
   }
  
   CMD_MINUS_CHK;

   if ( opt_f > 0 && opt_n == 0) {
      fprintf ( stderr, "%s -- please give option n with option f\n\n", cmd);
      return LVM_EINVALID_CMD_LINE;
   }

   if ( opt_l == 0) {
      if ( optind != argc - 1) {
         printf ( "%s -- please enter a physical volume name\n\n", cmd);
         return LVM_EVGCFGRESTORE_PV_MISSING;
      }
      pv_name = argv[optind];
      if ( pv_check_name ( pv_name) < 0) {
         fprintf ( stderr, "%s -- invalid physical volume \"%s\"\n\n",
                           cmd, pv_name);
         return LVM_EVGCFGRESTORE_PV_CHECK_NAME;
      }
      if ( opt_f == 0 && opt_n == 0) {
         fprintf ( stderr, "%s -- please enter a volume group name or a "
                           "volume group backup\n\n", cmd);
         return LVM_EINVALID_CMD_LINE;
      }
   }

   if ( opt_t == 0 && opt_l == 0 && vg_check_active ( vg_name) == TRUE) {
      fprintf ( stderr, "%s -- can't restore part of active "
                        "volume group \"%s\"\n\n",
                        cmd, vg_name);
      return LVM_EVGCFGRESTORE_VG_CHECK_ACTIVE;
   }

   if ( opt_f == 0) sprintf ( vg_backup_path,
                              "%s/%s.conf%c", VG_BACKUP_DIR, vg_name, 0);
   else strcpy ( vg_backup_path, vg_conf_path);


   LVM_LOCK ( 0);
   LVM_CHECK_IOP;
   lvm_dont_interrupt ( 0);

   if ( opt_v > 0) printf ( "%s -- restoring volume group \"%s\" from \"%s\"\n",
                            cmd, vg_name, vg_backup_path);
   if ( ( ret = vg_cfgrestore ( vg_name, vg_backup_path,
                                opt_v, &vg)) < 0) {
      if ( ret == -LVM_EVG_CFGRESTORE_OPEN) {
         fprintf ( stderr, "%s -- a backup for volume group "
                           "%s is not available\n\n",
                           cmd, vg_name);
      } else {
         fprintf ( stderr, "%s -- ERROR %d restoring volume group \"%s\"\n\n",
                           cmd, ret, vg_name);
      }
      return LVM_EVGCFGRESTORE_VG_CFGRESTORE;
   }

   if ( opt_n == 0) {
      vg_name = vg.vg_name;
      printf ( "%s -- this is a backup of volume group \"%s\"\n", cmd, vg_name);
   } else if ( strcmp ( vg_name, vg.vg_name) != 0) {
      fprintf ( stderr, "%s -- this is no backup of volume group \"%s\"\n\n",
                cmd, vg_name);
      return LVM_EVGCFGRESTORE_INVALID;
   }

   if ( opt_v > 0) printf ( "%s -- checking volume group consistency"
                            " of \"%s\"\n", cmd, vg_name);
   if ( ( ret = vg_check_consistency_with_pv_and_lv ( &vg)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d: volume group \"%s\" "
                        "is inconsistent\n\n",
                        cmd, ret, vg_name);
      return LVM_EVGCFGRESTORE_VG_CHECK_CONSISTENCY;
   } else if ( opt_t > 0) printf ( "%s -- backup of volume group \"%s\" "
                                   " is consistent\n",
                                   cmd, vg_name);

   pv_index = pv_index_old = -1;
   if ( opt_o > 0) {
      if ( pv_check_in_vg ( &vg, pv_name_old) == FALSE) {
         fprintf ( stderr, "%s -- physical volume \"%s\" doesn't belong "
                           "to volume group \"%s\" backup\n\n",
                           cmd, pv_name_old, vg_name);
         return LVM_EVGCFGRESTORE_PV_CHECK_IN_VG;
      } else pv_index_old = pv_get_index_by_name ( &vg, pv_name_old);
      if ( pv_check_in_vg ( &vg, pv_name) == TRUE) {
         fprintf ( stderr, "%s -- \"%s\" may not be an actual physical volume "
                           "of volume group \"%s\"\n\n", cmd, pv_name, vg_name);
         return LVM_EVGCFGRESTORE_PV_CHECK_IN_VG;
      } else pv_index = pv_get_index_by_name ( &vg, pv_name);
   } else {
      if ( pv_check_in_vg ( &vg, pv_name) == FALSE) {
         fprintf ( stderr, "%s -- physical volume \"%s\" doesn't belong "
                           "to volume group \"%s\"\n\n",
                           cmd, pv_name, vg_name);
         return LVM_EVGCFGRESTORE_PV_CHECK_IN_VG;
      } else pv_index = pv_get_index_by_name ( &vg, pv_name);
   }

   if ( opt_l > 0) {
      if ( opt_l == 1) vg_show ( &vg);
      else             vg_show_with_pv_and_lv ( &vg);
      printf ( "\n");
   } else {
      /* no test run */
      if ( opt_t == 0) {
         /* check given physical volume */
         if ( opt_v > 0) printf ( "%s -- reading physical volume \"%s\"\n",
                                  cmd, pv_name);
         if ( ( ret = pv_read ( pv_name, &pv, NULL)) < 0 &&
              ret != -LVM_EPV_READ_MD_DEVICE) {
            fprintf ( stderr, "%s -- ERROR %d reading physical "
                              "volume \"%s\"\n\n",
                              cmd, ret, pv_name);
            return LVM_EVGCFGRESTORE_PV_READ;
         }

         if ( opt_v > 0) printf ( "%s -- checking for new physical "
                                  "volume \"%s\"\n",
                                  cmd, pv_name);
         if ( pv_check_new ( pv) == FALSE) {
            if ( vg_check_name ( pv->vg_name) == 0 &&
                 strcmp ( pv->vg_name, vg_name) != 0) {
               fprintf ( stderr, "%s -- physical volume \"%s\" belongs "
                                 "to volume group \"%s\"\n\n",
                                 cmd, pv->pv_name, pv->vg_name);
               return LVM_EVGCFGRESTORE_PV_CHECK_IN_VG;
            }
         }

         if ( pv_index_old > -1) index = pv_index_old;
         else                    index = pv_index;

         if ( pv_get_size ( pv_name, NULL) !=  vg.pv[index]->pv_size) {
            fprintf ( stderr, "%s -- size of physical volume \"%s\" differs "
                              "from backup\n\n", cmd, pv_name);
            return LVM_EVGCFGRESTORE_PV_GET_SIZE;
         }

         memcpy ( &vg_write, &vg, sizeof ( vg));
         vg_write.pv[0] = vg.pv[index];
         vg_write.pv[1] = NULL;

         /* store VGDA from backup on physical volume */
         if ( opt_v) printf ( "%s -- writing VGDA of \"%s\" to physical "
                              "volume \"%s\"\n",
                              cmd, vg_name, vg_write.pv[0]->pv_name);
         if ( ( ret = vg_write_with_pv_and_lv ( &vg_write)) < 0) {
            fprintf ( stderr, "%s -- ERROR %d storing VGDA "
                              "of \"%s\" on disks\n\n", 
                              cmd, ret, vg_name);
            return LVM_EVGCFGRESTORE_VG_WRITE;
         }
   
         if ( opt_v) printf ( "%s -- removing any special files "
                              "for \"%s\"\n", cmd, vg_name);
         vg_remove_dir_and_group_and_nodes ( vg_name);
   
         if ( ( ret = vg_create_dir_and_group_and_nodes ( &vg, opt_v)) < 0) {
            fprintf ( stderr, "%s -- %d errors creating volume group "
                              "directory and special files\n\n",
                              cmd, -ret);
            return LVM_EVGCFGRESTORE_VG_REMOVE_DIR;
         }
   
         if ( opt_v > 0) printf ( "%s -- inserting volume group \"%s\" "
                                  "into \"%s\"\n",
                                  cmd, vg_name, LVMTAB);
         if ( ( ret = lvm_tab_vg_insert ( vg_name)) < 0 &&
              ret != -LVM_ELVM_TAB_VG_INSERT_VG_EXISTS) {
            fprintf ( stderr, "%s -- ERROR %d inserting volume group into "
                              "\"%s\"\n\n",
                              cmd, ret, LVMTAB);
            return LVM_EVGCFGRESTORE_LVM_TAB_VG_INSERT;
         }
         
         if ( ( ret = vg_cfgbackup ( vg_name, LVMTAB_DIR,
                                     opt_v, &vg)) < 0) {
            fprintf ( stderr, "%s -- ERROR %d writing lvmtab\n\n",
                              cmd, ret);
            return LVM_EVGCFGRESTORE_VG_CFGBACKUP;
         }

         printf ( "%s -- VGDA for \"%s\" successfully restored "
                  "to physical volume \"%s\"\n",
                  cmd, vg_name, pv_name);
         if ( opt_f > 0)
            printf ( "%s -- you may not have an actual backup of "
                     "restored volume group \"%s\"\n", cmd, vg_name);
      } else printf ( "%s -- test run for volume group \"%s\" end\n",
                      cmd, vg_name);
   }

   printf ( "\n");

   lvm_interrupt ();
   LVM_UNLOCK ( 0);

   return 0;
}
