// -*- C++ -*-
// Copyright (C) 2000 Red Hat, Inc.
// Example/test code using Inti signals
/*
  The following license applies to this example code, but not to
  the Inti library itself:
  
    Permission is hereby granted, free of charge, to any person obtaining
    a copy of this software and associated documentation files (the
    "Software"), to deal in the Software without restriction, including
    without limitation the rights to use, copy, modify, merge, publish,
    distribute, sublicense, and/or sell copies of the Software, and to
    permit persons to whom the Software is furnished to do so, subject to
    the following conditions:

    The above copyright notice and this permission notice shall be
    included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
    NONINFRINGEMENT.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
    DISTRIBUTOR OF THE SOFTWARE BE LIABLE FOR ANY CLAIM, DAMAGES OR
    OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
    OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
    OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <inti/signal-system.h>
#include <string>
#include <iostream>

using namespace Inti;

class Object : public SignalEmitter
{
public:
  Object ()
  {
  }

  ~Object ()
  {

  }

  Connection connect_whoop (Slot0<void> * s)
  {
    return sigwhoop.connect (this, s);
  }

  Connection connect_blah (Slot0<int> * s)
  {
    return sigblah.connect (this, s);
  }
  
  void whoop ();
  
  int blah ();
  
protected:
  static Signal0<void> sigwhoop;
  static Signal0<int> sigblah;
  
};

Signal0<void> Object::sigwhoop;
Signal0<int> Object::sigblah;

void
Object::whoop ()
{
  sigwhoop.emit (this);
}

int
Object::blah ()
{
  return sigblah.emit (this);
}

class A
{
public:
  void stuff ()
  {
    int i;
    i = 0;
    while (i < 10)
      ++i;
    //    cout << "A::stuff called" << endl;
  }

  void nothing ()
  {

    //    cout << "A::nothing called" << endl;
  }
  
  int frobate ()
  {
    //    cout << "A::frobate called" << endl;
    string s = "String!";
    string t = s;
    return 10;
  }
};


#include <time.h>
#include <stdio.h>
#include <vector>

void do_timings (int C, int N)
{
  Object o;
  A a;
  int i;
  clock_t start;
  clock_t end;
  vector<Connection> cns;

  printf ("===\n");
  
  start = clock ();
  i = 0;
  while (i < C)
    {
      cns.push_back (o.connect_whoop (Inti::slot (&a, &A::stuff)));
      cns.push_back (o.connect_blah (Inti::slot (&a, &A::frobate)));
      ++i;
    }
  end = clock ();

  printf (" connected %d handlers to each of 2 signals: %g seconds\n",
          C, ((double)end - (double)start)/CLOCKS_PER_SEC);
  
  start = clock ();
  i = 0;
  while (i < N)
    {
      o.whoop ();
      o.blah ();
      o.whoop ();
      ++i;
    }
  end = clock ();

  double seconds = ((double)end - (double)start)/CLOCKS_PER_SEC;
  printf (" %d emits, %d handlers connected: %g secs (%f seconds per emit)\n",
          N*3, C, seconds,
          seconds/(N*3));

  start = clock ();
  vector<Connection>::iterator iter = cns.begin ();
  vector<Connection>::iterator stop = cns.end ();
  while (iter != stop)
    {
      (*iter).disconnect ();
      ++iter;
    }
  end = clock ();

  seconds = ((double)end - (double)start)/CLOCKS_PER_SEC;
  printf (" %d handlers disconnected: %g secs (%f seconds per disconnect)\n",
          C, seconds,
          seconds/C);
  
  
}

void do_nothing_timings (int C, int N)
{
  Object o;
  A a;
  int i;
  clock_t start;
  clock_t end;  

  printf ("===\n");
  
  start = clock ();
  i = 0;
  while (i < C)
    {
      o.connect_whoop (Inti::slot (&a, &A::nothing));
      ++i;
    }
  end = clock ();

  printf (" connected %d empty handlers to one signal: %g seconds\n",
          C, ((double)end - (double)start)/CLOCKS_PER_SEC);
  
  start = clock ();
  i = 0;
  while (i < N)
    {
      o.whoop ();
      o.whoop ();
      o.whoop ();
      ++i;
    }
  end = clock ();

  double seconds = ((double)end - (double)start)/CLOCKS_PER_SEC;
  printf (" %d emits, %d empty handlers/emit: %g secs (%f seconds per emit)\n",
          N*3, C, seconds,
          seconds/(N*3));
}
  
int
main (int argc, char **argv)
{
  do_timings (1, 10000);
  do_timings (10, 10000);
  do_timings (100, 10000);
  do_timings (200, 7500);
  do_timings (500, 2000);
  do_timings (1000, 1000);
  do_timings (2000, 500);
  
  do_nothing_timings (1, 10000);
  do_nothing_timings (10, 10000);
  do_nothing_timings (100, 10000);
  do_nothing_timings (200, 7500);
  do_nothing_timings (500, 2000);
  do_nothing_timings (1000, 1000);
  do_nothing_timings (2000, 500);

  /*
  If everything is working properly, these timings should show:

    a) emission time is pretty much a function of how much user
       callback code an emission is executing; Inti overhead
       is not.

    b) connection time is basically 0 (this affects your application
       startup speed). A connection is pretty much just an
       allocation or two, and then an array assignment. Super-fast.

    c) disconnect time is O(n) if you disconnect specific handlers,
       but this doesn't even matter because for realistic values
       of n it's still very fast, and because almost always
       you allow all your handlers to die along with the
       emitter object. The emitter object's destructor
       can destroy all handlers in less time than it would take
       to do an emission.

    d) C++ signal systems are a lot faster than the GTK+ native
       signal system.
     
  */
  
  return 0;
}

