/* Generate an index file for an existing GNATS database.
   Copyright (C) 1993, 1995 Free Software Foundation, Inc.
   Contributed by Brendan Kehoe (brendan@cygnus.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"
#include "gnats-dirs.h"

/* The name this program was run with.  */
char *program_name;

/* If 1, we're running the daemon.  */
int is_daemon = 0;

/* If 1, we're running a network client.  */
int is_client = 0;

/* Where the list of categories should come from.  */
char *catfile = (char *)NULL;

/* Where the results of query-pr should go.  */
FILE *outfile = stdout;

/* If TRUE, we should sort the index list by number.  */
int sort_numerical = FALSE;

struct option long_options[] =
{
  {"directory", 1, NULL, 'd'},
  {"catfile", 1, NULL, 'c'},
  {"outfile", 1, NULL, 'o'},
  {"numerical", 0, NULL, 'n'},
  {"help", 0, NULL, 'h'},
  {"version", 0, NULL, 'V'},
  {NULL, 0, NULL, 0}
};

typedef struct category_list
{
  Category *c;
  struct category_list *next;
} Categories;

typedef struct entry
{
  unsigned int number;
  char *string;
} Entry;

#define MAX_ENTRIES 1024
Entry *entries;
int max_entries = MAX_ENTRIES, num_entries = 0;

void usage (), version ();

int
entry_cmp (e1, e2)
     Entry *e1, *e2;
{
  return e1->number - e2->number;
}

/* For a given category C, go into its directory and either emit an
   index entry for each file in it, or swallow its index entry so we
   can sort it later.  */
void
do_category (c)
     char *c;
{
  register DIR *d;
  register struct dirent *next;
  FILE *fp;
  int len = strlen (gnats_root) + 1 + strlen (c) + 2;
  /* Allocate for a PR with 8 digits.  */
  char *path = (char *) xmalloc (len + 9);
  char *p;
  char *line = (char *) xmalloc (STR_MAXLONG);

  errno = 0;
  if (chdir (gnats_root) < 0)
    {
      fprintf (stderr, "%s: can't read the database directory %s: %s\n",
	       program_name, gnats_root, strerror (errno));
      exit (3);
    }

  if (c == NULL || *c == '\0')
    {
      xfree (line);
      xfree (path);
      return;
    }

  errno = 0;
  d = opendir (c);
  if (! d)
    {
      fprintf (stderr, "%s: can't open the category directory %s/%s: %s\n",
	       program_name, gnats_root, c, strerror (errno));
      xfree (line);
      xfree (path);
      return;
    }

  sprintf (path, "%s/%s/", gnats_root, c);

  /* Process each file in the directory; ignore files that have periods
     in their names; either they're the . and .. dirs, or they're a
     lock file (1234.lock).  */
  while ((next = readdir (d)))
    if (strchr (next->d_name, '.') == NULL)
      {
	p = path + len - 1;
	strcat (p, next->d_name);

	fp = fopen (path, "r");
	if (fp == (FILE *) NULL)
	  {
	    fprintf (stderr, "%s: couldn't read pr %s: %s\n",
		     program_name, path, strerror (errno));

	    *p = '\0';
	    continue;
	  }

	if (read_header (fp) < 0)
	  {
	    fprintf (stderr, "%s: error reading pr %s\n", program_name, path);
	    *p = '\0';
	    continue;
	  }

	*p = '\0';
	read_pr (fp, 1);
	fclose (fp);

	create_index_entry (line);

	if (sort_numerical == TRUE)
	  {
	    if (num_entries == max_entries - 1)
	      {
		max_entries += MAX_ENTRIES;
		entries = (Entry *) xrealloc ((char *) entries,
					       max_entries * sizeof (Entry));
	      }

	    entries[num_entries].number = atoi (field_value (NUMBER));
	    entries[num_entries].string = (char *) strdup (line);
	    num_entries++;
	  }
	else
	  fprintf (outfile, "%s", line);
      }

  closedir (d);
  xfree (line);
  xfree (path);
}

/* Get the next entry in the categories file.  */
Categories *
next_category (fp)
     FILE *fp;
{
  char *start, *end, *b;
  char *buf;
  Categories *p;
  Category *c;
  static int lineno = 0;

  if (feof (fp))
    return (Categories *) NULL;

  buf = (char *) xmalloc (STR_MAX);
  p = (Categories *) xmalloc (sizeof (Categories));
  c = p->c = (Category *) xmalloc (sizeof (Category));
  p->next = NULL;

  b = buf;

  while (1)
    {
      do
	{
	  if (fgets (buf, STR_MAX, fp) == NULL)
	    goto no_entry;
	  lineno++;
	}
      while (buf[0] == '#' || buf[0] == '\n' || buf[0] == ' ');

      start = b;
      end = strchr (start, ':');
      if (end == NULL)
	goto retry;
      *end = '\0';
      c->key = start;

      start = end + 1;
      end = strchr (start, ':');
      if (end == NULL)
	goto retry;
      *end = '\0';
      c->fullname = start;

      start = end + 1;
      end = strchr (start, ':');
      if (end == NULL)
	goto retry;
      *end = '\0';
      c->person = start;

      start = end + 1;
      end = strchr (start, '\n');
      if (end == NULL)
	goto retry;
      *end = '\0';
      c->notify = start;

      return p;
retry:
      fprintf (stderr, "%s: malformed category line %d\n", program_name, lineno);
    }

no_entry:
  xfree ((char *)p->c);
  xfree ((char *)p);
  xfree (buf);
  return (Categories *) NULL;
}

/* Return a linked list of categories.  */
Categories *
get_categories ()
{
  char *path = (char *) xmalloc (PATH_MAX);
  FILE *fp;
  Categories *cat_list = (Categories *) NULL;
  Categories *cat_list_end = (Categories *) NULL;
  Categories *c;

  if (! catfile)
    sprintf (path, "%s/gnats-adm/%s", gnats_root, CATEGORIES);
  else
    path = catfile;

  errno = 0;
  fp = fopen (path, "r");
  if (fp == (FILE *) NULL)
    {
      fprintf (stderr, "%s: couldn't read the categories file %s: %s\n",
	       program_name, path, strerror (errno));
      exit (3);
    }

  while ((c = next_category (fp)))
    {
      if (cat_list == NULL)
	cat_list = cat_list_end = c;
      else
	{
	  cat_list_end->next = c;
	  cat_list_end = c;
	}

      c->next = NULL;
    }

  fclose (fp);
  xfree (path);

  return cat_list;
}
  
int
main (argc, argv)
     int argc;
     char **argv;
{
  int optc, i;
  Categories *clist, *c;

  program_name = (char *) basename (argv[0]);

  while ((optc = getopt_long (argc, argv, "o:c:hd:nV",
			      long_options, (int *) 0)) != EOF)
    {
      switch (optc)
	{
	case 'd':
	  gnats_root = optarg;
	  break;

	case 'o':
	  outfile = fopen (optarg, "w+");
	  if (outfile == (FILE *) NULL)
	    {
	      fprintf (stderr, "%s: can't write to %s: %s\n", program_name,
		       optarg, strerror (errno));
	      exit (3);
	    }
	  break;

	case 'c':
	  catfile = (char *) strdup (optarg);
	  break;

	case 'n':
	  sort_numerical = TRUE;
	  break;

	case 'V':
	  version ();
	  exit (0);
	  break;

	case 'h':
	  usage (0);
	  break;

	default:
	  usage (1);
	}
    }

  init_gnats (program_name);
  umask (022);

  if (sort_numerical)
    entries = (Entry *) xmalloc (max_entries * sizeof (Entry));

  clist = get_categories ();

  for (c = clist ; c ; c = c->next)
    do_category (c->c->key);

  if (sort_numerical)
    {
      qsort (entries, num_entries, sizeof (Entry), entry_cmp);
      for (i = 0; i < num_entries; i++)
	fprintf (outfile, "%s", entries[i].string);
    }

  fclose (outfile);
  free_db_conf();
  exit (0);
}

void
usage (s)
     int s;
{
  fprintf (stderr, "\
Usage: %s [-hnV] [-d directory] [-o outfile] [-c filename ]\n\
	  [--directory=directory] [--catfile=filename]\n\
          [--outfile=filename] [--numerical] [--version] [--help]\n",
	   program_name);
  exit (s);
}

void
version ()
{
  printf ("gen-index %s\n", version_string);
}
