/* Remote GNATS server.
   Copyright (C) 1994, 95, 96, 1997 Free Software Foundation, Inc.
   Contributed by Brendan Kehoe (brendan@cygnus.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"
#include "gnatsd.h"
#include "query.h"
#include "db_conf.h"

char myname[MAXHOSTNAMELEN];

char *program_name;

/* If 1, we're running the daemon.  */
int is_daemon = 1;

/* The name and IP address of the host talking with us and their access level.  */
char *current_host;
char *current_addr;
int   host_access;
int   user_access;

/* Whether or not we've made sure they are who they say they are.  */
int host_verified = 0;

/* Whether or not they need to execute CHDB; reset to 0 when they do */
int require_db    = 0;

struct option long_options[] =
{
  {"directory", 1, NULL, 'd'},
  {"not-inetd", 0, NULL, 'n'},
  {"require-db", 0, NULL, 'r'},
  {"version", 0, NULL, 'V'},
  {"help", 0, NULL, 'h'},
  {NULL, 0, NULL, 0}
};

/* The search state we use.  */
Index *search;

/* Whether we should emit a linefeed and a newline.  */
extern int doret;

void usage (), version ();

#ifdef HAVE_KERBEROS
/* Non-zero if the client has used Kerberos authentication.  */
int client_authorized;
#endif

/* Non-zero if the host has to use Kerberos.  */
int kerberos_host;

typedef struct _command
{
  char *s;
  void (*fn)();
  int flags;
} Command;

static Command cmds[] =
{
  /* Various unrestricted commands */
  { "HELO", GNATS_helo, 0 },
  { "QUIT", GNATS_quit, 0 },
  { "AUTH", GNATS_auth, 0 },
  { "USER", GNATS_user, 0 },
  { "CHDB", GNATS_chdb, 0 },
  { "VDAT", GNATS_vdat, FLAG_ONE_ARG },

  /* Set query constraints (no need to restrict) */
  { "RSET", GNATS_rset, 0 },
  { "NOCL", GNATS_nocl, 0 },
  { "RESP", GNATS_resp, 0 },
  { "CATG", GNATS_catg, 0 },
  { "SYNP", GNATS_synp, FLAG_ONE_ARG },
  { "STAT", GNATS_stat, 0 },
  { "CONF", GNATS_conf, 0 },
  { "SVTY", GNATS_svty, 0 },
  { "ORIG", GNATS_orig, FLAG_ONE_ARG },
  { "RLSE", GNATS_rlse, FLAG_ONE_ARG },
  { "CLSS", GNATS_clss, FLAG_ONE_ARG },
  { "PRIO", GNATS_prio, 0 },
  { "SUBM", GNATS_subm, FLAG_ONE_ARG },
  { "TEXT", GNATS_text, FLAG_ONE_ARG },
  { "MTXT", GNATS_mtxt, FLAG_ONE_ARG },
  { "ABFR", GNATS_abfr, FLAG_ONE_ARG },
  { "ARAF", GNATS_araf, FLAG_ONE_ARG },
  { "CBFR", GNATS_cbfr, FLAG_ONE_ARG },
  { "CAFT", GNATS_caft, FLAG_ONE_ARG },
  { "MBFR", GNATS_mbfr, FLAG_ONE_ARG },
  { "MAFT", GNATS_maft, FLAG_ONE_ARG },
#ifdef GNATS_RELEASE_BASED
  { "BFOR", GNATS_bfor, FLAG_ONE_ARG },
  { "AFTR", GNATS_aftr, FLAG_ONE_ARG },
  { "QRTR", GNATS_qrtr, FLAG_ONE_ARG },
  { "KYWD", GNATS_kywd, FLAG_ONE_ARG },
#endif

  /* Query a PR. */
  { "QURY", GNATS_qury, ACCESS_VIEW },
  { "SUMM", GNATS_summ, ACCESS_VIEW },
  { "FULL", GNATS_full, ACCESS_VIEW },
  { "SQLF", GNATS_sqlf, ACCESS_VIEW },
  { "SQL2", GNATS_sql2, ACCESS_VIEW },

  /* List GNATS internals */
  { "LCAT", GNATS_lcat, ACCESS_VIEW },
  { "LCFG", GNATS_lcfg, ACCESS_VIEW },
  { "LCLA", GNATS_lcla, ACCESS_VIEW },
  { "LRES", GNATS_lres, ACCESS_VIEW },
  { "LSUB", GNATS_lsub, ACCESS_VIEW },
  { "LSTA", GNATS_lsta, ACCESS_VIEW },
  { "DBLS", GNATS_dbls, ACCESS_VIEW },
  { "DBLA", GNATS_dbla, ACCESS_VIEW },
  { "HELP", GNATS_help, ACCESS_VIEW },

  /* Return the people that should be mailed about a given PR, category or submitter. */
  { "MLPR", GNATS_mlpr, ACCESS_VIEW },
  { "MLCT", GNATS_mlct, ACCESS_VIEW },
  { "MLSU", GNATS_mlsu, ACCESS_VIEW },

  /* Return the type of submitter of a given name.  */
  { "TYPE", GNATS_type, ACCESS_VIEW },

  /* Edit a PR. */
  { "LOCK", GNATS_lock, ACCESS_EDIT },
  { "EDIT", GNATS_edit, ACCESS_EDIT },
  { "UNLK", GNATS_unlk, ACCESS_EDIT },

  /* Stop and start the database.  */
  { "LKDB", GNATS_lkdb, ACCESS_EDIT },
  { "UNDB", GNATS_undb, ACCESS_EDIT },

  { NULL, NULL },
};

int
match (line, p)
     char *line;
     char *p;
{
  int matched;

  for ( ; *p; line++, p++)
    {
      if (*line == '\0' && *p != '*')
	return 0;
      switch (*p)
	{
	case '\\':
	  /* Literal match with following character. */
	  p++;
	default:
	  if (tolower (*line) != tolower (*p))
	    return 0;
	  continue;
	case '?':
	  /* Match anything. */
	  continue;
	case '*':
	  while (*++p == '*')
	    /* Consecutive stars act just like one. */
	    continue;
	  if (*p == '\0')
	    /* Trailing star matches everything. */
	    return 1;
	  while (*line)
	    if ((matched = match (line++, p)) != 0)
	      return matched;
	  return 0;
	}
    }

  return 1;
}

char *
get_name (host)
     struct in_addr *host;
{
  char *buf;
  int i;
  struct hostent *hp;
#ifdef h_addr
  char **pp;
#endif

  hp = gethostbyaddr ((char *) host, sizeof (host), AF_INET);
  if (hp == NULL)
    return NULL;
  i = strlen (hp->h_name);
  buf = (char *) xmalloc (i + 1);
  strcpy (buf, hp->h_name);
  hp = gethostbyname (buf);
  if (hp == NULL)
    return NULL;

#ifdef h_addr
  for (pp = hp->h_addr_list; *pp; pp++)
    if (memcmp ((char *) &host->s_addr, (char *) *pp, hp->h_length) == 0)
      break;
  if (*pp == NULL)
    return NULL;
#else
  if (memcmp ((char *) &host->s_addr, (char *) hp->h_addr, hp->h_length) != 0)
    return NULL;
#endif

  return buf;
}

void
set_confidential_access ()
{
     if (search->confidential)
       {
         xfree (search->confidential);
	 search->confidential = (char *) NULL;
       }
     if (user_access < ACCESS_VIEWCONF)
       {
         searching = 1;
	 search->confidential = (char *) strdup ("no");
       }
}

int
access_level (key)
     char *key;
{
     int access;
     if (!strcasecmp (key, "none"))
       access = ACCESS_NONE;
     else if (!strcasecmp (key, "view"))
       access = ACCESS_VIEW;
     else if (!strcasecmp (key, "viewconf"))
       access = ACCESS_VIEWCONF;
     else if (!strcasecmp (key, "edit") || !strcasecmp (key, ""))
       access = ACCESS_EDIT;
     else if (!strcasecmp (key, "deny"))
       access = ACCESS_DENY;
     else if (!strcasecmp (key, "admin"))
       access = ACCESS_ADMIN;
     else
       access = ACCESS_NONE;
     return (access);
}

char *
access_level_str (access)
     int access;
{
     switch (access) {
     case ACCESS_NONE:     return ("none");
          break;
     case ACCESS_VIEW:     return ("view");
          break;
     case ACCESS_VIEWCONF: return ("viewconf");
          break;
     case ACCESS_EDIT:     return ("edit");
          break;
     case ACCESS_DENY:     return ("deny");
          break;
     case ACCESS_ADMIN:    return ("admin");
          break;
     default: return ("none");
     }
}

/* See if we know about this host.  If we don't, then run with it.  */
int
allowed (host, ipaddr, database, access)
     char *host;
     char *ipaddr;
     char *database;
     int  *access;
{
  FILE *acc;
  char buffer[BUFSIZ], *x;
  char *fields[3];
  int i;
  int found = 0;
  char *filename = (char *) alloca (PATH_MAX);

  /* The format of the file is:
        sitename:access:
     where `sitename' is the hostname allowed; `access' is the access
     level, and the third field is undefined for now, but may be the
     field to control what PRs, categories, or submitter-id PRs are
     allowed, or whatever. */
  
  sprintf (filename, "%s/gnats-adm/%s", database, ACCESS_FILE);
  acc = fopen (filename, "r");
  if (acc == (FILE *)NULL)
    {
      syslog (LOG_ERR, "%s for %s cannot open %m", host, filename);
      return 0;
    }

  while (fgets (buffer, sizeof (buffer), acc) != NULL)
    {
      x = strchr (buffer, '\n');
      if (x != NULL)
	*x = '\0';
      x = strchr (buffer, '#');
      if (x != NULL)
	*x = '\0';
      if (buffer[0] == '\0')
	continue;

      for (fields[0] = buffer, i = 0, x = buffer; *x && i < 3; x++)
	if (*x == ':')
	  {
	    *x = '\0';
	    fields[++i] = x + 1;
	  }
      /* Is there a mistake on the setup of this line?  */
      if (i != 2)
	continue;

      /* check hostname and IP */
      if (! (match (host, fields[0]) || (ipaddr && match (ipaddr, fields[0]))))
	continue;
      found = 1;
      *access = access_level (fields[1]);
      if (*access == ACCESS_DENY)
           found = 0;
      break;
    }

  fclose (acc);

#ifdef HAVE_KERBEROS
  if (! found) {
    *access = ACCESS_EDIT;
    found = kerberos_host = 1;
  }
#endif

  return found;
}

/* Get the access lever for this user. */
int
get_user_access (database, filename, user, passwd)
     char *database;
     char *filename;
     char *user;
     char *passwd;
{
  FILE *acc;
  char buffer[BUFSIZ], *x;
  char *fields[4], *db;
  int i, access;
  int found = 0;

  /* The format of the file is:
        userid:passwd:access:database */

  acc = fopen (filename, "r");
  if (acc == (FILE *)NULL)
    {
      return (-1);
    }

  access = -1;
  
  while (fgets (buffer, sizeof (buffer), acc) != NULL)
    {
      x = strchr (buffer, '\n');
      if (x != NULL)
	*x = '\0';
      x = strchr (buffer, '#');
      if (x != NULL)
	*x = '\0';
      if (buffer[0] == '\0')
	continue;

      for (fields[0] = buffer, i = 0, x = buffer; *x && i < 3; x++)
	if (*x == ':')
	  {
	    *x = '\0';
	    fields[++i] = x + 1;
	  }
      
      /* Is there a mistake on the setup of this line? (must have 3 or 4 fields) */
      if (i < 2 || i > 3)
	continue;

      /* check userid */
      if (!match (user, fields[0]))
	continue;

      /* check passwd */
      if (!match (passwd, fields[1]))
        {
          access = ACCESS_NONE;
          break;
        }

      /* check for optional database alias, but just in the global access file */
      if (i > 2 && !strcmp (filename, GLOBAL_ACCESS_FILE))
        {
	  char alias[STR_MAX];
	  char *l, *l2;
          char **aliases = NULL;
          int  count = 0;

          get_conf_aliases(&aliases, &count);
          
          /* Compare all given aliases against all the aliases for the database in use */
          for (l2 = fields[3], l = get_next_field(l2, alias, ',');
               l || l2;
               (l2 = l) && (l = get_next_field(l, alias, ',')))
            {
              for (i = 0; i < count; i++)
                {
                  if (!strcmp (database, lookup_conf_alias(aliases[i])) &&
                      match (aliases[i], alias))
                    {
                      found = 1;
                      break;
                    }
                }
              if (found)
                break;
            }
          for (i = 0; i < count; i++)
            xfree (aliases[i]);
          xfree((char*) aliases);
          if (!found)
            continue;
        }
      
      access = access_level (fields[2]);

      break;
    }

  fclose (acc);

  return access;
}

void
start_connection ()
{
  struct sockaddr_in s;
  int len, access;
  len = sizeof (s);
  if (getpeername (0, (struct sockaddr *)&s, &len) < 0)
    {
      if (! isatty (0))
	{
	  syslog (LOG_ERR, "%s: can't get peername %m", "?");
	  printf ("%d Why can't I figure out your name?  Later.\r\n",
		  CODE_NO_ACCESS);
	  exit (1);
	}
      current_host = "stdin";
      current_addr = current_host;
    }
  else
    {
      if (s.sin_family != AF_INET)
	{
	  syslog (LOG_ERR, "%s: bad address family %ld",
		  "?", (long) s.sin_family);
	  printf ("%d You're a member of a bad address family.  Later.\r\n",
		  CODE_NO_ACCESS);
	  exit (1);
	}
      current_addr = (char *) inet_ntoa (s.sin_addr);
      current_host = get_name (&s.sin_addr);
      if (! current_host) {
        current_host = current_addr;
      }
    }

  if (!allowed (current_host, current_addr, gnats_root, &access))
    {
      syslog (LOG_NOTICE, "%s (%s) not allowed access", current_host, current_addr);
      printf ("%d You are not on the host access list: %s (%s).\r\n",
              CODE_NO_ACCESS, current_host, current_addr);
      exit (1);
    }
  host_access = user_access = access;

  syslog (LOG_INFO, "%s connect", current_host);
}


#include <setjmp.h>
jmp_buf env;

void /* FIXME */
read_timeout(sig)
     int sig;
{
     longjmp(env,1);
}

MsgType
get_line (start, size, timeout)
     char *start;
     int size;
     int timeout;
{
  static int count;
  static char buffer[BUFSIZ];
  static char *bp;
  register char *p;
  register char *end;
  struct timeval t;
  fd_set rmask;
  int i;
  char c;

  for (p = start, end = &start[size - 1]; ; ) {
    if (count == 0) {
      /* Fill the buffer. */
    Again:
      FD_ZERO(&rmask);
      FD_SET(0, &rmask);
      t.tv_sec = timeout;
      t.tv_usec = 0;
      i = select (0 + 1, &rmask, (fd_set *)NULL, (fd_set *)NULL, &t);
      if (i < 0) {
	if (errno == EINTR)
	  goto Again;
	return PR_timeout;
      }
      if (i == 0 || !FD_ISSET(0, &rmask))
	return PR_timeout;
      count = read (0, buffer, sizeof buffer);
      if (count < 0) {
	return PR_timeout;
      }
      if (count == 0)
	return PR_eof;
      bp = buffer;
    }

    /* Process next character. */
    count--;
    c = *bp++;
    if (c == '\n')
      break;
    if (p < end)
      *p++ = c;
  }

  /* If last two characters are \r\n, kill the \r as well as the \n. */
  if (p > start && p < end && p[-1] == '\r')
    p--;
  *p = '\0';
  return p == end ? PR_long : PR_ok;
}

int
Argify(line, argvp, cp)
    char		*line;
    char		***argvp;
    Command		**cp;
{
    register char	**argv;
    register char	*p;
    register int	i;

    if (*argvp != NULL) {
      xfree (*argvp[0]);
      xfree ((char *)*argvp);
    }

    /*  Copy the line, which we will split up. */
    while (*line == ' ' || *line == '\n')
	line++;
    i = strlen(line);
    p = (char *) xmalloc (sizeof (char) * (i + 1));
    strcpy(p, line);

    /* Allocate worst-case amount of space. */
    *argvp = argv = (char **) xmalloc (sizeof (char *) * (i + 2));
    argv[0] = p;
    if ((p = strpbrk(p, " \n")))
      *p++ = '\0';
    for (*cp = cmds; (*cp)->s; (*cp)++) {
      if (strncasecmp ((*cp)->s, argv[0], strlen ((*cp)->s)) == 0)
	break;
    }
    if (! (*cp)->s)
      *cp = NULL;
  
    if (! p) {
      argv[1] = NULL;
      return 1;
    }

    if (*cp && (((*cp)->flags & FLAG_MASK) == FLAG_ONE_ARG)) {
      while (*p && (*p == ' ' || *p == '\n'))
	p++;
      argv[1] = p;
      argv[2] = NULL;
      return 2;
    }
      
    for (argv++; *p; ) {
	/* Mark start of this word, find its end. */
	for (*argv++ = p; *p && *p != ' ' && *p != '\n'; )
	    p++;
	if (*p == '\0')
	    break;

	/* Nip off word, skip whitespace. */
	for (*p++ = '\0'; *p == ' ' || *p == '\n'; )
	    p++;
    }
    *argv = NULL;
    return argv - *argvp;
}

int
main (argc, argv)
     int argc;
     char **argv;
{
  char buff[512];
  struct hostent *hp;
  char **av;
  int ac;
  Command *cp;
  MsgType r;
  int optc;
  int not_inetd  = 0;
  int access, access_required;

  program_name = basename (argv[0]);

  search = (Index *) xmalloc (sizeof (Index));
  memset (search, 0, sizeof (Index));

  while ((optc = getopt_long (argc, argv, "d:nrVh",
			      long_options, (int *) 0)) != EOF)
    {
      switch (optc)
	{
	case 'd':
	  gnats_root = optarg;
	  break;

	case 'n':
	  not_inetd = 1;
	  break;

	case 'r':
	  require_db = 1;
	  break;

	case 'V':
	  version ();
	  exit (0);
	  break;

	case 'h':
	default:
	  usage ();
	}
    }

  configure ();
  init_gnats ();
  local_chdb(&gnats_root, program_name);

  umask (022);
  re_set_syntax ((RE_SYNTAX_POSIX_EXTENDED | RE_BK_PLUS_QM) & ~RE_DOT_NEWLINE);

  /* Don't emit any errors from get_pr.  */
  quiet = 1;
#ifdef HAVE_KERBEROS
  kerberos_host = client_authorized = 0;
#endif
  if (not_inetd)
    {
      host_verified = 1;
#ifdef HAVE_KERBEROS
      client_authorized = 1;
#endif
    }

  /* chdir */

  fflush (stderr);
  fflush (stdout);
  /* close file descriptors */

  closelog ();
#ifdef LOG_DAEMON
  openlog ("gnatsd", (LOG_PID|LOG_CONS), LOG_DAEMON);
#else
  openlog ("gnatsd", LOG_PID);
#endif

  if (gethostname (myname, MAXHOSTNAMELEN) != 0)
    {
      fprintf (stderr, "%s: cannot find out the name of this host\n",
	       program_name);
      exit (1);
    }

  /* Now see if we can get a FQDN for this host.  */
  hp = (struct hostent *) gethostbyname (myname);
  if (hp && (strlen (hp->h_name) > strlen (myname)))
    strncpy (myname, hp->h_name, MAXHOSTNAMELEN);

  /* Don't pay attention to SIGPIPE; read will give us the problem
     if it happens.  */
  signal (SIGPIPE, SIG_IGN);

  /* We want to emit both a linefeed and a newline for each line.  */
  doret = 1;

  if (! not_inetd)
    start_connection ();
  else
    {
      /* added host checking to plug a security hole. Otherwise you
         could fire the daemon up in non inet mode and access any data
         base. */
      current_host = myname;
      if (hp)
        current_addr = (char *) inet_ntoa (*(struct in_addr *) hp->h_addr);
      else
        current_addr = myname;
      if (!allowed (current_host, current_addr, gnats_root, &access))
	{
          printf ("%d You are not on the host access list: %s (%s).\r\n",
                  CODE_NO_ACCESS, current_host, current_addr);
	  exit (1);
	}
      host_access = user_access = access;
    }

  set_confidential_access ();

/* printf ("host: %s addr: %s access: %d\r\n", current_host, current_addr, host_access); */
/* fflush(stdout); */
  
  printf ("%d%c%s GNATS server %s ready.\r\n", CODE_GREETING,
	  kerberos_host ? '-' : ' ', myname, version_string);

  if (kerberos_host)
    printf ("%d Hi %s; you must use Kerberos authentication.\r\n",
	    CODE_GREETING, current_host);

  for (av = NULL; ; )
    {
      fflush (stdout);
      r = get_line (buff, (int) sizeof (buff), 2 * 60 * 60);
      switch (r)
	{
	case PR_timeout:
	  printf ("%d Read timeout.\r\n", CODE_TIMEOUT);
	  exit (1);
	  break;
	case PR_long:
	  printf ("%d Line too long\r\n", CODE_ERROR);
	  continue;
	case PR_ok:
	  if (buff[0] == '\0' || (buff[4] != ' ' && buff[4] != '\0'))
	    {
	      printf ("%d Unrecognized command.\r\n", CODE_ERROR);
	      continue;
	    }
	  ac = Argify (buff, &av, &cp);
	  if (ac == 0)
	    continue;
	  break;
	case PR_eof:
	  /* Handled below.  */
	  break;
	}

      if (r == PR_eof || strcasecmp (av[0], "quit") == 0)
	break;

      if (! cp)
	{
	  printf ("%d Unrecognized command.\r\n", CODE_ERROR);
	  continue;
	}
      
      /* Check access level */
      access_required = cp->flags & ACCESS_MASK;
      
/* printf ("cmd=%s access_required=%d user_access=%d host_access=%d\r\n", */
/*          av[0], access_required, user_access, host_access); */
      
#ifdef HAVE_KERBEROS
      /* Did they send AUTH? */
      if (kerberos_host && (access_required > user_access) && !client_authorized)
	{
	  syslog (LOG_ERR, "GNATS command `%s' without authentication for %s",
		  cp->s, current_host);
	}
#endif
      
      /* require_db means the first command MUST be CHDB and it must succeed */
      if (require_db || !strcasecmp (av[0], "chdb")) {
        if (!strcasecmp (av[0], "chdb"))
          (*cp->fn) (ac - 1, av + 1);
        if (require_db)
          exit (1);
        continue;
      }
      
      if (access_required > user_access)
        {
          printf ("%d You are not authorized to perform this operation (%s).\r\n",
                  CODE_NO_ACCESS, cp->s);
          continue;
        }

      /* For performance, we've delayed reading the index until now in case the user
         gives a CHDB command first, which is handled above */
      if (!index_chain) {
        index_chain = get_index ();
        /*   if (index_chain == NULL) */
        /*     { */
        /*       printf ("%d GNATS server cannot read the index.\r\n", CODE_FILE_ERROR); */
        /*       exit (1); */
        /*     } */
      }
      
      /* Call the function.  We skip the command itself in the arg list.  */
      (*cp->fn) (ac - 1, av + 1);
    }

  printf ("%d Later.\r\n", CODE_CLOSING);

  free_db_conf();
  exit (0);
}

void
usage ()
{
  fprintf (stderr, "\
Usage: %s [-h] [-n] [-d directory] [--help] [--directory=directory] [--not-inetd]\n",
	   program_name);
  exit (1);
}

void
version ()
{
  printf ("gnatsd %s\n", version_string);
}
