/* Commands for the GNATS server.
   Copyright (C) 1994, 95, 95, 1997 Free Software Foundation, Inc.
   Contributed by Brendan Kehoe (brendan@cygnus.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more detailmails.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"
#include "gnatsd.h"
#include "query.h"
#include "db_conf.h"

char *current_pr = (char *)NULL;

#ifdef HAVE_KERBEROS
#include <des.h>
#include <krb.h>
#ifndef HAVE_KRB_GET_ERR_TEXT
#define krb_get_err_text(status) krb_err_txt[status]
#endif
#endif

extern char myname[MAXHOSTNAMELEN];
extern int require_db;
extern int searching;
extern int doret;
extern Index *search;

#ifdef HAVE_KERBEROS
extern int client_authorized;
#endif

static int chk_nargs			PARAMS((int, char*));

static int
chk_nargs (ac, what)
     int ac;
     char *what;
{
  if (ac == 1)
    return 1;

  if (ac > 1)
    printf ("%d Only one %s allowed.\r\n", CODE_ERROR, what);
  else
    printf ("%d One %s required.\r\n", CODE_ERROR, what);
  return 0;
}

void
noargs (s)
     char *s;
{
  printf ("%d %s takes no arguments.\r\n", CODE_ERROR, s);
}

char *
lookup_pr_path (p)
     char *p;
{
  char *path;
  FILE *fp;

  /* They gave it to us with the category (possibly).  */
  if (((char *) strchr (p, '/')) == NULL)
    path = get_category (p);
  else
    {
      path = (char *) xmalloc (PATH_MAX);
      sprintf (path, "%s/%s", gnats_root, p);
    }
  fp = fopen (path, "r");
  if (fp == (FILE *)NULL)
    {
      printf ("%d Invalid PR %s.\r\n", CODE_INVALID_PR, p);
      return NULL;
    }

  fclose (fp);
  return path;
}

int
addr_cmp (a, b)
     const void *a, *b;
{
     return strcmp (*(char **)a, *(char **)b);
}

/* Given an array of possible addresses, make them into a real list with
   no duplicates.  Return the number of entries that we wiped out.
   FIXME:  We don't separate out single addr entries that have
   multiple people listed with commas.  */
int
unify_addrs (addrs, naddrs)
     char **addrs;
     int naddrs;
{
  char *last;
  int i, j = 0;

  qsort ((char *)addrs, naddrs, sizeof (char *), addr_cmp);
    
  if (naddrs == 1)
    return 0;

  last = addrs[0];
  for (i = 1; i < naddrs; i++)
    {
      if (strcmp (last, addrs[i]) != 0)
	last = addrs[i];
      else
	{
	  j++;
	  xfree (addrs[i]);
	  addrs[i] = NULL;
	}
    }

  return j;
}

char *
get_text ()
{
  register FILE *tf;
  char *path = (char *) xmalloc (PATH_MAX);
#ifndef HAVE_MKTEMP
  char name[L_tmpnam];
#endif
  char *buf, *tmpdir;
  MsgType r;

  tmpdir = getenv ("TMPDIR");
  if (tmpdir == NULL)
    tmpdir = "/tmp"; /* XXX */
#ifdef HAVE_MKTEMP
  sprintf (path, "%s/gnatsXXXXXX", tmpdir);
  mktemp (path);
#else
  tmpnam (name);
  strcpy (path, name);
#endif

  if ((tf = fopen (path, "w")) == (FILE *) NULL)
    {
      /* give error that we can't create the temp and leave. */
      xfree (path);
      return NULL;
    }

  buf = (char *) xmalloc (MAXLINE + 1);
  for (;;)
    {
      r = get_line (buf, MAXLINE, 2 * 60 * 60);
      switch (r)
	{
	case PR_timeout:
	  printf ("%d Read timeout.\r\n", CODE_TIMEOUT);
	  goto die;
	  break;
	case PR_long:
	  /* XXX what to do when it's too long?? */
	  continue;
	case PR_ok:
	  {
	    register char *bp = buf;
	    if (buf[0] == '.' && (buf[1] == '\n' || buf[1] == '\0'))
	      goto done;
	    fputs (bp, tf);
	    fputs ("\n", tf);
	    if (ferror (tf))
	      goto die; /* XXX */
	  }
	  break;
	case PR_eof:
	  goto die;
	}
    }

 done:
  if (feof (stdin) || ferror (stdin))
    {
      printf ("%d error reading\n", CODE_ERROR);
      goto die;
    }
  if (fflush (tf) != 0)
    {
      printf ("%d error flushing read input\n", CODE_ERROR);
      goto die;
    }

  fclose (tf);
  xfree (buf);
  return path;

 die:
  fclose (tf);
  if (unlink (path) && errno != ENOENT)
    syslog (LOG_ERR, "can't remove %s: %m", path);
  xfree (buf);
  xfree (path);
  return NULL;
}

void
daemon_pr (p, print)
     char *p;
     int print;
{
  char *path;

  if (p[0] != '/')
    path = lookup_pr_path (p);
  else
    {
      path = p;
      p = NULL;
    }

  if (path)
    {
      if (get_pr (path, p, quiet))
	{
	  if (print)
	    print_pr (path, p, 1, (Index *) NULL);
	  else
	    {
	      /* Even though we shouldn't do multiple loads of different
		 PRs in one session, let's be safe.  */
	      if (current_pr)
		xfree (current_pr);
	      current_pr = path;
	    }
	}
      else
	printf ("%d-Invalid PR %s.\r\n", CODE_INVALID_PR, p);

      if (print)
	xfree (path);
    }
  else
    /* We should have always verified each PR.  */
    abort ();
}

void
do_query (ac, av)
     int ac;
     char **av;
{
  char pat[40], *p, *n;
  int i, opened;
  Index *j;
  int hit = 0;

  reset_started ();

  if (ac == 0)
    {
      /* We weren't given a list of PRs to check, so we do the
	 whole shooting match.  */
      for (j = index_chain; j ; j = j->next)
	if (pr_matches (search, j, &opened) && do_pr (j, opened))
	  {
	    hit = 1;
	    printf ("\r\n");
	  }
    }
  else
    {
      for (i = 0; i < ac; i++)
        {
          p = av[i];
          if ((n = (char *) strchr (p, '/')) != NULL) /* Remove the category */
               p = ++n;
          strcpy (pat, p);
          strcat (pat, "\\'");
          for (j = index_chain ; j ; j = j->next)
       	    if (regcmp (pat, j->number) == 0)
              {
		if (pr_matches (search, j, &opened) && do_pr (j, opened))
                  {
                    hit = 1;
                    printf ("\r\n");
                  }
                /* Only go thru for a particular pattern until we make a
                   match; once we do, move on to the next argument.  */
                break;
              }
        }
    }
  
  if (! hit)
    printf ("%d No PRs match.\r\n", CODE_NO_PRS);
  else
    printf (".\r\n");
}

#if 0
static int
name_verify (n)
     char *n;
{
  return strcasecmp (current_host, n) == 0;
}
#endif

void
GNATS_helo (ac, av)
     int ac;
     char **av;
{
  printf ("%d Hello; the HELO command is no longer needed.\r\n",
	  CODE_HELLO);
}

void
GNATS_quit (ac, av)
     int ac;
     char **av;
{
  printf ("Hi, I'M GNATS_quit.\r\n");
}

void
GNATS_qury (ac, av)
     int ac;
     char **av;
{
  query_format = FORMAT_REG;
  do_query (ac, av);
}

void
GNATS_summ (ac, av)
     int ac;
     char **av;
{
  query_format = FORMAT_SUMM;
  do_query (ac, av);
}

void
GNATS_full (ac, av)
     int ac;
     char **av;
{
  query_format = FORMAT_FULL;
  do_query (ac, av);
}

void
GNATS_sqlf (ac, av)
     int ac;
     char **av;
{
  query_format = FORMAT_SQL;
  do_query (ac, av);
}

void
GNATS_sql2 (ac, av)
     int ac;
     char **av;
{
  query_format = FORMAT_SQL2;
  do_query (ac, av);
}

/* Used in edit.c to force creation of a PR's text.  We'll never do that
   from here.  */
int force = 0;

char *
daemon_lock_pr (p, w, processid)
     char *p, *w, *processid;
{
  char *path;

  path = lookup_pr_path (p);
  if (! path)
    return NULL;

  if (! lock_pr (path, w, processid))
    return NULL;

  return path;
}

void
GNATS_lock (ac, av)
     int ac;
     char **av;
{
  char *path, *l = (char *)NULL;
  int i, len;

  if (ac < 2)
    {
      printf ("%d Must lock with PR number, username, and process locking it.\r\n",
	      CODE_ERROR);
      return;
    }

  if (ac > 2)
    {
      /* XXX FIXME -- we need a cleaner approach to this.  */
      for (i = 2, len = 0; i < ac; i++)
	len += strlen (av[i]);
      l = (char *) xmalloc (sizeof (char) * len + ac - 2);
      sprintf (l, "%s", av[2]);
      for (i = 3; i < ac; i++)
	{
	  strcat (l, " ");
	  strcat (l, av[i]);
	}
    }
  path = daemon_lock_pr (av[0], av[1], l);
  if (l)
    xfree (l);
  if (! path)
    return;

  query_format = FORMAT_FULL;
  daemon_pr (path, 1);
  printf (".\r\n");
}

void
GNATS_unlk (ac, av)
     int ac;
     char **av;
{
  char *path;

  if (! chk_nargs (ac, "PR to be unlocked"))
    return;

  path = lookup_pr_path (av[0]); /* XXX what if it's null? */
  if (unlock_pr (path))
    printf ("%d PR %s unlocked.\r\n", CODE_OK, av[0]);
  xfree (path);
}

int
check_locked (path, p)
     char *path, *p;
{
  struct stat buf;
  char *lock_path = (char *) xmalloc (strlen (path) + 6);

  strcpy (lock_path, path);
  strcat (lock_path, ".lock");
  if (stat (lock_path, &buf) != 0)
    {
      if (errno == ENOENT)
	printf ("%d You must lock PR %s and get its current content first.\r\n",
		CODE_PR_NOT_LOCKED, p);
      else
	printf ("%d Could not detect if the lock for %s was there.\r\n",
		CODE_FILE_ERROR, p);
      return 0;
    }
  xfree (lock_path);
  return 1;
}

void
GNATS_lkdb (ac, av)
     int ac;
     char **av;
{
  if (ac)
    {
      noargs ("LKDB");
      return;
    }

  /* This does the error for us.  */
  if (lock_gnats () == 0)
    printf ("%d GNATS database is now locked.\r\n", CODE_OK);
}

void
GNATS_undb (ac, av)
     int ac;
     char **av;
{
  if (ac)
    {
      noargs ("UNDB");
      return;
    }

  if (unlock_gnats () < 0)
    {
      printf ("%d failed unlocking GNATS database.\r\n",
	      CODE_ERROR);
      return;
    }

  printf ("%d GNATS database is now unlocked.\r\n", CODE_OK);
}

void
GNATS_user (ac, av)
     int ac;
     char **av;
{
  char *path;
  int   access;

  if (ac == 0)
    {
      printf ("%d User access level set to %s\r\n", CODE_OK, access_level_str (user_access));
      return;
    }
  
  if (ac != 2)
    {
      printf ("%d USER requires two arguments, username and password.\r\n",
	      CODE_ERROR);
      return;
    }
  
  path = (char *) xmalloc (PATH_MAX);
  sprintf (path, "%s/gnats-adm/%s", gnats_root, DB_ACCESS_FILE);
  access = get_user_access (gnats_root, path, av[0], av[1]);
  xfree (path);

  if (access < 0)
    access = get_user_access (gnats_root, GLOBAL_ACCESS_FILE, av[0], av[1]);

  if (access < 0)
    access = ACCESS_NONE;
  
  if (access > host_access)
    user_access = access;

  if (user_access <= ACCESS_NONE)
    {
      printf ("%d You are not on the user access list: %s/%s.\r\n",
              CODE_NO_ACCESS, av[0], av[1]);
      exit (1);
    }
  
  set_confidential_access ();

  printf ("%d User access level set to %s\r\n", CODE_OK, access_level_str (user_access));
}

/* change database command */
void
GNATS_chdb (ac, av)
     int ac;
     char **av;
{
  static char *new_root = NULL;

  char  *new_db = NULL;
  char  *alias_db = NULL;
  int    access;

  struct stat buf;

  if (! chk_nargs(ac, "Database name"))
    return;

  /* verify new root exists and is a directory */
  new_db = av[0];

  /* We only allow database aliases in gnatsd, not full pathnames */
  alias_db = new_db;
  new_db = lookup_conf_alias(alias_db);
  if (!new_db)
    {
      printf ("%d Could not resolve database alias (%s).\r\n",
	      CODE_FILE_ERROR, alias_db);
      return;
    }

  if (stat(new_db, &buf) != 0)
    {
      if (errno == ENOENT)
	printf ("%d Could not change to database (%s): does not exist.\r\n",
		CODE_FILE_ERROR, new_db);
      else
	printf ("%d Error while getting status of (%s): errno = %d.\r\n",
		CODE_FILE_ERROR, new_db, errno);
      return;
    }
  else if ((buf.st_mode & S_IFMT) != S_IFDIR)
    {
      printf ("%d Specified database (%s) is not a directory.\r\n",
	      CODE_FILE_ERROR, new_db);
      return;
    }

  /* free existing copy then make new copy */
  if (new_root)
    xfree(new_root);
  new_root = (char *) xstrdup(new_db);

  /* re-initialize using new gnats_root */
  gnats_root = new_root;

  /* This doret stuff is wierd, but the following is necessary. <rickm@vsl.com> */
  doret = 0;
  configure  ();
  init_gnats ();
  doret = 1;

  /* make sure that a host that is allowed to access one database does
     not switch to a database where permission is not granted */
  if (!allowed (current_host, current_addr, gnats_root, &access))
    {
      syslog (LOG_NOTICE, "%s (%s) not allowed access for %s",
              current_host, current_addr, gnats_root);
      printf ("%d You are not on the host access list: %s (%s).\r\n",
              CODE_NO_ACCESS, current_host, current_addr);
      exit (1);
    }

  /* Clear the gnatsd --require-db flag and set the user access back to the default host access */
  require_db  = 0;
  user_access = host_access;
  
  set_confidential_access ();
  
  free_index(index_chain);
  index_chain = get_index ();
/*   if (index_chain == NULL) */
/*     { */
/*       printf ("%d GNATS server cannot read the index.\r\n", CODE_FILE_ERROR); */
/*       exit (1); */
/*     } */

  printf ("%d Now accessing GNATS database '%s'\r\n", CODE_OK, gnats_root);
}

/* list databases command */
void
GNATS_dbls (ac, av)
     int ac;
     char **av;
{
  char **databases = NULL;
  int  count = 0;
  int  access;
  int i;

  /* get complete list */
  get_conf_databases(&databases, &count);
  
  printf ("%d List follows.\r\n", CODE_PR_READY);
  for(i = 0; i < count; i++)
    {
      /* only list hosts with permitted access */
      if (allowed (current_host, current_addr, databases[i], &access))
	{
	  printf ("%s\r\n", databases[i]);
	}
      xfree(databases[i]);
    }
  printf (".\r\n");
  xfree((char*)databases);
}

/* list database alaises command */
void
GNATS_dbla (ac, av)
     int ac;
     char **av;
{
  char **aliases = NULL;
  int  count = 0;
  int  access;
  int  i;

  get_conf_aliases(&aliases, &count);

  printf ("%d List follows.\r\n", CODE_PR_READY);
  
  for(i = 0; i < count; i++)
    {
      /* only list hosts with permitted access */
      if (allowed (current_host, current_addr, lookup_conf_alias(aliases[i]), &access))
	{
	  /* if alias is a dot send a double dot to avoid end of data
             confusion */
	  if (strcmp(aliases[i], ".") == 0)
	    printf(".");
	  printf ("%s\r\n", aliases[i]);
	  xfree(aliases[i]);
	}
    }
  printf (".\r\n");
  xfree((char*) aliases);
}

void
GNATS_edit (ac, av)
     int ac;
     char **av;
{
  char *given, *path;
  FILE *fp;
  /* By default, unlock the PR after it's been filed.  */
  int unlockp = 1;

  if (! chk_nargs (ac, "PR to be edited"))
    return;

  path = lookup_pr_path (av[0]);
  if (! path)
    return;

  if (! check_locked (path, av[0]))
    return;

  if (is_gnats_locked ())
    {
      printf ("%d GNATS is currently locked; try again later.\r\n",
	      CODE_GNATS_LOCKED);
      return;
    }

  printf ("%d Ok.\r\n", CODE_OK); 
  fflush (stdout);
  given = get_text ();

  if (! given)
    {
      unlock_pr (path);
      return; /* XXX */
    }

  fp = fopen (given, "r");
  if (fp == (FILE *)NULL)
    {
      printf ("%d Failed reading saved text.\r\n", CODE_ERROR);
      goto editout;
    }

  /* block_signals (); */
  if (lock_gnats () < 0)
    goto editout;

  if (! modify_pr (fp))
    {
      printf ("%d Failed writing out PR.\r\n", CODE_ERROR);
      /* If we couldn't write it, we shouldn't unlock it.  */
      unlockp = 0;
    }
  else
    printf ("%d PR %s filed.\r\n", CODE_OK, av[0]);
  fclose (fp);
  if (unlockp)
    unlock_gnats ();
  unlock_pr (path); /* XXX */
  /* unblock_signals (); */
  if (unlink (given) && errno != ENOENT)
    syslog (LOG_ERR, "can't remove %s: %m", given);
 editout:
  xfree (path);
  xfree (given);
  return;

}

void
GNATS_mlpr (ac, av)
     int ac;
     char **av;
{
  char *path;

  if (! chk_nargs (ac, "PR"))
    return;

  path = lookup_pr_path (av[0]);
  if (! path)
    return;

  /* GNATS_mlpr must call get_pr() because of the need to get the X_GNATS_NOTIFY header */
  if (get_pr (path, av[0], 1))
    {
      Category category;
      Submitter submitter;
      Responsible *responsible;
      char *addrs[7]; /* XXX 6 possible ones, more later? */
      int naddrs = 0, i = 0;
      char *h;

      memset ((void *) &category, 0, sizeof (Category));
      memset ((void *) &submitter, 0, sizeof (Submitter));
      if (find_category (&category, pr[CATEGORY].value) == -1)
	{
	  printf ("%d No such category as %s.\r\n",
		  CODE_INVALID_CATEGORY, pr[CATEGORY].value);
	  return;
	}
      if (find_submitter (&submitter, pr[SUBMITTER].value) == -1)
	{
	  printf ("%d No such submitter as %s.\r\n",
		  CODE_INVALID_SUBMITTER, pr[SUBMITTER].value);
	  return;
	}
      responsible = get_responsible_address (category.person);
      if (responsible == (Responsible *)NULL)
	printf ("%d- No responsible entry for %s.\r\n",
		CODE_INVALID_RESPONSIBLE, category.person);

      addrs[naddrs++] = strdup (category.person);
      if (category.notify[0])
        addrs[naddrs++] = strdup (category.notify);
      if (submitter.contact[0])
        addrs[naddrs++] = strdup (submitter.contact);
      if (submitter.notify[0])
        addrs[naddrs++] = strdup (submitter.notify);
      if (responsible)
        addrs[naddrs++] = strdup (responsible->alias);
      h = raw_header_value (X_GNATS_NOTIFY);
      if (h)
	{
	  char *h2 = strchr (h, '\n'); h2[0] = '\0';
	  addrs[naddrs++] = strdup (h);
	}
      addrs[naddrs] = NULL;

      if (responsible)
	{
	  free_responsible (responsible);
	  xfree ((char *) responsible);
	}
      free_category  (&category);
      free_submitter (&submitter);

      i = unify_addrs (addrs, naddrs);
      if (i == naddrs)
	{
	  /* This should never happen.  */
	  printf ("%d No people listed for that PR.\r\n", CODE_ERROR);
	  return;
	}

      printf ("%d-Addresses to notify for PR %s/%s:\r\n",
	      CODE_INFORMATION, pr[CATEGORY].value, pr[NUMBER].value);
      for (i = 0; i < naddrs; i++)
	if (addrs[i])
	  {
	    printf ("%d-%s\r\n", CODE_INFORMATION, addrs[i]);
	    xfree (addrs[i]);
	  }
      printf ("%d End of list.\r\n", CODE_INFORMATION);
    }
  else
    printf ("%d Invalid PR %s.\n", CODE_INVALID_PR, av[0]);

  xfree (path);
}

void
GNATS_mlct (ac, av)
     int ac;
     char **av;
{
  Category category;

  if (! chk_nargs (ac, "category"))
    return;

  memset ((void *) &category, 0, sizeof (Category));
  if (find_category (&category, av[0]) == -1)
    {
      printf ("%d No such category as %s.\r\n", CODE_ERROR, av[0]);
      return;
    }
  printf ("%d-Addresses to notify for category %s:\r\n",
	  CODE_INFORMATION, av[0]);
  printf ("%d-%s\r\n", CODE_INFORMATION, category.person);
  if (category.notify[0])
    printf ("%d-%s\r\n", CODE_INFORMATION, category.notify);
  printf ("%d End of list.\r\n", CODE_INFORMATION);
}

void
GNATS_mlsu (ac, av)
     int ac;
     char **av;
{
  Submitter submitter;

  if (! chk_nargs (ac, "submitter"))
    return;

  memset ((void *) &submitter, 0, sizeof (Submitter));
  if (find_submitter (&submitter, av[0]) == -1)
    {
      printf ("%d No such submitter as %s.\r\n",
	      CODE_INVALID_SUBMITTER, av[0]);
      return;
    }
  printf ("%d-Addresses to notify for submitter %s:\r\n",
	  CODE_INFORMATION, av[0]);
  printf ("%d-%s\r\n", CODE_INFORMATION, submitter.contact);
  if (submitter.notify[0])
    printf ("%d-%s\r\n", CODE_INFORMATION, submitter.notify);
  printf ("%d End of list.\r\n", CODE_INFORMATION);
}

void
GNATS_type (ac, av)
     int ac;
     char **av;
{
  Submitter submitter;

  if (! chk_nargs (ac, "submitter"))
    return;

  memset ((void *) &submitter, 0, sizeof (Submitter));
  if (find_submitter (&submitter, av[0]) == -1)
    {
      printf ("%d No such submitter as %s.\r\n",
	      CODE_INVALID_SUBMITTER, av[0]);
      return;
    }
  printf ("%d-Type of submitter for %s.\r\n",
	  CODE_INFORMATION, av[0]);
  printf ("%d-%s\r\n", CODE_INFORMATION, submitter.type);
  printf ("%d End of type.\r\n", CODE_INFORMATION);
}

/* Various query options.  */
void
GNATS_resp (ac, av)
     int ac;
     char **av;
{
  Responsible *responsible;

  if (! chk_nargs (ac, "responsible entry"))
    return;

  responsible = get_responsible_address (av[0]);
  if (responsible == (Responsible *)NULL)
    {
      printf ("%d No responsible entry for %s.\r\n",
	      CODE_INVALID_RESPONSIBLE, av[0]);
      return;
    }

  free_responsible (responsible);
  xfree ((char *)responsible);

  searching = 1;
  search->responsible = (char *) strdup (av[0]);
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_catg (ac, av)
     int ac;
     char **av;
{
#if 0
  Category category;
#endif
  if (! chk_nargs (ac, "category"))
    return;
#if 0
  /* XXX but what if it's a regexp? we can't check it then */
  memset ((void *) &category, 0, sizeof (Category));
  if (find_category (&category, av[0]) == -1)
    {
      printf ("%d No such category as %s.\r\n",
	      CODE_INVALID_CATEGORY, av[0]);
      return;
    }
#endif
  searching = 1;
  search->category = (char *) strdup (av[0]);
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_synp (ac, av)
     int ac;
     char **av;
{
  if (! chk_nargs (ac, "synopsis"))
    return;

  searching = 1;
  search->synopsis = (char *) strdup (av[0]);
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_rlse (ac, av)
     int ac;
     char **av;
{
  if (! chk_nargs (ac, "release"))
    return;

  searching = 1;
  search->release = (char *) strdup (av[0]);
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_conf (ac, av)
     int ac;
     char **av;
{
  if (! chk_nargs (ac, "yes or no"))
    return;

  set_confidential_access ();
  if (!search->confidential)
    {
      searching = 1;
      search->confidential = (char *) strdup (av[0]);
    }
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_svty (ac, av)
     int ac;
     char **av;
{
  if (! chk_nargs (ac, "severity"))
    return;

  searching = 1;
  search->severity = (char *) strdup (av[0]);
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_stat (ac, av)
     int ac;
     char **av;
{
  if (! chk_nargs (ac, "state"))
    return;

  searching = 1;
  search->state = (char *) strdup (av[0]);
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_orig (ac, av)
     int ac;
     char **av;
{
  if (! chk_nargs (ac, "name"))
    return;

  searching = 1;
  search->originator = (char *) strdup (av[0]);
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_clss (ac, av)
     int ac;
     char **av;
{
  if (! chk_nargs(ac, "class list"))
    return;

  searching = 1;
  search->class = (char *) strdup (av[0]);
  printf ("%d Ok.\r\n", CODE_OK);
}

#ifdef GNATS_RELEASE_BASED
void
GNATS_qrtr (ac, av)
     int ac;
     char **av;
{
  if (! chk_nargs(ac, "quarter"))
    return;

  searching = 1;
  search->quarter = (char *) strdup (av[0]);
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_kywd (ac, av)
     int ac;
     char **av;
{
  if (! chk_nargs(ac, "keyword list"))
    return;

  searching = 1;
  search->keywords = (char *) strdup (av[0]);
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_bfor (ac, av)
     int ac;
     char **av;
{
  time_t t;

  if (! chk_nargs(ac, "date"))
    return;

  t = get_date (*av, NULL);
  if (t < 0)
    printf ("%d Cannot parse the date: %s.\r\n", CODE_INVALID_DATE, *av);
  else
    {
      required_before = t;
      searching = 1;
      printf ("%d Ok.\r\n", CODE_OK);
    }
}

void
GNATS_aftr (ac, av)
     int ac;
     char **av;
{
  time_t t;

  if (! chk_nargs(ac, "date"))
    return;

  t = get_date (*av, NULL);
  if (t < 0)
    printf ("%d Cannot parse the date: %s.\r\n", CODE_INVALID_DATE, *av);
  else
    {
      required_after = t;
      searching = 1;
      printf ("%d Ok.\r\n", CODE_OK);
    }
}
#endif

void
GNATS_prio (ac, av)
     int ac;
     char **av;
{
  if (! chk_nargs (ac, "priority"))
    return;

  searching = 1;
  search->priority = (char *) strdup (av[0]);
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_abfr (ac, av)
     int ac;
     char **av;
{
  time_t t;

  if (! chk_nargs(ac, "date"))
    return;

  t = get_date (*av, NULL);
  if (t < 0)
    printf ("%d Cannot parse the date: %s.\r\n", CODE_INVALID_DATE, *av);
  else
    {
      searching = 1;
      arrived_before = t;
      printf ("%d Ok.\r\n", CODE_OK);
    }
}

void
GNATS_araf (ac, av)
     int ac;
     char **av;
{
  time_t t;

  if (! chk_nargs(ac, "date"))
    return;

  t = get_date (*av, NULL);
  if (t < 0)
    printf ("%d Cannot parse the date: %s.\r\n", CODE_INVALID_DATE, *av);
  else
    {
      searching = 1;
      arrived_after = t;
      printf ("%d Ok.\r\n", CODE_OK);
    }
}

void
GNATS_cbfr (ac, av)
     int ac;
     char **av;
{
  time_t t;

  if (! chk_nargs(ac, "date"))
    return;

  t = get_date (*av, NULL);
  if (t == -1)
    printf ("%d Cannot parse the date: %s.\r\n", CODE_INVALID_DATE, *av);
  else
    {
      searching = 1;
      closed_before = t;
      printf ("%d Ok.\r\n", CODE_OK);
    }
}

void
GNATS_caft (ac, av)
     int ac;
     char **av;
{
  time_t t;

  if (! chk_nargs(ac, "date"))
    return;

  t = get_date (*av, NULL);
  if (t == -1)
    printf ("%d Cannot parse the date: %s.\r\n", CODE_INVALID_DATE, *av);
  else
    {
      searching = 1;
      closed_after = t;
      printf ("%d Ok.\r\n", CODE_OK);
    }
}

void
GNATS_mbfr (ac, av)
     int ac;
     char **av;
{
  time_t t;

  if (! chk_nargs(ac, "date"))
    return;

  t = get_date (*av, NULL);
  if (t == -1)
    printf ("%d Cannot parse the date: %s.\r\n", CODE_INVALID_DATE, *av);
  else
    {
      searching = 1;
      modified_before = t;
      printf ("%d Ok.\r\n", CODE_OK);
    }
}

void
GNATS_maft (ac, av)
     int ac;
     char **av;
{
  time_t t;

  if (! chk_nargs(ac, "date"))
    return;

  t = get_date (*av, NULL);
  if (t == -1)
    printf ("%d Cannot parse the date: %s.\r\n", CODE_INVALID_DATE, *av);
  else
    {
      searching = 1;
      modified_after = t;
      printf ("%d Ok.\r\n", CODE_OK);
    }
}

void
GNATS_vdat (ac, av)
     int ac;
     char **av;
{
  time_t t;

  if (! chk_nargs (ac, "date"))
    return;

  t = get_date (*av, NULL);
  if (t < 0)
    printf ("%d Couldn't parse the date: %s.\r\n", CODE_INVALID_DATE, *av);
  else
    printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_subm (ac, av)
     int ac;
     char **av;
{
  if (! chk_nargs (ac, "name"))
    return;

  searching = 1;
  search->submitter = (char *) strdup (av[0]);
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_nocl (ac, av)
     int ac;
     char **av;
{
  if (ac)
    {
      noargs ("NOCL");
      return;
    }
  searching = 1;
  skip_closed = 1;
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_text (ac, av)
     int ac;
     char **av;
{
  if (! chk_nargs (ac, "regular expression"))
    return;

  searching = 1;
  text_search = (char *) strdup (av[0]);
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_mtxt (ac, av)
     int ac;
     char **av;
{
  if (! chk_nargs (ac, "regular expression"))
    return;
  searching = 1;
  m_text_search = (char *) strdup (av[0]);
  printf ("%d Ok.\r\n", CODE_OK);
}

void
GNATS_rset (ac, av)
     int ac;
     char **av;
{
  int new_index = 0;

  if (ac != 0)
    {
      printf ("%d No arguments expected for RSET.\r\n", CODE_ERROR);
      return;
    }

  xfree (search->category);
  xfree (search->number);
  xfree (search->submitter);
  xfree (search->responsible);
  xfree (search->state);
  xfree (search->confidential);
  xfree (search->severity);
  xfree (search->priority);
  xfree (search->originator);
  xfree (search->class);
  xfree (search->release);
  xfree (search->synopsis);
  memset (search, 0, sizeof (Index));

  clean_pr ();
  clean_header ();
  if (check_index ())
    new_index = 1;

  searching = skip_closed = 0;
  xfree (text_search);
  text_search = (char *) NULL;
  xfree (m_text_search);
  m_text_search = (char *) NULL;
#ifdef GNATS_RELEASE_BASED
  required_before = required_after = (time_t)0;
  xfree (search->quarter);
  xfree (search->keywords);
#endif
  arrived_before  = arrived_after  = (time_t)0;
  modified_before = modified_after = (time_t)0;
  closed_before   = closed_after   = (time_t)0;

  set_confidential_access ();

  if (new_index)
    printf ("%d Reset state...reloaded the index.\r\n", CODE_OK);
  else
    printf ("%d Reset state.\r\n", CODE_OK);
}

#ifdef HAVE_KERBEROS
/* Get an input line, or die.  */
static void
simple_get_line (buf, size, timo)
     char *buf;
     int size, timo;
{
  switch (get_line (buf, size, timo))
    {
    case PR_long:
      printf ("%d Too long.\r\n", CODE_ERROR);
    case PR_timeout:
    case PR_eof:
      /* for these two, assume the client simply died */
      exit (1);
    case PR_ok:
      break;
    default:
      abort ();
    }
}

void
tohex (ptr, buf, sz)
     char *ptr, *buf;
     int sz;
{
  static const char hex[16] = "0123456789abcdef";
  int c;
  while (sz--)
    {
      c = *ptr++ & 0xff;
      *buf++ = hex[c >> 4];
      *buf++ = hex[c & 15];
    }
  *buf++ = 0;
}

int
hexdigit (c)
     int c;
{
  if (c >= '0' && c <= '9')
    return c - '0';
  if (c >= 'a' && c <= 'f')
    return c - 'a' + 10;
  if (c >= 'A' && c <= 'F')
    return c - 'A' + 10;
  return -1;
}
void
fromhex (ptr, buf, sz)
     char *ptr, *buf;
     int sz;
{
  int val;
  while (sz--)
    {
      val = hexdigit (*buf++);
      val <<= 4;
      val |= hexdigit (*buf++);
      *ptr++ = val;
    }
}
#endif

void
GNATS_auth (ac, av)
     int ac;
     char **av;
{
  if (! chk_nargs(ac, "authentication type name"))
    return;
#ifdef HAVE_KERBEROS
  /* This isn't the right way to do this ... we ought to exchange data
     in printable ascii form or something like that.  This'll do for
     now.  */
  else if (!strcmp (av[0], "krb4-simple"))
    {
      extern char *getusershell ();

      char keyfile[PATH_MAX], desired_user[32];
      struct sockaddr_in peer, laddr;
      int len = sizeof (peer);
      int status;
      char instance[INST_SZ+1], version[KRB_SENDAUTH_VLEN+1];
      char *p;
      struct passwd *pwd;
      /* Must be exactly 4 bytes.  */
      int cksum;
      /* Bundle these together to make it easy to erase them later.  */
      struct {
	AUTH_DAT auth;
	Key_schedule sched;
	KTEXT_ST ticket;
	char buf[MAX_KTXT_LEN * 3 + 10];
      } k;

      if (getpeername (0, (struct sockaddr *) &peer, &len) < 0)
	{
	  syslog (LOG_ERR, "can't get peername: %m");
	  goto addr_error;
	}
      if (getsockname (0, (struct sockaddr *) &laddr, &len) < 0)
	{
	  syslog (LOG_ERR, "can't get sockname: %m");
	addr_error:
	  printf ("%d Server system error checking addresses: %s\r\n",
		  CODE_ERROR, strerror (errno));
	  return;
	}

      strcpy (keyfile, gnats_root);
      strcat (keyfile, "/gnats-adm/srvtab");
      /* Sanity-check installation.  */
      {
	struct stat statbuf;
	if (stat (keyfile, &statbuf) < 0)
	  {
	    syslog (LOG_ERR, "can't find Kerberos key file %s: %m", keyfile);
	  bad_key_file:
	    printf ("%d Server installation error, cannot access key file\r\n",
		    CODE_ERROR);
	    return;
	  }
	else if (statbuf.st_mode & 0044)
	  {
	    syslog (LOG_ERR,
		    "Kerberos key file %s is readable to others; recommend changing it",
		    keyfile);
	    goto bad_key_file;
	  }
	else if (access (keyfile, R_OK))
	  {
	    syslog (LOG_ERR, "can't read Kerberos key file %s: %m", keyfile);
	    goto bad_key_file;
	  }
      }

      printf ("%d Go for it!\r\n", CODE_OK);
      fflush (stdout);

      /* Do authentication step.  */

      /* This is ugly, because krb_sendauth/krb_recvauth don't
	 maintain a sensible connection state if authentication fails.
	 This is unacceptable here.  So I've broken out most of the
	 interesting bits of those routines here.  I've also added a
	 hex encoding of the data exchanged instead of the standard
	 authenticator format, to maintain the text-only protocol used
	 in GNATS.  */

      /* Kerberos authenticator timeout is five minutes; give the client
	 nearly that long.  */
      simple_get_line (desired_user, sizeof (desired_user) - 1, 4 * 60);
      simple_get_line (k.buf, sizeof (k.buf) - 1, 4 * 60);
      if (k.buf[0] != 'x')
	{
	  printf ("%d Bad hex encoding.\r\n", CODE_ERROR);
	  return;
	}
      p = k.buf + 1;
      fromhex (version, p, KRB_SENDAUTH_VLEN * 2);
      p += KRB_SENDAUTH_VLEN * 2;
      version[KRB_SENDAUTH_VLEN] = 0;
      if (memcmp ("AUTHV0.1", version, KRB_SENDAUTH_VLEN))
	{
	  printf ("%d Unrecognized `sendauth' version `%s'.\r\n",
		  CODE_ERROR, version);
	  return;
	}
      fromhex (version, p, KRB_SENDAUTH_VLEN * 2);
      p += KRB_SENDAUTH_VLEN * 2;
      if (memcmp (GNATS_KRB4_VERSIONID, version, KRB_SENDAUTH_VLEN))
	{
	  printf ("%d Unrecognized authentication protocol version `%s'\r\n",
		  CODE_ERROR, version);
	  return;
	}
      p += 8;
      k.ticket.length = strlen (p) / 2;
      fromhex (k.ticket.dat, p, k.ticket.length);
      strcpy (instance, "*");
      status = krb_rd_req (&k.ticket, GNATS_KRB4_PRINCIPAL_NAME, instance,
			   peer.sin_addr.s_addr, &k.auth, keyfile);
      if (status != KSUCCESS)
	{
	  syslog (LOG_NOTICE, "auth failed for %s: %s",
		  current_host, krb_get_err_text (status));
	  printf ("%d Authentication failed: %s.\r\n",
		  CODE_ERROR, krb_get_err_text (status));
	  return;
	}
      if (sizeof (cksum) != 4)
	abort ();
      cksum = ntohl (k.auth.checksum + 1);
      key_sched (k.auth.session, k.sched);
      len = krb_mk_priv ((unsigned char *) &cksum, k.buf, 4, k.sched,
			 &k.auth.session, &laddr, &peer);
      if (len < 0)
	{
	  printf ("%d Kerberos error making private message.\r\n",
		  CODE_ERROR);
	  return;
	}
      if (len + 5 > sizeof (k.buf) / 3)
	{
	  syslog (LOG_ERR, "Need more internal buffer space for krb_mk_priv.");
	  printf ("%d Server bug: krb_mk_priv overruns buffer.\r\n",
		  CODE_ERROR);
	  return;
	}
      p = k.buf + len + 1;
      tohex (k.buf, p, len);
      printf ("x%s\r\n", p);

      /* Okay, authentication step is done.  Now check authorization.  */
      if (kuserok (&k.auth, desired_user))
	{
	  printf ("%d You are not allowed access as user `%s'.\r\n",
		  CODE_NO_ACCESS, desired_user);
	  return;
	}
      pwd = getpwnam (desired_user);
      if (!pwd)
	{
	  printf ("%d Cannot find a password file entry for `%s'.\r\n",
		  CODE_NO_ACCESS, desired_user);
	  return;
	}
      for (setusershell (); p; p = getusershell ())
	if (!strcmp (pwd->pw_shell, p))
	  break;
      endusershell ();
      if (p == 0)
	{
	  printf ("%d Access denied for user `%s'.\r\n",
		  CODE_NO_ACCESS, desired_user);
	  return;
	}

      printf ("%d Okie-dokey.\r\n", CODE_OK);
      client_authorized = 1;
      host_access = user_access = ACCESS_EDIT;
      return;
    }
#endif
  else
    {
      printf ("%d Authentication type `%s' not supported.\r\n",
	      CODE_AUTH_TYPE_UNSUP, av[0]);
      return;
    }
}

void
GNATS_lcat (ac, av)
     int ac;
     char **av;
{
  if (ac)
    {
      noargs ("LCAT");
      return;
    }
  printf ("%d List follows.\r\n", CODE_PR_READY);
  if (get_gnats_file (LIST_CATEGORIES, NULL) == 0)
    printf (".\r\n");
}

void
GNATS_lres (ac, av)
     int ac;
     char **av;
{
  if (ac)
    {
      noargs ("LRES");
      return;
    }
  printf ("%d List follows.\r\n", CODE_PR_READY);
  if (get_gnats_file (LIST_RESPONSIBLE, NULL) == 0)
    printf (".\r\n");
}

void
GNATS_lsub (ac, av)
     int ac;
     char **av;
{
  if (ac)
    {
      noargs ("LSUB");
      return;
    }
  printf ("%d List follows.\r\n", CODE_PR_READY);
  if (get_gnats_file (LIST_SUBMITTERS, NULL) == 0)
    printf (".\r\n");
}

void
GNATS_lsta (ac, av)
     int ac;
     char **av;
{
  if (ac)
    {
      noargs ("LSTA");
      return;
    }
  printf ("%d List follows.\r\n", CODE_PR_READY);
  if (get_gnats_file (LIST_STATES, NULL) == 0)
    printf (".\r\n");
}

void
GNATS_lcla (ac, av)
     int ac;
     char **av;
{
  if (ac)
    {
      noargs ("LCLA");
      return;
    }
  printf ("%d List follows.\r\n", CODE_PR_READY);
  if (get_gnats_file (LIST_CLASSES, NULL) == 0)
    printf (".\r\n");
}

void
GNATS_lcfg (ac, av)
     int ac;
     char **av;
{
  if (ac)
    {
      noargs ("LCFG");
      return;
    }
  printf ("%d List follows.\r\n", CODE_PR_READY);
  if (get_gnats_file (LIST_CONFIG, NULL) == 0)
    printf (".\r\n");
}

void
GNATS_help (ac, av)
     int ac;
     char **av;
{
  printf ("%d-The GNATS daemon accepts the following commands:\r\n",
	  CODE_INFORMATION);
  printf ("%d-   HELO and QUIT           say hello and goodbye\r\n",
	  CODE_INFORMATION);
  printf ("%d-   NOCL                    skip closed PRs\r\n",
	  CODE_INFORMATION);
  printf ("%d-   LCAT                    list the available categories\r\n",
	  CODE_INFORMATION);
  printf ("%d-   LRES                    list the available responsible names\r\n",
	  CODE_INFORMATION);
  printf ("%d-   LSUB                    list the available submitters\r\n",
	  CODE_INFORMATION);
  printf ("%d-   LSTA                    list the available states\r\n",
	  CODE_INFORMATION);
  printf ("%d-   LCLA                    list the available classes\r\n",
	  CODE_INFORMATION);
  printf ("%d-   LCFG                    list the GNATS config file\r\n",
	  CODE_INFORMATION);
  printf ("%d-   CATG <category>         search for PRs in <category>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   SYNP <synopsis>         search for PRs have the synopsis <synopsis>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   RESP <person>           search for PRs that <person> is responsible for\r\n",
	  CODE_INFORMATION);
  printf ("%d-   SUBM <submitter>        search for PRs with a submitter <submitter>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   ORIG <originator>       search for PRs with an originator <originator>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   RLSE <release>          search for PRs from the release <release>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   CLSS <class>            search for PRs of class <class>\r\n",
	  CODE_INFORMATION);
#ifdef GNATS_RELEASE_BASED
  printf ("%d-   QRTR <quarter>          search for PRs with Quarter <quarter>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   KYWD <keywords>         search for PRs with Keywords <keywords>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   BFOR <date>             find PRs with Date-Required: before <date>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   AFTR <date>             find PRs with Date-Required: after <date>\r\n",
	  CODE_INFORMATION);
#endif
  printf ("%d-   PRIO <priority>         search for PRs which are of the priority <priority>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   ABFR <date>             find all PRs that arrived before <date>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   ARAF <date>             find all PRs that arrived before <date>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   MBFR <date>             find all PRs that were modified before <date>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   MAFT <date>             find all PRs that were modified before <date>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   CBFR <date>             find all PRs that were closed before <date>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   CAFT <date>             find all PRs that were closed before <date>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   VDAT <date>             validate the format of <date>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   SVTY <severity>         search for PRs which are of the severity <severity>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   STAT <state>            search for PRs in the state <state>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   CONF <yes|no>           set whether or not to include confidential PRs\r\n",
	  CODE_INFORMATION);
  printf ("%d-   TEXT <text>             search for PRs with single-line <text>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   MTXT <text>             search for PRs with multiple-line <text>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   MLPR <PR>               list who should get mail about <PR>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   MLCT <category>         list who should get mail about <category>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   MLSU <submitter>        list who should get mail about <submitter> \r\n",
	  CODE_INFORMATION);
  printf ("%d-   TYPE <submitter>        give the type of a given submitter\r\n",
	  CODE_INFORMATION);
  printf ("%d-   RSET                    reset internal settings to initial startup\r\n",
	  CODE_INFORMATION);
  printf ("%d-   LKDB                    lock the main GNATS database\r\n",
	  CODE_INFORMATION);
  printf ("%d-   UNDB                    unlock the main GNATS database\r\n",
	  CODE_INFORMATION);
  printf ("%d-   LOCK <PR> <user> <pid>  lock <PR> for <user> and optional <pid> and return PR text\r\n",
	  CODE_INFORMATION);
  printf ("%d-   UNLK <PR>               unlock <PR>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   EDIT <PR>               check in edited <PR>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   CHDB <database>         change GNATS ROOT to <database>\r\n",
	  CODE_INFORMATION);
  printf ("%d-   DBLS                    list databases known to server\r\n",
	  CODE_INFORMATION);
  printf ("%d-   DBLA                    list database aliases known to server\r\n",
	  CODE_INFORMATION);

#ifdef HAVE_KERBEROS
  printf ("%d-   AUTH <kerberos>	 Kerberos authentication\r\n",
	  CODE_INFORMATION);
#endif
  printf ("%d-   USER <userid> <passwd>	 UserId and password for database access.\r\n",
	  CODE_INFORMATION);
  printf ("%d-The search itself is taken by either QURY, SUMM, or FULL.  Each\r\n",
	  CODE_INFORMATION);
  printf ("%d-  may be invoked with no arguments, which will search the entire\r\n",
	  CODE_INFORMATION);
  printf ("%d-  database given the constraints above, or with one or more PRs\r\n",
	  CODE_INFORMATION);
  printf ("%d-  in its command, which will display those PRs specifically.\r\n",
	  CODE_INFORMATION);
  printf ("%d-The QURY command will show a given PR in a multi-line summary\r\n",
	  CODE_INFORMATION);
  printf ("%d-  format, SUMM will show that PR in a single-line summary, and the\r\n",
	  CODE_INFORMATION);
  printf ("%d-  FULL command will show the entire header and body of the PR.\r\n",
	  CODE_INFORMATION);
  printf ("%d-You may also use the option SQLF or SQL2, which will give you an SQL\r\n",
	  CODE_INFORMATION);
  printf ("%d-  format line for each PR, which you can bring into your own database.\r\n",
	  CODE_INFORMATION);
  printf ("%d End of HELP info for GNATS daemon.\r\n", CODE_INFORMATION);
}
