/* Functionality for packing and unpacking core file libraries. 

   Copyright 1986, 1987, 1989, 1991-1994, 2000
   Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
   
   Designed and developed by Hewlett-Packard for HP-UX on PA-RISC
   and Itanium.	 Author: Bharath.  Date: 17 May 2004. */

/*----------------------------------------------------------------------------*/
/* Includes.
 */
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <libgen.h>

#include <sys/stat.h>

#include "defs.h"
#include "symtab.h"
#include "objfiles.h"
#include "call-cmds.h"	/* for core_file_command() prototype */
#include "top.h"	/* for cd_command() prototype */

/*----------------------------------------------------------------------------*/
/* External declarations.
 */
extern struct objfile *symfile_objfile;
extern char *current_directory;
extern char **gdb_shlib_path_list;
extern int gdb_shlib_path_length;

/*----------------------------------------------------------------------------*/
/* Local definitions.
 */
#define PACKCORE_BUF_SIZE 4096

static char *tar_output_file;  /* call tempnam() to initialize */

/*----------------------------------------------------------------------------*/
/* Local function declarations.
 */
static int file_readable (char *file);
static int file_exists (char *file);
static char* get_arg (char *arg_str, char *err_msg);
static void add_to_str (char **main_str, int *main_str_buf_len, char *add_str);
static void execute_tar_cmd (char *tar_cmd, char *err_msg, int ignore_err);
static void dump_tar_output (void);

/*----------------------------------------------------------------------------*/
/* Function	: packcore_command ()
 * Description	: Takes a user specified core file, identifies the shared 
 *		    libraries used by the core and creates a tar file called 
 *		    core.tar with all those libraries along with the executable.
 *		    Usage information can be obtained by doing a 'help packcore'
 *		    on at the gdb prompt.
 * Input	: The command string issued by the user containing the core
 *		    file's name and the directory in which core.tar is to be 
 *		    created.
 * Output	: Nothing is returned by this function, but a successful call
 *		    would have created a core.tar in the user specified 
 *		    directory.
 * Globals	: symfile_objfile is used to get the executable and shared 
 *		    library paths and names.  This is a (struct objfile *)
 *		    which points to the first objfile structure, i.e, the
 *		    executable.
 * Notes	: The directory structure of the libraries are discarded.  All 
 *		    libraries are stored relative to the current directory 
 *		    without any path.  This was done to simplify the design of 
 *		    unpacking the core.  Other wise, unpack core would have to
 *		    do a lot of directory juggling to work properly, which in 
 *		    turn complicates the design.  This is a reasonable 
 *		    simplification because this is how people debugging cores 
 *		    tend to operate.  The only draw back is if there are two 
 *		    shared libraries with the same name in different 
 *		    directories.  Apparently, this is very rare (no one in the
 *		    linker team had ever come across this before) and linking
 *		    it itself is likely to cause problems.  A check and a 
 *		    warning for this may be a good future enhancement.  Also,
 *		    as a part of packing, the core file is loaded so that 
 *		    symfile_objfile is initialized.  Another point of interest
 *		    is that the tar command is left to handle some failures
 *		    like out of disk space and lack of permissions; this is why
 *		    dump_tar_ouput exists.
 */
void 
packcore_command (char *command_line, int from_tty)
{
  char *core_file, *tar_dir, *tar_file;
  char *dir_name, *file;
  char *tar_cmd = NULL;
  int tar_cmd_len = 0;
  char *tmp;
  char *saved_command_line;
  struct objfile *obj_file;

  saved_command_line = strdup (command_line);

  /* Get core file & tar directory name from the command string.
   * Create the tar file name.
   */
  core_file = get_arg (command_line, "Core file not specified");
  tar_dir = get_arg (NULL, "Tar directory not specified");
  if (get_arg (NULL, NULL) != NULL)
    error ("packcore takes only two arguments");

  if (xmode_exec_format_is_different (core_file))	/* Fix for JAGaf26491 */
    xmode_launch_other_gdb (saved_command_line, xmode_packcore);

  tar_file = (char *) malloc (strlen (tar_dir) + strlen ("/core.tar") + 1);
  if (tar_file == NULL)
    error ("Can't packcore.  Out of memory");

  strcpy (tar_file, tar_dir);
  strcat (tar_file, "/core.tar");

  /* Load the core.  This will check core file availability and validity.
   */
  core_file_command (core_file, 1);

  /* Create tar string.
   */
  add_to_str (&tar_cmd, &tar_cmd_len, "/usr/bin/tar -chlof ");
  add_to_str (&tar_cmd, &tar_cmd_len, tar_file);
  for (obj_file = symfile_objfile; obj_file; obj_file = obj_file->next)
  {
    tmp = strdup (obj_file->name);
    if (tmp == NULL)
      error ("Can't packcore. Out of memory");

    if (!file_readable (tmp))	/* Is the library or executable readable? */
      error ("Can't read %s: %s", tmp, strerror (errno));

    dir_name = dirname (tmp);
    file = basename (tmp);

    /* -C is used to remove path from the library while tarring up */
    add_to_str (&tar_cmd, &tar_cmd_len, " -C ");
    add_to_str (&tar_cmd, &tar_cmd_len, dir_name);
    add_to_str (&tar_cmd, &tar_cmd_len, " ");
    add_to_str (&tar_cmd, &tar_cmd_len, file);

    free (tmp);
  }
  add_to_str (&tar_cmd, &tar_cmd_len, " 2>");
  tar_output_file = tar_output_file 
		  ? tar_output_file 
		  : tempnam(NULL, "__gdb_core_tar.out");
  add_to_str (&tar_cmd, &tar_cmd_len, (char*)tar_output_file);

  /* Create the tar.
   */
  if (file_exists (tar_file))	/* Does the tar file exist already? */
    error ("%s already exists", tar_file);

  execute_tar_cmd (tar_cmd, "Can't packcore", 0);

  if (!file_exists (tar_file))	/* Was the tar successfully created? */
    error ("Unable to create %s", tar_file);

  unlink (tar_output_file);  /* delete the file, keep the name string */
  free (tar_file);
  free (tar_cmd);
  free (saved_command_line);
}
/*----------------------------------------------------------------------------*/
/* Function	: unpackcore_command ()
 * Description	: Takes a user specified core file, its corresponding tar file
 *		    which was created by packcore, the name of a directory to 
 *		    unpack the tar in and then unpacks the tar file in that 
 *		    directory.  It then proceeds to set the current working 
 *		    directory to the unpack directory and loads the core file 
 *		    for debugging by using the executable and shared libraries 
 *		    in the unpack directory for symbol information.
 * Input	: The command string issued by the user containing the core
 *		    file's name, the tar file's name and the directory in which
 *		    tar is to be unpacked.
 * Output	: Nothing is returned by this function, but a successful call
 *		    would have unpacked the tar file in the unpack directory.
 * Globals	: current_directory is a (char *) which points to a string
 *		    containing the current directory's path name.  Used to
 *		    access the core after cd'ing to the unpack directory.
 *		    gdb_shlib_path_list and gdb_shlib_path_length are saved,
 *		    modified to load libraries from the unpack directory and
 *		    then restored to their original values.
 * Notes	: This routine will over write existing libraries in the unpack
 *		    directory with the same names as those in the tar file, if
 *		    permissions are right.  Warning the user in this case
 *		    a fair amount of work on gdb's part and it is reasonable
 *		    to expect the user is aware of what directory he or she is
 *		    unpacking to, so the warning logic is not done.  As with
 *		    packcore_command, the tar command is allowed to handle some
 *		    error cases.  Unpacking and loading core file will ignore
 *		    GDB_SHLIB_PATH because the libraries have to be picked up
 *		    from the unpack directory, not some other path.
 */
void 
unpackcore_command (char *command_line, int from_tty)
{
  char *core_file, *tar_file, *unpack_dir;
  char *core_file_with_path, *tar_file_with_path;
  char *tar_cmd = NULL;
  int tar_cmd_len = 0;
  char *tmp;
  char *saved_cwd;
  char **saved_gdb_shlib_path_list;
  int saved_gdb_shlib_path_length;

  /* Get core file, tar file & untar directory name from the command string.
   */
  core_file = get_arg (command_line, "Core file not specified");
  tar_file = get_arg (NULL, "Tar file not specified");
  unpack_dir = get_arg (NULL, "Unpack directory not specified");
  if (get_arg (NULL, NULL) != NULL)
    error ("unpackcore takes only three arguments");
 
  /* cd to the unpack directory and unpack the tar file. */
  saved_cwd = current_directory;
  cd_command (unpack_dir, 0);

  if (tar_file[0] == '/') 	/* Tar file has absolute path. */
    tar_file_with_path = tar_file;
  else
  {
    tar_file_with_path = (char *) malloc (strlen (tar_file) +
  					   strlen (saved_cwd) + 2);
    strcpy (tar_file_with_path, saved_cwd);
    strcat (tar_file_with_path, "/");
    strcat (tar_file_with_path, tar_file);
  }

  /* Since we are ignoring status 5 from the untar, try to verify that
   * it is OK by listing the contents.
   */
  tar_cmd = NULL;
  tar_cmd_len = 0;
  add_to_str (&tar_cmd, &tar_cmd_len, "/usr/bin/tar -tf ");
  add_to_str (&tar_cmd, &tar_cmd_len, tar_file_with_path);
  add_to_str (&tar_cmd, &tar_cmd_len, " 2>&1 > ");
  tar_output_file = tar_output_file 
		  ? tar_output_file 
		  : tempnam(NULL, "__gdb_core_tar.out");
  add_to_str (&tar_cmd, &tar_cmd_len, (char*)tar_output_file);
  execute_tar_cmd (tar_cmd, "Can't unpackcore", 0);
  
  tar_cmd = NULL;
  tar_cmd_len = 0;
  add_to_str (&tar_cmd, &tar_cmd_len, "/usr/bin/tar -xof ");
  add_to_str (&tar_cmd, &tar_cmd_len, tar_file_with_path);
  add_to_str (&tar_cmd, &tar_cmd_len, " 2>");
  tar_output_file = tar_output_file 
		  ? tar_output_file 
		  : tempnam(NULL, "__gdb_core_tar.out");
  add_to_str (&tar_cmd, &tar_cmd_len, (char*)tar_output_file);
  execute_tar_cmd (tar_cmd, "Can't unpackcore", 5);

  /* Set the internal gdb_shlib_path to unpack directory.  Load the core; this
   * will check core file availability and validity.  Restore gdb_shlib_path.
   */
  saved_gdb_shlib_path_list = gdb_shlib_path_list;
  saved_gdb_shlib_path_length = gdb_shlib_path_length;

  tmp = strdup (unpack_dir);
  gdb_shlib_path_list = &tmp;
  gdb_shlib_path_length = 1;

  if (core_file[0] == '/') 	/* Core file has absolute path. */
    core_file_with_path = core_file;
  else
  {
    core_file_with_path = (char *) malloc (strlen (core_file) +
  					   strlen (saved_cwd) + 2);
    strcpy (core_file_with_path, saved_cwd);
    strcat (core_file_with_path, "/");
    strcat (core_file_with_path, core_file);
  }

  core_file_command (core_file_with_path, 1);

  gdb_shlib_path_list = saved_gdb_shlib_path_list;
  gdb_shlib_path_length = saved_gdb_shlib_path_length;

  if (core_file[0] != '/') 	/* Free only for the allocated case. */
    free (core_file_with_path);
  if (tar_file[0] != '/') 	/* Free only for the allocated case. */
    free (tar_file_with_path);
  free (tar_cmd);
  unlink (tar_output_file);  /* delete the file, keep the name string */
}
/*----------------------------------------------------------------------------*/
/* Function	: file_readable ()
 * Description	: Determines whether or not a file is readable by the current
 *		    gdb session.
 * Input	: Pointer to a file name string.
 * Output	: Returns TRUE if the file is readable, FALSE otherwise.
 * Globals	: - N/A -
 * Notes	: Relies on open() rather than on stat(), getpid(), getgid(), 
 *		    etc.  Makes it simple.
 */
static int 
file_readable (char *file)
{
  int fd;

  fd = open (file, O_RDONLY);
  close (fd);

  if (fd == -1)
    return FALSE;

  return TRUE;
}
/*----------------------------------------------------------------------------*/
/* Function	: file_exists ()
 * Description	: Determines whether or not a given file exists.
 * Input	: Pointer to a file name string.
 * Output	: Returns TRUE if the file is readable, FALSE otherwise.
 * Globals	: - N/A -
 * Notes	: - N/A -
 */
static int 
file_exists (char *file)
{
  int status;
  struct stat stat_buf;

  status = stat (file, &stat_buf);
  if (status == -1)
  {
    if (!(errno == ENOENT || errno == ENOTDIR))
      error ("stat() on %s failed: %s", file, strerror (errno));
    return FALSE;
  }
  else	/* If stat succeeded, then the file exists. */
    return TRUE;
}
/*----------------------------------------------------------------------------*/
/* Function	: get_arg ()
 * Description	: Used to get one argument at a time from a user command string.
 *		    Builds on strtok() and provides error reporting facility if
 *		    argument obtained from the command string is NULL.  Can be
 *		    called upon subsequently just like strtok () which maintains
 *		    state between calls.
 * Input	: Pointer to a user command string and a pointer to an error 
 *		    message.
 * Output	: Pointer to the next argument in the command string.
 * Globals	: - N/A -
 * Notes	: Trashes the argument string by writing '\0' at different 
 *		    points.
 */
static char* 
get_arg (char *arg_str, char *err_msg)
{
  char *tmp;

  tmp = strtok (arg_str, " \t");
  if (tmp == NULL && err_msg != NULL)
    error (err_msg);

  return tmp;
}
/*----------------------------------------------------------------------------*/
/* Function	: add_to_str ()
 * Description	: Adds string s2 to string s1.  This is similar to strcat except
 *		    in one way.  If the size of the buffer in which s1 resides 
 *		    isn't enough to do the concatenation, then s1's buffer is
 *		    expanded with a realloc().  If s1 is NULL, a buffer is 
 *		    allocated for it.
 * Input	: A pointer to string s1's pointer, pointer to the length of s1
 *		    and pointer to string s2.
 * Output	: Nothing is returned, but s1 is modified by appending s2 at the
 *		    end and s1's size is updated.
 * Globals	: - N/A -
 * Notes	: The size of s1's buffer is independent of s1's size.
 */
static void 
add_to_str (char **main_str, int *main_str_buf_len, char *add_str)
{
  if (*main_str == NULL)
  {
    *main_str = (char *) malloc (PATH_MAX);
    if (*main_str == NULL)
      error ("Can't packcore. Out of memory");
    *main_str_buf_len = PATH_MAX;
    **main_str = '\0';
  }

  if ((strlen (*main_str) + strlen (add_str) + 1) > *main_str_buf_len)
  {
    *main_str = (char *) realloc (*main_str, *main_str_buf_len * 2);
    if (*main_str == NULL)
      error ("Can't packcore. Out of memory");
    *main_str_buf_len *= 2;
  }

  strcat (*main_str, add_str);
}
/*----------------------------------------------------------------------------*/
/* Function	: execute_tar_cmd ()
 * Description	: Takes a user specified tar command and executes it.  If the 
 *		    tar command was unsuccessful, it dumps the output of the 
 *		    command along with an error message.
 * Input	: Pointer to a command string and a pointer to an error message.
 *		  and an error status to treat as zero (may be zero).
 *		  For untarring, we get status 5 if files are not writable.
 * Output	: Nothing on a success, error messages are printed on a failure.
 * Globals	: - N/A -
 * Notes	: This routine can be used to execute any command and have its
 *		    output displayed, with some minor modifications.
 */
static void 
execute_tar_cmd (char *tar_cmd, char *err_msg, int ignore_err)
{
  int status;

  status = system (tar_cmd);
  status = WEXITSTATUS (status);
  /* When we extract files with read-only permissions, we end up with a
   * status of 5, which we treat as success.
   */
  if (info_verbose || (status != 0 && status != ignore_err))
  {
    dump_tar_output ();
  }
  if (status != 0 && status != ignore_err)
    error ("%s.  Tar cmd failed.  Tar cmd: %s", err_msg, tar_cmd);
}

/*----------------------------------------------------------------------------*/
/* Function	: dump_tar_output ()
 * Description	: Displays the contents of the tar output file.  
 * Input	: - N/A -
 * Output	: - N/A -
 * Globals	: tar_output_file is used to find the file from which the 
 *		    contents are displayed.
 * Notes	: With slight modification, this routine can be made generic 
 *		    enough to dump the contents of any text file.
 */
static void 
dump_tar_output (void)
{
  FILE *fd;
  char buf[PACKCORE_BUF_SIZE];

  tar_output_file = tar_output_file 
		  ? tar_output_file 
		  : tempnam(NULL, "__gdb_core_tar.out");
  fd = fopen (tar_output_file, "r");
  if (fd == NULL)
    error ("Can't open tar output file %s: %s", tar_output_file, 
    						strerror (errno));
    
  while (!feof(fd))
  {
    fscanf (fd, "%[^\n]s", buf);
    fgetc (fd);
    printf_filtered ("%s\n", buf);
  }
  fclose (fd);
}
/*----------------------------------------------------------------------------*/
