

#include "defs.h"
#include "symtab.h"
#include "gdb-stabs.h"       
#include "objfiles.h"
#include "target.h"         
#include "command.h"
#include "gdbcmd.h"
#include "bfd.h"
#include "value.h"
#include "breakpoint.h"
#include "inferior.h"
#include "language.h"
#include "rtc.h"
#include "top.h"
#include "demangle.h"
#include "annotate.h"
#include <strings.h>
#include <ctype.h>

#if defined (RTC)

#include <shl.h>
#ifndef HPUX_1020
#include <sys/ttrace.h>
#endif /* HPUX_1020 */


/* Bindu 101702: On IA64, we will need to do swizzling of certain addresses.
   This should be a nop for other platforms. You don't want to put
   ifdef's all around..... */
#ifdef SWIZZLE
#define SWZ SWIZZLE
#else
#define SWZ
#endif /* !SWIZZLE */

/* Size of the pointer in the inferior. */
#ifdef HP_IA64
#define PTR_SIZE \
          (is_swizzled? 4: 8)
#else
#define PTR_SIZE (sizeof (CORE_ADDR))
#endif /* !HP_IA64 */

#define DEMANGLE_NEW(nm) \
    if (new_cxx_abi && is_cplus_name (nm))  \
      { \
         nm = cplus_demangle(nm, DMGL_PARAMS | DMGL_ANSI);\
         free_function = 1;\
      } \
      else\
      {\
         if (!strcmp (nm, "__nwa(unsigned int)"))\
           nm = "operator new[](unsigned int)";\
         else if (!strcmp (nm, "__nwa(unsigned long)"))\
           nm = "operator new[](unsigned long)";\
      }\

#define HEAP_INFO	 1 /* to print memory allocations only */
#define BOUNDS_INFO	 2 /* to print memory overruns */

int gc_in_progress = 0;
int dld_can_preload = 0;
int check_heap_in_this_run = 0;
extern int check_heap;  /* next run */
extern int inferior_pid;

extern void switch_to_thread (int pid);
extern void foreach_live_thread ( void (*function )(void *), void *);

#ifdef GDB_TARGET_IS_HPUX
extern char * wdb_version_nbr;
#endif

/* Bindu 040303 attach & rtc: to save the information about if we are
   debugging an attached process. */
int att_flag = 0;

/* various tunable knobs for RTC. The mother of all knobs is the
   variable check_heap_in_this_run. If this is off, none of the others
   matter.
*/
int rtc_initialized = 0;
static int frame_count = DEFAULT_FRAME_COUNT;
static int check_leaks = 1;
static int check_bounds = 0;
static int scramble_blocks = 0;
static int min_leak_size = DEFAULT_MIN_LEAK_SIZE;
static CORE_ADDR watch_address = 0;

/* null-check variables */
#define DEBUG(x)
#define DEFAULT_NULL_CHECK_RANDOM_RANGE  100
#define DEFAULT_NULL_CHECK_SEED_VALUE    5 
static int null_check_count = -1; 
static int null_check_size = -1;  
static int null_check_random_range = DEFAULT_NULL_CHECK_RANDOM_RANGE;
static int null_check_seed_value = DEFAULT_NULL_CHECK_SEED_VALUE;

/* incremental heap analysis */
#define TIME_STR_LEN 25

struct interval {
char  start_time[TIME_STR_LEN]; /* dyn: get the correct size */
char  end_time[TIME_STR_LEN]; /* dyn: get the correct size */
int   interval;
int   num_blocks;
int   unq_blocks;
int   total_bytes;
struct memory_info *block_info;
}; 

/* heap_idx is used to hold the heap records for each interval.
   The size of this array is determined by the number of records
   int "/tmp/__heap_interval_info.idx" file.
 */
static struct interval *heap_idx;
static int check_heap_interval_value = 0;
static int check_heap_interval_reset_value = 0;
static int check_heap_interval_repeat_value = DEFAULT_REPEAT_CNT;

static const char *heap_interval_filename =  "/tmp/__heap_interval_info";
static const char *heap_interval_idxfile  = "/tmp/__heap_interval_info.idx";

/* big_block_size : If this value is non-zero, we will stop
   the inferior whenever an allocation size exceeds this size.
*/
static long big_block_size = 0;  

/* heap_growth_size : If this value is non-zero, we will stop
   the inferior whenever an allocation causes the process
   address space to grow by a size greater than this value.
   The difference between this option and the former is
   subtle but important. 

   p = malloc (huge_size);
   ...
   free (p);
   p = malloc (huge_size);

   will trigger two events in one the former case, but in the 
   latter, may trigger zero, one or two event(s). 
*/

static long heap_growth_size = 0;
static int check_free = 0;
static int is_threaded = 0;
static int is_cma_threaded = 0;
static int is_dce_threaded = 0;

/* The structure gdb_chunk_info is a clone of rtc_chunk_info defined in
   rtc.h. Sharing this structure between a 32 bit gdb and a potentially
   64 bit librtc does not work well. Anytime you touch one, make
   sure the other one is updated too.
*/

typedef struct gdb_chunk_info {

    unsigned scanned     : 1;
    unsigned old_leak    : 1;
    unsigned heap_block  : 1;
    unsigned padded_block  : 1;

    CORE_ADDR next_leak;
    CORE_ADDR next;
    CORE_ADDR base;        
    CORE_ADDR end;         
    CORE_ADDR pc;

} gdb_chunk_info;

/* We use a slightly different structure for maintaining leak info
   in gdb, compared to the data structures used by the inferior.
   The structure here is influenced by presentation needs i.e., the
   need to collate multiple leaks from the same allocating stack and
   the need to sort the leaks by size. The memory pressure in
   the inferior is intense since there is one instance of rtc_chunk_info
   per allocated block. In gdb, the memory pressure is expected to be
   very low as we store only the (collated) leaks.
*/

static struct memory_info {

  /* where art thou ? */
  CORE_ADDR address;

  long long size;

  long long min;
  long long max;

  unsigned count;

  CORE_ADDR * pc;

  int	corrupted; /* set to 1 if the block is corrupted 
                      at header or footer by librtc */

  /* The field pointer_to_stack_trace is the pointer to an array of PC
     values in the inferior. We use it here to identify the leaks from
     the same place. Confusingly it is called pc in rtc_chunk_info.
  */

  CORE_ADDR pointer_to_stack_trace;

} * leak_info, *block_info;

static long long total_leak_bytes = 0;
static long total_leak_blocks = 0;
static long long total_bytes = 0;
static long total_blocks = 0;

static int leak_info_capacity = 0;  /* how big is leak_info[] ? */

CORE_ADDR leaks_info_plabel;
CORE_ADDR heap_info_plabel;
CORE_ADDR interval_info_plabel;

static struct objfile * rtc_dot_sl;
static struct objfile * libc_dot_sl = 0;
static struct objfile * libdld_dot_sl = 0;
#if !defined(HP_IA64) && !defined(GDB_TARGET_IS_HPPA_20W)
/* This is defined only for PA32 as this variable is used only by
   som_patchup_definitions. */
static CORE_ADDR rtc_dot_sl_got_value;
#endif

static int leak_count;
static int block_count;

/* To remember if we already patched the a.out's definations. */
static int symfile_objfile_patched = 0;

boolean threaded_program (void);

static struct wrappers {
    char * name;
    char * rtc_name;
    CORE_ADDR function_pointer;
    CORE_ADDR pointer_to_plabel;
} wrappers[] = {
   { "malloc", "__rtc_malloc",  0, 0 }, { "calloc",  "__rtc_calloc",  0, 0 },
   { "valloc", "__rtc_valloc",  0, 0 }, { "realloc", "__rtc_realloc", 0, 0 },
   { "free",   "__rtc_free",    0, 0 }, { "mmap",    "__rtc_mmap",    0, 0 },
   { "munmap", "__rtc_munmap",  0, 0 }, { "sbrk",    "__rtc_sbrk",    0, 0 }, 
   { "brk",    "__rtc_brk",     0, 0 }, { "shmat",   "__rtc_shmat",   0, 0 },
   { "shmdt",  "__rtc_shmdt",   0, 0 }, { "mprotect", "__rtc_mprotect", 0, 0 },
   { "shl_unload",  "__rtc_shl_unload",   0, 0 }, { "dlclose", "__rtc_dlclose", 0, 0 },

#if !defined(GDB_TARGET_IS_HPPA_20W) && !defined(HP_IA64)
   { "U_get_shLib_text_addr",  "__rtc_U_get_shLib_text_addr", 0, 0 },
   { "U_get_shLib_unw_tbl",    "__rtc_U_get_shLib_unw_tbl",   0, 0 },
   { "U_get_shLib_recv_tbl",   "__rtc_U_get_shLib_recv_tbl",  0, 0 },
#endif
};

int pending_dld_relocations = 0;

static int
dld_understands_preload ()
{
  static int dld_can_preload = -1;
  char * file;
  char * version;
  int minor;
  char buffer[BUFSIZ + 1] = "";
  FILE * fp;

  /* srikanth, we have no clean way of knowing if dld on the user system
     supports LD_PRELOAD. Rather than grep for version strings, just
     say yes. Here are the cases :

     (1) dld actually supports LD_PRELOAD and the user has not linked with
         librtc. This library will be preloaded : fine.
     (2) dld actually supports LD_PRELOAD and the user has already linked
         in librtc : This is ok as dld guarantees that preload would
         happen anyway and the implicitly linked in copy would not
         additionally be loaded.
     (3) dld does not support LD_PRELOAD and the user has not linked in
         librtc. Gdb would export LD_PRELOAD environment variable
         but this would be ignored by dld. Later on gdb would discover that
         librtc is not linked in and complain about it. This is the
         required behavior.
     (4) dld does not support LD_PRELOAD and the user has linked in
         librtc. Gdb would export LD_PRELOAD environment variable
         but this would be ignored by dld. But leak detection can progress.

     The only minor issue with (3) and (4) is the environment pollution
     that happens. This happens only when RTC is turned on. I think this
     is no big deal as we do this anyway if dld version is > 11.19.
   */

  return 1;

#if 0
  if (dld_can_preload != -1)
    return dld_can_preload;

  file = tmpnam (0);
  if (!file)
    return 0;

  /* This check is somewhat flaky. Requests for a more reboust method
     have been turned down by the linker project. Let us just hope that
     problems if any would be caught by integrated testing. Thing is
     even if this check fails incorrectly it is not catastrophic : the
     only observed symptom would be that gdb would insist that the user
     link in librtc explicitly even on a system where this should
     not be needed.
  */
  sprintf (buffer, "/usr/bin/what /usr/lib/dld.sl | /usr/bin/grep 'B.11' > %s",
file);
  system (buffer);
 
  fp = fopen (file, "r");
  if (!fp)
    {
      unlink (file);
      return dld_can_preload = 0;
    }

  fgets (buffer, BUFSIZ, fp);
  fclose (fp);
  unlink (file);
  if ((version = strstr (buffer, "B.11")) == 0)
    return dld_can_preload = 0;

  version += 5;
  minor = atoi (version);

  dld_can_preload = (minor >= 19);
  return dld_can_preload;
#endif
}

/* Reset all the globals to prepare for runtime-checking. This function is
   called everytime gdb starts debugging a process.  */
void
prepare_for_rtc (int attached)
{
  struct objfile * objfile;
  int i;
  struct minimal_symbol * m;

  libdld_dot_sl = libc_dot_sl = rtc_dot_sl = 0;
#if !defined(HP_IA64) && !defined(GDB_TARGET_IS_HPPA_20W)
  rtc_dot_sl_got_value = 0;
#endif

  for (i=0; i < leak_count; i++)
    free (leak_info[i].pc);

  for (i=0; i < block_count; i++)
    free (block_info[i].pc);

  free (leak_info);
  free (block_info);
  is_threaded = 0;
  is_cma_threaded = 0;
  is_dce_threaded = 0;
  block_count = 0;
  leak_count = 0;
  leak_info_capacity = 0;
  leak_info = 0;
  block_info = 0;
  total_leak_bytes = total_bytes = 0;
  total_leak_blocks = 0;
  total_blocks = 0;
  heap_info_plabel = 0;
  leaks_info_plabel = 0;
  interval_info_plabel = 0;
  att_flag = 0;

  /* Enable RTC after attach */
#if 0
  if (attached && check_heap)
    {
      warning ("GDB cannot check the heap in a process already running.");
      warning ("You will have to restart the program for heap debugging.");
      check_heap_in_this_run = 0;
    }
  else
#endif

#if !defined(HP_IA64) && !defined(GDB_TARGET_IS_HPPA_20W)
/* JAGaf37695 - RTC_INIT should be turned 'on' to collect heap information 
   from startup of the process for PA32 application. 
   $ LD_PRELOAD=/opt/langtools/lib/librtc.sl RTC_INIT=on <executable>
   
   Caution : If RTC_INIT is turned 'on', librtc starts recording 
   heap information by default for PA32 process. So, user should 
   avoid export-ing this environment variable for shell and set 
   only when appropriate. */

if (attached && check_heap)
  {
    m = lookup_minimal_symbol ("__pthreads_not_ready", 0, rtc_dot_sl);
    if (m)
      {
        int librtc_not_initialized = 1;
        target_read_memory (SYMBOL_VALUE_ADDRESS (m), (char *)&librtc_not_initialized, 4);
        if (librtc_not_initialized) /* set to 0 by librtc once initialized */
    	  warning ("Heap information provided may not be accurate for the attached process.\nHint: To get complete heap information from startup of the attached process set environment variable RTC_INIT to `on' in command line.");
      }
  }
#endif

  check_heap_in_this_run = check_heap;

  /* Remove any RTC event breakpoint we inserted in the previous run */
  remove_rtc_event_breakpoints ();
  rtc_initialized = 0;


  if (check_heap_in_this_run)
    dld_can_preload = dld_understands_preload ();

  symfile_objfile_patched = 0;
  att_flag = attached; /* is debuggee an attached process. */
}

/* Check if the debugee is a threaded program. */
boolean
threaded_program ()
{
  struct objfile * objfile;
  int is_threaded = 0;

  ALL_OBJFILES (objfile)
    {
      if (strstr (objfile->name, "/libcma."))
        is_threaded = is_cma_threaded = 1;
      else
      if (strstr (objfile->name, "/libdce."))
        is_threaded = is_dce_threaded = 1;
      else
      if (strstr (objfile->name, "/libpthread."))
        is_threaded = 1;
    }

  return is_threaded;
}

/* Return the objfile associated with libc. */
static struct objfile * 
locate_libc ()
{
  struct objfile * objfile;
  struct minimal_symbol * m;

  ALL_OBJFILES (objfile)
    {
      if (strstr (objfile->name, "/libc."))
        {
          /* paranoia could destroya ... */
          m = lookup_minimal_symbol_text ("fopen", NULL, objfile);
          if (m)
            return objfile;
        }
    }

  return 0;
}

/* Return the librtc's objfile. */
static struct objfile * 
locate_libdld ()
{
  struct objfile * objfile;
  struct minimal_symbol * m;

  ALL_OBJFILES (objfile)
    {
      if (strstr(objfile -> name,"/libdl.")||strstr(objfile -> name,"/libdld."))
        {
          /* paranoia could destroya ... */
          m = lookup_minimal_symbol_text ("shl_unload", NULL, objfile);
          if (m)
            return objfile;
        }
    }
    return 0;
}


static struct objfile * 
locate_librtc ()
{
  struct objfile * objfile;
  struct minimal_symbol *m;
  int i;

  ALL_OBJFILES (objfile)
    {
      /* in future we could make it a full path name */
      if (!strstr (objfile->name, "/librtc"))
        continue;
      
      for (i=0; i < (sizeof(wrappers) / sizeof(wrappers[0])); i++)
        {
          char name[100];
          strcpy (name, wrappers[i].rtc_name);
          strcat (name, "_plabel");
          m = lookup_minimal_symbol (name, 0, objfile);
          target_read_memory (SYMBOL_VALUE_ADDRESS (m),
                              (char *) &wrappers[i].pointer_to_plabel,
                              sizeof (CORE_ADDR));

          m = lookup_minimal_symbol_text
                           (wrappers[i].rtc_name, 0, objfile);
          wrappers[i].function_pointer = SYMBOL_VALUE_ADDRESS (m);
        }
 
#if !defined(HP_IA64) && !defined(GDB_TARGET_IS_HPPA_20W)
      rtc_dot_sl_got_value = 
                    SOLIB_GET_GOT_BY_PC (SYMBOL_VALUE_ADDRESS (m));
#endif
      
      return objfile;
    }
    return 0;
}

#if !defined(GDB_TARGET_IS_HPPA_20W) && !defined(HP_IA64)
static int
som_patchup_definitions (struct objfile * objfile)
{
  int i;
  struct minimal_symbol * m;
  CORE_ADDR address;
  int inst;

  for (i=0; i < sizeof(wrappers) / sizeof(wrappers[0]); i++)
    {
      m = lookup_minimal_symbol_text (wrappers[i].name, 0, objfile);

      /* ignore local definitions, if any */
      if (!m || MSYMBOL_TYPE (m) == mst_file_text)
        continue;

      /* Now for the dirty work ... bulk of the code below is
         courtesy of f&c effort. We need room for seven instructions.
         Since we are going by the assumption that any exported function
         by the name of malloc/free (and so on) is actually malloc and
         free (and so on), it is reasonable to expect that there will
         be atleast seven instructions in it.

         Even where malloc simply forwards the call to a "real_malloc",
         this has to be true. For, we need one instruction each to store
         the return pointer in the caller's frame, set up a stack frame,
         branch to the real routine, idle in the delay slot, retrieve
         the return address, branch to the caller, and finally idle in
         the delay slot.
      */

      address = SYMBOL_VALUE_ADDRESS (m);

      /* LDIL rtc_dot_sl_got_value , r19 */
      inst = 0x22600000;
      inst = deposit_21 (rtc_dot_sl_got_value >> 11, inst);
      target_write_memory (address, (char *)&inst, 4);

      /* LDO rtc_dot_sl_got_value(r19), r19 */
      inst = 0x36730000;
      inst = deposit_14 (rtc_dot_sl_got_value & MASK_11, inst);
      target_write_memory (address + 4, (char *)&inst, 4);

      /* LDIL rtc_wrapper() , r22 */
      inst = 0x22c00000;
      inst = deposit_21 (wrappers[i].function_pointer >> 11, inst);
      target_write_memory (address + 8, (char *)&inst, 4);

      /* LDO rtc_wrapper(r22), r22 */
      inst = 0x36d60000;
      inst = deposit_14 (wrappers[i].function_pointer & MASK_11, inst);
      target_write_memory (address + 12, (char *)&inst, 4);

      /* let us work with the least common denominator (1.1) */
      inst = 0x02c010bf; /*  ldsid 0(r22), r31 */
      target_write_memory (address + 16, (char *)&inst, 4);

      inst = 0x001f1820; /*  mtsp r31,sr0 */
      target_write_memory (address + 20, (char *)&inst, 4);

      inst = 0xe2c00002; /* BE,n (sr0, r22) */
      target_write_memory (address + 24, (char *)&inst, 4);
    }
    return 1;
}
#endif

#ifdef GDB_TARGET_IS_HPPA_20W
static int
pa64_patchup_definitions (struct objfile * objfile)
{
  int i;
  struct minimal_symbol * m;
  CORE_ADDR address;
  int inst;

  for (i=0; i < sizeof(wrappers) / sizeof(wrappers[0]); i++)
    {
      int temp;
      m = lookup_minimal_symbol_text (wrappers[i].name, 0, objfile);

      /* ignore local definitions, if any */
      if (!m || MSYMBOL_TYPE (m) == mst_file_text)
        continue;

      /* For the time being disable RTC ... We did not implement
	 patchup definitions for PA64 */
      return 0;
    }

    return 1;
}
#endif

#ifdef HP_IA64
static int
ia64_patchup_definitions (struct objfile * objfile)
{
  int i;
  struct minimal_symbol * m;
  CORE_ADDR address;
  int inst;
  for (i=0; i < sizeof(wrappers) / sizeof(wrappers[0]); i++)
    {
      int temp;
      m = lookup_minimal_symbol_text (wrappers[i].name, 0, objfile);

      /* ignore local definitions, if any */
      if (!m || MSYMBOL_TYPE (m) == mst_file_text)
        continue;

      /* For the time being disable RTC ... We didnot implement 
	 patchup_definitions on IA64. */
      return 0;
    }

    return 1;
}
#endif

/* Set the value of the variable (in librtc) whose type is 
   pointer in the inferior */
static void
set_inferior_pointer_var_value (char * name, CORE_ADDR value)
{
  struct minimal_symbol * m;
  CORE_ADDR anaddr;
  char buf[8];
  int status;

  if (rtc_dot_sl == 0)  /* Can happen for PA64 or IA64 */
    return;

  m = lookup_minimal_symbol (name, NULL, rtc_dot_sl);
  if (!m)
#ifdef GDB_TARGET_IS_HPUX
    error ("Internal error : %s missing in librtc!\nIt might be due to version mismatch of librtc and gdb(Version %s).\nHint: Use environment variable LIBRTC_SERVER to pick correct version of librtc.", name, wdb_version_nbr);
#else 
    error ("Internal error : %s missing in librtc!", name);
#endif

  anaddr = SYMBOL_VALUE_ADDRESS (m);

  if (PTR_SIZE == 8)
    store_address (buf, PTR_SIZE, value);
  else
    {
      /* If the ptr size is 4, just write it as an interger.
         This is important for ilp32 when the value is swizzled. */
      store_unsigned_integer (buf, PTR_SIZE, value);
    }

  status = target_write_memory (anaddr, buf, PTR_SIZE);
  if (status != 0)
    error ("Internal error : Unable to update %s!", name);
}


/* Set the value of a variable with the given integer value. */
static void
set_inferior_var_value (char * name, int value)
{
  struct minimal_symbol * m;
  CORE_ADDR anaddr;
  char buf[4];
  int status;

  if (rtc_dot_sl == 0)  /* Can happen for PA64 or IA64 */
    return;

  m = lookup_minimal_symbol (name, NULL, rtc_dot_sl);
  if (!m)
#ifdef GDB_TARGET_IS_HPUX
    error ("Internal error : %s missing in librtc!\nIt might be due to version mismatch of librtc and gdb(Version %s).\nHint: Use environment variable LIBRTC_SERVER to pick correct version of librtc.", name, wdb_version_nbr);
#else
    error ("Internal error : %s missing in librtc!", name);
#endif  

  anaddr = SYMBOL_VALUE_ADDRESS (m);
  store_unsigned_integer (buf, 4, value);
  status = target_write_memory (anaddr, buf, 4);
  if (status != 0)
    error ("Internal error : Unable to update %s!", name);
}


/* This function is called at startup and everytime there is a library load.
   It initializes the librtc and makes it ready to start memory checking.
 */
void 
snoop_on_the_heap ()
{
  struct minimal_symbol *m;
  CORE_ADDR pc;

  if (!target_has_stack)
    return;

  if (!check_heap_in_this_run)
    return;

  if (!is_threaded)
    {
      is_threaded = threaded_program ();
#ifdef HPUX_1020
      /* Cannot do heap checking on userspace thread. */
      if (is_threaded)
        {
          check_heap = check_heap_in_this_run = 0;
          warning ("GDB cannot detect leaks in a user space threaded program.");
          return;
        }
#else
      if (is_cma_threaded || is_dce_threaded)
        {
          check_heap = check_heap_in_this_run = 0;
          warning ("GDB cannot detect leaks in a user space threaded program.");
          return;
        }
#endif
    }

  /* Make sure that librtc and libc.sl are linked in. */


  /* There is a minor difference in the startup model between PA32 and
     PA64 which needs to be dealt with: on PA32, when we receive the
     first notification from dld about shared libraries being mapped,
     all implicitly linked shared libraries have been mapped in. On
     PA64, we receive one call back per library. So if a library we 
     are looking for is missing, we cannot complain and disable RTC
  */
  if (rtc_dot_sl == 0)
    if ((rtc_dot_sl = locate_librtc ()) == 0)
      return;
 
  if (libc_dot_sl == 0)
    if ((libc_dot_sl = locate_libc ()) == 0)
      return;

  if (libdld_dot_sl == 0)
    if ((libdld_dot_sl = locate_libdld ()) == 0)
      return;

  DEBUG(printf("calling snoop_on_the_heap\n");)

  /* The way we intercept and reroute calls to [de]allocation
     functions is by  preloading the librtc before any other
     libraries get loaded and forcing the dld to bind any calls to
     these symbols to the symbols in librtc.

     However, this would not work if the a.out *defines* any of the
     symbols that we want to intercept.

     This situation is not uncommon. There are numerous programs
     including gdb itself that have there own homegrown versions
     of mallocators. Many products in CLL (compilers) also have
     a private implementation of malloc package that gets bound
     into the a.out.
     
     We will cope with these bad programs as follows : We will
     simply patch the definitions and turn them into a special
     kind of import stub : one that does not consult the PLT, but
     in effect does the same : Jump to the appropriate wrapper
     routine after updating the linkage table pointer with the
     correct value.
  */

  if (symfile_objfile &&
      !symfile_objfile_patched)
    {
      if (!PATCHUP_DEFINITIONS (symfile_objfile))
        {
          warning ("Your program seems to override the C library functions.");
          warning ("This behavior is not supported yet for this platform !");
          check_heap = check_heap_in_this_run = 0;
          return;
        }
      symfile_objfile_patched = 1;
    }

  /* The leak detector used to determine the list of currently mapped
     libraries by using shl_get (). This is unsafe in a threaded
     program, as the call could block. Since a library has been loaded
     or unloaded just now, let the RTC module know that its library
     list is not valid. The list will be automatically updated on the
     next call to malloc () or free ().  Garbage collection is not
     possible in the interim. This should not be an issue as most
     programs don't shl_load () anyway and cannot shl_unload unless
     they shl_load (). Even in the case where libraries are dynamically
     loaded, this interval should be negligible.

     The alternative to this scheme is ugly, requiring gdb to update
     the data structures in the inferior and it is not always doable
     without advancing the inferior and that could trigger breakpoints
     and really mess up things.
  */

  set_inferior_var_value ("shlib_info_invalid", 1);
     
  if (!rtc_initialized)  /* one time chore */
    {
       /* Added for batch rtc support. Following initializations are 
          not needed for batch rtc as they have been already been done 
          thru config file in librtc. Also, when gdb is invoked in batch
          rtc, you are at the very end of the program - libc.so.1 is being 
          unloaded by dynamic loade; Nothing much other than printing the 
          leaks information would happen before the a.out is unloaded. So, 
          there is no need to do that. We still need to get plabel info of
          commands which are executed to get leaks and heap info.
       */
 
       if (!batch_rtc) 
         {
           /* Insert an internal breakpoint in rtc_event () to trap sos
              signals and calls to free with non heap objects.
            */

           m = lookup_minimal_symbol_text ("__rtc_event", 0, rtc_dot_sl);
           if (m)
             {
                pc = SYMBOL_VALUE_ADDRESS (m);
                /*  JAGaf36306
                   We want to have breakpoint at the absolute 0 of the
                   function because, later, we read arg registers of 
		   __rtc_event. If the control is stopped after the 
		   prologue, it is possible that some of the arg 
                   registers have been trashed by the compiler. Since
                   the user is not going to do source debugging of this
                   function, it is ok not to skip the prologue. The side
                   effect is thta the arguments of the top frame will not
                   show correctly but this is true for a lot of places
                   where we stop at the absolute 0 of a function.
                */
                /*pc = SKIP_PROLOGUE (pc);*/
                create_rtc_event_breakpoint (pc);
             }


           /* Tune the knobs in the inferior per user's request ... */
           set_inferior_var_value ("__rtc_check_bounds", check_bounds);
           set_inferior_var_value ("__rtc_check_free", check_free);
           set_inferior_var_value ("__rtc_scramble_blocks", scramble_blocks);

           /* If we attached, user might have set the frame_count by calling
	      the __rtc_init_leaks. We don't want to overwrite that. */
           if (att_flag)
	     {
                struct minimal_symbol *m = lookup_minimal_symbol 
				     ("__rtc_frame_count", 0, rtc_dot_sl);
                if (m)
	          {
	             int fc;
	             target_read_memory (SYMBOL_VALUE_ADDRESS (m), (char *)&fc, 4);
	             frame_count = fc;
                  }
             }
           set_inferior_var_value ("__rtc_frame_count", frame_count);
           set_inferior_var_value ("__rtc_min_leak_size", min_leak_size);

           /* null check related inits */
           set_inferior_var_value ("__rtc_null_check_size", null_check_size);
           set_inferior_var_value ("__rtc_null_check_count", null_check_count);
           set_inferior_var_value ("__rtc_null_check_seed_value", 
                                     null_check_seed_value);
           set_inferior_var_value ("__rtc_null_check_random_range", 
                                     null_check_random_range);

           /* interval check related inits */
           set_inferior_var_value ("__rtc_check_heap_interval", 
                                     check_heap_interval_value);
           set_inferior_var_value ("__rtc_check_heap_repeat", 
                                   check_heap_interval_repeat_value);
           set_inferior_var_value ("__rtc_check_heap_reset", 
                                     check_heap_interval_reset_value);
            check_heap_interval_reset_value = 0;

           set_inferior_var_value ("__rtc_big_block_size", big_block_size);
           set_inferior_var_value ("__rtc_heap_growth_size", heap_growth_size);
           set_inferior_pointer_var_value ("__rtc_watch_address", 
							watch_address);

           /* Initialize the stack pointer in librtc. We should be stopped 
              at _start (or dld on PA64.) This SP value is not the same as the 
              stack base, but the RTC module will recompute it. The reason we 
              want to do it here is to prevent the LIBCL bug from from rearing 
              its ugly head right at startup. See comments in infrtc.c
            */

           set_inferior_pointer_var_value ("stack_base",
                                  read_register (SP_REGNUM));
#if !defined(HP_IA64) && !defined(GDB_TARGET_IS_HPPA_20W)
           set_inferior_var_value ("__pthreads_not_ready", 0);
#endif
        } /* if (!batch_rtc) ... */

      /* Locate plabels to debugger hooks. This is to avoid doing it
         when making the command line call. It is not safe to postpone
         it till then since shl_findsym uses mutexes and a deadlock may
         result. (See find_stub_with_shl_get in hppa-tdep.c)
      */
      m = lookup_minimal_symbol ("leaks_info_plabel", 0, rtc_dot_sl);
      if (!m)
        error ("Internal error : Leak detector missing !\n");
      target_read_memory (SYMBOL_VALUE_ADDRESS (m),
                              (char *) &leaks_info_plabel,
                              sizeof (CORE_ADDR));
                      
      m = lookup_minimal_symbol ("heap_info_plabel", 0, rtc_dot_sl);
      if (!m)
        error ("Internal error : Heap analyzer missing !\n");
      target_read_memory (SYMBOL_VALUE_ADDRESS (m),
                              (char *) &heap_info_plabel,
                              sizeof (CORE_ADDR));
                      
      m = lookup_minimal_symbol ("interval_info_plabel", 0, rtc_dot_sl);
      if (!m)
        error ("Internal error : Heap analyzer missing !\n");
      target_read_memory (SYMBOL_VALUE_ADDRESS (m),
                              (char *) &interval_info_plabel,
                              sizeof (CORE_ADDR));
      
      rtc_initialized = 1;
    }
}

void
decode_nomem_event ()
{
  enum rtc_event ecode = 0;;
  CORE_ADDR pointer = 0;;
  char address[128];

  ecode = (CORE_ADDR) read_register (ARG0_REGNUM);
  pointer = (CORE_ADDR) read_register (ARG1_REGNUM);
  strcpy (address,
              longest_local_hex_string_custom ((LONGEST) pointer,"08l"));

  switch (ecode)
    {
    case RTC_NOMEM : 
      warning ("Malloc is returning simulated %s value", address);
      return;
    /* JAGaf48255 - gdb should report if malloc returns null pointer */
    case RTC_MEM_NULL :  
      warning ("Memory Allocation is returning %s value", address);
      return;
    default : 
      warning ("Unknown RTC null-check event at %s", address);
      return;
    }
}

void
print_context_for_overrun(char *blkp, CORE_ADDR pclist, size_t sz, int type)
{
   int i, status;
   int free_function = 0;

   CORE_ADDR addr;

   char local_buf[4096]; 
   char *function = "???";

   struct minimal_symbol *m;
   struct symtab_and_line sal;

   if (!pclist)
     {
        if (type) /* header corrupted */
          sprintf(local_buf,
            "Memory block (size = %d address = %s) appears to be corrupted at the beginning.\nAllocation context not found\n\t", sz, blkp);
        else
          sprintf(local_buf,
            "Memory block (size = %d address = %s) appears to be corrupted at the end.\nAllocation context not found\n\t", sz, blkp);
        warning(local_buf); 
        return;
     }

   if (type)
      sprintf(local_buf,
        "Memory block (size = %d address = 0x%x) appears to be corrupted at the beginning.\nLikely block allocation context is\n\t", sz, blkp);
   else
      sprintf(local_buf,
        "Memory block (size = %d address = 0x%x) appears to be corrupted at the end.\nLikely block allocation context is\n\t", sz, blkp);

   warning(local_buf); 

   /* get the frames for this pclist */
   for (i = 0; pclist && i < frame_count; i++)
     {
       status = target_read_memory (pclist + PTR_SIZE*i, (char*) &addr, 
                  PTR_SIZE);
       if (status != 0)
         error("Error downloading data !");

       /* Shift if pointer size is not equal to the sizeof CORE_ADDR */
       if (PTR_SIZE != sizeof (CORE_ADDR))
	 addr = addr >> (sizeof (CORE_ADDR) - PTR_SIZE) * 8;
          
       if (m = lookup_minimal_symbol_by_pc ((CORE_ADDR) SWZ (addr)))
         {
            function = SYMBOL_SOURCE_NAME (m);
            DEMANGLE_NEW(function);
            sal = find_pc_line (addr, 0);
            printf_filtered("#%-2d %s()", i+1, function);

            if (sal.symtab && sal.symtab->filename && sal.symtab->filename[0])
              printf_filtered (" at %s:%d\n", sal.symtab->filename, sal.line);
            else 
              printf_filtered (" from %s\n",PC_SOLIB(addr));

            if (free_function)
              {
                 free (function);
                 free_function = 0;
              }
          }
        else 
          {
             if (i == 0)
               printf_filtered("#%-2d %s()", i+1, "unknown_procedure");
          }
          addr = 0;
       }

    strcpy(local_buf, "Use command backtrace (bt) to see the current context.\n\nIgnore top 3 frames belonging to leak detection library of gdb\n");
    warning(local_buf); 

    return;
}

void
decode_rtc_event ()
{
  size_t size;
  enum rtc_event ecode;
  CORE_ADDR pointer = 0, pcl;
  char address[128];
 
  /* At this point, the inferior is stopped in __rtc_event (). The first
     parameter is the event code and second one is a pointer, which
     may not make sense for all event codes. While reporting the event
     use warnings rather than printf_* since warning automatically
     annotates the message suitably.
  */

  ecode = (CORE_ADDR) read_register (ARG0_REGNUM);
  DEBUG(printf ("ecode = %x reg =%x\n", ecode, ARG0_REGNUM);)
  pointer = (CORE_ADDR) read_register (ARG1_REGNUM);

  pcl = (CORE_ADDR) read_register (ARG2_REGNUM);
  size = (CORE_ADDR) read_register (ARG3_REGNUM);

  strcpy (address,
              longest_local_hex_string_custom ((LONGEST) pointer,"08l"));

  switch (ecode)
    {
    case RTC_BAD_FREE : 
      warning ("Attempt to free unallocated or already freed object at %s", address);
      return;

    case RTC_HEAP_GROWTH :
      warning ("Attempt to grow the heap at %s", address);
      return;

    case RTC_HUGE_BLOCK :
      warning ("Attempt to allocate a large object at %s", address);
      return;

    case RTC_BAD_REALLOC :
      warning ("Attempt to realloc an unallocated object at %s",
                                                 address);
      return;

    case RTC_ALLOCATED_WATCHED_BLOCK :
      warning ("Watched address %s allocated", address);
      return;

    case RTC_DEALLOCATED_WATCHED_BLOCK :
      warning ("Watch address %s deallocated", address);
      return;

    case RTC_BAD_HEADER :
      print_context_for_overrun(address, pcl, size, 1);
      return;

    case RTC_BAD_FOOTER :
      print_context_for_overrun(address, pcl, size, 0);
      return;

    /* As of now all other event codes indicate some kind of internal
       failure. We cannot continue and the RTC module will disable 
       itself. The king is dead, long live the king !
    */

    case RTC_NO_MEMORY : 
      warning ("The child process ran out of memory.");
      return;

    case RTC_SBRK_NEGATIVE :
      warning ("Attempt to shrink the heap ! Heap debugging disabled.");
      return;

    /* If for some reason, we had trouble determining the stack base or
       the C library entry points, let us not wash the dirty linen in
       public, just quit with a generic "internal error" message.
    */

    case RTC_MALLOC_MISSING :
    case RTC_STACK_MISSING :
      warning ("Internal error in GDB. Please notify HP.");
      warning ("Heap debugging disabled. Process otherwise debuggable.");
      return;
    }
}

 
/* Print the information about a leak (leak_no). Write it either to
   stdout or to fp if provided. */ 
static void
report_this_leak (int leak_no, FILE * fp)
{
    int i;
    int free_function = 0;
    struct memory_info * leak;
    char local_buf[4096];

    if (leak_no < 0 || leak_no >= leak_count)
      error ("Not a valid leak number.");

    leak = leak_info + leak_no;

    if (leak->count == 1)
      {
        /* Only 1 block. Print the address then. */
        /* srikanth, 000221, for some reason, printf_filtered ()
           crashed when we try to print long long values. This
           does not always happen, only sometimes. Just be safe
           and use sprintf () and puts_filtered ().
        */
        sprintf (local_buf, "%lld bytes leaked at %s "
                       "(%2.2f%% of all bytes leaked)\n",
                               leak->size,
           longest_local_hex_string_custom ((LONGEST) leak->address, "08l"),
                          ((((float) leak->size) / total_leak_bytes) * 100));
        if (fp)
          fputs (local_buf, fp);
        else 
          puts_filtered (local_buf);
      }
    else 
      {
        sprintf (local_buf, "%lld bytes leaked in %d blocks "
                      "(%2.2f%% of all bytes leaked)\n",
                                  leak->size, leak->count,
                          ((((float) leak->size) / total_leak_bytes) * 100));
        if (fp)
          fputs (local_buf, fp);
        else 
          puts_filtered (local_buf);

        sprintf (local_buf, "These range in size from "
                            "%lld to %lld bytes %s\n",
                                  leak->min, leak->max,
                                  leak->pc ? "and are allocated" : "");
        if (fp)
          fputs (local_buf, fp);
        else 
          puts_filtered (local_buf);
      }

    if (!leak->pc)
      return;

    /* Print the frame stack. */
    for (i=0; i < frame_count; i++)
      {
        struct symtab_and_line sal;
        struct minimal_symbol *m;
        char * name = "???";
        CORE_ADDR pc_addr = (CORE_ADDR) SWZ (leak->pc[i]);
      
        /* For some reason when the stack is completely unwound
           we end up with a pc value of 0xfffffffc. This is not
           the documented behavior of libcl. Just deal with it ...
        */
        if (leak->pc[i] == 0 ||
            leak->pc[i] == (CORE_ADDR) 0xfffffffc)
          break;

        m = lookup_minimal_symbol_by_pc (pc_addr);
        if (m)
          name = SYMBOL_SOURCE_NAME (m);

	/* JAGae74363 Diwakar Nag */
        DEMANGLE_NEW(name)

        sal = find_pc_line (pc_addr, 0);

	/* JAGae78845 Diwakar Nag */
        if (fp)
          fprintf (fp, "#%-2d %s()", i, name);
        else 
          printf_filtered("#%-2d %s()", i, name);

        if (free_function) {
          free(name);
	  free_function = 0;
        }

        if (sal.symtab && sal.symtab->filename && 
                                sal.symtab->filename[0])
          {
            if (fp)
              fprintf (fp, " at %s:%d\n", sal.symtab->filename, sal.line);
            else 
              printf_filtered (" at %s:%d\n", sal.symtab->filename, sal.line);
          }
        else 
          {
            if (fp)
              fprintf (fp, " from %s\n",PC_SOLIB(pc_addr));
            else 
              printf_filtered (" from %s\n",PC_SOLIB(pc_addr));
          }
      }
}


static void
print_rtc_stack_frame (struct memory_info *block, FILE *fp)
{
   int i;
   int free_function = 0;
   char local_buf[4096];

   for (i=0; i < frame_count; i++)
     {
        struct symtab_and_line sal;
        struct minimal_symbol *m;
        char * name = "???";
        CORE_ADDR pc_addr = (CORE_ADDR) SWZ (block->pc[i]);

        /* For some reason when the stack is completely unwound
           we end up with a pc value of 0xfffffffc. This is not
           the documented behavior of libcl. Just deal with it ...
        */
        if (block->pc[i] == 0 ||
            block->pc[i] == (CORE_ADDR) 0xfffffffc)
          break;

        m = lookup_minimal_symbol_by_pc (pc_addr);
        if (m)
          name = SYMBOL_SOURCE_NAME (m);

	/* JAGae74363 Diwakar Nag */
        DEMANGLE_NEW(name);

        /* JAGae78845 Diwakar Nag */
        sal = find_pc_line (pc_addr, 0);

        if (fp)
          fprintf (fp, "#%-2d %s()", i, name);
        else 
          printf_filtered("#%-2d %s()", i, name);

        if (free_function) 
          {
             free(name);
	     free_function = 0;
          }

       if (sal.symtab && sal.symtab->filename && sal.symtab->filename[0])
	 {
            if (fp)
              fprintf (fp, " at %s:%d\n", sal.symtab->filename, sal.line);
            else 
              printf_filtered (" at %s:%d\n", sal.symtab->filename, sal.line);
	 }
       else 
         {
            if (fp)
              fprintf (fp, " from %s\n",PC_SOLIB(pc_addr));
            else 
              printf_filtered (" from %s\n",PC_SOLIB(pc_addr));
         }
      }
   return;
}


static void
report_this_interval_block (struct memory_info *block, FILE * fp, int tot_bytes)
{
    char local_buf[200];
    int free_function = 0;

    if (block->count == 1)
      {
        sprintf (local_buf, "%lld bytes at %s "
                    "(%2.2f%% of all bytes allocated)\n",
                  block->size, longest_local_hex_string_custom (
                        (LONGEST) block->address, "08l"),
               ((((float) block->size) / tot_bytes) * 100));
        if (fp)
          fputs (local_buf, fp);
        else 
          puts_filtered (local_buf);
      }
    else 
      {
        sprintf (local_buf, "%lld bytes in %d blocks "
                      "(%2.2f%% of all bytes allocated)\n",
                             block->size, block->count,
                       ((((float) block->size) / tot_bytes) * 100));
        if (fp)
          fputs (local_buf, fp);
        else 
          puts_filtered (local_buf);

        sprintf (local_buf, "These range in size from "
                            "%lld to %lld bytes %s\n",
                             block->min, block->max,
                             block->pc ? "and are allocated" : "");

        if (fp)
          fputs (local_buf, fp);
        else 
          puts_filtered (local_buf);
      }

    if (!block->pc)
      return;

    print_rtc_stack_frame(block, fp);
    return;
}

/* Print out the information about a heap block (block_no). Write it
   either to stdout or to fp if provided. 
*/

static void
report_this_block (int block_no, FILE * fp, int type)
{
    int i;
    char local_buf[200];
    int free_function = 0;
    struct memory_info * block;

    if (block_no < 0 || block_no >= block_count)
      error ("Not a valid block number.");
  
    block = block_info + block_no;

    if (type == BOUNDS_INFO && !block->corrupted)
      error ("Block number not corrupted.");

    if (block->count == 1)
      {
        sprintf (local_buf, "%lld bytes at %s "
                    "(%2.2f%% of all bytes allocated)\n",
                  block->size, longest_local_hex_string_custom (
                        (LONGEST) block->address, "08l"),
               ((((float) block->size) / total_bytes) * 100));
        if (fp)
          fputs (local_buf, fp);
        else 
          puts_filtered (local_buf);
      }
    else 
      {
        sprintf (local_buf, "%lld bytes in %d blocks "
                      "(%2.2f%% of all bytes allocated)\n",
                             block->size, block->count,
                       ((((float) block->size) / total_bytes) * 100));

        if (fp)
          fputs (local_buf, fp);
        else 
          puts_filtered (local_buf);

        sprintf (local_buf, "These range in size from "
                            "%lld to %lld bytes %s\n",
                             block->min, block->max,
                             block->pc ? "and are allocated" : "");

        if (fp)
          fputs (local_buf, fp);
        else 
          puts_filtered (local_buf);
      }

    if (!block->pc)
      return;

    print_rtc_stack_frame(block, fp);
    return;
}

/* Print out a report on all the leaks provided by leaks_info.
   If fp is provided print a detailed report. If not write to
   stdout a brief desc of all the leaks. */
static void
print_leaks (struct memory_info *leak_info, FILE * fp)
{
  int i = 0;
  int free_function = 0;
  char local_buf[4096];

  if (!leak_count)
    return;

  sprintf (local_buf, "\n%lld bytes leaked in %d blocks\n\n",
                      total_leak_bytes, total_leak_blocks);
  if (fp)
    fputs (local_buf, fp);
  else
    puts_filtered (local_buf);

#ifdef GDB_TARGET_IS_HPPA_20W
  if (!fp)
    printf_filtered ("No.   Total bytes     Blocks         Address        "
                                             " Function\n");
  else 
    fprintf (fp, "No.   Total bytes     Blocks         Address        "
                                             " Function\n");
#else
  if (!fp)
    printf_filtered ("No.   Total bytes     Blocks     Address     Function\n");
  else 
    fprintf (fp, "No.   Total bytes     Blocks     Address     Function\n");
#endif

  for (i=0; i < leak_count; i++, leak_info++)
    {
      struct minimal_symbol * m;
      char * function = "???";

      if (leak_info->pc)
        {
          m = lookup_minimal_symbol_by_pc ((CORE_ADDR) SWZ (leak_info->pc[0]));
          if (m)
            function = SYMBOL_SOURCE_NAME (m);

	  /* JAGae74363 Diwakar Nag */
          DEMANGLE_NEW(function);
        }

        sprintf (local_buf, "%-3d      %-10lld     %-6d  %s   %s()\n", 
                                        i, leak_info->size,
                                             leak_info->count,
        longest_local_hex_string_custom ((LONGEST) leak_info->address,
                                            "08l"),
                                             function);
        if (free_function)
          {
             free (function);
             free_function = 0;
          }

        if (!fp)
          puts_filtered (local_buf);
        else 
          fputs (local_buf, fp);
       }

  if (fp)
    {
        fprintf (fp, "\n\n-------------------------------------------------------------------------\n");
        fprintf (fp, "\n"
"                              Detailed Report\n");
        fprintf (fp, "\n-------------------------------------------------------------------------\n");

        for (i=0; i < leak_count; i++)
          {
            report_this_leak (i, fp);
        fprintf (fp, "\n-------------------------------------------------------------------------\n");
          }
    }
  return;
}

static void
print_interval_blocks (FILE * fp, int total_records)
{
  int i = 0, idx;
  int free_function = 0;

  char *t_p;
  char local_buf[4096];

  struct memory_info *block_ptr;

  for (idx = 0; idx < total_records; idx++)
    {
       t_p = local_buf;
       t_p += sprintf (local_buf, 
        "\n===========================================================\n\n");

       t_p += sprintf (t_p, "Start Time: %s End Time: %s\nInterval: %d\n", 
           heap_idx[idx].start_time, heap_idx[idx].end_time, 
           heap_idx[idx].interval);

       t_p += sprintf (t_p, "%lld bytes allocated in %d blocks\n this n\n",
           heap_idx[idx].total_bytes,heap_idx[idx].num_blocks);

       if (fp)
         fputs (local_buf, fp);
       else
         puts_filtered (local_buf);

#ifdef GDB_TARGET_IS_HPPA_20W
       if (fp)
         fprintf (fp, "No.   Total bytes     Blocks         Address        "
                                             " Function\n");
       else 
         printf_filtered ("No.   Total bytes     Blocks         Address        "
                                             " Function\n");
#else
       if (fp)
         fprintf (fp, "No.   Total bytes     Blocks     Address     Function\n");
       else 
         printf_filtered ("No.   Total bytes     Blocks     Address     Function\n");
#endif
       for (i=0, block_ptr = heap_idx[idx].block_info; 
             i < heap_idx[idx].unq_blocks; i++, block_ptr++)
         {
            struct minimal_symbol * m;
            char * function = "???";

            if (block_ptr->pc)
              {
                 m = lookup_minimal_symbol_by_pc (
                   (CORE_ADDR) SWZ (block_ptr->pc[0]));
                 if (m)
                   function = SYMBOL_SOURCE_NAME (m);

                 DEMANGLE_NEW(function);
              }

            if (fp)
              fprintf (fp, "%-3d      %-10lld     %-6d  %s   %s()\n", i, 
                 block_ptr->size, block_ptr->count,
                 longest_local_hex_string_custom (
                   (LONGEST) block_ptr->address, "08l"), function);
            else 
              {
                 sprintf (local_buf, "%-3d      %-10lld     %-6d  %s   %s()\n",
                   i, block_ptr->size, block_ptr->count,
                   longest_local_hex_string_custom (
                     (LONGEST) block_ptr->address, "08l"), function);
                 puts_filtered (local_buf);
              }

           if (free_function) 
             {
                free(function);
	        free_function = 0;
             }
         } /* for */

      if (fp)
        {
            fprintf (fp, "\n\n-------------------------------------------------------------------------\n");
            fprintf (fp, "\n"
"                              Detailed Report\n");
            fprintf (fp, "\n-------------------------------------------------------------------------\n");
            for (i=0; i < heap_idx[idx].unq_blocks; i++)
               {
                  report_this_interval_block (
                      &(heap_idx[idx].block_info[i]), fp, heap_idx[i].total_bytes);
                  fprintf (fp, "\n-------------------------------------------------------------------------\n");
               }
         }
    }
}

/* Print out a report on all the heap blocks provided by block_info.
   If fp is provided print a detailed report. If not write to
   stdout a brief desc of all the blocks. */
static void
print_blocks (struct memory_info *block_info, FILE * fp, int type)
{
  int i = 0;
  int free_function = 0;
  char local_buf[4096];

  if (type == HEAP_INFO)
    sprintf (local_buf, "\n%lld bytes allocated in %d blocks\n\n",
                      total_bytes, total_blocks);
  else
    sprintf (local_buf, "Following blocks appear to be corrupted\n");
  if (fp)
    fputs (local_buf, fp);
  else
    puts_filtered (local_buf);

#ifdef GDB_TARGET_IS_HPPA_20W
  if (fp)
    fprintf (fp, "No.   Total bytes     Blocks         Address        "
                                             " Function\n");
  else 
    printf_filtered ("No.   Total bytes     Blocks         Address        "
                                             " Function\n");
#else
  if (fp)
    fprintf (fp, "No.   Total bytes     Blocks     Address     Function\n");
  else 
    printf_filtered ("No.   Total bytes     Blocks     Address     Function\n");
#endif

  for (i=0; i < block_count; i++, block_info++)
    {
      struct minimal_symbol * m;
      char * function = "???";

      if (type == BOUNDS_INFO && block_info->corrupted == 0)
      {
        continue;
      }

      if (block_info->pc)
        {
          m = lookup_minimal_symbol_by_pc ((CORE_ADDR) SWZ (block_info->pc[0]));
          if (m)
            function = SYMBOL_SOURCE_NAME (m);

	  /* JAGae74363 Diwakar Nag */
          DEMANGLE_NEW(function);
        }

      if (fp)
        fprintf (fp, "%-3d      %-10lld     %-6d  %s   %s()\n", i, 
                                             block_info->size,
                                             block_info->count,
           longest_local_hex_string_custom ((LONGEST) block_info->address,
                                            "08l"),
                                             function);
      else 
        {
          sprintf (local_buf, "%-3d      %-10lld     %-6d  %s   %s()\n",
                                   i, block_info->size,
                                             block_info->count,
           longest_local_hex_string_custom ((LONGEST) block_info->address,
                                            "08l"),
                                             function);
          puts_filtered (local_buf);
        }

        if (free_function) {
          free(function);
	  free_function = 0;
        }
    }

    if (fp)
      {
        fprintf (fp, "\n\n-------------------------------------------------------------------------\n");
        fprintf (fp, "\n"
"                              Detailed Report\n");
        fprintf (fp, "\n-------------------------------------------------------------------------\n");
        for (i=0; i < block_count; i++)
          {
            report_this_block (i, fp, type);
        fprintf (fp, "\n-------------------------------------------------------------------------\n");
          }
       }
}


/* Return true if the size of p2 is greater than size of p1.
   Used by qsort. */
static int
compare_leaks (const void *p1, const void *p2)
{
  struct memory_info * l1, *l2;
  
  l1 = (struct memory_info *) p1;
  l2 = (struct memory_info *) p2;

  return l2->size - l1->size;
}

/* Add this leak to our leak_info list. */
static void
add_a_leak (struct gdb_chunk_info * a_leak)
{

  int i;
  long long this_leak_size;

  this_leak_size = a_leak->end - a_leak->base;
  total_leak_bytes += this_leak_size;
  total_leak_blocks++;

  /* Have we met Mr. Leaky already ? If this is Mr. Anonymous (i.e.,
     no associated stack trace,) let us not lump it with such others.
  */

  if (a_leak->pc)
    {
      for (i=0; i < leak_count; i++)
        {
          if (leak_info[i].pointer_to_stack_trace == (CORE_ADDR) a_leak->pc)
            {
              leak_info[i].count++;
              leak_info[i].size += this_leak_size;

              if (leak_info[i].min > this_leak_size)
                leak_info[i].min = this_leak_size;
              if (leak_info[i].max < this_leak_size) 
                leak_info[i].max = this_leak_size;

              return;
            }
        }
    }

  i = leak_count;
  /* expand leak info if needed */

  if (i == leak_info_capacity)
    {
      leak_info_capacity += 4096;
      leak_info = xrealloc (leak_info, leak_info_capacity * 
                                  sizeof (struct memory_info));
    }

  leak_info[i].address = a_leak->base;
  leak_info[i].size = this_leak_size;
  leak_info[i].count = 1;
  leak_info[i].pointer_to_stack_trace = a_leak->pc;
  leak_info[i].min = leak_info[i].max = this_leak_size;

  if (a_leak->pc)
    {
      int j;

      /* srikanth, 000317, allocate an extra NULL word at the end.
         This is necessary to know where the `pc' array ends. We can't
         rely on frame_count since the user could change it.
      */
      leak_info[i].pc = calloc (frame_count + 1, sizeof (CORE_ADDR));

      if (!leak_info[i].pc)
        error ("Out of memory !");

      for (j=0; j < frame_count; j++)
        {
	  (void) target_read_memory ((CORE_ADDR) a_leak->pc +
                                         (j * PTR_SIZE),
                                       (char *) &leak_info[i].pc[j],
                                        PTR_SIZE);
	  /* Let's shift, if the read-in size is not equal to
             CORE_ADDR. */
	  if (PTR_SIZE != sizeof (CORE_ADDR))
	    leak_info[i].pc[j] = leak_info[i].pc[j] >>
				  ((sizeof (CORE_ADDR) - PTR_SIZE) * 8);

          if (leak_info[i].pc[j] == 0)
            break;
        }
      }
    else 
      leak_info[i].pc = 0;

  leak_count ++;
}

/* Returns true if the input string does not look like a number.
   This is used to determine whether the output of `info heap'
   and `info leaks' should be directed to a file. In the command
   info [heap|leaks] 123, the string 123 should be treated as a 
   heap-block|leak number. In the command `info [heap|leaks] junk',
   the output should be sent to file junk. 
*/

static int
could_be_filename (char * string)
{
  int retval = 0;
  while (*string && isdigit (*string))
    string++;
  if (*string)
    return 1;
  else 
    return 0;
}

/* Save current thread's registers onto the inferior space.
   Input is the pointer to a CORE_ADDR, which says where the
   next register value should go. CORE_ADDR will be advanced
   after save.
*/
  
static void
save_thread_registers (void * registers)
{

  register int regnum;
  char buffer[MAX_REGISTER_RAW_SIZE];

  CORE_ADDR * reg_file;

  reg_file = (CORE_ADDR*) registers;

  /* Dump the register values onto a malloc block allocated by librtc
     specifically for this purpose.
  */
#ifndef HP_IA64
  for (regnum = 1; regnum < 32; regnum++)
#else
  /* Bindu 101702: On IPF, the general registers are gr0 to gr31. */
  for (regnum = GR0_REGNUM + 1; regnum < GR0_REGNUM + 32; regnum++)
#endif
    {
      read_register_gen (regnum, buffer);
      target_write_memory (*reg_file, buffer, REGISTER_RAW_SIZE (regnum));
      *reg_file = *reg_file + REGISTER_RAW_SIZE (regnum);
    }

  /* Now for the TP register */
  regnum = TP_REGNUM;
  read_register_gen (regnum, buffer);
  target_write_memory (*reg_file, buffer, REGISTER_RAW_SIZE (regnum));
  *reg_file = *reg_file + REGISTER_RAW_SIZE (regnum);

  /* Is there any need to push floating point registers, space
     registers, control registers other than CR27, PSW, PC Q
     etc ... ?
   */

#ifdef HP_IA64
  /* Bindu 101702: For IA64, lets save all the dirty registers too.
     On IA64 (upto Madison) there can be a maximum of 96 dirty registers
     as the number of stacked physical registers are 96. */
  {
    CORE_ADDR addr_bsp, addr_bspstore;

    addr_bsp = read_register (AR0_REGNUM + 17);
    addr_bspstore = read_register (AR0_REGNUM + 18);

    if (addr_bsp > addr_bspstore)
      {
        char rse_buffer[REGISTER_RAW_SIZE (GR0_REGNUM) * 96];
        int dirty_size = addr_bsp - addr_bspstore;

        /* Make sure not to write beyond 96 in the reg_file. So set the 
           size to 96 if it is > 96. This can never happen though. */
	if (dirty_size > 96 * REGISTER_RAW_SIZE (GR0_REGNUM))
	  dirty_size = 96 * REGISTER_RAW_SIZE (GR0_REGNUM);

        target_read_memory (addr_bspstore, rse_buffer, dirty_size);
        target_write_memory (*reg_file, rse_buffer, dirty_size);
        *reg_file = *reg_file + dirty_size;
      }
  }
#endif

}


#ifdef HP_IA64
/* Bindu 101702: On IA64, we need to save the bottom/top(BSPSTORE) of
   RSE whose info is already flushed into memory for each thread.
   Librtc can use these pointers to scan this memory. 
   This function takes as an argument a pointer to the memory location
   where an inferior memory addr(CORE_ADDR) is saved. (__rtc_RSE_file).
 */
static void
save_thread_rse_state (void *rse_file)
{
  CORE_ADDR *rse_addr = (CORE_ADDR *) rse_file;
  int regnum;
  char buffer[MAX_REGISTER_RAW_SIZE];
  struct frame_info *frame = NULL, *prev_frame = NULL;

  /* Get the bottom most frame. */
  prev_frame = get_current_frame ();
  while ((prev_frame != NULL))
    {
      frame = prev_frame;
      prev_frame = get_prev_frame (frame);
      if (frame == prev_frame)
        break;
    }
  if (frame != NULL)
    {
      /* rse_bottom is bottom most frame's rse_fp. */
      CORE_ADDR rse_bottom = frame->rse_fp;
      CORE_ADDR rse_top;
      regnum = AR0_REGNUM + 18; /* BSPSTORE */
      rse_top = read_register (regnum);
      if (rse_top > rse_bottom)
      {
        /* Write the bottom of rse */
        target_write_memory (*rse_addr, 
			     (char*) &rse_bottom, 
			     sizeof (CORE_ADDR));
	*rse_addr = *rse_addr + sizeof (CORE_ADDR);


        /* Write the top of rse. */
        target_write_memory (*rse_addr, (char*) &rse_top, sizeof (CORE_ADDR));
        *rse_addr = *rse_addr + sizeof (CORE_ADDR);
      }
    }
}
#endif


/* This gets invoked when info leaks is called by the user.*/
static void  
leaks_info_command (leaknum_exp, from_tty)
     char *leaknum_exp;
     int from_tty;
{
  value_ptr funcval, val;
  struct gdb_chunk_info a_leak;
  CORE_ADDR new_leaks;
  int status; 
  int leak_num = -1;
  int leaked_more = 0;
  FILE * fp = 0;
  struct cleanup * old_chain =0; /* initialized for cmpiler warning */
  CORE_ADDR register_file = 0;
  struct minimal_symbol * m;
  int pid;
  int current_thread;
  CORE_ADDR main_stack_pointer;
  CORE_ADDR thread_stack_pointer = 0; /* initialized for cmpiler warning */
  int thread_state;

  if (!check_heap_in_this_run)
    error ("Leak detection is not enabled now.");

  /* On PA64, it is possible that we get here without the necessary libs */ 
  if (!rtc_dot_sl)
    error ("The heap debugging library (librtc) is not linked in !\nHint: Use environment variable LIBRTC_SERVER to pick correct version of librtc and upgrade to the latest linker version (greater than B.11.19);  Or link in librtc with your application to enable gdb support for heap debugging.");

  if (!libc_dot_sl)
    error ("The shared C library is not linked in !");

#ifdef HPUX_1020
  if (is_threaded)
    error ("GDB cannot detect leaks in a user space threads program.\n");
#endif 

#ifdef LOG_BETA_RTC_USED
  log_test_event (LOG_BETA_RTC_USED, 1);
#endif

  dont_repeat ();

  if (leaknum_exp)
    {
      if (could_be_filename(leaknum_exp))
        {
          /* treat it as a filename ... */
          fp = fopen (leaknum_exp, "w");
          if (!fp)
            error ("Cannot open file %s\n", leaknum_exp);
          old_chain = make_cleanup ((make_cleanup_ftype *) fclose, fp);
        }
      else 
        leak_num = atoi (leaknum_exp);

      /* If a user asks for the n'th leak and we only know about
         m leaks, where m < n, don't complain and quit. Look for
         more leaks if the process is still running.
      */
 
      if (!fp)
        {
          if (leak_num < leak_count || !target_has_stack)
            {
              report_this_leak (leak_num, 0);
              return;
            }
          /* else fall through ... */
        }
    }

  if (!target_has_stack)
    {
      if (leak_info)
        print_leaks (leak_info, fp);
      else
        error ("There is no running process.");
      goto cleanup;
    }

#ifdef HAS_END_DOT_O
  /* Making *any* command line call after a Ctrl-C confuses end.o so
     thoroughly, it sigsegvs (only on 11.00.) Even when we don't pass
     the signal. No idea why. Just bail out.
  */ 
  if (stop_signal && stop_signal != TARGET_SIGNAL_TRAP)
    error ("There are pending signals. Cannot detect leaks now.");
#endif /* HAS_END_DOT_O */

#ifndef HPUX_1020

  /* srikanth, we may be garbage collecting in the context of some
     thread other than the main thread. This poses two problems:
     Firstly, we may overrun the smaller thread stacks by using
     recursion in mark(). We cannot predict how recursive it could get.

     Secondly, the function __rtc_leaks_info () scans only the main
     stack explicitly. All thread stacks are scanned only implicitly.
     These are allocated by using mmap and there is somewhere in the
     user space a pointer to the different thread stacks. This works
     fortunately for us because pthreads is essentially a user threads
     package. The fact that these user space threads are bound 1 to 1
     to a kernel lwp is incidental.

     What we will do is this : if we are not collecting in the
     context of the main thread, then we will simply switch the
     stack to a main thread's stack and switch it back once we are done.
  */

  /* srikanth, 000928, we used to refuse to oblige if there are pending
     events for other threads. With the K threads overhaul, this
     is no longer necessary. The only condition is that the current
     thread should not have unhandled event. This is checked for
     by thread_could_run_gc() below.
  */ 

  current_thread = inferior_pid;
  pid = get_pid_for (inferior_pid);


  thread_state = thread_could_run_gc (current_thread);

  switch (thread_state) {

    case TTS_WASSUSPENDED : 
      error ("Current thread is suspended. Cannot detect leaks now.");
    case TTS_WASSLEEPING :
      error ("Current thread is blocked. Cannot detect leaks now.");
    case TTS_INSYSCALL :
      error ("Current thread is inside a syscall. Cannot detect leaks now.");
    case TTS_ATEXIT :
      error ("Current thread is about to terminate. Cannot detect leaks now.");
    case 0 :
      error ("Current thread has pending events. Cannot detect leaks now.");
  }

  /* So far so good. Save the registers. */
  m = lookup_minimal_symbol ("__rtc_register_file", NULL, rtc_dot_sl);
  if (!m)
    error ("Internal error : __rtc_register_file missing !\n");
  target_read_memory (SYMBOL_VALUE_ADDRESS (m),
                      (char *) &register_file,
                      sizeof (CORE_ADDR));

  if (PTR_SIZE != sizeof (CORE_ADDR))
    register_file = register_file >> (sizeof (CORE_ADDR) - PTR_SIZE) * 8;

  gc_in_progress = 1;
  foreach_live_thread (save_thread_registers, &register_file);

#ifdef HP_IA64
  /* Let's fill up __rtc_RSE_file with a set of bottom of rse stack
     and top of rse stack (bspstore) for each thread. */
  m = lookup_minimal_symbol ("__rtc_RSE_file", NULL, rtc_dot_sl);
  if (!m)
    error ("Internal error : __rtc_RSE_file missing !\n");
  target_read_memory (SYMBOL_VALUE_ADDRESS (m),
                      (char *) &register_file,
                      sizeof (CORE_ADDR));

  if (PTR_SIZE != sizeof (CORE_ADDR))
    register_file = register_file >> (sizeof (CORE_ADDR) - PTR_SIZE) * 8;

  gc_in_progress = 1;
  foreach_live_thread (save_thread_rse_state, &register_file);
#endif /* HP_IA64 */

  /* If we are not running in the main thread, switch the stack ... */
  if (current_thread != pid)
    {
      switch_to_thread (pid);
      main_stack_pointer = read_register (SP_REGNUM);
      switch_to_thread (current_thread);
      thread_stack_pointer = read_register (SP_REGNUM);
      write_register (SP_REGNUM, main_stack_pointer);
      registers_changed ();
    }

#endif /* !HP_1020 */

  if (!batch_rtc)
      printf_filtered ("Scanning for memory leaks...\n\n");

  funcval = find_function_in_inferior ("__rtc_leaks_info");
  gc_in_progress = 1;
  val = call_function_by_hand (funcval, 0, 0);
  gc_in_progress = 0;
  new_leaks = value_as_pointer (val);

#ifndef HPUX_1020
  /* If we switched stacks above, restore the original SP */
  if (current_thread != pid)
    {
      write_register (SP_REGNUM, thread_stack_pointer);
      registers_changed ();
    }
#endif 

  if (PTR_SIZE != sizeof (CORE_ADDR))
    new_leaks = SWZ ((long) new_leaks);

  if (new_leaks == (CORE_ADDR) RTC_NOT_RUNNING)
    error ("Heap debugging has not commenced or has been disabled.\n");
  else 
  if (new_leaks == (CORE_ADDR) RTC_MUTEX_LOCK_FAILED)
    error ("Some other thread has locked the mutex. Try again later.\n");
  else 
  if (new_leaks == (CORE_ADDR) RTC_UNSAFE_NOW)
    error ("The current thread is inside the allocator. Try again later.\n");

  if (!new_leaks)
    { 
      if (leak_num != -1)
       error ("Not a valid leak number.\n");

      if (!leak_info)
        error ("No new leaks.\n");
      else if (!query ("No new leaks. Would you like to look at "
                       "the previous leak(s) ? "))
             goto cleanup;
    }

  while (new_leaks)
    {
      CORE_ADDR addr = 0;
      int field_itr;

      QUIT;

      /* First 4 fields are 1-bit fields. Remaining 5 are pointers in
	 inferior. See rtc.h for rtc_chunk_info structure.*/

      /* Read-in the bit fields first. */
      status = target_read_memory (new_leaks, (char *) &a_leak,
                                           PTR_SIZE);
      if (status != 0)
        error("Error downloading data !");

      /* Read in the next 5 pointer fields into our CORE_ADDR fields. */
      for (field_itr = 0; field_itr < 5; field_itr++)
        {
          status = target_read_memory (new_leaks + PTR_SIZE + field_itr * PTR_SIZE,
					 (char*) &addr, PTR_SIZE);
	  if (status != 0)
            error("Error downloading data !");

	  /* Let's shift if the pointer size is not equal to the 
	     sizeof CORE_ADDR. */
          if (PTR_SIZE != sizeof (CORE_ADDR))
	    addr = addr >> (sizeof (CORE_ADDR) - PTR_SIZE) * 8;

          memcpy ((char*)(&a_leak) + ((field_itr + 1) * sizeof (CORE_ADDR)),
		  (char*)&addr, sizeof (CORE_ADDR));

	  addr = 0;
        }
	
      add_a_leak (&a_leak);
      leaked_more = 1;
      new_leaks = a_leak.next_leak;
    }

  if (leaked_more)
    qsort (leak_info, leak_count, sizeof (struct memory_info),
                                    compare_leaks);

  if (leak_num != -1)
    report_this_leak (leak_num, 0);
  else 
    print_leaks (leak_info, fp);

cleanup :

  if (fp)
    {
      if (fclose (fp) == -1)
        error ("Error writing to file\n");
      if (old_chain)
        discard_cleanups (old_chain);
    }
}

/* Return 1 if the p1's pointer_to_stack_trace is greater than
   to that of p2, -1 other way round. 0 if equal. */
static int
compare_blocks (const void *p1, const void *p2)
{
  struct memory_info * b1, *b2;
  
  b1 = (struct memory_info *) p1;
  b2 = (struct memory_info *) p2;

  if (b1->pointer_to_stack_trace < b2->pointer_to_stack_trace)
    return -1;
  else if (b1->pointer_to_stack_trace > b2->pointer_to_stack_trace)
    return 1;

  return 0;
}

/* Returns > 0 if p2's size is greater than that of p1, 0 if equal, else
   returns < 0. Used to sort heap blocks by size. */
static int
compare_blocks_by_size (const void *p1, const void *p2)
{
  struct memory_info * b1, *b2;
  
  b1 = (struct memory_info *) p1;
  b2 = (struct memory_info *) p2;

  return b2->size - b1->size;
}

/* At this point the inferior has gathered the profile statistics
   and has dumped it to a file named /tmp/__heap_info. This
   binary file has the following format : For each allocated block
   there is one record of output. (The field separator is not
   actually there.)

   size|address|pointer-to-stack-trace|pc[0]|pc[1]|...|pc[n-1]

   where size is the size of the block.
         address is the base address of the block.
         pointer-to-stack-trace is an inferior pointer which points to
                 the stack trace collected for the allocation. The
                 reason we are interested in it is because, if two
                 or more blocks are from the same allocation call
                 stack, they will have the same value for this field.
                 Thus it is useful for collating blocks based on
                 the allocation stack.
         pc[x]   is the PC value of the x'th caller in the call
                 stack.

  There will be a total of 'frame-count' number of these.
*/
                 
static int
read_interval_idx_file()
{
  FILE *fp;
  int i = 0;

  if ((fp = fopen (heap_interval_idxfile, "r")) == NULL)
    return 0;

  while (fread (&heap_idx[i].start_time, TIME_STR_LEN, 1, fp))
    {
       heap_idx[i].start_time[TIME_STR_LEN -1] = NULL;
       fread (&heap_idx[i].end_time, TIME_STR_LEN, 1, fp);
       heap_idx[i].end_time[TIME_STR_LEN -1] = NULL;
       fread (&heap_idx[i].interval, sizeof (heap_idx[i].interval), 1, fp);
       fread (&heap_idx[i].num_blocks, sizeof (heap_idx[i].num_blocks), 1, fp);
       i++;
     }

   if (!feof(fp))
       error("Heap interval index file is corrupted");

   fclose(fp);
   return i;
}

#define FREAD_IDX_DATA_FILE(a,b,c,d) if (!fread(a, b, c, d)) \
                         error("Error reading heap interval data file");
static int
down_load_interval_data ()
{
  int i, idx;
  int j;
  int tmp_flag = 0;
  int total_records = 0;
  int unique_allocations = 0;

  FILE * fp;
  struct memory_info * interval_data;

  CORE_ADDR tmp_size;
  CORE_ADDR last_allocation_address;

  if ((total_records = read_interval_idx_file()) == 0)
    error ("No heap interval index file.");

  if ((fp = fopen (heap_interval_filename, "r")) == NULL)
    error ("No heap interval data file.");

  for (idx = 0; idx < total_records; idx++)
    {
       interval_data = xmalloc (heap_idx[idx].num_blocks * sizeof (struct memory_info));

       for (i = 0; i < heap_idx[idx].num_blocks; i++)
         {
            interval_data[i].count = 1;
            interval_data[i].pc = 0;

            FREAD_IDX_DATA_FILE (&tmp_size, sizeof (CORE_ADDR), 1, fp);

            interval_data[i].size = tmp_size;

            heap_idx[idx].total_bytes += interval_data[i].size;

            FREAD_IDX_DATA_FILE (&interval_data[i].address, sizeof (CORE_ADDR), 1, fp);
            FREAD_IDX_DATA_FILE (&interval_data[i].pointer_to_stack_trace, sizeof (CORE_ADDR), 1, fp);
            FREAD_IDX_DATA_FILE (&tmp_flag, sizeof (int), 1, fp);

            interval_data[i].corrupted = tmp_flag;

            interval_data[i].pc = calloc (frame_count + 1, sizeof (CORE_ADDR));
            if (!interval_data[i].pc)
              error ("Out of memory reading heap interval data!");

            FREAD_IDX_DATA_FILE (interval_data[i].pc, sizeof (CORE_ADDR), frame_count, fp);

            DEBUG(printf("i= %d count=%d pc=%x add=%x size=%d\n", i, interval_data[i].count, interval_data[i].pc, interval_data[i].address, interval_data[i].size);)
         }

      /* Sort on the 'pointer to stack trace' field.  */
      qsort (interval_data, heap_idx[idx].num_blocks, sizeof (struct memory_info),
        compare_blocks);

      /* Find total unique blocks */
      unique_allocations = 0;
      last_allocation_address = (CORE_ADDR) -1;
      for (i=0; i < heap_idx[idx].num_blocks; i++)
        {
           if (interval_data[i].pointer_to_stack_trace != last_allocation_address)
             {
                unique_allocations++;
                last_allocation_address = interval_data[i].pointer_to_stack_trace;
             }
        }

     heap_idx[idx].block_info = xmalloc (unique_allocations * 
                           sizeof (struct memory_info));
     heap_idx[idx].unq_blocks = unique_allocations;

     /* Aggregate the blocks */
     unique_allocations = 0;
     last_allocation_address = (CORE_ADDR) -1;
     for (i=0, j=-1; i < heap_idx[idx].num_blocks; i++)
       {
          if (interval_data[i].pointer_to_stack_trace != last_allocation_address)
            {
               heap_idx[idx].block_info[++j] = interval_data[i];
               heap_idx[idx].block_info[j].min = 
               heap_idx[idx].block_info[j].max = interval_data[i].size;

               last_allocation_address = interval_data[i].pointer_to_stack_trace;
            }
          else 
            {
               heap_idx[idx].block_info[j].count++;
               heap_idx[idx].block_info[j].size += interval_data[i].size;

               if (heap_idx[idx].block_info[j].min > interval_data[i].size)
                  heap_idx[idx].block_info[j].min = interval_data[i].size;
               if (heap_idx[idx].block_info[j].max < interval_data[i].size)
                  heap_idx[idx].block_info[j].max = interval_data[i].size;

               free (interval_data[i].pc);
             }
       } /* for */
     free (interval_data);

     /* Sort based on the size */
     qsort (heap_idx[idx].block_info, heap_idx[idx].unq_blocks,
       sizeof (struct memory_info), compare_blocks_by_size);
   } /* for */
   fclose (fp);
   return total_records;
}

static void
down_load_data ()
{
  struct memory_info * data;
  int i;
  FILE * fp;
  int j;
  int unique_allocations = 0;
  char filename[1024];
  int tmp_flag = 0;
  CORE_ADDR last_allocation_address;
  CORE_ADDR tmp_size;

  total_bytes = 0;
  total_blocks = block_count;

  data = xmalloc (block_count * sizeof (struct memory_info));
  sprintf (filename, "/tmp/__heap_info");
  fp = fopen (filename, "r");
  if (!fp)
    error ("Heap analysis file missing !");

  /* Read in the raw data ... */
  for (i = 0; i < block_count; i++)
  {
      data[i].count = 1;
      data[i].pc = 0;

      /* librtc writes 'size' of size of CORE_ADDR. */
      fread (&tmp_size, sizeof (CORE_ADDR), 1, fp);
      data[i].size = tmp_size;

      total_bytes += data[i].size;

      fread (&data[i].address, sizeof (CORE_ADDR), 1, fp);
      fread (&data[i].pointer_to_stack_trace, sizeof (CORE_ADDR), 1, fp);
      fread (&tmp_flag, sizeof (int), 1, fp);
      data[i].corrupted = tmp_flag;

      data[i].pc = calloc (frame_count + 1, sizeof (CORE_ADDR));
      if (!data[i].pc)
        error ("Out of memory!");
      fread (data[i].pc, sizeof (CORE_ADDR), frame_count, fp);
    }

  fclose (fp);
  unlink (filename);
  qsort (data, block_count, sizeof (struct memory_info),
                                    compare_blocks);

  /* At this point the blocks are sorted on the 'pointer to
     stack trace' field. So all blocks originating in the
     same call stack are adjacent to each other ...
  */
  unique_allocations = 0;
  last_allocation_address = (CORE_ADDR) -1;
  for (i=0; i < block_count; i++)
    {
      if (data[i].pointer_to_stack_trace != last_allocation_address)
        {
          unique_allocations++;
          last_allocation_address = data[i].pointer_to_stack_trace;
        }
    }

  block_info = xmalloc (unique_allocations *
                           sizeof (struct memory_info));

  /* Collate the blocks now so that allocations from the same
     call stack show up together.
  */

  unique_allocations = 0;
  last_allocation_address = (CORE_ADDR) -1;
  for (i=0, j=-1; i < block_count; i++)
    {
      if (data[i].pointer_to_stack_trace != last_allocation_address)
        {
          block_info[++j] = data[i];
          block_info[j].min = block_info[j].max = data[i].size;
          last_allocation_address = data[i].pointer_to_stack_trace;
        }
      else
        {
          block_info[j].count++;
          block_info[j].size += data[i].size;

          if (block_info[j].min > data[i].size)
            block_info[j].min = data[i].size;
          if (block_info[j].max < data[i].size)
            block_info[j].max = data[i].size;

          free (data[i].pc);

        }
    }

  free (data);
  block_count = j + 1;

  /* Now sort based on the size ... */
  qsort (block_info, block_count, sizeof (struct memory_info),
                                    compare_blocks_by_size);
}

static void
allocate_heap_idx ()
{
   int i;
  
   /* Data is freed only when user resets the interval slot so that
      he can perform heap_interval_info as many time as he wants.
    */
   for (i = 0; heap_idx[i].block_info; i++) 
        if (!heap_idx[i].block_info)
           free(heap_idx[i].block_info);
   free(heap_idx);

   heap_idx = xmalloc (check_heap_interval_repeat_value * 
                sizeof (struct interval));
   memset(heap_idx, NULL, check_heap_interval_repeat_value *sizeof (struct interval));
   return;
}

static void
heap_interval_info_command (output_file, from_tty)
     char *output_file;
     int from_tty;
{
  int i;
  long ret_val;
  int thread_state;
  int block_num = -1;

  FILE * fp = 0;
  value_ptr funcval, val;
  struct cleanup * old_chain = 0; /* initialized for cmpiler warning */

  if (!check_heap_in_this_run)
    error ("Heap analysis is not enabled now.");

  if (!rtc_dot_sl)
    error ("The heap debugging library (librtc) is not linked in !\nHint:\
            Use environment variable LIBRTC_SERVER to pick correct version \
            of librtc and upgrade to the latest linker version (greater \
            than B.11.19);  Or link in librtc with your application to \
            enable gdb support for heap debugging.");

  if (!libc_dot_sl)
    error ("The shared C library is not linked in !");

  dont_repeat ();

  if (output_file)
    if ((fp = fopen (output_file, "w")) == NULL)
      error ("Cannot open file %s\n", output_file);

  if (!target_has_stack)
    {
        error ("There is no running process.");
        goto cleanup;
    }

#ifdef HAS_END_DOT_O
  if (stop_signal && stop_signal != TARGET_SIGNAL_TRAP)
    error ("There are pending signals. Try again later.");
#endif /* HAS_END_DOT_O */

#ifndef HPUX_1020
  thread_state = thread_could_run_gc (inferior_pid);
  switch (thread_state) {

    case TTS_WASSUSPENDED : 
      error ("Current thread is suspended. Cannot profile now.");
    case TTS_WASSLEEPING :
      error ("Current thread is blocked. Cannot profile now.");
    case TTS_INSYSCALL :
      error ("Current thread is inside a syscall. Cannot profile now.");
    case TTS_ATEXIT :
      error ("Current thread is about to terminate. Cannot profile now.");
    case 0 :
      error ("Current thread has pending events. Cannot profile now.");
  }
#endif

  allocate_heap_idx();

  if (!batch_rtc)
      printf_filtered ("Analyzing heap ...\n\n");

  funcval = find_function_in_inferior ("__rtc_heap_interval_info");
  gc_in_progress = 1;
  val = call_function_by_hand (funcval, 0, 0);
  gc_in_progress = 0;

  ret_val = value_as_long (val);
  
  if (ret_val == RTC_NOT_RUNNING)
    error ("Heap debugging has not commenced or has been disabled.\n");
  else 
  if (ret_val == RTC_MUTEX_LOCK_FAILED)
    error ("Some other thread has locked the mutex. Try again later.\n");
  else 
  if (ret_val == RTC_UNSAFE_NOW)
    error ("Current thread is inside the allocator. Try again later.\n");
  else
  if (ret_val == RTC_FOPEN_FAILED)
    error ("Failed to open temporary file.\n");
  else 
  if (block_count == RTC_FCLOSE_FAILED)
    error ("Failed to close temporary file.\n");
  print_interval_blocks (fp, down_load_interval_data ());

cleanup :
  if ((fp) && (fclose (fp) == -1))
    error ("Error writing to output file\n");
}



static void
heap_info_command_common (blocknum_exp, from_tty, type)
     char *blocknum_exp;
     int from_tty;
     int type;
{
  value_ptr funcval, val;
  FILE * fp = 0;
  int block_num = -1;
  int i;
  struct cleanup * old_chain = 0; /* initialized for cmpiler warning */
  int thread_state;

  if (!check_heap_in_this_run)
    error ("Heap analysis is not enabled now.");

  /* On PA64 and IPF, it is possible that we get here without the
     necessary libs */ 
  if (!rtc_dot_sl)
    error ("The heap debugging library (librtc) is not linked in !\nHint: Use environment variable LIBRTC_SERVER to pick correct version of librtc and upgrade to the latest linker version (greater than B.11.19);  Or link in librtc with your application to enable gdb support for heap debugging.");

  if (!libc_dot_sl)
    error ("The shared C library is not linked in !");

#ifdef LOG_BETA_RTC_USED
  log_test_event (LOG_BETA_RTC_USED, 1);
#endif

  dont_repeat ();

  if (blocknum_exp)
    {
      if (could_be_filename (blocknum_exp))
        {
          /* treat it as a filename ... */
          fp = fopen (blocknum_exp, "w");
          if (!fp)
            error ("Cannot open file %s\n", blocknum_exp);
          old_chain = make_cleanup ((make_cleanup_ftype *) fclose, fp);
        }
      else 
        block_num = atoi (blocknum_exp);

      if (!fp)
        {
          if (block_num < 0 || block_num >= block_count)
            error ("Bad block number.\n");
          else  
            report_this_block (block_num, 0, type);  /* always to the screen */
          return;
        }
    }

  if (!target_has_stack)
    {
      if (block_count)
        print_blocks (block_info, fp, type);
      else 
        error ("There is no running process.");
      goto cleanup;
    }

#ifdef HAS_END_DOT_O
  /* Making *any* command line call after a Ctrl-C confuses end.o so
     thoroughly, it sigsegvs (only on 11.00.) Even when we don't pass
     the signal. No idea why. Just bail out  -- srikanth, 000311.
  */ 
  if (stop_signal && stop_signal != TARGET_SIGNAL_TRAP)
    error ("There are pending signals. Try again later.");
#endif /* HAS_END_DOT_O */

#ifndef HPUX_1020

  /* srikanth, 000928, we used to refuse to oblige if there are pending
     events for other threads. With the K threads overhaul, this
     is no longer necessary. The only condition is that the current
     thread should not have unhandled event. This is checked for
     by thread_could_run_gc() below.
  */ 

  thread_state = thread_could_run_gc (inferior_pid);

  switch (thread_state) {

    case TTS_WASSUSPENDED : 
      error ("Current thread is suspended. Cannot profile now.");
    case TTS_WASSLEEPING :
      error ("Current thread is blocked. Cannot profile now.");
    case TTS_INSYSCALL :
      error ("Current thread is inside a syscall. Cannot profile now.");
    case TTS_ATEXIT :
      error ("Current thread is about to terminate. Cannot profile now.");
    case 0 :
      error ("Current thread has pending events. Cannot profile now.");
  }
  
#endif

  /* release any old data .. */
  for (i=0; i < block_count; i++)
    free (block_info[i].pc);

  free (block_info);
  block_info = 0;
  block_count = 0;
  total_blocks = 0;
  total_bytes = 0;

  if (!batch_rtc)
      printf_filtered ("Analyzing heap ...\n\n");
  funcval = find_function_in_inferior ("__rtc_heap_info");
  gc_in_progress = 1;
  val = call_function_by_hand (funcval, 0, 0);
  gc_in_progress = 0;

  block_count = value_as_long (val);
  
  if (block_count == RTC_NOT_RUNNING)
    error ("Heap debugging has not commenced or has been disabled.\n");
  else 
  if (block_count == RTC_MUTEX_LOCK_FAILED)
    error ("Some other thread has locked the mutex. Try again later.\n");
  else 
  if (block_count == RTC_UNSAFE_NOW)
    error ("Current thread is inside the allocator. Try again later.\n");
  else
  if (block_count == RTC_FOPEN_FAILED)
    error ("Failed to open temporary file.\n");
  else 
  if (block_count == RTC_FCLOSE_FAILED)
    error ("Failed to close temporary file.\n");
  
  if (block_count > 0)
    {
      down_load_data ();
      print_blocks (block_info, fp, type);
    }  
  else   /* do not leave 0 byte output file hanging around ... */
    {
      char filename[1024];
      
      sprintf (filename, "/tmp/__heap_info");
      unlink (filename);
      error ("No blocks were found.\n");
    }

cleanup :

  if (fp)
    {
      if (fclose (fp) == -1)
        error ("Error writing to file\n");
      if (old_chain)
        discard_cleanups (old_chain);
    }
}

/* This function is called when user does an 'info heap'. */
static void
bounds_info_command (blocknum_exp, from_tty)
     char *blocknum_exp;
     int from_tty;
{
    /* bounds checking should be enabled for this command to work. */
    if (check_bounds == 1) 
      {
         set_inferior_var_value ("__rtc_bounds_or_heap", BOUNDS_INFO);
         heap_info_command_common(blocknum_exp, from_tty, BOUNDS_INFO);
      }
    else
      error ("Bounds checking not enabled. Use help heap-check for syntax");
}

/* This function is called when user does an 'info heap'. */
static void
heap_info_command (blocknum_exp, from_tty)
     char *blocknum_exp;
     int from_tty;
{
    set_inferior_var_value ("__rtc_bounds_or_heap", HEAP_INFO);
    heap_info_command_common(blocknum_exp, from_tty, HEAP_INFO);
}


/* this will be automatically updated by the `set heap-check' command */
static char * heap_check_args;

static void
set_heap_check_command (args, from_tty, c)
     char *args;
     int from_tty;
     struct cmd_list_element *c;
{
  int found = 0, switching_heap_checking_on = 0;
  struct objfile * objfile;
  int value;

  if (target_has_execution)
    {
       ALL_OBJFILES (objfile)
         {
            if (strstr (objfile->name, "/librtc")) 
              {
                 found = 1;
                 break;
              }
         }
         if (!found) 
              error ("librtc is not loaded: Either use -leaks command line option, set heap-check before starting the program, or link librtc explicitly");
     }
      
   /* First of all cause to emit an `rtc-parameters-invalid' annotation.
      This is necessary so that the GUI and GDB speak the same language
      (have the same values for parameters.) In response to the
      annotation, the GUI will issue the command, "show heap-check"
      which dumps out all the values.
   */

   annotate_rtc_parameter_change ();

   /* The input args is useless, we need to look in heap_check_args.
      GDB's set command handler leaves this pointer pointing after
      the "set heap-check". For example, if the user issued the
      command "set heap-check leaks on", when control reaches this
      point, the char * heap_check_args points to the substring
      "leaks on".
   */

   args = heap_check_args;

   if (args == 0)
     error ("Incomplete command : try `help set heap-check'");

   /* We will only do minimal validation of input arguments. For 
      example the code below would accept "set heap-check leaks ontario"
      happily as though the user said "set heap-check leaks on". This 
      should be good enough.

      Many of the heap checking features are controlled by variables in
      the inferior. Update the variables suitably. There is no variable
      for toggling leak detection however. For this case, we will simply
      update the call stack frame count to be zero. The thing is even 
      when the user does not care for a leak report, we still need to 
      maintain the exact same data structures to be able to issue any 
      other heap debugging diagonostic.
   */

   while (*args == ' ' || *args == '\t')
     args++;

   if (!strncmp (args, "leaks ", 6) || !strncmp (args, "leaks\t", 6))
     {
       args += 6;
       while (*args == ' ' || *args == '\t')
         args++;

       if (!strncmp (args, "on", 2))
         {
           switching_heap_checking_on = check_leaks == 0 ? 1 : 0; /* leaks is being changed ot on from off */
           check_heap = check_leaks = 1;
         }
       else if (!strncmp (args, "off", 3))
         check_leaks = 0;
     }
   else if (!strncmp (args, "bounds ", 7) ||
            !strncmp (args, "bounds\t", 7))
     {
       args += 7;
       while (*args == ' ' || *args == '\t')
         args++;

       if (!strncmp (args, "on", 2))
         {
           switching_heap_checking_on = check_bounds == 0 ? 1 : 0; /* bounds is being changed to on from off */
           check_heap = check_bounds = 1;
           if (target_has_stack && check_heap_in_this_run)
             set_inferior_var_value ("__rtc_check_bounds", 1);
         }
       else if (!strncmp (args, "off", 3))
         {
           check_bounds = 0;
           if (target_has_stack && check_heap_in_this_run)
             set_inferior_var_value ("__rtc_check_bounds", 0);
         }
     }
   else if (!strncmp (args, "free ", 5) || !strncmp (args, "free\t", 5))
     {
       args += 5;
       while (*args == ' ' || *args == '\t')
         args++;

       if (!strncmp (args, "on", 2))
         {
           switching_heap_checking_on = check_free == 0 ? 1 : 0; /* free is being changed to on from off */
           check_heap = check_free = 1;
           if (target_has_stack && check_heap_in_this_run)
             set_inferior_var_value ("__rtc_check_free", 1);
         }
       else if (!strncmp (args, "off", 3))
         {
           check_free = 0;
           if (target_has_stack && check_heap_in_this_run)
             set_inferior_var_value ("__rtc_check_free", 0);
         }
     }
   else if (!strncmp (args, "scramble ", 9) || 
            !strncmp (args, "scramble\t", 9))
     {
       args += 9;
       while (*args == ' ' || *args == '\t')
         args++;

       if (!strncmp (args, "on", 2))
         {
           switching_heap_checking_on = scramble_blocks == 0 ? 1 : 0; /* scramble_blocks is being changed to on from off */
           check_heap = scramble_blocks = 1;
           if (target_has_stack && check_heap_in_this_run)
             set_inferior_var_value ("__rtc_scramble_blocks", 1);
         }
       else if (!strncmp (args, "off", 3))
         {
           scramble_blocks = 0;
           if (target_has_stack && check_heap_in_this_run)
             set_inferior_var_value ("__rtc_scramble_blocks", 0);
         }
     }
   else if (!strncmp (args, "frame-count ", 12) ||
            !strncmp (args, "frame-count\t",12))
     {
       args += 12;
       while (*args == ' ' || *args == '\t')
         args++;

       frame_count = atoi (args);
       check_heap = 1;

       if (target_has_stack && check_heap_in_this_run)
         {
           set_inferior_var_value ("__rtc_frame_count", frame_count);
           /* Invalidate the buffers allocated using old parameters. */
           set_inferior_var_value ("__rtc_stack_trace_count", 0);
         }
     }
   else if (!strncmp (args, "min-leak-size ",14) ||
            !strncmp (args, "min-leak-size\t",14))
     {
       args += 14;
       while (*args == ' ' || *args == '\t')
         args++;

       min_leak_size = atoi (args);
       check_heap = 1;

       if (target_has_stack && check_heap_in_this_run)
         set_inferior_var_value ("__rtc_min_leak_size", min_leak_size);
     }
   else if (!strncmp (args, "watch ",6) ||
            !strncmp (args, "watch\t",6))
     {
       args += 6;
       while (*args == ' ' || *args == '\t')
         args++;

       watch_address = parse_and_eval_address (args);
       check_heap = 1;

       if (target_has_stack && check_heap_in_this_run)
         set_inferior_pointer_var_value ("__rtc_watch_address", 
                                                     watch_address);
     }
   else if (!strncmp (args, "block-size ",11) ||
            !strncmp (args, "block-size\t",11))
     {
       args += 11;
       while (*args == ' ' || *args == '\t')
         args++;

       big_block_size = atoi (args);
       check_heap = 1;

       if (target_has_stack && check_heap_in_this_run)
         set_inferior_var_value ("__rtc_big_block_size", 
                                                     big_block_size);
     }
   else if (!strncmp (args, "heap-size ",10) ||
            !strncmp (args, "heap-size\t",10))
     {
       args += 10;
       while (*args == ' ' || *args == '\t')
         args++;

       heap_growth_size = atoi (args);
       check_heap = 1;

       if (target_has_stack && check_heap_in_this_run)
         set_inferior_var_value ("__rtc_heap_growth_size", 
                                                     heap_growth_size);
     }
   else if (!strncmp (args, "null-check-size ",16) ||
            !strncmp (args, "null-check-size\t",16))
     {
       args += 16;
       while (*args == ' ' || *args == '\t')
         args++;

       value = atoi (args);
       if (value <= 0)
           error("Invalid value for null-check-size\n");

       if (null_check_count != -1)
           error("null-check-count already specified. Ignoring null-check-size value\n");

       null_check_size = value;
       check_heap = 1;

       if (target_has_stack && check_heap_in_this_run)
         set_inferior_var_value ("__rtc_null_check_size", 
                                   null_check_size);
     }
   else if (!strncmp (args, "seed-value ",11) ||
            !strncmp (args, "seed-value\t",11))
     {
       args += 11;
       while (*args == ' ' || *args == '\t')
         args++;

       value = atoi (args);
       if (value <= 0)
           error("Invalid value for seed-value\n");

       null_check_seed_value = value;

       check_heap = 1;

       if (target_has_stack && check_heap_in_this_run)
         set_inferior_var_value ("__rtc_null_check_seed_value", 
                                   null_check_seed_value);
     }
   else if (!strncmp (args, "random-range ",13) ||
            !strncmp (args, "random-range\t",13))
     {
       args += 13;
       while (*args == ' ' || *args == '\t')
         args++;

       value = atoi (args);
       if (value <= 0)
           error("Invalid value for random-range\n");

       null_check_random_range = value;

       check_heap = 1;

       if (target_has_stack && check_heap_in_this_run)
         set_inferior_var_value ("__rtc_null_check_random_range", 
                                    null_check_random_range);
     }
   else if (!strncmp (args, "null-check ",11) ||
            !strncmp (args, "null-check\t",11))
     {
       args += 11;
       while (*args == ' ' || *args == '\t')
         args++;

       if (!strncmp(args, "random", 6))
         {
            check_heap = 1;
            null_check_count = 0;
            if (target_has_stack && check_heap_in_this_run)
              set_inferior_var_value ("__rtc_null_check_count", 
                                        null_check_count);
         }
       else if ((value = atoi(args)))
         {
            check_heap = 1;
            null_check_count = value;

            if (target_has_stack && check_heap_in_this_run)
              set_inferior_var_value ("__rtc_null_check_count", 
                                         null_check_count);
         }
       else
           error("Invalid value for null-check\n");
     }
   else if (!strncmp (args, "interval ",9) ||
            !strncmp (args, "interval\t",9))
     {
       args += 9;
       while (*args == ' ' || *args == '\t')
         args++;

       value = atoi (args);
       if (value < 0)
           error("Invalid value for interval\n");

       check_heap_interval_value = value;
       check_heap = 1;

       if (target_has_stack && check_heap_in_this_run)
         set_inferior_var_value ("__rtc_check_heap_interval", 
                                   check_heap_interval_value);
     }
   else if (!strncmp (args, "repeat ",7) ||
            !strncmp (args, "repeat\t",7))
     {
       args += 7;
       while (*args == ' ' || *args == '\t')
         args++;

       value = atoi (args);
       if (value < 0)
           error("Invalid value for interval repetition\n");

       check_heap_interval_repeat_value = value;
       check_heap = 1;
       if (target_has_stack && check_heap_in_this_run)
         set_inferior_var_value ("__rtc_check_heap_repeat", 
                                   check_heap_interval_repeat_value);
     }
   else if (!strncmp (args, "reset",5))
     {
       check_heap_interval_reset_value = 1;
       check_heap = 1;
       if (target_has_stack && check_heap_in_this_run)
         {
            set_inferior_var_value ("__rtc_check_heap_reset", 
                                   check_heap_interval_reset_value);
            check_heap_interval_reset_value = 0;
         }
     }
   else if (!strncmp (args, "on",2))
     {
       /* The command "set heap-check on" is provided as a short hand
          for the following settings : leak check on, bounds check on,
          free check on, scrambling on.
       */
          
       switching_heap_checking_on = check_heap == 0 ? 1 : 0; /* check_heap is being changed to on from off */
       check_heap = check_leaks = check_bounds =
           check_free = scramble_blocks = 1;
       if (target_has_stack && check_heap_in_this_run)
         {
           set_inferior_var_value ("__rtc_check_bounds", 1);
           set_inferior_var_value ("__rtc_check_free", 1);
           set_inferior_var_value ("__rtc_scramble_blocks", 1);
         }
     }
   else if (!strncmp (args, "off",3))
     {
       /* The command "set heap-check off" is the master switch */
          
       check_heap = check_leaks = check_bounds =
           check_free = scramble_blocks = 0;
       if (target_has_stack && check_heap_in_this_run)
         {
           set_inferior_var_value ("__rtc_check_bounds", 0);
           set_inferior_var_value ("__rtc_check_free", 0);
           set_inferior_var_value ("__rtc_scramble_blocks", 0);
         }
     }
   else
     error ("Malformed command, try `help set heap-check'.");

   if (found && switching_heap_checking_on)
     warning("Program is already running. Heap information provided may not be accurate.");

  /* Bindu 040303 attach & rtc: If the user started the heap check in the
     middle of execution (or attach). Let's start-up the stuff. In the
     normal case, this is done when the program starts up. */
  if (target_has_stack && check_heap && !check_heap_in_this_run)
    {
      prepare_for_rtc (1);
      snoop_on_the_heap ();
    }
}

static void
show_heap_check_command (args, from_tty, c)
     char *args;
     int from_tty;
     struct cmd_list_element *c;
{
  if (!check_heap)
    {
      printf_filtered ("Heap checking is not enabled now.\n");
      return;
    }

  printf_filtered ("Current heap check settings are :\n");
  printf_filtered ("\tCheck leaks      : %s\n", check_leaks ? "on" : "off");
  printf_filtered ("\tCheck bounds     : %s\n", check_bounds ? "on" : "off");
  printf_filtered ("\tCheck free()     : %s\n", check_free ? "on" : "off");
  printf_filtered ("\tScrambling       : %s\n", scramble_blocks ? "on" : "off");
  printf_filtered ("\tFrame count      : %d\n", frame_count);
  printf_filtered ("\tMin-leak-size    : %d\n", min_leak_size);
  printf_filtered ("\tBlock-size       : %d\n", big_block_size);
  printf_filtered ("\tSeed-value       : %d\n", null_check_seed_value);
  printf_filtered ("\tRandom-range     : %d\n", null_check_random_range);
  printf_filtered ("\tNull-check       : %d\n", null_check_count);
  printf_filtered ("\tNull-check-size  : %d\n", null_check_size);
  printf_filtered ("\tHeap-size        : %d\n", heap_growth_size);
  printf_filtered ("\tHeap Interval    : %d\n", check_heap_interval_value);
  printf_filtered ("\tRepeat Count     : %d\n", 
                                         check_heap_interval_repeat_value);
  printf_filtered ("\tWatch Address    : %s\n", 
              longest_local_hex_string_custom ((LONGEST) watch_address, "08l"));
}

void 
set_rtc_catch_nomem()
{
    if (target_has_stack && check_heap_in_this_run)
      set_inferior_var_value ("__rtc_catch_nomem", 1);
    return;
}
/* Baskar, Fucntion to set "command_line_call" var in inferior 
   Used in call_function_by_hand()
*/
void
set_command_line_call_in_inf (int value)
{
   if (target_has_stack && check_heap_in_this_run)
      set_inferior_var_value ("command_line_call", value);
}

void
_initialize_gdbrtc ()
{
  struct cmd_list_element *set, *show;

  add_info ("leaks", leaks_info_command, 
	"Memory leak report for all leaks, or for a specified LEAK.\nIf a file name is provided the output of the command is written to the given file.");
  add_info ("heap", heap_info_command, 
	"Memory profile report for all allocations, or for specified ALLOCATION.\nIf a file name is provided the output of the command is written to the given file.");
  add_info ("corruption", bounds_info_command, 
	"Memory profile report of all allocations which appear to be corrupted.\nIf a file name is provided the output of the commonad is written to the given file.");

  add_info ("heap-interval", heap_interval_info_command, 
	"Incremental memory profile report of all allocations.");

  set = add_set_cmd ("heap-check", class_support, var_string_noescape,
         (char *) &heap_check_args, "", 
                     &setlist);

    set->function.sfunc = set_heap_check_command;
    show = add_show_from_set (set, &showlist);
    show->function.sfunc = show_heap_check_command;
    set->doc = 
      "Set heap-check { on/off | <option> on/off | <option> <num> }.\n\
    where <option> on/off can be one of the following:\n\n\
        leaks on/off        : Toggle the leak detection capability.\n\
        bounds on/off       : Toggle bounds check on heap blocks.\n\
        free on/off         : Toggle validation of calls to free()\n\
        scramble on/off     : Specify whether freed blocks should be scrambled\n\
        frame-count <num>   : Specify the depth of call stack to be captured\n\
        min-leak-size <num> : Specify the minimum size of a block for stack\n\
                              trace collection. GDB will still be able to \n\
                              report leaks of blocks smaller than this value.\n\
                              However, no stack trace would be provided.\n\
        block-size <num>    : Stop the program whenever it makes an allocation\n\
                              request exceeding <num> bytes (0 to disable).\n\
        heap-size <num>     : Stop target program whenever it's heap grows by\n\
                              <num> bytes (0 to disable).\n\
        watch <address>     : Instruct GDB to stop whenever a block at the\n\
                              given address gets allocated or deallocated.\n\n\
        null-check <num>    : Instructs malloc to return null after <num>\n\
			      invocations of malloc\n\
	null-check random   : Instructs malloc to return null after a random\n\
			      number of invocations of malloc\n\
        null-check-size <num> : Instructs malloc to return null after <num>\n\
			      bytes have been allocated by malloc\n\
        seed-value <num>    : Specifies the seed value to be used for\n\
                              generating random null-check-count\n\
        random-range <num>  : Specifies the random range to be used by random\n\
			      range\n\
        interval <num>      : Specifies the time interval to be used for\n\
                              incremental memory profile\n\
        repeat <num>        : Perform incremental profile for <num>\n\
                              interval periods where each period duration\n\
                              is defined by set heap-check interval command.\n\
                              The default value is 100.\n\
        reset               : Discard the heap interval data from the\n\
                              internal data file.\n\
Simply typing `set heap-check on' uses the following defaults :\n\
    leak check on, bounds check on, free check on, scrambling on\n";
}

#else   /* RTC */

  /* just to keep the linkers happy */
snoop_on_the_heap () {} prepare_for_rtc () {} decode_rtc_event () {}
_initialize_gdbrtc () {}

#endif 
