/* Target-vector operations for controlling VMS child processes, for GDB.
   Copyright 1995, 1996
   Free Software Foundation, Inc.

   Contributed by Ada Core Technologies
   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without eve nthe implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* by Douglas B. Rupp, rupp@gnat.com */

#include "defs.h"
#include "frame.h"		/* required by inferior.h */
#include "inferior.h"
#include "target.h"
#include "wait.h"
#include "gdbcore.h"
#include "command.h"
#include <signal.h>
#include <types.h>
#include <fcntl.h>
#include "buildsym.h"
#include "symfile.h"
#include "objfiles.h"
#include "gdb_string.h"
#include "gdbthread.h"
#include "gdbcmd.h"

#include <unistd.h>

#include <remote-utils.h>
#include <lib$routines.h>
#include <libicb.h>
#include <libwaitdef.h>
#include <pdscdef.h>
#include <descrip.h>
#include <ssdef.h>
#include <errnodef.h>
#include <stat.h>
#include <prcdef.h>
#include <dvidef.h>
#include <clidef.h>
#include <eihddef.h>
#include <jpidef.h>

#define CMA$_EXIT_THREAD 4227492

#define CHECK(x) 	check (x, __FILE__,__LINE__)
#define DEBUG_EXEC(x)	if (debug_exec)		printf x
#define DEBUG_EVENTS(x)	if (debug_events)	printf x
#define DEBUG_MEM(x)	if (debug_memory)	printf x
#define DEBUG_EXCEPT(x)	if (debug_exceptions)	printf x

#define PBUFSIZ 2048

#define MAXBUFBYTES ((PBUFSIZ-32)/2)

#if REGISTER_BYTES > MAXBUFBYTES
#undef PBUFSIZ
#define	PBUFSIZ	(REGISTER_BYTES * 2 + 32)
#endif

serial_t remote_desc;

/* Forward declaration */
extern struct target_ops vms_ops;

static int current_process_id;
static int current_thread_id;
static unsigned long ast_counter; 

/* Counts of things. */
static int exception_count = 0;
static int event_count = 0;

/* User options. */
static int new_console = 0;
static int new_group = 0;
static int vms_path_style = 0;
static int debug_exec = 0;		/* show execution */
static int debug_events = 0;		/* show events from kernel */
static int debug_memory = 0;		/* show target memory accesses */
static int debug_exceptions = 0;	/* show target exceptions */
/* static int remote_timeout = 2; */

static __int64 regs [NUM_REGS];

/* This table contains the mapping between VMS condition types, and
   signals, which are primarily what GDB understands. */

static struct hard_trap_info
{
  unsigned int tt;	       /* Trap type code */
  unsigned char signo;	       /* Signal that we map this trap into */
} hard_trap_info[] = {
  {SS$_OPCCUS,	SIGABRT},      /* The abort function            */
  {SS$_ASTFLT,	SIGALRM},      /* The alarm function            */
  {SS$_ACCVIO,	SIGBUS},       /* Access violation              */
  {SS$_CMODUSER,SIGBUS},       /* Change mode user              */
  {C$_SIGCHLD,	SIGCHLD},      /* Child process stopped         */
  {SS$_COMPAT,	SIGEMT},       /* Compatibility mode trap       */
  {SS$_DECDIV,	SIGFPE},       /* Decimal divide trap           */
  {SS$_DECINV,	SIGFPE},       /* Decimal invalid operand trap  */
  {SS$_DECOVF,	SIGFPE},       /* Decimal overflow trap         */
  {SS$_HPARITH,	SIGFPE},       /* Floating point trap           */
  {SS$_INTDIV,	SIGFPE},       /* Integer div by zero           */
  {SS$_SUBRNG,	SIGFPE},       /* Subscript out of range        */
  {SS$_SUBRNG1,	SIGFPE},       /* Subscript1 out of range       */
  {SS$_SUBRNG2,	SIGFPE},       /* Subscript2 out of range       */
  {SS$_SUBRNG3,	SIGFPE},       /* Subscript3 out of range       */
  {SS$_SUBRNG4,	SIGFPE},       /* Subscript4 out of range       */
  {SS$_SUBRNG5,	SIGFPE},       /* Subscript5 out of range       */
  {SS$_SUBRNG6,	SIGFPE},       /* Subscript6 out of range       */
  {SS$_SUBRNG7,	SIGFPE},       /* Subscript7 out of range       */
  {SS$_HANGUP,	SIGHUP},       /* Data set hangup               */
  {SS$_OPCDEC,	SIGILL},       /* Reserved instruction          */
  {SS$_ROPRAND,	SIGILL},       /* Reserved operand              */
  {5,           SIGINT},       /* ^C seems to generate this ... */
  {SS$_CONTROLC,SIGINT},       /* OpenVMS Ctrl/C interrupt      */
  {SS$_OPCCUS,	SIGIOT},       /* Customer-reserved opcode      */
  {SS$_ABORT,	SIGKILL},      /* External signal only          */
  {SS$_CONTROLY,SIGQUIT},      /* The raise function            */
  {SS$_NOMBX,	SIGPIPE},      /* No mailbox                    */
  {SS$_ACCVIO,	SIGSEGV},      /* Length violation              */
  {SS$_CMODUSER,SIGSEGV},      /* Change mode user              */
  {SS$_BADPARAM,SIGSYS},       /* Bad argument to system call   */
  {CMA$_EXIT_THREAD, SIGTERM}, /* DECthread exit                */
  {SS$_BREAK,	SIGTRAP},      /* Breakpoint fault instruction  */
  {C$_SIGUSR1,	SIGUSR1},      /* The raise function            */
  {C$_SIGUSR2,	SIGUSR2},      /* The raise function            */
  {0, 0}		       /* Must be last                  */
};

static struct {
  unsigned short buflen, item_code;
  void *bufaddr;
  void *retlenaddr;
  unsigned int terminator;
} itm_lst;

static int fromhex PARAMS ((int a));

static void interrupt_query PARAMS ((void));

static void getpkt PARAMS ((char *buf, int forever));

static void putpkt PARAMS ((char *buf));

static int read_frame PARAMS ((char *buf));

static int readchar PARAMS ((int timeout));

static void remote_interrupt PARAMS ((int signo));

static void remote_interrupt_twice PARAMS ((int signo));

static int remote_read_bytes PARAMS ((CORE_ADDR memaddr, char *myaddr, int len));

static void remote_send PARAMS ((char *buf));

static int remote_write_bytes PARAMS ((CORE_ADDR memaddr, char *myaddr, int len));

static int tohex PARAMS ((int nib));

static int vmschild_clear_breakpoints PARAMS((void));

static int vmschild_wait PARAMS((int pid, struct target_waitstatus *status));

static void (*ofunc)();

static void 
check (int ok, const char *file, int line)
{
  if (!ok)
    printf_filtered ("error return %s:%d was %d\n", file, line, errno);
}

static int
computeSignal(tt)
     int tt;
{
  struct hard_trap_info *ht;

  for (ht = hard_trap_info; ht->tt && ht->signo; ht++)
    if (ht->tt == tt)
      return ht->signo;

  return SIGHUP;		/* default for things we don't know about */
}

static DCACHE *remote_dcache;

/* Start an inferior VMS child process and sets inferior_pid to its pid.
   EXEC_FILE is the file to run.
   ALLARGS is a string containing the arguments to the program.
   ENV is the environment vector to pass.  Errors reported with error().  */
static int inferior_status;
extern struct target_ops exec_ops;

static int spawnstatus;
char *evax_target_pid;

static void
vmschild_create_inferior (exec_file, allargs, env)
     char *exec_file;
     char *allargs;
     char **env;
{
  unsigned int pid;
  long status;
  struct target_waitstatus dummy;
  char *libdebug = "LIB$DEBUG";
  char *gdbstub = "GDB$DEBUG";
  struct dsc$descriptor_s libdebugdsc;
  struct dsc$descriptor_s gdbstubdsc;
  $DESCRIPTOR(tabledsc, "LNM$JOB");
  struct dsc$descriptor_s commanddsc;
  struct dsc$descriptor_s procnamdsc;
  char *args;
  int len, maxargs;
  char *pos, *oldpos;
  int newargc = 0;
  char **newargv;
  int i;
  char mboxbuff[256];
  char vms_exec_file [255];
  char *ext;
  char commandbuff [1024];
  int spawnflags;
  char *exetempfullname = 0;
  struct dsc$descriptor_s deccimagname
    = {8, DSC$K_DTYPE_T, DSC$K_CLASS_S, "decc$shr"};
  struct dsc$descriptor_s deccsymname
    = {21, DSC$K_DTYPE_T, DSC$K_CLASS_S, "decc$$set_get_foreign"};
  void (*fn_set_get_foreign)() = 0;

  unsigned int input_term_len;
  $DESCRIPTOR (input_lognam, "SYS$INPUT");
  unsigned int output_term_len;
  $DESCRIPTOR (output_lognam, "SYS$OUTPUT");
  unsigned int error_term_len;
  $DESCRIPTOR (error_lognam, "SYS$ERROR");

  if (!exec_file)
    {
      error ("No executable specified, use `target exec'.\n");
    }

  target_preopen (0);

  unpush_target (&vms_ops);

  args = alloca (strlen (exec_file) + strlen (allargs) + 2);

  strcpy (args, exec_file);
  strcat (args, " ");
  strcat (args, allargs);

  maxargs = strlen (args);
  newargv = (char **) malloc (maxargs+1);
  pos = args;
  while (*pos)
    {
      int quoted = 0;
      int backq = 0;
      int argc = 0;
      char *buff;

      oldpos = pos;
      while (*pos)
	{
	  if (!backq && !quoted && *pos == ' ')
	    break;
	  else if (!backq && !quoted && *pos == '"')
	    quoted = 1;
	  else if (!backq && quoted && *pos == '"')
	    {
	      pos ++;
	      break;
	    }
	  else if (*pos == '\\')
	    backq = 1;
	  else if (backq)
	    backq = 0;

	  pos++;
	}
      len = pos - oldpos;
      buff = (char *)alloca (len + 1);
      strncpy (buff, oldpos, len);
      buff[len] = 0;
      newargv[newargc] = buff;
      newargc++;

      /* Skip extraneous spaces */
      while (*pos && *pos == ' ')
         pos++;

      if (*pos == 0)
	break;
    }
  newargv[newargc] = 0;

  libdebugdsc.dsc$w_length = strlen (libdebug);
  libdebugdsc.dsc$b_dtype = DSC$K_DTYPE_T;
  libdebugdsc.dsc$b_class = DSC$K_CLASS_S;
  libdebugdsc.dsc$a_pointer = libdebug;

  gdbstubdsc.dsc$w_length = strlen (gdbstub);
  gdbstubdsc.dsc$b_dtype = DSC$K_DTYPE_T;
  gdbstubdsc.dsc$b_class = DSC$K_CLASS_S;
  gdbstubdsc.dsc$a_pointer = gdbstub;

  status = lib$set_logical (&libdebugdsc, &gdbstubdsc, &tabledsc, 0, 0);
  if ((status & 1) != 1)
    {
      error ("failed to set LIB$DEBUG");
      LIB$SIGNAL (status);
    }

  lib$find_image_symbol
    (&deccimagname, &deccsymname, (int *)&fn_set_get_foreign, 0);

  if (fn_set_get_foreign == 0)
    {
      error ("Can't find decc$$set_get_foreign");
    }

  /* It is generally good practice to flush any possible pending stdio
     output prior to doing a fork, to avoid the possibility of both the
     parent and child flushing the same data after the fork. */

  gdb_flush (gdb_stdout);
  gdb_flush (gdb_stderr);

  /* Ask the tty subsystem to switch to the one we specified earlier
     (or to share the current terminal, if none was specified).  */

  /*  new_tty (); */

  strcpy (vms_exec_file, TO_HOST_FILE_SPEC (exec_file));

  strcpy (commandbuff, "run/debug ");
  strcat (commandbuff, vms_exec_file);

  commanddsc.dsc$w_length = strlen (commandbuff);
  commanddsc.dsc$b_dtype = DSC$K_DTYPE_T;
  commanddsc.dsc$b_class = DSC$K_CLASS_S;
  commanddsc.dsc$a_pointer = commandbuff;

#if 0
  procnamdsc.dsc$w_length = strlen ("debug");
  procnamdsc.dsc$b_dtype = DSC$K_DTYPE_T;
  procnamdsc.dsc$b_class = DSC$K_CLASS_S;
  procnamdsc.dsc$a_pointer = "debug";
#endif

  if (inferior_io_terminal)
    {
      input_lognam.dsc$w_length = strlen (inferior_io_terminal);
      input_lognam.dsc$a_pointer = inferior_io_terminal;
      output_lognam.dsc$w_length = strlen (inferior_io_terminal);
      output_lognam.dsc$a_pointer = inferior_io_terminal;
      error_lognam.dsc$w_length = strlen (inferior_io_terminal);
      error_lognam.dsc$a_pointer = inferior_io_terminal;
    }

#if 0
  itm_lst.buflen = 64;
  itm_lst.item_code = DVI$_DEVNAM;
  itm_lst.bufaddr = input_term;
  itm_lst.retlenaddr = &input_term_len;
  itm_lst.terminator = 0;
  status = SYS$GETDVI (0, 0, &input_lognam, &itm_lst, 0, 0, 0, 0);
  if ((status & 1) != 1)
    {
      LIB$SIGNAL (status);
      error ("failed to retrieve SYS$INPUT equivalence name");
    }

  itm_lst.bufaddr = output_term;
  itm_lst.retlenaddr = &output_term_len;
  status = SYS$GETDVI (0, 0, &output_lognam, &itm_lst, 0, 0, 0, 0);
  if ((status & 1) != 1)
    {
      LIB$SIGNAL (status);
      error ("failed to retrieve SYS$OUTPUT equivalence name");
    }

  itm_lst.bufaddr = error_term;
  itm_lst.retlenaddr = &error_term_len;
  status = SYS$GETDVI (0, 0, &error_lognam, &itm_lst, 0, 0, 0, 0);
  if ((status & 1) != 1)
    {
      LIB$SIGNAL (status);
      error ("failed to retrieve SYS$ERROR equivalence name");
    }
#endif
  spawnflags = CLI$M_NOWAIT;

  status = lib$spawn
    (&commanddsc,           /* command-string */
     0,                     /* input-file  */
     &output_lognam,        /* output-file */
     &spawnflags,           /* flags */
     0,                     /* process-name */
     &pid,                  /* process-id */
     &spawnstatus,          /* completion-status-address */
     0,                     /* byte-integer-event-flag-num */
     0,                     /* AST-address */
     0,                     /* AST-argument */
     0,                     /* prompt-string */
     0,                     /* cli */
     0);                    /* table */

  if ((status & 1) != 1)
    {
      LIB$SIGNAL (status);
      error ("failed to create inferior process");
    }

  remote_dcache = dcache_init (remote_read_bytes, remote_write_bytes);
  sprintf (mboxbuff, "gdbmbox%x", pid);
  remote_desc = SERIAL_OPEN (mboxbuff);
  if (!remote_desc)
    perror_with_name (mboxbuff);

  exception_count = 0;
  event_count = 0;

  inferior_pid = pid;
  current_process_id = pid;
  current_thread_id = pid;
  push_target (&vms_ops);
  init_thread_list ();
  init_task_list ();
  init_wait_for_inferior ();
  clear_proceed_status ();
  target_terminal_init ();
  target_terminal_inferior ();

  vmschild_wait (inferior_pid, &dummy);

  status = lib$delete_logical (&libdebugdsc, &tabledsc);
  if ((status & 1) != 1)
    {
      LIB$SIGNAL (status);
      error ("failed to delete LIB$DEBUG");
    }

  free (newargv);

  {
    char buf[PBUFSIZ];
    char *p;

    p = buf;
    sprintf (p, "A");
    strcat (p, allargs);
    remote_send (p);
  }

  SOLIB_CREATE_INFERIOR_HOOK (pid);

  proceed ((CORE_ADDR) - 1, TARGET_SIGNAL_0, 0);
}

static void
vmschild_mourn_inferior ()
{
  struct dsc$descriptor_s jpi_imagename;
  char buff [1024];
  int jpi_itemcode = JPI$_IMAGNAME;
  short jpi_length;
  unsigned long status;
  float wait_time = 0.1;
  unsigned long wait_time_float_type = LIB$K_IEEE_S;

  jpi_imagename.dsc$w_length = 1024;
  jpi_imagename.dsc$b_dtype = DSC$K_DTYPE_T;
  jpi_imagename.dsc$b_class = DSC$K_CLASS_S;
  jpi_imagename.dsc$a_pointer = buff;

  /* 
     Sometimes the process hangs around for awhile, which messes up
     restarts. So wait until it disappears from the process tables.
     If gdb was started with run/debug, the inferior will be running
     in the same process as DCL which has no imagename. Testing for
     a null string (jpi_length) here handles that case (probably).
  */

  do
    {
      status = lib$getjpi (&jpi_itemcode, &inferior_pid, 0,
                           0, &jpi_imagename, &jpi_length);
      if ((status & 1) == 1)
	buff [jpi_length] = 0;
    } while (status != SS$_NONEXPR && jpi_length &&
	     lib$wait (&wait_time, 0, &wait_time_float_type));

  inferior_pid = 0;
  unpush_target (&vms_ops);
  generic_mourn_inferior ();
}


/* Send a SIGINT to the process group.  This acts just like the user typed a
   ^C on the controlling terminal. */

void
child_stop ()
{
  DEBUG_EVENTS (("gdb: GenerateConsoleCtrlEvent (CTRLC_EVENT, 0)\n"));
/*  CHECK (GenerateConsoleCtrlEvent (CTRL_C_EVENT, 0)); */
  registers_changed();		/* refresh register state */
}

static int
child_can_run ()
{
  return 1;
}

void
vmschild_open (args, from_tty)
     char *args;
     int from_tty;
{
  long pid, status;
  char mboxbuff[256];
  struct target_waitstatus dummy;

  sscanf (args, "%x", &pid);
  remote_dcache = dcache_init (remote_read_bytes, remote_write_bytes);

  sprintf (mboxbuff, "gdbmbox%x", pid);
  remote_desc = SERIAL_OPEN (mboxbuff);
  if (!remote_desc)
    perror_with_name (mboxbuff);

  exception_count = 0;
  event_count = 0;

  inferior_pid = pid;
  current_process_id = pid;
  current_thread_id = pid;
  push_target (&vms_ops);
  init_thread_list ();
  init_task_list ();
  init_wait_for_inferior ();
  clear_proceed_status ();
  target_terminal_init ();
  target_terminal_inferior ();

  vmschild_wait (inferior_pid, &dummy);
}

void
vmschild_attach (proc, from_tty)
     char *proc;
     int from_tty;
{
  long pid, status;
  char mboxbuff[256];
  struct target_waitstatus dummy;

  sscanf (proc, "%x", &pid);
  sprintf (mboxbuff, "gdbmbox%x", pid);
  remote_desc = SERIAL_OPEN (mboxbuff);
  if (!remote_desc)
    perror_with_name (mboxbuff);

  printf_filtered ("Signaling pid: %x\n", pid);
  status = SYS$SIGPRC (&pid, 0, 1132);
  if ((status & 1) != 1)
    LIB$SIGNAL (status);

  remote_dcache = dcache_init (remote_read_bytes, remote_write_bytes);

  exception_count = 0;
  event_count = 0;

  inferior_pid = pid;
  current_process_id = pid;
  current_thread_id = pid;
  push_target (&vms_ops);
  init_thread_list ();
  init_task_list ();
  init_wait_for_inferior ();
  clear_proceed_status ();
  target_terminal_init ();
  target_terminal_inferior ();

}

static int
vmschild_clear_breakpoints ()
{
  return(0);
}

static int register_bytes_found;

/* Read the remote registers into the block REGS.  */
/* Currently we just read all the registers, so we don't use regno.  */
/* ARGSUSED */
static void
vmschild_fetch_inferior_registers (regno)
     int regno;
{
  char buf[PBUFSIZ];
  int i;
  char *p;
  char regs[REGISTER_BYTES];

  sprintf (buf, "g");
  remote_send (buf);

  /* Unimplemented registers read as all bits zero.  */
  memset (regs, 0, REGISTER_BYTES);

  /* We can get out of synch in various cases.  If the first character
     in the buffer is not a hex character, assume that has happened
     and try to fetch another packet to read.  */
  while ((buf[0] < '0' || buf[0] > '9')
	 && (buf[0] < 'a' || buf[0] > 'f'))
    {
      if (remote_debug)
	printf_unfiltered ("Bad register packet; fetching a new packet\n");
      getpkt (buf, 0);
    }

  /* Reply describes registers byte by byte, each byte encoded as two
     hex characters.  Suck them all up, then supply them to the
     register cacheing/storage mechanism.  */

  p = buf;
  for (i = 0; i < REGISTER_BYTES; i++)
    {
      if (p[0] == 0)
	break;
      if (p[1] == 0)
	{
	  warning ("Remote reply is of odd length: %s", buf);
	  /* Don't change register_bytes_found in this case, and don't
	     print a second warning.  */
	  goto supply_them;
	}
      regs[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
      p += 2;
    }

  if (i != register_bytes_found)
    {
      register_bytes_found = i;
#ifdef REGISTER_BYTES_OK
      if (!REGISTER_BYTES_OK (i))
	warning ("Remote reply is too short: %s", buf);
#endif
    }

 supply_them:
  for (i = 0; i < NUM_REGS; i++)
    supply_register (i, &regs[REGISTER_BYTE(i)]);
}

static unsigned char break_insn[] = BREAKPOINT;

static int
vmschild_insert_breakpoint (addr, save)
     CORE_ADDR addr;
     char *save;
{
  int val;

  val = target_read_memory (addr, save, sizeof break_insn);

  if (val == 0)
    val = target_write_memory (addr, (char *)break_insn, sizeof break_insn);

  return val;
}

static void
vmschild_kill_inferior (void)
{
  long status;
  struct target_waitstatus dummy;
  char buf[PBUFSIZ];
  __int64 timeout = -1 * 10000000 * 3;

  if (inferior_pid != 0)
    {
      strcpy (buf, "k");
      putpkt (buf);
      SYS$SETIMR (TIMER_EF, &timeout, 0, 0, 0);
      vmschild_wait (inferior_pid, &dummy);
      vmschild_mourn_inferior ();
      inferior_pid = 0;
    }
}

static int
vmschild_remove_breakpoint (addr, save)
     CORE_ADDR addr;
     char *save;
{
  return target_write_memory (addr, save, sizeof break_insn);
}

/* Tell the remote machine to resume.  */

static void
vmschild_resume (pid, step, siggnal)
     int pid, step;
     enum target_signal siggnal;
{
  char buf[PBUFSIZ];

#if 0
  if (siggnal)
    {
      target_terminal_ours_for_output ();
      printf_filtered
	("Can't send signals to a remote system.  %s not sent.\n",
	 target_signal_to_name (siggnal));
      target_terminal_inferior ();
    }
#endif

  dcache_flush (remote_dcache);

  strcpy (buf, step ? "s": "c");

  putpkt (buf);
}

static void
vmschild_store_inferior_registers (regno)
     int regno;
{
  char buf[PBUFSIZ];

  sprintf(buf, "G%x:%16lx", regno, read_register(regno));
  remote_send (buf);
}

/* Wait until the remote machine stops, then return,
   storing status in STATUS just as `wait' would.
   Returns "pid". */

static int
vmschild_wait (pid, status)
     int pid;
     struct target_waitstatus *status;
{
  char buf[PBUFSIZ];

  status->kind = TARGET_WAITKIND_EXITED;
  status->value.integer = 0;

  while (1)
    {
      char *p;

      ofunc = (void (*)()) signal (SIGINT, remote_interrupt);
      getpkt ((char *) buf, 1);
      SYS$CANTIM (0, 0);
      signal (SIGINT, ofunc);

      switch (buf[0])
	{
	case 'E':		/* Error of some sort */
	  if (strcmp (buf, "E00") == 0)
	    {
	      warning
		("Timeout waiting for target exit confirmation, quitting anyway.");
	      status->value.integer = 0;
	      return inferior_pid;
	    }

	  warning ("Remote failure reply: %s", buf);
	  continue;
	case 'T':		/* Status with PC, SP, FP, ... */
	  {
	    int i;
	    long regno;
	    char regs[MAX_REGISTER_RAW_SIZE];

	    /* Expedited reply, containing Signal, {regno, reg} repeat */
	    /*  format is:  'Tss%message%n...:r...;n...:r...;#cc', where
		ss = signal number
		n... = register number
		r... = register contents
		*/

	    p = &buf[9];	/* after Txx */
	    if (*p == '%')
	      {
		p++;
		p = strchr (p, '%');
		p++;
	      }

	    while (*p)
	      {
		char *p1;

		/* Read the register number */
		regno = strtol (p, (char **) &p1, 16);

		if (p1 == p)
		  warning
		    ("Remote sent badly formed register number: %s\nPacket: '%s'\n",
		     p1, buf);

		p = p1;

		if (*p++ != ':')
		  warning
		    ("Malformed packet (missing colon): %s\nPacket: '%s'\n",
		     p, buf);

		if (regno >= NUM_REGS)
		  warning
		    ("Remote sent bad register number 0x%x: %s\nPacket: '%s'\n",
		     regno, p, buf);

		for (i = 0; i < REGISTER_RAW_SIZE (regno); i++)
		  {
		    if (p[0] == 0 || p[1] == 0)
		      warning ("Remote reply is too short: %s", buf);
		    regs[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
		    p += 2;
		  }

		if (*p++ != ';')
		  warning ("Remote register badly formatted: %s", buf);

		supply_register (regno, regs);
	      }
	  }
	  /* fall through */
	case 'S':		/* Old style status, just signal only */
	  {
	    char *q;
	    int i, vms_status;


	    p = &buf[1];
	    q = (char *) &vms_status;
	    for (i = 0; i < 4; i++)
	      {
		q[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
		p += 2;
	      }

	    status->kind = TARGET_WAITKIND_STOPPED;
	    status->value.sig
	      = (enum target_signal) computeSignal(vms_status);

	    p = &buf[9];
	    if (*p == '%')
	      {
		p++;
		q = strchr (p, '%');
		*q = 0;

		if (*p)
		  target_signal_set_string (status->value.sig, xstrdup (p));
	      }
	  }

	  return inferior_pid;
	case 'W':		/* Target exited */
	  {
	    char *q;
	    int i;

	    /* The remote process exited.  */
	    status->kind = TARGET_WAITKIND_EXITED;
	    p = &buf[1];
	    status->value.integer = 0;
	    q = (char *)&status->value.integer;
	    for (i = 0; i < 4; i++)
	      {
		if (p[0] == 0 || p[1] == 0)
		  warning ("Remote reply is too short: %s", buf);
		q[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
		p += 2;
	      }
	    return inferior_pid;
	  }
	case 'O':		/* Console output */
	  fputs_filtered (buf + 1, gdb_stdout);
	  continue;
	default:
	  warning ("Invalid remote reply: %s", buf);
	  continue;
	}
    }
  return inferior_pid;
}

/* Read or write LEN bytes from inferior memory at MEMADDR, transferring
   to or from debugger address MYADDR.  Write to inferior if SHOULD_WRITE is
   nonzero.  Returns length of data written or read; 0 for error.  */

/* ARGSUSED */
static int
vmschild_xfer_memory(memaddr, myaddr, len, should_write, target)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
     int should_write;
     struct target_ops *target;			/* ignored */
{
  int xfersize;
  int bytes_xferred;
  int total_xferred = 0;

  while (len > 0)
    {
      if (len > MAXBUFBYTES)
	xfersize = MAXBUFBYTES;
      else
	xfersize = len;

      if (should_write)
        bytes_xferred = remote_write_bytes (memaddr, myaddr, xfersize);
      else
	bytes_xferred = remote_read_bytes (memaddr, myaddr, xfersize);

      /* If we get an error, we are done xferring.  */
      if (bytes_xferred == 0)
	break;

      memaddr += bytes_xferred;
      myaddr  += bytes_xferred;
      len     -= bytes_xferred;
      total_xferred += bytes_xferred;
    }
  return total_xferred;
}

void
vmschild_remote_call (funaddr, nargs, args, r0buff)
     CORE_ADDR funaddr;
     int nargs;
     value_ptr *args;
     char *r0buff;
{
  char buf[PBUFSIZ], *bufptr;
  int i;

  if (!target_has_stack)
    error ("No stack.");

  /* Cfuncaddr:numargs;arg1addr:arg2addr:...:argnaddr; */
  bufptr = buf;
  sprintf (bufptr, "C%lx:%x;", funaddr, nargs);

  for (i=0; i<nargs; i++)
    {
      bufptr = buf+strlen (buf);
      switch (args[i]->lval)
	{
	case not_lval:
	  sprintf (bufptr, "%lx:", value_as_long (args [i]));
	  break;
	case lval_memory:
	  if (TYPE_CODE (VALUE_TYPE (args [i])) == TYPE_CODE_ARRAY)
	    {
	      sprintf (bufptr, "%lx:", value_as_pointer (args [i]));
	    }
	  else
	    {
	      sprintf (bufptr, "%lx:", value_as_long (args [i]));
	    }
	  break;
	case lval_register:
	  sprintf (bufptr, "%%%lx:", args [i]->regno);
	  break;
	case lval_internalvar:
	  sprintf (bufptr, "%lx:", value_as_long (args [i]));
	  warning ("unsupported call arg storage class: lval_internalvar\n");
	  break;
	case lval_internalvar_component:
	  sprintf (bufptr, "%lx:", value_as_long (args [i]));
	  warning ("unsupported call arg storage class: lval_internalvar_component\n");
	  break;
	case lval_reg_frame_relative:
	  sprintf (bufptr, "%lx:", value_as_long (args [i]));
	  warning ("unsupported call arg storage class: lval_reg_frame_relative\n");
	  break;
	default:
	  sprintf (bufptr, "%lx:", value_as_long (args [i]));
	  warning ("unsupported call arg storage class: %d\n", args[i]->lval);
	}
    }
  bufptr = buf+strlen (buf);
  *(bufptr-1) = ';';

  remote_send (buf);

  bufptr = (char *) buf;
  for (i = 0; i < 8; i++)
    {
      r0buff[i] = fromhex (bufptr[0]) * 16 + fromhex (bufptr[1]);
      bufptr += 2;
    }
}

void
vmschild_read_pd_from_invo_handle (handle, frameflags, basereg, pd,
				   prev_invo_pc)
     long handle;
     char *frameflags;
     CORE_ADDR *basereg;
     PDSCDEF *pd;
     CORE_ADDR *prev_invo_pc;
{
  char buf[PBUFSIZ], *p, *q;
  int i, size;

  if (!target_has_stack)
    error ("No stack.");

  if (!handle)
    error ("No frame.");

  p = buf;
  sprintf (p, "p%x", handle);
  remote_send (buf);

  /* Read frame flags */
  p = buf;
  q = (char *) frameflags;
  for (i = 0; i < 1; i++)
    {
      q[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
      p += 2;
    }
  if (*p++ != ';')
    error ("Bad pd packet format");


  /* Read base register */
  q = (char *) basereg;
  for (i = 0; i < 8; i++)
    {
      q[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
      p += 2;
    }
  if (*p++ != ';')
    error ("Bad pd packet format");

  q = (char *) pd;
  for (i = 0; i < 2; i++)
    {
      q[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
      p += 2;
    }

  if (pd->pdsc$v_kind == PDSC$K_KIND_NULL)
    size = PDSC$K_NULL_SIZE;
  else if (pd->pdsc$v_kind == PDSC$K_KIND_FP_REGISTER)
    size = PDSC$K_MIN_REGISTER_SIZE;
  else if (pd->pdsc$v_kind == PDSC$K_KIND_FP_STACK)
    size = PDSC$K_MIN_STACK_SIZE;

  for (i = 2; i < size; i++)
    {
      if (p[0] == 0 || p[1] == 0) /* Reply is short. */
	error ("Short pd packet");
      q[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
      p += 2;
    }
  if (*p++ != ';')
    error ("Bad pd packet format");

  /* Read previous frame's PC */
  q = (char *) prev_invo_pc;
  for (i = 0; i < 8; i++)
    {
      q[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
      p += 2;
    }
}

void
vmschild_goto_unwind (handle)
     long handle;
{
  char buf[PBUFSIZ], *p;

  if (!target_has_stack)
    error ("No stack.");

  /* whandle */
  p = buf;
  if (handle)
    sprintf (p, "w%x", handle);

  putpkt (buf);
}

long
vmschild_get_prev_invo_handle (handle)
     long handle;
{
  char buf[PBUFSIZ], *p, *q;
  long prev_handle;
  int i;

  if (!target_has_stack)
    error ("No stack.");

  /* vhandle or v */
  p = buf;
  if (handle)
    sprintf (p, "h%x", handle);
  else
    sprintf (p, "h");

  remote_send (buf);

  p = buf;
  prev_handle = 0;
  q = (char *)&prev_handle;
  for (i = 0; i < 4; i++)
    {
      if (p[0] == 0 || p[1] == 0)
	warning ("Remote reply is too short: %s", buf);
      q[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
      p += 2;
    }

  return prev_handle;
}

unsigned long
vmschild_get_imglstptr ()
{
  char buf[PBUFSIZ], *p, *q;
  unsigned long imglstptr;
  int i;

  if (!target_has_stack)
    error ("No stack.");

  p = buf;
  sprintf (p, "i");

  remote_send (buf);

  p = buf;
  q = (char *)&imglstptr;
  for (i = 0; i < 4; i++)
    {
      if (p[0] == 0 || p[1] == 0)
	warning ("Remote reply is too short: %s", buf);
      q[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
      p += 2;
    }

  return imglstptr;
}

/* Convert hex digit A to a number.  */

static int
fromhex (a)
     int a;
{
  if (a >= '0' && a <= '9')
    return a - '0';
  else if (a >= 'a' && a <= 'f')
    return a - 'a' + 10;
  else
    error ("Reply contains invalid hex digit: %c", a);
}

/* Read a packet from the remote machine, with error checking,
   and store it in BUF.  BUF is expected to be of size PBUFSIZ.
   If FOREVER, wait forever rather than timing out; this is used
   while the target is executing user code.  */

/* All this timeout crap is ignored, the mailbox driver is assumed
   to be 100% reliable */

static void
getpkt (buf, forever)
     char *buf;
     int forever;
{
  char *bp;
  int c;
  int tries;
  int val;

  do
    {
      c = readchar (-1);
      if (c == SERIAL_TIMEOUT)
	{
	  strcpy (buf, "E00");
	  return;
	}
    }
  while (c != '$');

  /* We've found the start of a packet, now collect the data.  */

  val = read_frame (buf);

  if (val == 1)
    {
      if (remote_debug)
	fprintf_unfiltered (gdb_stderr, "Packet received: %s\n", buf);
      SERIAL_WRITE (remote_desc, "+", 1);
      return;
    }
}

/* Send a packet to the remote machine, with error checking.
   The data of the packet is in BUF.  */
static void
putpkt (buf)
     char *buf;
{
  int i;
  unsigned char csum = 0;
  char buf2[PBUFSIZ];
  int cnt = strlen (buf);
  int ch;
  char *p;

  /* Copy the packet into buffer BUF2, encapsulating it
     and giving it a checksum.  */

  if (cnt > sizeof(buf2) - 5)		/* Prosanity check */
    abort();

/*  printf ("putpkt: %s\n", buf); */

  p = buf2;
  *p++ = '$';

  for (i = 0; i < cnt; i++)
    {
      csum += buf[i];
      *p++ = buf[i];
    }
  *p++ = '#';
  *p++ = tohex ((csum >> 4) & 0xf);
  *p++ = tohex (csum & 0xf);

  /* Send it over and over until we get a positive ack.  */

  while (1)
    {
      int started_error_output = 0;

      if (remote_debug)
	{
	  *p = '\0';
	  printf_unfiltered ("Sending packet: %s...", buf2);
	  gdb_flush(gdb_stdout);
	}
      if (SERIAL_WRITE (remote_desc, buf2, p - buf2))
	perror_with_name ("putpkt: write failed");

      /* read until either a timeout occurs (-2) or '+' is read */
      while (1)
	{
	  ch = readchar (remote_timeout);

	  if (remote_debug)
	    {
	      switch (ch)
		{
		case '+':
		case SERIAL_TIMEOUT:
		case '$':
		  if (started_error_output)
		    {
		      putchar_unfiltered ('\n');
		      started_error_output = 0;
		    }
		}
	    }

	  switch (ch)
	    {
	    case '+':
	      if (remote_debug)
		printf_unfiltered("Got Ack\n");
	      return;
	    case SERIAL_TIMEOUT:
	      break;		/* Retransmit buffer */
	    case '$':
	      {
		char junkbuf[PBUFSIZ];

	      /* It's probably an old response, and we're out of sync.  Just
		 gobble up the packet and ignore it.  */
		getpkt (junkbuf, 0);
		continue;		/* Now, go look for + */
	      }
	    default:
	      if (remote_debug)
		{
		  if (!started_error_output)
		    {
		      started_error_output = 1;
		      printf_unfiltered ("putpkt: Junk: ");
		    }
		  putchar_unfiltered (ch & 0177);
		}
	      continue;
	    }
	  break;		/* Here to retransmit */
	}

#if 0
      /* This is wrong.  If doing a long backtrace, the user should be
	 able to get out next time we call QUIT, without anything as violent
	 as interrupt_query.  If we want to provide a way out of here
	 without getting to the next QUIT, it should be based on hitting
	 ^C twice as in remote_wait.  */
      if (quit_flag)
	{
	  quit_flag = 0;
	  interrupt_query ();
	}
#endif
    }
}

/* Come here after finding the start of the frame.  Collect the rest into BUF,
   verifying the checksum, length, and handling run-length compression.
   Returns 0 on any error, 1 on success.  */

static int
read_frame (buf)
     char *buf;
{
  unsigned char csum;
  int count;
  char *bp;
  int c;

  csum = 0;
  count = 0;
  bp = buf;

  while (1)
    {
      c = readchar (remote_timeout);

      switch (c)
	{
	case SERIAL_TIMEOUT:
	  if (remote_debug)
	    puts_filtered ("Timeout in mid-packet, retrying\n");
	  return 0;
	case '$':
	  if (remote_debug)
	    puts_filtered ("Saw new packet start in middle of old one\n");
	  return 0;		/* Start a new packet, count retries */
	case '#':
	  {
	    unsigned char pktcsum;

	    *bp = '\000';

	    pktcsum = fromhex (readchar (remote_timeout)) << 4;
	    pktcsum |= fromhex (readchar (remote_timeout));

/*	    puts_filtered (buf);
	    puts_filtered ("\n"); */
	    if (csum == pktcsum)
	      return 1;

	    printf_filtered ("Bad checksum, sentsum=0x%x, csum=0x%x, buf=",
			     pktcsum, csum);
	    puts_filtered (buf);
	    puts_filtered ("\n");

	    return 0;
	  }
	case '*':		/* Run length encoding */
	  csum += c;
	  count+= 1;
	  c = readchar (remote_timeout);
	  csum += c;
	  count+= 1;
	  c = c - ' ' + 3;	/* Compute repeat count */

	  if (bp + c - 1 < buf + PBUFSIZ - 1)
	    {
	      memset (bp, *(bp - 1), c);
	      bp += c;
	      continue;
	    }

	  *bp = '\0';
	  printf_filtered ("Repeat count %d too large for buffer: ", c);
	  puts_filtered (buf);
	  puts_filtered ("\n");
	  return 0;

	default:
	  if (bp < buf + PBUFSIZ - 1)
	    {
	      *bp++ = c;
	      csum += c;
	      count+= 1;
	      continue;
	    }

	  *bp = '\0';
	  puts_filtered ("Remote packet too long: ");
	  puts_filtered (buf);
	  puts_filtered ("\n");

	  return 0;
	}
    }
}

/* Read a single character from the remote end, masking it down to 7 bits. */

static int
readchar (timeout)
     int timeout;
{
  int ch;

  ch = SERIAL_READCHAR (remote_desc, timeout);

  switch (ch)
    {
    case SERIAL_EOF:
      error ("Remote connection closed");
    case SERIAL_ERROR:
      perror_with_name ("Remote communication error");
    case SERIAL_TIMEOUT:
      return ch;
    default:
      return ch & 0xff;
    }
}

static void
remote_close (quitting)
     int quitting;
{
  if (remote_desc)
    SERIAL_CLOSE (remote_desc);
  remote_desc = NULL;
}

/* Send ^C to target to halt it.  Target will respond, and send us a
   packet.  */

static void
remote_interrupt (signo)
     int signo;
{
  int status;

  /* If this doesn't work, try more severe steps.  */
  signal (signo, remote_interrupt_twice);
  
  if (remote_debug)
    printf_unfiltered ("remote_interrupt called\n");

  status = SYS$SIGPRC (&inferior_pid, 0, SIGTRAP);
  if ((status & 1) != status) lib$stop (status);
}

/* The user typed ^C twice.  */
static void
remote_interrupt_twice (signo)
     int signo;
{
  signal (signo, ofunc);
  
  interrupt_query ();

  signal (signo, remote_interrupt);
}

/* Ask the user what to do when an interrupt is received.  */

static void
interrupt_query ()
{
  target_terminal_ours ();

  if (query ("Interrupted while waiting for the program.\n\
Give up (and stop debugging it)? "))
    {
      vmschild_mourn_inferior ();
      return_to_top_level (RETURN_QUIT);
    }

  target_terminal_inferior ();
}


/* Read memory data directly from the remote machine.
   This does not use the data cache; the data cache uses this.
   MEMADDR is the address in the remote memory space.
   MYADDR is the address of the buffer in our space.
   LEN is the number of bytes.

   Returns number of bytes transferred, or 0 for error.  */

static int
remote_read_bytes (memaddr, myaddr, len)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
{
  char buf[PBUFSIZ];
  int i;
  char *p;

  if (len > PBUFSIZ / 2 - 1)
    abort ();

  /* FIXME-32x64: Need a version of print_address_numeric which puts the
     result in a buffer like sprintf.  */
  sprintf (buf, "m%lx,%x", (unsigned long) memaddr, len);
  putpkt (buf);
  getpkt (buf, 0);

  if (buf[0] == 'E')
    {
      /* There is no correspondance between what the remote protocol uses
	 for errors and errno codes.  We would like a cleaner way of
	 representing errors (big enough to include errno codes, bfd_error
	 codes, and others).  But for now just return EIO.  */
      errno = EIO;
      return 0;
    }

  if (buf[0] == 'W')
    {
      error ("Target has exited\n");
      return 0;
    }

  /* Reply describes memory byte by byte,
     each byte encoded as two hex characters.  */

  p = buf;
  for (i = 0; i < len; i++)
    {
      if (p[0] == 0 || p[1] == 0)
	/* Reply is short.  This means that we were able to read only part
	   of what we wanted to.  */
	break;
      myaddr[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
      p += 2;
    }
  return i;
}
/* Send the command in BUF to the remote machine,
   and read the reply into BUF.
   Report an error if we get an error reply.  */

static void
remote_send (buf)
     char *buf;
{

  putpkt (buf);
  getpkt (buf, 0);

  if (buf[0] == 'E')
    error ("Remote failure reply: %s", buf);
}

/* Write memory data directly to the remote machine.
   This does not inform the data cache; the data cache uses this.
   MEMADDR is the address in the remote memory space.
   MYADDR is the address of the buffer in our space.
   LEN is the number of bytes.

   Returns number of bytes transferred, or 0 for error.  */

static int
remote_write_bytes (memaddr, myaddr, len)
     CORE_ADDR memaddr;
     char *myaddr;
     int len;
{
  char buf[PBUFSIZ];
  int i;
  char *p;

  /* FIXME-32x64: Need a version of print_address_numeric which puts the
     result in a buffer like sprintf.  */
  sprintf (buf, "M%lx,%x:", (unsigned long) memaddr, len);

  /* We send target system values byte by byte, in increasing byte addresses,
     each byte encoded as two hex characters.  */

  p = buf + strlen (buf);
  for (i = 0; i < len; i++)
    {
      *p++ = tohex ((myaddr[i] >> 4) & 0xf);
      *p++ = tohex (myaddr[i] & 0xf);
    }
  *p = '\0';

  putpkt (buf);
  getpkt (buf, 0);

  if (buf[0] == 'E')
    {
      /* There is no correspondance between what the remote protocol uses
	 for errors and errno codes.  We would like a cleaner way of
	 representing errors (big enough to include errno codes, bfd_error
	 codes, and others).  But for now just return EIO.  */
      errno = EIO;
      return 0;
    }
  return len;
}

/* Convert number NIB to a hex digit.  */

static int
tohex (nib)
     int nib;
{
  if (nib < 10)
    return '0'+nib;
  else
    return 'a'+nib-10;
}

struct target_ops vms_ops =
{
  "vmschild",			/* to_shortname */
  "VMS child process",          /* to_longname */
  "VMS child process (started by the \"run\" command).",	/* to_doc */
  vmschild_open,		/* to_open */
  remote_close, 		/* to_close */
  vmschild_attach,		/* to_attach */
  gr_detach,			/* to_detach */
  vmschild_resume,		/* to_resume */
  vmschild_wait,		/* to_wait */
  vmschild_fetch_inferior_registers,  /* to_fetch_registers */
  vmschild_store_inferior_registers,  /* to_store_registers */
  gr_prepare_to_store,          /* to_child_prepare_to_store */
  vmschild_xfer_memory,		/* to_xfer_memory */
  gr_files_info,		/* to_files_info */
  vmschild_insert_breakpoint,	/* to_insert_breakpoint */
  vmschild_remove_breakpoint,	/* to_remove_breakpoint */
  0,                            /* to_terminal_init */
  0,                            /* to_terminal_inferior */
  0,                            /* to_terminal_ours_for_output */
  0,                            /* to_terminal_ours */
  0,                            /* to_terminal_info */
  vmschild_kill_inferior,	/* to_kill */
  0,				/* to_load */
  0,				/* to_lookup_symbol */
  vmschild_create_inferior,	/* to_create_inferior */
  vmschild_mourn_inferior,	/* to_mourn_inferior */
  child_can_run,		/* to_can_run */
  0,				/* to_notice_signals */
  0,				/* to_thread_alive */
  child_stop,			/* to_stop */
  process_stratum,		/* to_stratum */
  0,				/* to_next */
  1,				/* to_has_all_memory */
  1,				/* to_has_memory */
  1,				/* to_has_stack */
  1,				/* to_has_registers */
  1,				/* to_has_execution */
  0,				/* to_sections */
  0,				/* to_sections_end */
  OPS_MAGIC			/* to_magic */
};

void
_initialize_inftarg ()
{
  add_target (&vms_ops);
}
