/* Read EVAX (Alpha/VMS) format executable files for GDB.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include "defs.h"
#include "gdb_string.h"
#include "bfd.h"
#include "symtab.h"
#include "gdbtypes.h"
#include "symfile.h"
#include "objfiles.h"
#include "buildsym.h"
#include "expression.h"
#include "language.h"
#include "gdb-stabs.h"
#include "evax-solib.h"
#include <dscdef>
#include "../../digital/vmsdbg.h"
#include <unixlib.h>
#include "evax-debug.h" /* from bfd */

#define DSTDEBUG 0

struct evaxinfo {
  file_ptr dstfilepos;
  unsigned char *dmtcontents;
  unsigned int dmtsize;
};

struct dstinfo {
  /* Always the absolute file offset to the module begin record */
  DST_MODULE_BEGIN *dstmodbeg;
  /* Size of the DST's for this module ??? */
  unsigned int dstmodsiz;
};

typedef struct pd_hash_struct
{
  struct bfd_hash_entry root;
  CORE_ADDR pd_address;
} pd_hash_entry;

struct bfd_hash_table *pd_hash_table;

typedef struct modnam_hash_struct
{
  struct bfd_hash_entry root;
} modnam_hash_entry;

struct bfd_hash_table *modnam_hash_table;

static enum language cu_language;
static const struct language_defn *cu_language_defn;
static struct pending **list_in_scope = &file_symbols;

#define DSTMODBEG(p) (((struct dstinfo *)((p)->read_symtab_private))->dstmodbeg)
#define DSTMODSIZ(p) (((struct dstinfo *)((p)->read_symtab_private))->dstmodsiz)

static void
evax_new_init PARAMS ((struct objfile *));

static void
evax_psymtab_to_symtab PARAMS ((struct partial_symtab *));

static void
evax_symfile_init PARAMS ((struct objfile *));

static void
evax_symfile_read PARAMS ((struct objfile *, struct section_offsets *, int));

static void
evax_symfile_finish PARAMS ((struct objfile *));

static void
evax_symtab_read PARAMS ((bfd *, struct section_offsets *, struct objfile *));

static struct section_offsets *
evax_symfile_offsets PARAMS ((struct objfile *, CORE_ADDR));

static void
record_minimal_symbol PARAMS ((char *, CORE_ADDR, enum minimal_symbol_type,
			       struct objfile *, CORE_ADDR));

static void
evax_read_debug_sections PARAMS ((bfd *, asection *, void *));

static void
evax_build_psymtabs PARAMS ((struct objfile*, struct section_offsets*, int,
			     file_ptr,
			     unsigned char *, unsigned int));

static struct textsec
{
  asection *textsec;
};

static void
find_text_section (abfd, sectp, textsecp)
     bfd *abfd;
     asection *sectp;
     PTR textsecp;
{
  if (sectp->flags & SEC_CODE)
    (*(struct textsec *)textsecp).textsec = sectp;
}

static void
evax_build_psymtabs (objfile, section_offsets, mainline,
		     dstfilepos,
		     dmtcontents, dmtsize)
     struct objfile *objfile;
     struct section_offsets *section_offsets;
     int mainline;
     file_ptr dstfilepos;
     unsigned char *dmtcontents;
     unsigned int dmtsize;
{
  bfd *abfd = objfile ->obfd;
  struct cleanup *back_to;
  unsigned char *dmtptr;
  CORE_ADDR baseaddr;
  struct partial_symtab *pst;
  asection *textsec;
  struct textsec atextsec;

  current_objfile = objfile;

  back_to = make_cleanup (free, dmtcontents);

  /* Guess in initial size */
  if (mainline
      || objfile -> global_psymbols.size == 0
      || objfile -> static_psymbols.size == 0)
    {
      init_psymbol_list (objfile, 1024);
    }

  atextsec.textsec = 0;
  bfd_map_over_sections (abfd, find_text_section, (PTR) &atextsec);

  textsec = atextsec.textsec;

#if 0
  printf ("text section: %llx - %llx\n",
	  textsec->vma,
	  textsec->vma + textsec->_raw_size);
#endif


  baseaddr = ANOFFSET (section_offsets, 0);

  dmtptr = dmtcontents;
  while (dmtptr < dmtcontents + dmtsize)
    {
      DBG_DMT_HEADER *dmtheader;

      dmtheader = (DBG_DMT_HEADER *) dmtptr;
      dmtptr += DBG_K_DMT_HEADER_SIZE;

      {
	int i;
	int count = dmtheader->dbg_w_dmt_psect_count;
	DBG_DMT_PSECT *dmtpsect [255];
	int textsecnum = -1;

	for (i=0; i<count; i++)
	  {
	    dmtpsect [i] = (DBG_DMT_PSECT *)dmtptr;
	    if (((CORE_ADDR) dmtpsect[i] -> dbg_l_dmt_psect_start
		 >= textsec->vma)
		&&
		(((CORE_ADDR) dmtpsect[i] -> dbg_l_dmt_psect_start
		  + dmtpsect[i] -> dbg_l_dmt_psect_length)
		 <= textsec->vma + textsec->_raw_size))
	      {
		if (textsecnum != -1)
		  warning ("multiple text sections in DMT psect array");
		textsecnum = i;
#if DSTDEBUG
		printf ("text section number: %d\n", textsecnum);
#endif
	      }
	    dmtptr += DBG_K_DMT_PSECT_SIZE;
	  }

	if (count > 0)
	  {

	    char modname [40];
	    DST_MODULE_BEGIN dst;
	    long dstmodbeg;
	    unsigned int dstmodsiz;
	    CORE_ADDR textsecaddr;

	    dstmodsiz = min (dmtheader->dbg_l_dmt_dst_size,
			     DST_K_MODBEG_SIZE + 39);
	    dstmodbeg = (long) dstfilepos + (long) dmtheader->dbg_l_dmt_modbeg;

	    if ((bfd_seek (abfd, dstmodbeg, SEEK_SET) != 0)
		|| (bfd_read (&dst, dstmodsiz, 1, abfd) != dstmodsiz))
	    {
	      error ("can't read debug symbol table from '%s'",
		     bfd_get_filename (abfd));
	    }

	    strncpy (modname,
		     (char *) &dst.dst_b_modbeg_name+1,
		     dst.dst_b_modbeg_name);
	    modname [dst.dst_b_modbeg_name] = 0;

#if DSTDEBUG
	    printf ("Module: %s psects:\n", modname);
	    for (i=0; i<count; i++)
	      {
		printf ("psect %d: %llx - %llx\n",
			i,
			(CORE_ADDR) dmtpsect[i] -> dbg_l_dmt_psect_start,
			(CORE_ADDR) dmtpsect[i] -> dbg_l_dmt_psect_start
			+ dmtpsect[i] -> dbg_l_dmt_psect_length);
	      }
#endif

	    if (!module_loaded (modname))
	      {
#if DSTDEBUG
		printf ("Loading VMS debug info for module: %s\n", modname);
		for (i=0; i<count; i++)
		  {
		    printf ("psect %d: %llx - %llx\n",
			    i,
			    (CORE_ADDR) dmtpsect[i] -> dbg_l_dmt_psect_start,
			    (CORE_ADDR) dmtpsect[i] -> dbg_l_dmt_psect_start
			    + dmtpsect[i] -> dbg_l_dmt_psect_length);
			    
		  }
#endif

		if (textsecnum == -1)
		  {
		    textsecaddr = 0;
		  }
		else
		  {
		    textsecaddr =
		    (CORE_ADDR) dmtpsect [textsecnum]
		      -> dbg_l_dmt_psect_start;
		  }

		pst = start_psymtab_common
		  (objfile, section_offsets, modname, 
		   textsecaddr,
		   objfile->global_psymbols.next,
		   objfile->static_psymbols.next);
		
		pst -> read_symtab_private
		  = (char *) obstack_alloc (&objfile -> psymbol_obstack,
					    sizeof (struct dstinfo));
		
		if (textsecnum != -1)
		  pst -> texthigh
		    = (CORE_ADDR) ((char *) dmtpsect [textsecnum]
				   -> dbg_l_dmt_psect_start
				   + dmtpsect [textsecnum]
				   -> dbg_l_dmt_psect_length);
		else
		  pst -> texthigh = (CORE_ADDR) 0;
		
		pst -> read_symtab = evax_psymtab_to_symtab;

		DSTMODBEG(pst) = (DST_MODULE_BEGIN *)
		  (dstfilepos + (int) dmtheader->dbg_l_dmt_modbeg);
		DSTMODSIZ(pst) = dmtheader->dbg_l_dmt_dst_size;

		/* To do: Read in partial symbols */

		pst->n_global_syms = objfile->global_psymbols.next -
		  (objfile->global_psymbols.list + pst->globals_offset);
		pst->n_static_syms = objfile->static_psymbols.next -
		  (objfile->static_psymbols.list + pst->statics_offset);
		sort_pst_symbols (pst);

	      }
	  }
      }
    }

  do_cleanups (back_to);
  current_objfile = NULL;
    
}

static char *
create_name (name, obstackp)
     char *name;
     struct obstack *obstackp;
{
  int length;
  char *newname;

  length = strlen (name) + 1;
  newname = (char *) obstack_alloc (obstackp, length);
  strcpy (newname, name);
  return (newname);
}

static void
evax_psymtab_to_symtab (pst)
     struct partial_symtab *pst;
{
  struct objfile *objfile = pst->objfile;
  bfd *abfd = objfile->obfd;
  unsigned char *dst, *dstptr;
  unsigned int dstmodsiz = DSTMODSIZ(pst);
  struct cleanup *cleanup_dst, *cleanup_module;
  struct symtab *symtab;
  struct subfile *module_subfile = NULL;
  char *last_filename = NULL;
  char *last_dirname = NULL;
  struct lineinfo *lineinfo;
  struct srecinfo *srecinfo;
  struct srecinfo* save_srec = 0;

  if (pst != NULL)
    {
      if (pst->readin)
	{
	  warning ("psymtab for %s already read in.  Shouldn't happen.",
		   pst -> filename);
	}
      else
	{
	  struct module* module;

	  module = (struct module*) xmalloc (sizeof (struct module));
	  cleanup_module = make_cleanup (free, module);
	  module->file_table_max_srcfiles = MAX_SRCFILES;
	  module->file_table = xmalloc
	    (module->file_table_max_srcfiles * sizeof (struct fileinfo));
	  memset (module->file_table, 0,
		  module->file_table_max_srcfiles * sizeof (struct fileinfo));
	  module->abfd = abfd;

	  dst = (unsigned char *) xmalloc (dstmodsiz);
	  cleanup_dst = make_cleanup (free, dst);
	  dstptr = dst;
	  
	  if ((bfd_seek (abfd, (long) DSTMODBEG(pst), SEEK_SET) != 0)
	      || (bfd_read (dst, dstmodsiz, 1, abfd) != dstmodsiz))
	    {
	      free (dst);
	      free (module);
	      error ("can't read debug symbol table from '%s'",
		     bfd_get_filename (abfd));
	    }
	  
	  buildsym_init ();
	  parse_module (module, dst, dstmodsiz);
	  while (dstptr < dst + dstmodsiz)
	    {
	      int reclength;
	      int rectype;
	      
	      reclength = bfd_getl16 (dstptr);
	      rectype = bfd_getl16 (dstptr + 2);

#if DSTDEBUG
	      printf ("rectype: %x (%d), reclength: %x\n",
		      rectype, rectype, reclength);
#endif
	      switch (rectype)
		{
		case DST_K_MODBEG:
		  {
		    DST_MODULE_BEGIN *dstmodbeg;
		    
		    dstmodbeg = (DST_MODULE_BEGIN *) dstptr;
#if DSTDEBUG
		    printf ("module begin: %s\n", module->name);
#endif

		    switch (dstmodbeg->dst_l_modbeg_language)
		      {
		      case DST_K_ADA:
			cu_language = language_ada;
			break;
		      case DST_K_C:
			cu_language = language_c;
			break;
		      case DST_K_CXX:
			cu_language = language_cplus;
			break;
		      case DST_K_FORTRAN:
			cu_language = language_fortran;
			break;
		      case DST_K_MODULA:
			cu_language = language_m2;
			break;
		      default:
			cu_language = language_c;
		      }
		    cu_language_defn = language_def (cu_language);

		    start_symtab (module->name, "", pst->textlow);

		    /* Save the module subfile so it can be patched with the
		       proper file name later */
		    module_subfile = current_subfile;

		    list_in_scope = &file_symbols;

		    /* Print the message now, before starting serious work,
		       to avoid disconcerting pauses.  */
		    if (info_verbose)
		      {
			printf_filtered ("Reading in symbols for %s...",
					 module->name);
			gdb_flush (gdb_stdout);
		      }
		  }
		  break;
		case DST_K_MODEND:
#if DSTDEBUG
		  printf ("module end\n");
#endif
		  break;
		case DST_K_PROLOG:
#if DSTDEBUG
		  printf ("prolog\n");
#endif
		  break;
		case DST_K_EPILOG:
#if DSTDEBUG
		  printf ("epilog\n");
#endif
		  break;
		case DST_K_RTNBEG:
		  {
		    DST_ROUTINE_BEGIN *dstrtnbeg;
		    char rtnname [255];
		    char buff [20];
		    struct symbol *sym;
		    struct context_stack *new;
		    pd_hash_entry *pd_entry;

		    dstrtnbeg = (DST_ROUTINE_BEGIN *) dstptr;
		    strncpy (rtnname,
			     (char *) &dstrtnbeg->dst_b_rtnbeg_name + 1,
			     dstrtnbeg->dst_b_rtnbeg_name);
		    rtnname [dstrtnbeg->dst_b_rtnbeg_name] = 0;
#if DSTDEBUG
		    printf ("begin routine: %s at %x\n",
			    rtnname,
			    dstrtnbeg->dst_l_rtnbeg_address);
#endif

		    if (STREQ (rtnname, "main"))
		      objfile -> ei.main_func_lowpc
			= (CORE_ADDR) dstrtnbeg->dst_l_rtnbeg_address;

		    new = push_context
		      (0, (CORE_ADDR) dstrtnbeg->dst_l_rtnbeg_address);
		    sym = (struct symbol *)obstack_alloc
		      (&objfile ->symbol_obstack,
		       sizeof (struct symbol));
		    new->name = sym;
		    list_in_scope = &local_symbols;
		    OBJSTAT (objfile, n_syms++);
		    memset (sym, 0, sizeof (struct symbol));
		    SYMBOL_NAME (sym)
		      = create_name (rtnname, &objfile->symbol_obstack);

		    SYMBOL_NAMESPACE (sym) = VAR_NAMESPACE;
		    SYMBOL_VALUE_ADDRESS (sym)
		      = (CORE_ADDR) dstrtnbeg->dst_l_rtnbeg_address;
		    SYMBOL_AUX_ADDRESS (sym)
		      = (CORE_ADDR) dstrtnbeg->dst_l_rtnbeg_pd_address;

#if DSTDEBUG
		    printf ("hash insert: %s\n", rtnname);
#endif
		    sprintf (buff, "%llx", dstrtnbeg->dst_l_rtnbeg_address);
		    pd_entry = (pd_hash_entry *) bfd_hash_lookup
		      (pd_hash_table, buff, true, true);
		    if (pd_entry->pd_address == 0)
		      pd_entry->pd_address
			= (CORE_ADDR) dstrtnbeg->dst_l_rtnbeg_pd_address;
		    else if (pd_entry->pd_address
			     != (CORE_ADDR) dstrtnbeg->dst_l_rtnbeg_pd_address)
		      error ("duplicate pd hash entry for '%s'", rtnname);

		    SYMBOL_TYPE (sym)
		      = cu_language_defn -> la_fund_type (objfile, FT_VOID);
		    SYMBOL_TYPE (sym)
		      = lookup_function_type (SYMBOL_TYPE (sym));
		    SYMBOL_CLASS (sym) = LOC_BLOCK;
		    add_symbol_to_list (sym, &global_symbols);
		  }
		  break;
		case DST_K_RTNEND:
		  {
		    DST_ROUTINE_END *dstrtnend;
		    struct context_stack *new;

		    dstrtnend = (DST_ROUTINE_END *) dstptr;
		    new = pop_context ();
		    finish_block
		      (new->name, &local_symbols, new->old_blocks,
		       new->start_addr,
		       new->start_addr + dstrtnend->dst_l_rtnend_size,
		       objfile);
		    if (STREQ (SYMBOL_NAME (new->name), "main"))
		      objfile -> ei.main_func_highpc
			= new->start_addr + dstrtnend->dst_l_rtnend_size;
#if DSTDEBUG
		    printf ("end routine at %x, size of %x\n", 
			    new->start_addr + dstrtnend->dst_l_rtnend_size,
			    dstrtnend->dst_l_rtnend_size);
#endif
		    list_in_scope = &file_symbols;
		  }
		  break;
		case DST_K_BLKBEG:
		  {
		    DST_BLOCK_BEGIN *dstblkbeg;

		    dstblkbeg = (DST_BLOCK_BEGIN *) dstptr;
#if DSTDEBUG
		    printf ("begin block at %x\n",
			    dstblkbeg->dst_l_blkbeg_address);
#endif
		    push_context
		      (0, (CORE_ADDR) dstblkbeg->dst_l_blkbeg_address);
		  }
		  break;
		case DST_K_BLKEND:
		  {
		    DST_BLOCK_END *dstblkend;
		    struct context_stack *new;

		    dstblkend = (DST_BLOCK_END *) dstptr;
		    new = pop_context ();

		    if (local_symbols != NULL)
		      {
#if DSTDEBUG
			printf ("block has local symbols\n");
#endif
			finish_block
			  (0, &local_symbols, new->old_blocks,
			   new->start_addr,
			   new->start_addr + dstblkend->dst_l_blkend_size,
			   objfile);
		      }
#if DSTDEBUG
		    printf ("end block at %x, size: %x\n",
			    new->start_addr + dstblkend->dst_l_blkend_size,
			    dstblkend->dst_l_blkend_size);
#endif
		    local_symbols = new->locals;
		  }
		  break;
		case 0x17: /* Transfer address */
		  break;
		case DSC$K_DTYPE_L:
		  {
		    DST_DATA_DST *dstdata;
		    char dataname [255];
		    struct symbol *sym = NULL;

#if 1
		    printf ("data\n");
#endif
		    dstdata = (DST_DATA_DST *) dstptr;
		    strncpy (dataname,
			     (char *) &dstdata->dst_b_name + 1,
			     dstdata->dst_b_name);
		    dataname [dstdata->dst_b_name] = 0;
#if 1
		    printf ("dataname: %s = %x\n",
			    dataname, dstdata->dst_l_value);
#endif


		    sym = (struct symbol *) obstack_alloc
		      (&objfile -> symbol_obstack, sizeof (struct symbol));
		    OBJSTAT (objfile, n_syms++);
		    memset (sym, 0, sizeof (struct symbol));
		    SYMBOL_NAME (sym) = create_name
		      (dataname, &objfile->symbol_obstack);

		    SYMBOL_NAMESPACE (sym) = VAR_NAMESPACE;
		    SYMBOL_TYPE (sym)
		      = cu_language_defn -> la_fund_type (objfile, FT_LONG);
		    SYMBOL_LANGUAGE (sym) = cu_language;

		    if (dstdata->dst_a_data_dst_header.dst_b_vflags.
			 dst_v_valkind == DST_K_VALKIND_ADDR)
		      {
			if (dstdata->dst_a_data_dst_header.dst_b_vflags.
			    dst_v_disp == 1)
			  /* displacement off of a register */
			  {
			    SYMBOL_VALUE (sym) = dstdata->dst_l_value;
			    SYMBOL_CLASS (sym) = LOC_BASEREG_ARG;
			    
			    if (dstdata->dst_a_data_dst_header.dst_b_vflags.
				dst_v_regnum == DST_K_REG_ALPHA_REGNUM_FP)
			      SYMBOL_BASEREG (sym) = FP_REGNUM;
			    else if (dstdata->dst_a_data_dst_header.
				     dst_b_vflags. dst_v_regnum
				     == DST_K_REG_ALPHA_REGNUM_SP)
			      SYMBOL_BASEREG (sym) = SP_REGNUM;
			    else
			      warning ("unknown DSC$DTYPE_L base register: %d",
				       dstdata->dst_a_data_dst_header.
				       dst_b_vflags.dst_v_regnum);
			  }
			else
			  /* absolute displacement ??? */
			  {
			    SYMBOL_VALUE (sym) = dstdata->dst_l_value;
			    SYMBOL_CLASS (sym) = LOC_STATIC;
			  }
		      }
		    else if (dstdata->dst_a_data_dst_header.dst_b_vflags.
			     dst_v_valkind == DST_K_VALKIND_REG)
		      {
#if DSTDEBUG
			printf ("valkind reg\n");
#endif
			SYMBOL_CLASS (sym) = LOC_REGISTER;
			SYMBOL_VALUE (sym)
			  = dstdata->dst_l_value - DST_K_REG_ALPHA_MIN;
		      }
		    else
		      {
			warning
			  ("unknown DSC$DTYPE_L  valkind: %d, disp: %d, regnum: %d",
				 dstdata->dst_a_data_dst_header.
				 dst_b_vflags.dst_v_valkind,
				 dstdata->dst_a_data_dst_header.
				 dst_b_vflags.dst_v_disp,
				 dstdata->dst_a_data_dst_header.
				 dst_b_vflags.dst_v_regnum);
		      }
		    add_symbol_to_list (sym, list_in_scope);
		  }
		  break;
		default:
		  ;
#if DSTDEBUG
		  printf ("Ignoring debug record: %x\n", rectype);
#endif
		}
	      dstptr += reclength + 1;
	    }

	  for (lineinfo = module->line_table;
	       lineinfo->next_line;
	       lineinfo = lineinfo->next_line)
	    {

	      if ((save_srec == 0) || (lineinfo->line < save_srec->line))
		save_srec = module->srec_table;

	      for (srecinfo = save_srec;
		   srecinfo;
		   srecinfo = srecinfo->next_srec)
		{
		  if (srecinfo->sfile != 0
		      && lineinfo->line < srecinfo->next_srec->line)
		    {
		      save_srec = srecinfo;
		      
		      /* Heuristic to avoid excessive calls to
			 start_subfile () */

		      if (last_filename !=
			  module->file_table [srecinfo->sfile].filename
			  ||
			  last_dirname !=
			  module->file_table [srecinfo->sfile].dirname)
			{
			  start_subfile
			    (module->file_table [srecinfo->sfile].filename,
			     module->file_table [srecinfo->sfile].dirname);

			  last_filename =
			    module->file_table [srecinfo->sfile].filename;
			  last_dirname =
			    module->file_table [srecinfo->sfile].dirname;
			}

		      record_line
			(current_subfile,
			 srecinfo->srec
			 + (lineinfo->line - srecinfo->line),
			 lineinfo->address);

		      break;
		    }
		}
	    }

	  if (module->file_table [1].filename)
	    {
	      module_subfile->name = module->file_table [1].filename;
	      module_subfile->dirname = module->file_table [1].dirname;
	    }

	  symtab = end_symtab (pst->texthigh, objfile, 0);
	  if (symtab != NULL)
	    {
	      symtab -> language = cu_language;
	    }      
	  module_subfile = NULL;

#if DSTDEBUG
	  printf ("evax_psymtab_to_symtab called for module: %s at ", modname);
	  printf ("range: %llx - %llx\n", pst->textlow, pst->texthigh);
#endif

	  pst->readin = 1;
	  pst->symtab = pst->objfile->symtabs;

	  do_cleanups (cleanup_dst);
	  do_cleanups (cleanup_module);
	}
    }
}

static void
evax_read_debug_sections (abfd, sectp, eip)
     bfd *abfd;
     asection *sectp;
     PTR eip;
{
  register struct evaxinfo *ei;

  ei = (struct evaxinfo *) eip;

  if (STREQ (sectp->name, "$DST$"))
    {
      ei->dstfilepos = sectp->filepos;
    }
  else if (STREQ (sectp->name, "$DMT$"))
    {
      ei->dmtsize = bfd_get_section_size_before_reloc (sectp);
      ei->dmtcontents = xmalloc (ei->dmtsize);
      bfd_get_section_contents (abfd, sectp, ei->dmtcontents, 0, ei->dmtsize);
    }
}

/* Initialize anything that needs initializing when a completely new symbol
   file is specified (not just adding some symbols from another file, e.g. a
   shared library).

   We reinitialize buildsym, since gdb will be able to read stabs from an EVAX
   file at some point in the near future.  ??? */


struct bfd_hash_entry *
pd_hash_newfunc (entry, table, string)
     struct bfd_hash_entry *entry;
     struct bfd_hash_table *table;
     const char *string;
{
  pd_hash_entry *ret = (pd_hash_entry *)entry;

  if (ret == (pd_hash_entry *)NULL)
    ret = (pd_hash_entry *) bfd_hash_allocate
      (table, sizeof (pd_hash_entry));

  if (ret == (pd_hash_entry *)NULL)
    {
      bfd_set_error (bfd_error_no_memory);
      return (struct bfd_hash_entry *)NULL;
    }

  ret = (pd_hash_entry *) bfd_hash_newfunc
    ((struct bfd_hash_entry *)ret, table, string);

  ret->pd_address = (CORE_ADDR) 0;

  return (struct bfd_hash_entry *)ret;
}

struct bfd_hash_entry *
modnam_hash_newfunc (entry, table, string)
     struct bfd_hash_entry *entry;
     struct bfd_hash_table *table;
     const char *string;
{
  modnam_hash_entry *ret = (modnam_hash_entry *)entry;

  if (ret == (modnam_hash_entry *)NULL)
    ret = (modnam_hash_entry *) bfd_hash_allocate
      (table, sizeof (modnam_hash_entry));

  if (ret == (modnam_hash_entry *)NULL)
    {
      bfd_set_error (bfd_error_no_memory);
      return (struct bfd_hash_entry *)NULL;
    }

  ret = (modnam_hash_entry *) bfd_hash_newfunc
    ((struct bfd_hash_entry *)ret, table, string);

  return (struct bfd_hash_entry *)ret;
}

static void
evax_new_init (ignore)
     struct objfile *ignore;
{
  buildsym_new_init ();

  pd_hash_table = ((struct bfd_hash_table *)
		   xmalloc (sizeof (struct bfd_hash_table)));
  bfd_hash_table_init (pd_hash_table, pd_hash_newfunc);

  modnam_hash_table = ((struct bfd_hash_table *)
		       xmalloc (sizeof (struct bfd_hash_table)));
  bfd_hash_table_init (modnam_hash_table, modnam_hash_newfunc);
}


/* EVAX specific initialization routine for reading symbols.

   It is passed a pointer to a struct sym_fns which contains, among other
   things, the BFD for the file whose symbols are being read, and a slot for
   a pointer to "private data" which we can fill with goodies.

   For now at least, we have nothing in particular to do, so this function is
   just a stub. */

static void
evax_symfile_init (ignore)
     struct objfile *ignore;
{
}

static void
record_minimal_symbol (name, address, ms_type, objfile, en_address)
     char *name;
     CORE_ADDR address;
     enum minimal_symbol_type ms_type;
     struct objfile *objfile;
     CORE_ADDR en_address;
{

  name = obsavestring (name, strlen (name), &objfile -> symbol_obstack);
  prim_record_minimal_symbol (name, address, ms_type, objfile);
}


/*

LOCAL FUNCTION

	evax_symtab_read -- read the symbol table of an EVAX file

SYNOPSIS

	void evax_symtab_read (bfd *abfd, CORE_ADDR addr,
			      struct objfile *objfile)

DESCRIPTION

	Given an open bfd, a base address to relocate symbols to, and a
	flag that specifies whether or not this bfd is for an executable
	or not (may be shared library for example), add all the global
	function and data symbols to the minimal symbol table.
*/

static void
evax_symtab_read (abfd, section_offsets, objfile)
     bfd *abfd;
     struct section_offsets *section_offsets;
     struct objfile *objfile;
{
  long storage_needed;
  asymbol *sym;
  asymbol **symbol_table;
  long number_of_symbols;
  long i;
  struct cleanup *back_to;
  CORE_ADDR symaddr;
  enum minimal_symbol_type ms_type;
  asection *section;
  
  storage_needed = bfd_get_symtab_upper_bound (abfd);
  if (storage_needed < 0)
    error ("Can't read symbols from %s: %s",
	   TO_HOST_FILE_SPEC (bfd_get_filename (abfd)),
	   bfd_errmsg (bfd_get_error ()));
  if (storage_needed > 0)
    {
      symbol_table = (asymbol **) xmalloc (storage_needed);
      back_to = make_cleanup (free, symbol_table);
      number_of_symbols = bfd_canonicalize_symtab (abfd, symbol_table); 
      if (number_of_symbols < 0)
	error ("Can't read symbols from %s: %s",
	       TO_HOST_FILE_SPEC (bfd_get_filename (abfd)),
	       bfd_errmsg (bfd_get_error ()));

      for (i = 0; i < number_of_symbols; i++)
	{
	  sym = symbol_table[i];
	  if (sym -> flags & BSF_GLOBAL)
	    {
	      CORE_ADDR symval = 0;

	      symaddr = sym -> value;
	      /* Bfd symbols are section relative. */
	      if (sym->section)
		symaddr += sym -> section -> vma;
	      /* Relocate all non-absolute symbols by base address.  */
	      if (sym -> section != &bfd_abs_section)
		symaddr += ANOFFSET (section_offsets, sym->section->index);

	      /* For non-absolute symbols, use the type of the section
		 they are relative to, to intuit text/data.  BFD provides
		 no way of figuring this out for absolute symbols. */
	      if (sym -> section -> flags & SEC_CODE)
		ms_type = mst_text;
	      else if (sym -> section -> flags & SEC_DATA)
		ms_type = mst_data;
	      else
		ms_type = mst_unknown;

	      if (sym -> flags & BSF_FUNCTION)
		{

		  bfd_get_section_contents
		    (abfd, sym->section, &symval,
		     sym->value + 8, sizeof (CORE_ADDR));

		  symval += ANOFFSET (section_offsets, 0);

#if 0
		  printf ("PD %s value %x has entry %x\n",
			  sym->name,
			  sym->value,
			  symval);
#endif

		}
	      record_minimal_symbol
		((char *) sym -> name, symaddr, ms_type, objfile, symval);
	    }
	}
      do_cleanups (back_to);
    }
}

struct dwarfxref {
  char *sym;
  char *sec;
};

static struct dwarfxref dwarfxrefs[] =
{
  {"$dwarf2.debug_abbrev",   ".debug_abbrev"},
  {"$dwarf2.debug_aranges",  ".debug_aranges"},
  {"$dwarf2.debug_frame",    ".debug_frame"},
  {"$dwarf2.debug_info",     ".debug_info"},
  {"$dwarf2.debug_line",     ".debug_line"},
  {"$dwarf2.debug_loc",      ".debug_loc"},
  {"$dwarf2.debug_macinfo",  ".debug_macinfo"},
  {"$dwarf2.debug_pubnames", ".debug_pubnames"},
  {"$dwarf2.debug_str",      ".debug_str"},
  {"$dwarf2.debug_zzzzzz",   ""},
  {0,0}
};

struct dwarfxinfo
{
  CORE_ADDR start;
  asection *dwarfsect;
  CORE_ADDR offset;
};

static void
find_dwarf2_section (abfd, sectp, dip)
     bfd *abfd;
     asection *sectp;
     PTR dip;
{
  register struct dwarfxinfo *di;

  di = (struct dwarfxinfo *) dip;

  if (di->dwarfsect == 0
      && (di->start >= sectp->vma + di->offset)
      && (di->start <= (sectp->vma  + di->offset + sectp->_raw_size)))
    {
      di->dwarfsect = sectp;
    }
}

static void
make_dwarf2_subsections (abfd, offset)
     bfd *abfd;
     CORE_ADDR offset;
{
  struct minimal_symbol *sym, *lsym;
  int i;
  char *p, *q;
  char *dwarfprefix;
  struct dwarfxinfo dix;
  asection *sect;

  if ((abfd->flags & DYNAMIC) && !(abfd->flags & EXEC_P))
    {
      char buff[256];

      strcpy (buff, basename ((char *)abfd->filename));

      if (q = strrchr (buff, '.'))
	*q = 0;

      dwarfprefix = xmalloc (strlen (buff) + 1);
      strcpy (dwarfprefix, buff);

      for (q=dwarfprefix; *q; q++)
	*q = toupper (*q);

      sprintf (buff, "%s$DWARF2.DEBUG_ABBREV", dwarfprefix);

      sym = lookup_minimal_symbol (buff, NULL, NULL);
    }
  else
    {
      sym = lookup_minimal_symbol ("$dwarf2.debug_abbrev", NULL, NULL);
      dwarfprefix = "";
    }
  
  if (sym == 0)
    return;

  dix.start = SYMBOL_VALUE_ADDRESS(sym);
  dix.dwarfsect = 0;
  dix.offset = offset;

  bfd_map_over_sections (abfd, find_dwarf2_section, (PTR) &dix);

  if (!dix.dwarfsect)
    return;

  lsym = NULL;
  for (i=0, p=dwarfxrefs[i].sym; p; i++, p=dwarfxrefs[i].sym)
    {
      char buff [64];

      strcpy (buff, dwarfprefix);
      strcat (buff, p);
      if (sym = lookup_minimal_symbol (buff, NULL, NULL))
	{
	  if (lsym)
	    {
	      sect = bfd_make_section (abfd, dwarfxrefs[i-1].sec);

	      sect->_raw_size
		= SYMBOL_VALUE_ADDRESS(sym) - SYMBOL_VALUE_ADDRESS(lsym);
	      sect->_cooked_size = sect->_raw_size;
	      sect->vma = SYMBOL_VALUE_ADDRESS(lsym) - dix.offset;
	      sect->filepos
		= dix.dwarfsect->filepos + (sect->vma - dix.dwarfsect->vma);

	    }
	  lsym = sym;
	}
  }
}

/* Scan and build partial symbols for a symbol file.
   We have been initialized by a call to evax_symfile_init, which 
   currently does nothing.

   SECTION_OFFSETS is a set of offsets to apply to relocate the symbols
   in each section.  We simplify it down to a single offset for all
   symbols.  FIXME.

   MAINLINE is true if we are reading the main symbol
   table (as opposed to a shared lib or dynamically loaded file).

   This function only does the minimum work necessary for letting the
   user "name" things symbolically; it does not read the entire symtab.
   Instead, it reads the external and static symbols and puts them in partial
   symbol tables.  When more extensive information is requested of a
   file, the corresponding partial symbol table is mutated into a full
   fledged symbol table by going back and reading the symbols
   for real.

   Note that EVAX files have two sets of information that is potentially
   useful for building gdb's minimal symbol table.  The first is a list
   of the publically exported symbols, and is currently used to build
   bfd's canonical symbol table.  The second is an optional native debugging
   format which contains additional symbols (and possibly duplicates of
   the publically exported symbols).  The optional native debugging format
   is not currently used. */

static void
evax_symfile_read (objfile, section_offsets, mainline)
     struct objfile *objfile;
     struct section_offsets *section_offsets;
     int mainline;
{
  bfd *abfd = objfile -> obfd;
  struct evaxinfo ei;
  struct cleanup *back_to;
  CORE_ADDR offset;
  struct symbol *mainsym;
  struct symbol *entrysym;
  int evax_mainline = mainline;
  asection *section;

  init_minimal_symbol_collection ();
  back_to = make_cleanup (discard_minimal_symbols, 0);

  memset ((char *) &ei, 0, sizeof (ei));

  /* FIXME, should take a section_offsets param, not just an offset.  */

  offset = ANOFFSET (section_offsets, 0);

  /* Process the EVAX export records, which become the bfd's canonical symbol
     table. */

#if 0 /* moved to bfd */
  if (!evax_slurp_symdbg(abfd))
    error ("Error reading symbol table: %s", bfd_errmsg (bfd_get_error ()));
#endif

  evax_symtab_read (abfd, section_offsets, objfile);

  /* Install any minimal symbols that have been collected as the current
     minimal symbols for this objfile. */

  install_minimal_symbols (objfile);

  make_dwarf2_subsections (abfd, offset);

  if (dwarf2_has_info (abfd))
    {
      /* DWARF 2 sections */
      dwarf2_build_psymtabs (objfile, section_offsets, mainline);
      evax_mainline = 0;
    }

  bfd_map_over_sections (abfd, evax_read_debug_sections, (PTR) &ei);
  if (ei.dmtcontents)
    {
      register struct partial_symtab *pst;
  
      ALL_OBJFILE_PSYMTABS (objfile, pst)
	{
	  char *p;
	  int i;
	  char modname [256];
	  modnam_hash_entry *modnam_entry;

	  if (strchr (pst->filename, ']') || strchr (pst->filename, ':'))
	    strcpy (modname, basename (decc$translate_vms (pst->filename)));
	  else
	    strcpy (modname, basename (pst->filename));

	  p = strrchr (modname, '.');
	  *p = 0;
	  for (i=0; i<strlen(modname); i++)
	    modname[i] = toupper (modname[i]);

	  modnam_entry = (modnam_hash_entry *) bfd_hash_lookup
	    (modnam_hash_table, modname, true, true);

	}

      evax_build_psymtabs (objfile,
			   section_offsets, evax_mainline,
			   ei.dstfilepos,
			   ei.dmtcontents, ei.dmtsize);
    }

  entrysym = lookup_symbol ("__MAIN", NULL, VAR_NAMESPACE, NULL, NULL);

  if (entrysym
      && SYMBOL_CLASS(entrysym) == LOC_BLOCK)
    {
      objfile->ei.entry_func_lowpc
	= BLOCK_START (SYMBOL_BLOCK_VALUE (entrysym));
      objfile->ei.entry_func_highpc
	= BLOCK_END (SYMBOL_BLOCK_VALUE (entrysym));
    }

  mainsym = lookup_symbol ("main", NULL, VAR_NAMESPACE, NULL, NULL);

  if (!mainsym)
    mainsym = lookup_symbol ("MAIN", NULL, VAR_NAMESPACE, NULL, NULL);

  if (mainsym
      && SYMBOL_CLASS(mainsym) == LOC_BLOCK)
    {
      objfile->ei.main_func_lowpc = BLOCK_START (SYMBOL_BLOCK_VALUE (mainsym));
      objfile->ei.main_func_highpc = BLOCK_END (SYMBOL_BLOCK_VALUE (mainsym));
    }

  /* FIXME:  We could locate and read the optional native debugging format
     here and add the symbols to the minimal symbol table. */

  do_cleanups (back_to);
}


/* Perform any local cleanups required when we are done with a particular
   objfile.  I.E, we are in the process of discarding all symbol information
   for an objfile, freeing up all memory held for it, and unlinking the
   objfile struct from the global list of known objfiles. */

static void
evax_symfile_finish (objfile)
     struct objfile *objfile;
{
  if (objfile -> sym_private != NULL)
    {
      mfree (objfile -> md, objfile -> sym_private);
    }
}

/* EVAX specific parsing routine for section offsets.
   FIXME:  This may or may not be necessary.  All the symbol readers seem
   to have similar code.  See if it can be generalized and moved elsewhere. */

#undef SECT_OFF_MAX
#define SECT_OFF_MAX 10

static
struct section_offsets *
evax_symfile_offsets (objfile, addr)
     struct objfile *objfile;
     CORE_ADDR addr;
{
  struct section_offsets *section_offsets;
  int i;

  objfile->num_sections = SECT_OFF_MAX;

  if (addr)
    {
      asection *lowest_sect = 0;
      CORE_ADDR addr1 = addr;

      bfd_map_over_sections (objfile->obfd, find_lowest_section,
			     (PTR) &lowest_sect);
      if (lowest_sect)
	addr1 += bfd_section_vma (objfile->obfd, lowest_sect);
      section_offsets = evax_solib_offsets (objfile, addr1, SECT_OFF_MAX);
    }
  else
    {
      section_offsets = (struct section_offsets *)
	obstack_alloc (&objfile -> psymbol_obstack,
		       sizeof (struct section_offsets) +
		       sizeof (section_offsets->offsets) * (SECT_OFF_MAX-1));
      for (i = 0; i < SECT_OFF_MAX; i++)
	{
	  ANOFFSET (section_offsets, i) = addr;
	}
    }
  
  return (section_offsets);
}

CORE_ADDR
convert_to_descriptor (address)
     CORE_ADDR address;
{
  pd_hash_entry *pd_entry;
  char buff[20];

  sprintf (buff, "%llx", address);
  pd_entry = (pd_hash_entry *) bfd_hash_lookup
    (pd_hash_table, buff, false, false);

  if (pd_entry)
    return pd_entry->pd_address;
  else
    return 0;
}

void
evax_add_pd_address (addr, pd_addr)
     CORE_ADDR addr;
     CORE_ADDR pd_addr;
{
  char buff [20];
  pd_hash_entry *pd_entry;

  sprintf (buff, "%llx", addr);
  pd_entry = (pd_hash_entry *) bfd_hash_lookup
    (pd_hash_table, buff, true, true);
  if (pd_entry->pd_address == 0)
    pd_entry->pd_address = pd_addr;
  else if (pd_entry->pd_address != pd_addr)
    error ("duplicate pd hash entry for '%s'", buff);
  
}

int
module_loaded (modname)
     char *modname;
{
  modnam_hash_entry *modnam_entry;

  modnam_entry = (modnam_hash_entry *) bfd_hash_lookup
    (modnam_hash_table, modname, false, false);

  return (int) modnam_entry;
}


/* Register that we are able to handle EVAX file format. */

static struct sym_fns evax_sym_fns =
{
  bfd_target_evax_flavour,
  evax_new_init,	/* sym_new_init: init anything gbl to entire symtab */
  evax_symfile_init,	/* sym_init: read initial info, setup for sym_read() */
  evax_symfile_read,	/* sym_read: read a symbol file into symtab */
  evax_symfile_finish,	/* sym_finish: finished with file, cleanup */
  evax_symfile_offsets,	/* sym_offsets:  Translate ext. to int. relocation */
  NULL			/* next: pointer to next struct sym_fns */
};

void
_initialize_evaxread ()
{
  add_symtab_fns (&evax_sym_fns);
}
