// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes:
// 
// Div
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_DIFFOPS_DIV_UR_H
#define POOMA_NEWFIELD_DIFFOPS_DIV_UR_H

//-----------------------------------------------------------------------------
// Overview: 
//
// Classes:
//
// Div	:  Divergence operator on Fields, using 2nd-order centered differences
//         These are used by the div() template function. See Div.h for
//         details, and the div() function template definition.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Tiny/Vector.h"
#include "NewField/DiffOps/FieldStencil.h"
#include "NewField/FieldInitializers.h"
#include "NewField/FieldEngine/FieldEngine.UR.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
// 
// Classes:
//
// Div:
// 
// Partial specializations of the generic Div (divergence) functor. See Div.h
// for general comments. These are for UniformRectilinearMesh-based 
// DiscreteGeometry.
// 
// Div is a functor class serving as the "Functor" template parameter for
// FieldStencil<Functor>. Its operator() functions implement 2nd
// centered differences on an input Field and return output elements of an
// output Field.  The types of the input and output Field differ in two ways:
//	1) The input centering is (possibly) different than the output 
//	   centering.
//	2) The input element type is Vector<Dim,T2> (or Tensor<Dim,T2>) and
//         the output type is a scalar type T2 (or Vector<Dim,T2>).
// Partial specializations implement various combinations of input and output
// centerings, for specific coordinate systems.
// 
// Exported typedefs:
//
// OutputCentering_t - Centering parameter of the output DiscreteGeometry class
// InputField_t      - Type of the input Field, which is restricted to 
//                     something from the
//                     DiscreteGeometry<Centering, UniformRectilinearMesh<Dim...> >
//                     geometry and a vector or tensor type..
// OutputElement_t   - Type of the elements in the output ConstField; 
//                     restricted to a scalar type (vector input) or vector
//                     (tensor input)
// 
// Accessors:
//
// lowerExtent(int d) - Returns the stencil width in direction d, at the "low"
//                      end of the (logically) rectilinear mesh. This is the
//                      maximim positive integer offset from the element 
//                      indexed by integer i in the input Field's index space
//                      along dimension d used in outputting the element
//                      indexed by integer i in the output Field's index space
//                      along dimension d
// upperExtent(int d) - Same as lowerExtent(), but for the "high" end of the 
//                      mesh. That is, the maximum (magnitude) *negative*
//                      offset from i in direction d.
// 
// Other methods:
// 
// operator()(...) - The actual implementation for the stencil. This acts on a 
//                   set of scalar-indexed values in the input Field's index
//                   space making up the stencil, as offset from the fixed
//                   index point specified by the function's input arguments
//                   (list of scalar index values).  The stencil must be
//                   written so that the same fixed index point specified by
//                   the input arguments where the values are to be assigned in
//                   the index space of the output Field. This means, for
//                   example, that if the operator is going from one centering
//                   to a different output centering, the index bookkeeping
//                   must be done correctly by this operator()() function
//                   implementation.
//-----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Partial specializations of DivVertToCell
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Divergence Vector/Vert -> Scalar/Cell
// ----------------------------------------------------------------------------

template<class T2, class Mesh>
class DivVertToCell;

template<class T2, int Dim, class TM, class CS>
class DivVertToCell<Vector<Dim, T2>, UniformRectilinear<Dim, TM, CS> >
{
public:

  typedef T2   OutputElement_t;

  Cell outputCentering() const
  {
    return Cell();
  }

  Loc<Dim> inputOffsets() const
  {
    // input centering is Vert:
    return Loc<Dim>(1);
  }

  // 
  // Constructors.
  // 

  // default version is required by default stencil engine constructor.

  DivVertToCell()
  {
    int d;
    for (d = 0; d < Dim; ++d)
    {
      fact_m(d) = 1.0;
    }
  }

  template<class FE>
  DivVertToCell(const FE &fieldEngine)
  {
    int d;
    for (d = 0; d < Dim; ++d)
    {
      fact_m(d) = 1 / fieldEngine.spacings()(d);
    }
  }

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 0; }
  int upperExtent(int d) const { return 1; }
      
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    return
      fact_m(0) *
      (f(i1 + 1)(0) - f(i1)(0));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    return 0.5 *
      (fact_m(0) *
       (f(i1 + 1, i2    )(0) - f(i1    , i2    )(0) +
	f(i1 + 1, i2 + 1)(0) - f(i1    , i2 + 1)(0)
	) +
       fact_m(1) *
       (f(i1    , i2 + 1)(1) - f(i1    , i2    )(1) +
	f(i1 + 1, i2 + 1)(1) - f(i1 + 1, i2    )(1)
	)
       );
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    return 0.25 *
      (fact_m(0) *
       (f(i1 + 1, i2    , i3    )(0) - f(i1    , i2    , i3    )(0) +
	f(i1 + 1, i2 + 1, i3    )(0) - f(i1    , i2 + 1, i3    )(0) +
	f(i1 + 1, i2    , i3 + 1)(0) - f(i1    , i2    , i3 + 1)(0) +
	f(i1 + 1, i2 + 1, i3 + 1)(0) - f(i1    , i2 + 1, i3 + 1)(0)
	) +
       fact_m(1) *
       (f(i1    , i2 + 1, i3    )(1) - f(i1    , i2    , i3    )(1) +
	f(i1 + 1, i2 + 1, i3    )(1) - f(i1 + 1, i2    , i3    )(1) +
	f(i1    , i2 + 1, i3 + 1)(1) - f(i1    , i2    , i3 + 1)(1) +
	f(i1 + 1, i2 + 1, i3 + 1)(1) - f(i1 + 1, i2    , i3 + 1)(1)
	) +
       fact_m(2) *
       (f(i1    , i2    , i3 + 1)(2) - f(i1    , i2    , i3    )(2) +
	f(i1 + 1, i2    , i3 + 1)(2) - f(i1 + 1, i2    , i3    )(2) +
	f(i1    , i2 + 1, i3 + 1)(2) - f(i1    , i2 + 1, i3    )(2) +
	f(i1 + 1, i2 + 1, i3 + 1)(2) - f(i1 + 1, i2 + 1, i3    )(2)
	)
       );
  }

private:

  Vector<Dim, TM> fact_m;
};

template<class T2, class Mesh, class OC>
class DivSameToSame;

template<class T2, int Dim, class TM, class CS, class OC>
class DivSameToSame<Vector<Dim, T2>, UniformRectilinear<Dim, TM, CS>, OC>
{
public:

  typedef T2   OutputElement_t;

  OC outputCentering() const
  {
    return OC();
  }

  Loc<Dim> inputOffsets() const
  {
    return inputOffsets_m;
  }

  // 
  // Constructors.
  // 

  // default version is required by default stencil engine constructor.

  DivSameToSame()
  {
    int d;
    for (d = 0; d < Dim; ++d)
    {
      fact_m(d) = 0.5;
    }
    inputOffsets_m = Loc<Dim>(0);
  }

  template<class FE>
  DivSameToSame(const FE &fieldEngine)
  {
    int d;
    for (d = 0; d < Dim; ++d)
    {
      fact_m(d) = 0.5 / fieldEngine.spacings()(d);
    }
    inputOffsets_m = fieldEngine.offsets();
  }

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 1; }
  int upperExtent(int d) const { return 1; }
      
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    return
      fact_m(0) *
      (f(i1 + 1)(0) - f(i1 - 1)(0));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    return
      (fact_m(0) *
       (f(i1 + 1, i2    )(0) - f(i1 - 1, i2    )(0)) +
       fact_m(1) *
       (f(i1    , i2 + 1)(1) - f(i1    , i2 - 1)(1))
       );
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    return
      (fact_m(0) *
       (f(i1 + 1, i2    , i3    )(0) - f(i1 - 1, i2    , i3    )(0)) +
       fact_m(1) *
       (f(i1    , i2 + 1, i3    )(1) - f(i1    , i2 - 1, i3    )(1)) +
       fact_m(2) *
       (f(i1    , i2    , i3 + 1)(2) - f(i1    , i2    , i3 - 1)(2))
       );
  }

private:

  Vector<Dim, TM> fact_m;
  Loc<Dim> inputOffsets_m;
};

#endif     // POOMA_NEWFIELD_DIFFOPS_DIV_UR_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Div.UR.h,v $   $Author: sa_smith $
// $Revision: 1.3 $   $Date: 2000/07/26 22:27:27 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
