// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   DiscreteGeometry<Centering, RectilinearMesh<...> >
//-----------------------------------------------------------------------------

#ifndef POOMA_GEOMETRY_DISCRETEGEOMETRY_RM_H
#define POOMA_GEOMETRY_DISCRETEGEOMETRY_RM_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// DiscreteGeometry<Centering, RectilinearMesh<...> >:
//
// DiscreteGeometry specialization for canned centerings and 
// RectilinearMesh.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Geometry/RectilinearGeometryBase.h"
#include "Meshes/RectilinearMesh.h"

//-----------------------------------------------------------------------------
// Full Description:
// 
// DiscreteGeometry provides geometry services to discrete field by tying
// together a centering and a mesh, both of which are specified in the template
// parameters.
//
// The versions in this file tie together the common centerings:
//   o Cell
//   o Vert
//   o RectilinearCentering<Dim, RectilinearCenteringTag, Componentwise, 
//                          TensorRank, NComponents>
// and RectilinearMesh.
//
// See DiscreteGeometry.h for the constraints on classes that are supposed
// to model the DiscreteGeometry concept.
//-----------------------------------------------------------------------------

///////////////////////////////////////////////////////////////////////////////
//
// Vert-centered version.
//
///////////////////////////////////////////////////////////////////////////////

template<int Dim, class CoordinateSystem, class T>
class DiscreteGeometry<Vert, RectilinearMesh<Dim, CoordinateSystem, T> > :
public RectilinearGeometryBase<
  Vert, RectilinearMesh<Dim, CoordinateSystem, T> >
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // The base class.
  
  typedef RectilinearGeometryBase<
    Vert, RectilinearMesh<Dim, CoordinateSystem, T> > Base_t;
    
  // The centering type.
  
  typedef typename Base_t::Centering_t Centering_t;
  
  // The mesh type.
  
  typedef typename Base_t::Mesh_t  Mesh_t;
  
  // This class.
  
  typedef DiscreteGeometry<Centering_t, Mesh_t> This_t;
  
  // The coordinate point type.
  
  typedef typename Base_t::PointType_t PointType_t;
  
  // The coordinate system.
  
  typedef typename Base_t::CoordinateSystem_t CoordinateSystem_t;
  
  // The domain type.
  
  typedef typename Base_t::Domain_t Domain_t;
  
  // The guard layers type.
  
  typedef typename Base_t::GuardLayers_t GuardLayers_t;
  
  // Dimensions.
  
  enum { dimensions = Base_t::dimensions };
  enum { coordinateDimensions = Base_t::coordinateDimensions };

  // Are we component-centered?
  
  static const bool componentCentered = Base_t::componentCentered;

  // Array types
  
  typedef typename Base_t::PositionArray_t PositionArray_t;

  // Bounding box types

  typedef typename Base_t::BoundingRegion_t BoundingRegion_t;

  //---------------------------------------------------------------------------
  // Default constructor. Must call initialize function to be a valid object.
  
  DiscreteGeometry() { }
  
  //---------------------------------------------------------------------------
  // Constructor using a mesh and (optionally) a GuardLayers object.
  
  DiscreteGeometry(const Mesh_t &m, const GuardLayers_t &gl = GuardLayers_t())
    : Base_t(m, gl)
  { }

  //---------------------------------------------------------------------------
  // Copy constructor.
  
  DiscreteGeometry(const This_t &model)
   : Base_t(model.mesh(), model.guardLayers()) 
 { }

  //---------------------------------------------------------------------------
  // Copy assignment operator (should not be called).
  
  This_t &operator=(const This_t &)
  {
    PAssert(false);
    return *this;
  }

  //---------------------------------------------------------------------------
  // Empty destructor is fine for us.
  
  ~DiscreteGeometry() { }

  //---------------------------------------------------------------------------
  // Initialize functions.

  void initialize(const Mesh_t &m, const GuardLayers_t &gl = GuardLayers_t())
  {
    Base_t::initialize(m, gl);
  }
  
  void initialize(const This_t &model)
  {
    Base_t::initialize(model.mesh(), model.guardLayers());
  }
};


///////////////////////////////////////////////////////////////////////////////
//
// Cell-centered version.
//
///////////////////////////////////////////////////////////////////////////////

template<int Dim, class CoordinateSystem, class T>
class DiscreteGeometry<Cell, RectilinearMesh<Dim, CoordinateSystem, T> > :
public RectilinearGeometryBase<
  Cell, RectilinearMesh<Dim, CoordinateSystem, T> >
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // The base class.
  
  typedef RectilinearGeometryBase<
    Cell, RectilinearMesh<Dim, CoordinateSystem, T> > Base_t;
    
  // The centering type.
  
  typedef typename Base_t::Centering_t Centering_t;
  
  // The mesh type.
  
  typedef typename Base_t::Mesh_t  Mesh_t;
  
  // This class.
  
  typedef DiscreteGeometry<Centering_t, Mesh_t> This_t;
  
  // The coordinate point type.
  
  typedef typename Base_t::PointType_t PointType_t;
  
  // The coordinate system.
  
  typedef typename Base_t::CoordinateSystem_t CoordinateSystem_t;
  
  // The domain type.
  
  typedef typename Base_t::Domain_t Domain_t;
  
  // The guard layers type.
  
  typedef typename Base_t::GuardLayers_t GuardLayers_t;
  
  // Dimensions.
  
  enum { dimensions = Base_t::dimensions };
  enum { coordinateDimensions = Base_t::coordinateDimensions };

  // Are we component-centered?
  
  static const bool componentCentered = Base_t::componentCentered;

  // Array types
  
  typedef typename Base_t::PositionArray_t PositionArray_t;

  // Bounding box types

  typedef typename Base_t::BoundingRegion_t BoundingRegion_t;

  //---------------------------------------------------------------------------
  // Default constructor. Must call initialize function to be a valid object.
  
  DiscreteGeometry() { }
  
  //---------------------------------------------------------------------------
  // Constructor using a mesh and (optionally) a GuardLayers object.
  
  DiscreteGeometry(const Mesh_t &m, const GuardLayers_t &gl = GuardLayers_t())
    : Base_t(m, gl)
  { }

  //---------------------------------------------------------------------------
  // Copy constructor.
  
  DiscreteGeometry(const This_t &model)
   : Base_t(model.mesh(), model.guardLayers()) 
 { }

  //---------------------------------------------------------------------------
  // Copy assignment operator (should not be called).
  
  This_t &operator=(const This_t &)
  {
    PAssert(false);
    return *this;
  }

  //---------------------------------------------------------------------------
  // Empty destructor is fine for us.
  
  ~DiscreteGeometry() { }

  //---------------------------------------------------------------------------
  // Initialize functions.

  void initialize(const Mesh_t &m, const GuardLayers_t &gl = GuardLayers_t())
  {
    Base_t::initialize(m, gl);
  }
  
  void initialize(const This_t &model)
  {
    Base_t::initialize(model.mesh(), model.guardLayers());
  }
};


///////////////////////////////////////////////////////////////////////////////
//
// RectilinearCentering<Dim,RectilinearCenteringTag,Componentwise,TensorRank,
// NComponents>-centered version.
// 
// For non-componentwise centerings, simpler.
// For componentwise centerings, partial specializations for TensorRank={1,2}
//
///////////////////////////////////////////////////////////////////////////////

// For non-componentwise centerings, only one partial specialization. Element
// type of position array is Position_t (a position vector type).

template<int Dim, class T, class CoordinateSystem, 
  class RectilinearCenteringTag, int TensorRank, int NComponents>
class DiscreteGeometry<RectilinearCentering<Dim, RectilinearCenteringTag, 
  false, TensorRank, NComponents>, 
  RectilinearMesh<Dim, CoordinateSystem, T> > : public
  RectilinearGeometryBase<RectilinearCentering<Dim, RectilinearCenteringTag, 
    false, TensorRank, NComponents>, 
    RectilinearMesh<Dim, CoordinateSystem, T> >
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // The base class.
  
  typedef RectilinearGeometryBase<
    RectilinearCentering<Dim, RectilinearCenteringTag, false, TensorRank, 
    NComponents>, 
    RectilinearMesh<Dim, CoordinateSystem, T> > Base_t;
    
  // The centering type.
  
  typedef typename Base_t::Centering_t Centering_t;
  
  // The mesh type.
  
  typedef typename Base_t::Mesh_t  Mesh_t;
  
  // This class.
  
  typedef DiscreteGeometry<Centering_t, Mesh_t> This_t;
  
  // The coordinate point type.
  
  typedef typename Base_t::PointType_t PointType_t;
  
  // The coordinate system.
  
  typedef typename Base_t::CoordinateSystem_t CoordinateSystem_t;
  
  // The domain type.
  
  typedef typename Base_t::Domain_t Domain_t;
  
  // The guard layers type.
  
  typedef typename Base_t::GuardLayers_t GuardLayers_t;
  
  // Dimensions.
  
  enum { dimensions = Base_t::dimensions };
  enum { coordinateDimensions = Base_t::coordinateDimensions };

  // Are we component-centered?
  
  static const bool componentCentered = Base_t::componentCentered;

  // Array types
  
  typedef typename Base_t::PositionArray_t PositionArray_t;

  // Bounding box types

  typedef typename Base_t::BoundingRegion_t BoundingRegion_t;

  //---------------------------------------------------------------------------
  // Default constructor. Must call initialize function to be a valid object.
  
  DiscreteGeometry() { }
  
  //---------------------------------------------------------------------------
  // Constructor using a mesh and (optionally) a GuardLayers object.
  
  DiscreteGeometry(const Mesh_t &m, const GuardLayers_t &gl = GuardLayers_t())
    : Base_t(m, gl)
  { }

  //---------------------------------------------------------------------------
  // Copy constructor.
  
  DiscreteGeometry(const This_t &model)
   : Base_t(model.mesh(), model.guardLayers()) 
 { }

  //---------------------------------------------------------------------------
  // Copy assignment operator (should not be called).
  
  This_t &operator=(const This_t &)
  {
    PAssert(false);
    return *this;
  }

  //---------------------------------------------------------------------------
  // Empty destructor is fine for us.
  
  ~DiscreteGeometry() { }

  //---------------------------------------------------------------------------
  // Initialize functions.

  void initialize(const Mesh_t &m, const GuardLayers_t &gl = GuardLayers_t())
  {
    Base_t::initialize(m, gl);
  }
  
  void initialize(const This_t &model)
  {
    Base_t::initialize(model.mesh(), model.guardLayers());
  }
};

// For componentwise centerings, partially specialize on tensor rank. Since the
// only one that exists so far is VectorFace, only need to put in tensor rank
// 1, really; but, since I already did TensorRank=2, leave it here for now.

// TensorRank = 1
// Elementype of position array is Vector<Dim,Position_t> (one position vector
// for each component of the rank-1 Field-element type).

template<int Dim, class CoordinateSystem, class T,
  class RectilinearCenteringTag, int NComponents>
class DiscreteGeometry<RectilinearCentering<Dim, RectilinearCenteringTag, 
  true, 1, NComponents>, 
  RectilinearMesh<Dim, CoordinateSystem, T> > : public
  RectilinearGeometryBase<RectilinearCentering<Dim, RectilinearCenteringTag, 
    true, 1, NComponents>, 
    RectilinearMesh<Dim, CoordinateSystem, T> >
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // The base class.
  
  typedef RectilinearGeometryBase<
    RectilinearCentering<Dim, RectilinearCenteringTag, true, 1, NComponents>, 
    RectilinearMesh<Dim, CoordinateSystem, T> > Base_t;
    
  // The centering type.
  
  typedef typename Base_t::Centering_t Centering_t;
  
  // The mesh type.
  
  typedef typename Base_t::Mesh_t  Mesh_t;
  
  // This class.
  
  typedef DiscreteGeometry<Centering_t, Mesh_t> This_t;
  
  // The coordinate point type.
  
  typedef typename Base_t::PointType_t PointType_t;
  
  // The coordinate system.
  
  typedef typename Base_t::CoordinateSystem_t CoordinateSystem_t;
  
  // The domain type.
  
  typedef typename Base_t::Domain_t Domain_t;
  
  // The guard layers type.
  
  typedef typename Base_t::GuardLayers_t GuardLayers_t;
  
  // Dimensions.
  
  enum { dimensions = Base_t::dimensions };
  enum { coordinateDimensions = Base_t::coordinateDimensions };

  // Are we component-centered?
  
  static const bool componentCentered = Base_t::componentCentered;

  // Array types
  
  typedef typename Base_t::PositionArray_t PositionArray_t;

  // Bounding box types

  typedef typename Base_t::BoundingRegion_t BoundingRegion_t;

  //---------------------------------------------------------------------------
  // Default constructor. Must call initialize function to be a valid object.
  
  DiscreteGeometry() { }
  
  //---------------------------------------------------------------------------
  // Constructor using a mesh and (optionally) a GuardLayers object.
  
  DiscreteGeometry(const Mesh_t &m, const GuardLayers_t &gl = GuardLayers_t())
    : Base_t(m, gl)
  { }

  //---------------------------------------------------------------------------
  // Copy constructor.
  
  DiscreteGeometry(const This_t &model)
   : Base_t(model.mesh(), model.guardLayers()) 
 { }

  //---------------------------------------------------------------------------
  // Copy assignment operator (should not be called).
  
  This_t &operator=(const This_t &)
  {
    PAssert(false);
    return *this;
  }

  //---------------------------------------------------------------------------
  // Empty destructor is fine for us.
  
  ~DiscreteGeometry() { }

  //---------------------------------------------------------------------------
  // Initialize functions.

  void initialize(const Mesh_t &m, const GuardLayers_t &gl = GuardLayers_t())
  {
    Base_t::initialize(m, gl);
  }
  
  void initialize(const This_t &model)
  {
    Base_t::initialize(model.mesh(), model.guardLayers());
  }
};

// TensorRank = 2
// Elementype of position array is Tensor<Dim,Position_t> (one position
// vector for each component of the rank-1 Field-element type).

template<int Dim, class CoordinateSystem, class T,
  class RectilinearCenteringTag, int NComponents>
class DiscreteGeometry<RectilinearCentering<Dim, RectilinearCenteringTag, 
  true, 2, NComponents>, 
  RectilinearMesh<Dim, CoordinateSystem, T> > : public
  RectilinearGeometryBase<RectilinearCentering<Dim, RectilinearCenteringTag, 
    true, 2, NComponents>, 
    RectilinearMesh<Dim, CoordinateSystem, T> >
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // The base class.
  
  typedef RectilinearGeometryBase<
    RectilinearCentering<Dim, RectilinearCenteringTag, true, 2, NComponents>, 
    RectilinearMesh<Dim, CoordinateSystem, T> > Base_t;
    
  // The centering type.
  
  typedef typename Base_t::Centering_t Centering_t;
  
  // The mesh type.
  
  typedef typename Base_t::Mesh_t  Mesh_t;
  
  // This class.
  
  typedef DiscreteGeometry<Centering_t, Mesh_t> This_t;
  
  // The coordinate point type.
  
  typedef typename Base_t::PointType_t PointType_t;
  
  // The coordinate system.
  
  typedef typename Base_t::CoordinateSystem_t CoordinateSystem_t;
  
  // The domain type.
  
  typedef typename Base_t::Domain_t Domain_t;
  
  // The guard layers type.
  
  typedef typename Base_t::GuardLayers_t GuardLayers_t;
  
  // Dimensions.
  
  enum { dimensions = Base_t::dimensions };
  enum { coordinateDimensions = Base_t::coordinateDimensions };

  // Are we component-centered?
  
  static const bool componentCentered = Base_t::componentCentered;

  // Array types
  
  typedef typename Base_t::PositionArray_t PositionArray_t;

  // Bounding box types

  typedef typename Base_t::BoundingRegion_t BoundingRegion_t;

  //---------------------------------------------------------------------------
  // Default constructor. Must call initialize function to be a valid object.
  
  DiscreteGeometry() { }
  
  //---------------------------------------------------------------------------
  // Constructor using a mesh and (optionally) a GuardLayers object.
  
  DiscreteGeometry(const Mesh_t &m, const GuardLayers_t &gl = GuardLayers_t())
    : Base_t(m, gl)
  { }

  //---------------------------------------------------------------------------
  // Copy constructor.
  
  DiscreteGeometry(const This_t &model)
   : Base_t(model.mesh(), model.guardLayers()) 
 { }

  //---------------------------------------------------------------------------
  // Copy assignment operator (should not be called).
  
  This_t &operator=(const This_t &)
  {
    PAssert(false);
    return *this;
  }

  //---------------------------------------------------------------------------
  // Empty destructor is fine for us.
  
  ~DiscreteGeometry() { }

  //---------------------------------------------------------------------------
  // Initialize functions.

  void initialize(const Mesh_t &m, const GuardLayers_t &gl = GuardLayers_t())
  {
    Base_t::initialize(m, gl);
  }
  
  void initialize(const This_t &model)
  {
    Base_t::initialize(model.mesh(), model.guardLayers());
  }
};

#endif // POOMA_GEOMETRY_DISCRETEGEOMETRY_RM_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: DiscreteGeometry.RM.h,v $   $Author: swhaney $
// $Revision: 1.6 $   $Date: 2000/03/07 13:17:22 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
