/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* librvngabw
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

/* "This product is not manufactured, approved, or supported by
 * Corel Corporation or Corel Corporation Limited."
 */
#ifndef _PAGESPAN_HXX_
#define _PAGESPAN_HXX_

#include <librevenge/librevenge.h>
#include <map>
#include <set>
#include <vector>

#include "FilterInternal.hxx"

#include "Style.hxx"

class DocumentElement;
class ABWDocumentHandler;

namespace librvngabw
{
class PageSpanManager;

class PageSpan
{
public:
	enum StorageType
	{
		C_Header=0, C_HeaderFirst, C_HeaderLeft, C_HeaderLast,
		C_Footer, C_FooterFirst, C_FooterLeft, C_FooterLast,
		C_MainContent,
		C_NumStorageTypes // keep this one last
	};
	//! class used to define a storage
	struct Storage
	{
		//! constructor
		Storage(int id, int xId) : m_content(), m_id(id), m_xId(xId), m_sectionPropList()
		{
		}
		//! write the data ( knowing the type )
		void appendContent(librevenge::RVNGString const &type, librevenge::RVNGPropertyList const &pageSpanData,
		                   librevenge::RVNGPropertyList const &extra, DocumentElementVector &storage) const;
		//! the storage content
		DocumentElementVector m_content;
		//! the storage id
		int m_id;
		//! the storage xId
		int m_xId;
		//! the section list
		librevenge::RVNGPropertyList m_sectionPropList;
	};

	PageSpan(PageSpanManager &manager,
	         librevenge::RVNGPropertyList const &xPropList,
	         bool isFirstPage);
	virtual ~PageSpan();
	//! append all auxiliary contents
	void appendAuxiliaryContents(DocumentElementVector &content);
	//! append the current main content ( and clean it )
	void appendMainContent(DocumentElementVector &content);
	//! returns true if we have sent at least one section
	bool isSomeMainSectionSent() const
	{
		return m_hasSentMainSection;
	}
	//! append the page size data
	void writePageSize(DocumentElementVector &content) const;
	//! update the section property of a current data
	void updateSectionPropList(librevenge::RVNGPropertyList const &sectionPropList);
	//! open a storage
	DocumentElementVector *openStorage(StorageType type, int &XId, bool createIfNeeded=true);
	//! close a storage
	void closeStorage()
	{
		m_currentStorage=nullptr;
	}
	//! reset a content storage
	void resetContent(StorageType type);
	//! reset all content
	void resetAllContents();

protected:
	//! returns the type corresponding to a storage string
	static librevenge::RVNGString getTypeName(StorageType type);
private:
	PageSpan(const PageSpan &);
	PageSpan &operator=(const PageSpan &);
	//! the page manager
	PageSpanManager &m_pageSpanManager;
	//! the page properties
	librevenge::RVNGPropertyList m_pagePropList;
	//! the storage list
	std::vector<std::shared_ptr<Storage> > m_storageList;
	//! the current storage list
	Storage *m_currentStorage;
	//! true if this is the first page
	bool m_isFirstPage;
	//! a flag to know if we have already sent at least one main section
	bool m_hasSentMainSection;
	//! list of header/footer that need to be created
	std::set<int> m_headerFooterToSent;
};

//! class used to store the list of created page
class PageSpanManager
{
public:
	//! constructor
	PageSpanManager() : m_pageSpanList(), m_sectionId(0)
	{
	}
	//! destructor
	~PageSpanManager()
	{
		clean();
	}
	//! clean data
	void clean();
	//! create a new page and set it to current. Returns a pointer to this new page
	PageSpan *add(const librevenge::RVNGPropertyList &xPropList);
	//! returns a new section id
	int getNewSectionId()
	{
		return m_sectionId++;
	}
protected:
	//! the list of page
	std::vector<std::shared_ptr<PageSpan> > m_pageSpanList;
	//! the number of section
	int m_sectionId;
};
}
#endif

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
