/******************************************************************************
*******************************************************************************
**
**  Copyright (C) Sistina Software, Inc.  1997-2003  All rights reserved.
**  Copyright (C) 2004 Red Hat, Inc.  All rights reserved.
**
**  This copyrighted material is made available to anyone wishing to use,
**  modify, copy, or redistribute it subject to the terms and conditions
**  of the GNU General Public License v.2.
**
*******************************************************************************
******************************************************************************/

#include <linux/module.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/smp_lock.h>
#include <linux/lm_interface.h>

#define RELEASE_NAME "1.00.00"

struct nolock_lockspace {
	unsigned int nl_lvb_size;
};

struct lm_lockops nolock_ops;

/**
 * nolock_mount - mount a nolock lockspace
 * @table_name: the name of the space to mount
 * @host_data: host specific data
 * @cb: the callback
 * @lockstruct: the structure of crap to fill in
 *
 * Returns: 0 on success, -EXXX on failure
 */

static int
nolock_mount(char *table_name, char *host_data,
	     lm_callback_t cb, lm_fsdata_t *fsdata,
	     unsigned int min_lvb_size, struct lm_lockstruct *lockstruct)
{
	char *c;
	unsigned int jid;
	struct nolock_lockspace *nl;

	/* If there is a "jid=" in the hostdata, return that jid.
	   Otherwise, return zero. */

	c = strstr(host_data, "jid=");
	if (!c)
		jid = 0;
	else {
		c += 4;
		sscanf(c, "%u", &jid);
	}

	nl = kmalloc(sizeof(struct nolock_lockspace), GFP_KERNEL);
	if (!nl)
		return -ENOMEM;

	memset(nl, 0, sizeof(struct nolock_lockspace));
	nl->nl_lvb_size = min_lvb_size;

	lockstruct->ls_jid = jid;
	lockstruct->ls_first = 1;
	lockstruct->ls_lvb_size = min_lvb_size;
	lockstruct->ls_lockspace = (lm_lockspace_t *)nl;
	lockstruct->ls_ops = &nolock_ops;
	lockstruct->ls_flags = LM_LSFLAG_LOCAL;

	return 0;
}

/**
 * nolock_others_may_mount - unmount a lock space
 * @lockspace: the lockspace to unmount
 *
 */

static void
nolock_others_may_mount(lm_lockspace_t *lockspace)
{
}

/**
 * nolock_unmount - unmount a lock space
 * @lockspace: the lockspace to unmount
 *
 */

static void
nolock_unmount(lm_lockspace_t *lockspace)
{
	struct nolock_lockspace *nl = (struct nolock_lockspace *)lockspace;
	kfree(nl);
}

/**
 * nolock_withdraw - withdraw from a lock space
 * @lockspace: the lockspace
 *
 */

static void
nolock_withdraw(lm_lockspace_t *lockspace)
{
}

/**
 * nolock_get_lock - get a lm_lock_t given a descripton of the lock
 * @lockspace: the lockspace the lock lives in
 * @name: the name of the lock
 * @lockp: return the lm_lock_t here
 *
 * Returns: 0 on success, -EXXX on failure
 */

static int
nolock_get_lock(lm_lockspace_t *lockspace, struct lm_lockname *name,
		lm_lock_t ** lockp)
{
	*lockp = (lm_lock_t *)lockspace;
	return 0;
}

/**
 * nolock_put_lock - get rid of a lock structure
 * @lock: the lock to throw away
 *
 */

static void
nolock_put_lock(lm_lock_t *lock)
{
}

/**
 * nolock_lock - acquire a lock
 * @lock: the lock to manipulate
 * @cur_state: the current state
 * @req_state: the requested state
 * @flags: modifier flags
 *
 * Returns: A bitmap of LM_OUT_*
 */

static unsigned int
nolock_lock(lm_lock_t *lock, unsigned int cur_state, unsigned int req_state,
	    unsigned int flags)
{
	return req_state | LM_OUT_CACHEABLE;
}

/**
 * nolock_unlock - unlock a lock
 * @lock: the lock to manipulate
 * @cur_state: the current state
 *
 * Returns: 0
 */

static unsigned int
nolock_unlock(lm_lock_t *lock, unsigned int cur_state)
{
	return 0;
}

/**
 * nolock_cancel - cancel a request on a lock
 * @lock: the lock to cancel request for
 *
 */

static void
nolock_cancel(lm_lock_t *lock)
{
}

/**
 * nolock_hold_lvb - hold on to a lock value block
 * @lock: the lock the LVB is associated with
 * @lvbp: return the lm_lvb_t here
 *
 * Returns: 0 on success, -EXXX on failure
 */

static int
nolock_hold_lvb(lm_lock_t *lock, char **lvbp)
{
	struct nolock_lockspace *nl = (struct nolock_lockspace *)lock;
	int error = 0;

	*lvbp = kmalloc(nl->nl_lvb_size, GFP_KERNEL);
	if (*lvbp)
		memset(*lvbp, 0, nl->nl_lvb_size);
	else
		error = -ENOMEM;

	return error;
}

/**
 * nolock_unhold_lvb - release a LVB
 * @lock: the lock the LVB is associated with
 * @lvb: the lock value block
 *
 */

static void
nolock_unhold_lvb(lm_lock_t *lock, char *lvb)
{
	kfree(lvb);
}

/**
 * nolock_sync_lvb - sync out the value of a lvb
 * @lock: the lock the LVB is associated with
 * @lvb: the lock value block
 *
 */

static void
nolock_sync_lvb(lm_lock_t *lock, char *lvb)
{
}

/**
 * nolock_plock_get - 
 * @lockspace: the lockspace
 * @name:
 * @file:
 * @fl:
 *
 * Returns: errno
 */

static int
nolock_plock_get(lm_lockspace_t *lockspace,
		 struct lm_lockname *name,
		 struct file *file, struct file_lock *fl)
{
	struct file_lock *tmp;

	lock_kernel();
	tmp = posix_test_lock(file, fl);
	fl->fl_type = F_UNLCK;
	if (tmp)
		memcpy(fl, tmp, sizeof(struct file_lock));
	unlock_kernel();

	return 0;
}

/**
 * nolock_plock -
 * @lockspace: the lockspace
 * @name:
 * @file:
 * @cmd:
 * @fl:
 *
 * Returns: errno
 */

static int
nolock_plock(lm_lockspace_t *lockspace,
	     struct lm_lockname *name,
	     struct file *file, int cmd, struct file_lock *fl)
{
	int error;
	lock_kernel();
	error = posix_lock_file_wait(file, fl);
	unlock_kernel();
	return error;
}

/**
 * nolock_punlock -
 * @lockspace: the lockspace
 * @name:
 * @file:
 * @fl:
 *
 * Returns: errno
 */

static int
nolock_punlock(lm_lockspace_t *lockspace,
	       struct lm_lockname *name,
	       struct file *file, struct file_lock *fl)
{
	int error;
	lock_kernel();
	error = posix_lock_file_wait(file, fl);
	unlock_kernel();
	return error;
}

/**
 * nolock_recovery_done - reset the expired locks for a given jid
 * @lockspace: the lockspace
 * @jid: the jid
 *
 */

static void
nolock_recovery_done(lm_lockspace_t *lockspace, unsigned int jid,
		     unsigned int message)
{
}

struct lm_lockops nolock_ops = {
	.lm_proto_name = "lock_nolock",
	.lm_mount = nolock_mount,
	.lm_others_may_mount = nolock_others_may_mount,
	.lm_unmount = nolock_unmount,
	.lm_withdraw = nolock_withdraw,
	.lm_get_lock = nolock_get_lock,
	.lm_put_lock = nolock_put_lock,
	.lm_lock = nolock_lock,
	.lm_unlock = nolock_unlock,
	.lm_cancel = nolock_cancel,
	.lm_hold_lvb = nolock_hold_lvb,
	.lm_unhold_lvb = nolock_unhold_lvb,
	.lm_sync_lvb = nolock_sync_lvb,
	.lm_plock_get = nolock_plock_get,
	.lm_plock = nolock_plock,
	.lm_punlock = nolock_punlock,
	.lm_recovery_done = nolock_recovery_done,
	.lm_owner = THIS_MODULE,
};

/**
 * init_nolock - Initialize the nolock module
 *
 * Returns: 0 on success, -EXXX on failure
 */

int __init
init_nolock(void)
{
	int error;

	error = lm_register_proto(&nolock_ops);
	if (error) {
		printk("lock_nolock: can't register protocol: %d\n", error);
		return error;
	}

	printk("Lock_Nolock %s (built %s %s) installed\n",
	       RELEASE_NAME, __DATE__, __TIME__);

	return 0;
}

/**
 * exit_nolock - cleanup the nolock module
 *
 */

void __exit
exit_nolock(void)
{
	lm_unregister_proto(&nolock_ops);
}

module_init(init_nolock);
module_exit(exit_nolock);

MODULE_DESCRIPTION("GFS Nolock Locking Module " RELEASE_NAME);
MODULE_AUTHOR("Red Hat, Inc.");
MODULE_LICENSE("GPL");
