/*
	DATE OF THIS VERSION:
	---------------------
	Sat Nov 27 23:00:20 MST 1993

	NEW SINCE 20 NOV 1993:
	----------------------
  	Two new functions were added: GETMATCHES() and GETRMATCHES()

	BUGS:
	-----
  	I had a script that used these functions that caused ircII to crash 
	once or twice, but I have been unable to trace the cause or reproduce 
	the crash.  I would appreciate any help or info if anyone else 
	experiences similar problems.  This IS my first time using writing code 
	to work with ircII code.

	ARGUMENTS:
	----------
  	array_name  : A string of some sort (no spaces, case insensitive) 
		identifying an array, either an existing array, or a new array.

  	item_number : A number or index into the array.  All array items are 
		numbered beginning at zero and continuing sequentially.  The 
		item number can be an existing item of an existing array, a 
		new item in an existing array (if the previous greatest item 
		number was 5, then the new item number must be 6, maintaining 
		the sequence), or a new item (must be item zero) in a new array.
  	data_...    : Basically any data you want.  It is stored internally as a
                character string, and in indexed (more to come) internally
                using the standard C library function strcmp().

	FUNCTIONS:
	----------
 	SETITEM(array_name item_number data_to_be_stored)
    	Use SETITEM to set an existing item of an existing array to a new value,
    	to create a new value in an existing array (see note about item_number),
    	or to create a new array (see note about item_number above).
    	RETURNS: 0 (zero) if it successfully sets and existing item in the array
             	 1 if it successfully creates a new array
             	 2 if it successfully adds a new item to an existing array 
            	-1 if it was unable to find the array specified (item_number > 0)
            	-2 if it was unable to find the item in an existing array
               		(item_number was too large)

  	GETITEM(array_name item_number)
    	Use this to retrieve previously stored data.
    	RETURNS: the data requested
             OR an empty string if the array did not exist or the item did not.

  	NUMITEMS(array_name)
    	RETURNS: the number of items in the array
             OR zero if the array name is invalid.  Useful for auto-adding to
             an array:
                 alias ADDDATA {
                     if (SETITEM(my-array $NUMITEMS(my-array) $*) >= 0) {
                         echo ADDED DATA
                     } {
                         echo FAILED TO ADD DATA
                     }
                 }
 
  	DELITEM(array_name item_number)
    	This deletes the item requested from the array.  If it is the last item
    	(item zero), it deletes the array.  If there are items numbered higher
    	than this item, they will each be moved down.  So if we had a 25 item
    	array called "MY-ARRAY" and deleted item 7, items 8 through 24 (remember
    	that a 25 item array is numbered 0 to 24) would be moved down and become
    	items 7 through 23.
    	RETURNS:  Zero on success,
              -1 if unable to find the array,
              -2 if unable find the item.

  	MATCHITEM(array_name pattern)
    	Searches through the items in the array for the item that best matches 
	the pattern, much like the MATCH() function does.
    	RETURNS: zero or a positive number which is the item_number of the match
             OR -1 if unable to find the array,
             OR -2 if no match was found in the array

  	RMATCHITEM(array_name data_to_look_for)
    	This treats the items in the array as patterns, searching for the 
	pattern in the array that best matches the data_to_look_for, working 
	similarly to the RMATCH() function.
    	RETURNS: zero or a positive number which is the item_number of the match
             OR -1 if unable to find the array,
             OR -2 if no match was found in the array

  	GETMATCHES(array_name pattern)
    	Seeks all array items that match the pattern.
   	RETURNS: A string containing a space-separated list of item numbers of
             array elements that match the pattern, or a null string if no
             matches were found, or if the array was not found.

  	GETRMATCHES(array_name data_to_look_for)
    	Treats all array items as patterns, and seeks to find all patterns that
    	match the data_to_look_for.
    	RETURNS: A string containing a space-separated list of item numbers of
             array elements that match the data, or a null string if no
             matches were found, or if the array was not found.

  	FINDITEM(array_name data_to_search_for)
    	This does a binary search on the items stored in the array and returns
    	the item number of the data.  It is an EXACT MATCH search.  It is highly
    	case sensitive, using C's strcmp() and not IRCII's caseless comparison
   	functions.  I did it this way, because I needed it, so there!  ;)
    	RETURNS: zero or a positive number on success -- this number IS the
             item_number of the first match it found
             OR -1 if unable to find the array,
             OR -2 if the item was not found in the array.

  	IGETITEM(array_name index_number)
    	This is exactly like GETITEM() except it uses a index number in the same
   	range as the item_number's.  It returns the item that corresponds to the
    	internal alphabetized index that these functions maintain.  Thus if you
    	access items 0 to 24 of "MY-ARRAY" with this function, you would observe
    	that the items returned came in an almost alphabetized manner.  They 
	would not be truly alphabetized because the ordering is done using 
	strcmp() which is case sensitive.
    	RETURNS: the data to which the index refers
             OR an empty string on failure to find the array or index.

  	INDEXTOITEM(array_name index_number)
    	This converts an index_number to an item_number.
    	RETURNS: the item_number that corresponds to the index_number for 
		the array
             OR -1 if unable to find the array,
             OR -2 if the index_number was invalid

  	ITEMTOINDEX(array_name item_number)
    	This converts an item_number to an index_number.
    	RETURNS: the index_number that corresponds to the item_number for 
		the array
             OR -1 if unable to find the array,
             OR -2 if the item_number was invalid

  	DELARRAY(array_name)
    	This deletes all items in an array.
    	RETURNS:  zero on success, -1 if it couldn't find the array.

  	NUMARRAYS()
    	RETURNS: the number of arrays that currently exist.

  	GETARRAYS()
    	RETURNS: a string consisting of all the names of all the arrays 
		 separated by spaces.

Thanks, 
Aaron Gifford
Karll on IRC
<agifford@sci.dixie.edu>

*/
/*
 * FILE:       alias.c
 * WRITTEN BY: Aaron Gifford (Karll on IRC)
 * DATE:       Sat Nov 27 23:00:20 MST 1993
 */

#include "irc.h"
#include "ircaux.h"

#define ARRAY_THRESHOLD	100

typedef struct an_array_struct {
	char **item;
	long *index;
	long size;
} an_array;

static an_array array_info = {
        null(char **),
        null(long *),
        0L
};

static an_array *array_array = null(an_array *);

/*
 * find_item() does a binary search of array.item[] using array.index[]
 * to find an exact match of the string *find.  If found, it returns the item
 * number (array.item[item_number]) of the match.  Otherwise, it returns a
 * negative number.  The negative number, if made positive again, and then
 * having 1 subtracted from it, will be the item_number where the string *find
 * should be inserted into the array.item[].  The function_setitem() makes use
 * of this attribute.
 */
long         find_item(array, find)
an_array     array;
char        *find;
{
        long top, bottom, key, cmp;

        top = array.size - 1;
        bottom = 0;
	
        while (top >= bottom)
	{
                key = (top - bottom) / 2 + bottom;
                cmp = strcmp(find, array.item[array.index[key]]);
                if (cmp == 0)
                        return key;
                if (cmp < 0)
                        top = key - 1;
                else
                        bottom = key + 1;
        } 
        return -bottom - 1;
}

/*
 * insert_index() takes a valid index (newIndex) and inserts it into the array
 * **index, then increments the *size of the index array.
 */
void          insert_index(index, size, newIndex)
long        **index;
long         *size;
long          newIndex;
{
	long cnt;

	if (*size)
		*index = (long *)new_realloc(*index, (*size + 1) * sizeof(long));
	else
	{
		*index = (long *)new_malloc(sizeof(long));
		newIndex = 0;
	}
	
	for (cnt = *size; cnt > newIndex; cnt--)
		(*index)[cnt] = (*index)[cnt - 1];
	(*index)[newIndex] = *size;
	(*size)++;
}

/*
 * move_index() moves the array.index[] up or down to make room for new entries
 * or to clean up so an entry can be deleted.
 */
void         move_index(array, oldindex, newindex)
an_array    *array;
long         oldindex;
long         newindex;
{
	long temp;

	if (newindex > oldindex)
		newindex--;
	if (newindex == oldindex)
		return;
	
	temp = array->index[oldindex];

	if (oldindex < newindex)
		for (; oldindex < newindex; oldindex++)
			array->index[oldindex] = array->index[oldindex + 1];
	else
		for(; oldindex > newindex; oldindex--)
			array->index[oldindex] = array->index[oldindex - 1];
	
	array->index[newindex] = temp;
}

/*
 * find_index() attempts to take an item and discover the index number
 * that refers to it.  find_index() assumes that find_item() WILL always return
 * a positive or zero result (be successful) because find_index() assumes that
 * the item is valid, and thus a find will be successful.  I don't know what
 * value ARRAY_THRESHOLD ought to be.  I figured someone smarter than I am
 * could figure it out and tell me or tell me to abandon the entire idea.
 */
long         find_index(array, item)
an_array    *array;
long         item;
{
	long search = 0;

	if (array->size >= ARRAY_THRESHOLD)
	{
		search = find_item(*array, array->item[item]);
		while (search >= 0 && !strcmp(array->item[array->index[search]], array->item[item]))
			search--;
		search++;
	}
	while(array->index[search] != item && search < array->size)
		search++;

	if (search == array->size)
		say("ERROR in find_index()");	
	return search;
}

/*
 * get_array() searches and finds the array referenced by *name.  It returns
 * a pointer to the array, or a null pointer on failure to find it.
 */
an_array        *get_array(name)
char            *name;
{
	long index;

	if (array_info.size && *name)
        {
                upper(name);
                if ((index = find_item(array_info, name)) >= 0)
                        return &array_array[array_info.index[index]];
	}
	return null(an_array*);
}

/*
 * delete_array() deletes the contents of an entire array.  It assumes that
 * find_item(array_info, name) will succeed and return a valid zero or positive
 * value.
 */
void         delete_array(name)
char        *name;
{
        char **ptr;
        long cnt;
        long index;
        long item;
        an_array *array;

        index = find_item(array_info, name);
        item = array_info.index[index];
        array = &array_array[item];
        for (ptr=array->item, cnt=0; cnt < array->size; cnt++, ptr++)
                new_free(*ptr);
        new_free(&array->item);
        new_free(&array->index);

        if (array_info.size > 1)
        {
                for(cnt = 0; cnt < array_info.size; cnt++)
                        if (array_info.index[cnt] > item)
                                (array_info.index[cnt])--;
                move_index(&array_info, index, array_info.size);
                array_info.size--;
                for(ptr=&array_info.item[item], cnt=item; cnt < array_info.size; cnt++, ptr++, array++)
                {
                        *ptr = *(ptr + 1);
                        *array = *(array + 1);
                }
                array_info.item = (char**)new_realloc(array_info.item, sizeof(char *) * array_info.size);
                array_info.index = (long *)new_realloc(array_info.index, sizeof(long) * array_info.size);
                array_array = (an_array *)new_realloc(array_array, sizeof(an_array) * array_info.size);
        }
        else
        {
                new_free(&array_info.item);
                new_free(&array_info.index);
                new_free(&array_array);
                array_info.size = 0;
        }
}

/*
 * Now for the actual alias functions
 * ==================================
 */


/*
 * function_matchitem() attempts to match a pattern to the contents of an array
 * RETURNS -1 if it cannot find the array, or -2 if no matches occur
 */
char        *function_matchitem(input)
char        *input;
{
        char    *result = null(char *);
	char	*name;
	long	index;
	an_array *array;
        long     current_match;
        long     best_match = 0;
        long     match = -1;
	char	tmp[40];

	if ((name = next_arg(input, &input)) && (array = get_array(name)) && (*input != '\0'))
	{
		match = -2;
        	if (*input)
        	{
			for (index = 0; index < array->size; index++)
                	{
                        	if ((current_match = wild_match(input, array->item[index])) > best_match)
                        	{
                               		match = index;
                               		best_match = current_match;
                        	}
                	}
		}
	}
        sprintf(tmp, "%ld", match);
        malloc_strcpy(&result, tmp);
        return result;
}

/*
 * function_getmatches() attempts to match a pattern to the contents of an
 * array and returns a list of item_numbers of all items that match the pattern
 * or it returns a null string if not items matches or if the array was not
 * found.
 */
char        *function_getmatches(input)
char        *input;
{
        char    *result = null(char *);
        char    *name = null(char *);
        long    index;
        an_array *array;
        long	match = 0;
	char	tmp[40];

        if ((name = next_arg(input, &input)) && (array = get_array(name)) && (*input != '\0'))
        {
                if (*input)
                {
                        for (index = 0; index < array->size; index++)
                        {
                                if (wild_match(input, array->item[index]) > 0)
                                {
					if (!match) {
						match = 1;
						sprintf(tmp, "%ld", index);
					}
					else
						sprintf(tmp, " %ld", index);
					malloc_strcat(&result, tmp);
                                }
                        }
                }
        }
        return result;
}

/*
 * function_rmatchitem() attempts to match the input text with an array of
 * patterns, much like RMATCH()
 * RETURNS -1 if it cannot find the array, or -2 if no matches occur
 */
char        *function_rmatchitem(input)
char        *input;
{
        char    *result = null(char *);
        char    *name = null(char *);
        long    index;
        an_array *array;
        long     current_match;
        long     best_match = 0;
        long     match = -1;
        char    tmp[40];

        if ((name = next_arg(input, &input)) && (array = get_array(name)) && (*input != '\0'))
        {
                match = -2;
                if (*input)
                {
                        for (index = 0; index < array->size; index++)
                        {
                                if ((current_match = wild_match(array->item[index], input)) > best_match)
                                {
                                        match = index;
                                        best_match = current_match;
                                }
                        }
                }
        }
        sprintf(tmp, "%ld", match);
        malloc_strcpy(&result, tmp);
        return result;
}

/*
 * function_getrmatches() attempts to match the input text with an array of
 * patterns, and returns a list of item_numbers of all patterns that match the
 * given text, or it returns a null string if no matches occur or if the array
 * was not found.
 */
char        *function_getrmatches(input)
char        *input;
{
        char    *result = null(char *);
        char    *name = null(char *);
        long    index;
        an_array *array;
        char     tmp[40];
	long	match = 0;

        if ((name = next_arg(input, &input)) && (array = get_array(name)) && (*input != '\0'))
        {
                if (*input)
                {
                        for (index = 0; index < array->size; index++)
                        {
                                if (wild_match(array->item[index], input) > 0)
                                {
                                        if (!match) {
                                                match = 1;
                                                sprintf(tmp, "%ld", index);
                                        }
                                        else
                                                sprintf(tmp, " %ld", index);
                                        malloc_strcat(&result, tmp);
                                }
                        }
                }
        }
        return result;
}

/*
 * function_numitems() returns the number of items in an array, or -1 if unable
 * to find the array
 */
char        *function_numitems(input)
char        *input;
{
        char *name = null(char *);
	char *result = null(char *);
	an_array *array;
	long items = 0;
	char tmp[40];

        if ((name = next_arg(input, &input)) && (array = get_array(name)))
                items = array->size;
	sprintf(tmp, "%ld", items);
        malloc_strcpy(&result, tmp);
	return result;
}

/*
 * function_getitem() returns the value of the specified item of an array, or
 * returns an empty string on failure to find the item or array
 */
char        *function_getitem(input)
char        *input;
{
	char *name = null(char *);
	char *itemstr = null(char *);
	long item;
	an_array *array;
	char *found = null(char *);
	char *result;

	if ((name = next_arg(input, &input)) && (array = get_array(name)))
		if (itemstr = next_arg(input, &input))
		{
			item = atol(itemstr);
			if (item >= 0 && item < array->size)
				found = array->item[item];
		}
	malloc_strcpy(&result, found);
	return result;
}

/*
 * function_setitem() sets an item of an array to a value, or creates a new
 * array if the array doesn not already exist and the item number is zero, or
 * it adds a new item to an existing array if the item number is one more than
 * the prevously largest item number in the array.
 * RETURNS: 0 on success
 *          1 on success if a new item was added to an existing array
 *          2 on success if a new array was created and item zero was set
 *         -1 if it is unable to find the array (and item number was not zero)
 *         -2 if it was unable to find the item (item < 0 or item was greater
 *            than 1 + the prevous maximum item number
 */
char        *function_setitem(input)
char        *input;
{
        char *name = null(char *);
	char *itemstr = null(char *);
	long item;
	long index = 0;
	long oldindex;
	an_array *array;
	int result = -1;
	char tmp[40];
	char *resultstr = null(char *);

        if (name = next_arg(input, &input))
        {
		if (strlen(name) && (itemstr = next_arg(input, &input)))
		{
			item = atol(itemstr);
			if (item >= 0)
			{
				upper(name);
				if (array_info.size && ((index = find_item(array_info, name)) >= 0))
       	         		{
					array =  &array_array[array_info.index[index]];
					result = -2;
					if (item < array->size)
					{
						oldindex = find_index(array, item);
						index = find_item(*array, input);
						index = (index >= 0) ? index : (-index) - 1;
						move_index(array, oldindex, index);
						new_free(&array->item[item]);
						malloc_strcpy(&array->item[item], input);
						result = 0;
					}
					else if (item == array->size)
					{
						array->item = (char **)new_realloc(array->item, (array->size + 1) * sizeof(char*));
						array->item[item] = null(char *);
						malloc_strcpy(&array->item[item], input);
						index = find_item(*array, input);
						index = (index >= 0) ? index : (-index) - 1;
						insert_index(&array->index, &array->size, index);
						result = 2;
					}
				}
				else
				{
					if (item == 0)
					{
						if (array_info.size)
							array_array = (an_array*)new_realloc(array_array, (array_info.size + 1) * sizeof(an_array));
						else
							array_array = (an_array*)new_malloc(sizeof(an_array));
						array = &array_array[array_info.size];
						array->size = 1;
						array->item = (char **)new_malloc(sizeof(char *));
						array->index = (long *)new_malloc(sizeof(long));
						array->item[0] = null(char*);
						array->index[0] = 0;
						malloc_strcpy(&array->item[0], input);
						if (array_info.size)
							array_info.item = (char **)new_realloc(array_info.item, (array_info.size + 1) * sizeof(char *));
						else
							array_info.item = (char **)new_malloc(sizeof(char *));
						array_info.item[array_info.size] = null(char *);
						malloc_strcpy(&array_info.item[array_info.size], name);
						insert_index(&array_info.index, &array_info.size, (-index) - 1);
						result = 1;
					}
				}
			}
		}
	}
	sprintf(tmp, "%d", result);
        malloc_strcpy(&resultstr, tmp);
        return resultstr;
}

/*
 * function_getarrays() returns a string containg the names of all currently
 * existing arrays separated by spaces
 */
char        *function_getarrays(input)
char        *input;
{
	char *result = null(char *);
	long index;
	char tmp[512];
	
	for (index = 0; index < array_info.size; index++)
	{
		if (index)
		{
			sprintf(tmp, " %s", array_info.item[array_info.index[index]]);
			malloc_strcat(&result, tmp);
		}
		else
			malloc_strcat(&result, array_info.item[array_info.index[index]]);
	}

	return result;
}

/*
 * function_numarrays() returns the number of currently existing arrays
 */
char        *function_numarrays(input)
char        *input;
{
	char tmp[40];
	char *result = null(char *);

	sprintf(tmp, "%ld", array_info.size);
	malloc_strcpy(&result, tmp);
	return result;
}

/*
 * function_finditem() does a binary search and returns the item number of
 * the string that exactly matches the string searched for, or it returns
 * -1 if unable to find the array, or -2 if unable to find the item.
 */
char        *function_finditem(input)
char        *input;
{
        char    *result = null(char *);
        char    *name = null(char *);
        an_array *array;
	long	item = -1;
        char    tmp[40];

	if ((name = next_arg(input, &input)) && (array = get_array(name)) && (*input != '\0'))
        {
		item = find_item(*array, input);
		item = (item >= 0) ? array->index[item] : -2;
        }
        sprintf(tmp, "%ld", item);
        malloc_strcpy(&result, tmp);
        return result;
}

/*
 * function_ifinditem() does a binary search and returns the index number of
 * the string that exactly matches the string searched for, or it returns
 * -1 if unable to find the array, or -2 if unable to find the item.
 */
char        *function_ifinditem(input)
char        *input;
{
        char    *result = null(char *);
        char    *name = null(char *);
        an_array *array;
        long    item = -1;
        char    tmp[40];

        if ((name = next_arg(input, &input)) && (array = get_array(name)) && (*input != '\0'))
        {
                item = find_item(*array, input);
		if (item < 0)
                	item = -2;
        }
        sprintf(tmp, "%ld", item);
        malloc_strcpy(&result, tmp);
        return result;
}

/*
 * function_igetitem() returns the item referred to by the passed-in index
 * or returns an empty string if unable to find the array or if the index was
 * invalid.
 */
char        *function_igetitem(input)
char        *input;
{
        char *name = null(char *);
        char *itemstr = null(char *);
        long item;
        an_array *array;
        char *found = null(char *);
        char *result;

        if ((name = next_arg(input, &input)) && (array = get_array(name)))
        {
                if (itemstr = next_arg(input, &input))
                {
                        item = atol(itemstr);
                        if (item >= 0 && item < array->size)
                                found = array->item[array->index[item]];
                }
        }
        malloc_strcpy(&result, found);
        return result;
}

/*
 * function_indextoitem() converts an index number to an item number for the
 * specified array.  It returns a valid item number, or -1 if unable to find
 * the array, or -2 if the index was invalid.
 */
char        *function_indextoitem(input)
char        *input;
{
        char *name = null(char *);
        char *itemstr = null(char *);
        long item;
        an_array *array;
        char tmp[40];
	long found = -1;
	char *result= null(char *);

        if ((name = next_arg(input, &input)) && (array = get_array(name)))
        {
		found = -2;
                if (itemstr = next_arg(input, &input))
                {
                        item = atol(itemstr);
                        if (item >= 0 && item < array->size)
                                found = array->index[item];
                }
        }
	sprintf(tmp, "%ld", found);
        malloc_strcpy(&result, tmp);
        return result;
}

/*
 * function_itemtoindex() takes an item number and searches for the index that
 * refers to the item.  It returns the index number, or -1 if unable to find
 * the array, or -2 if the item was invalid.
 */
char        *function_itemtoindex(input)
char        *input;
{
        char *name;
        char *itemstr;
        long item;
        an_array *array;
        char tmp[40];
        long found = -1;
        char *result;

        if ((name = next_arg(input, &input)) && (array = get_array(name)))
	{
                found = -2;
		if (itemstr = next_arg(input, &input))
                {
                        item = atol(itemstr);
                        if (item >= 0 && item < array->size)
                                found = find_index(array, item);
                }
        }
        sprintf(tmp, "%ld", found);
        malloc_strcpy(&result, tmp);
        return result;
}

/*
 * function_delitem() deletes an item of an array and moves the contents of the
 * array that were stored "above" the item down by one.  It returns 0 (zero)
 * on success, -1 if unable to find the array, -2 if unable to find the item.
 * Also, if the item is the last item in the array, it deletes the array.
 */
char        *function_delitem(input)
char        *input;
{
        char *name;
	char *itemstr;
        char **strptr;
        long item;
	long cnt;
	long oldindex;
        an_array *array;
        char tmp[40];
        long found = -1;
	char *result;

        if ((name = next_arg(input, &input)) && (array = get_array(name)))
        {
		found = -2;
                if (itemstr = next_arg(input, &input))
                {
                        item = atol(itemstr);
                        if (item >= 0 && item < array->size)
			{
                                if (array->size == 1)
				{
					delete_array(name);
				}
				else
				{
					oldindex = find_index(array, item);
					for(cnt = 0; cnt < array->size; cnt++)
						if (array->index[cnt] > item)
							(array->index[cnt])--;
					move_index(array, oldindex, array->size);
                                	new_free(&array->item[item]);
					array->size--;
					for(strptr=&(array->item[item]), cnt=item; cnt < array->size; cnt++, strptr++)
						*strptr = *(strptr + 1);
					array->item = (char**)new_realloc(array->item, sizeof(char *) * array->size);
					array->index = (long*)new_realloc(array->index, sizeof(long) * array->size);
				}
				found = 0;
                        }
                }
        }
	sprintf(tmp, "%ld", found);
        malloc_strcpy(&result, tmp);
        return result;
}

/*
 * function_delarray() deletes the entire contents of the array using the
 * delete_array() function above.  It returns 0 on success, -1 on failure.
 */
char        *function_delarray(input)
char        *input;
{
        char *name;
        an_array *array;
        char tmp[40];
        long found = -1;
        char *result;

	if ((name = next_arg(input, &input)) && (array = get_array(name)))
        {
                delete_array(name);
		found = 0;
	}
	sprintf(tmp, "%ld", found);
        malloc_strcpy(&result, tmp);
        return result;
}
/*
 * function_ifindfirst() returns the first index of an exact match with the
 * search string, or returns -2 if unable to find the array, or -1 if unable
 * to find any matches.
 */
char        *function_ifindfirst(input)
char        *input;
{
        char    *result = null(char *);
        char    *name;
        an_array *array;
        long    item = -1;
        char    tmp[40];

        if ((name = next_arg(input, &input)) && (array = get_array(name)) && (*input != '\0'))
        {
                item = find_item(*array, input);
                if (item >= 0)
                {
                        while (item >= 0 && !strcmp(array->item[array->index[item]], input))
                                item--;
                        item++;
                }
                else
                        item = -2;
        }
        sprintf(tmp, "%ld", item);
        malloc_strcpy(&result, tmp);
        return result;
}
