
/*
 * Copyright (C) 2002 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 * Copyright (C) 1998 Rasca, Berlin
 * EMail: thron@gmx.de
 *
 * Olivier Fourdan (fourdan@xfce.org)
*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <X11/Xlib.h>
#include <X11/Xproto.h>
#include <X11/Xatom.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include "glade_callbacks.h"
#include "glade_gui.h"
#include "glade_support.h"

#include "constants.h"
#include "types.h"

#include "aux.h"
#include "bookmarks.h"
#include "cpy.h"
#include "dnd.h"
#include "icons.h"
#include "entry.h"
#include "misc.h"
#include "monitor.h"
#include "smb_download.h"
#include "smb_misc.h"
#include "run.h"
#include "refresh.h"
#include "uri.h"
#include "widgets.h"


extern char *src_host;
extern gboolean tar_extraction;
static gboolean dragging = FALSE;
static GtkTreeRowReference *target_ref = NULL;

enum
{
    DND_NONE,
    DND_MOVE,
    DND_COPY,
    DND_LINK
};

enum
{
    TARGET_URI_LIST,
    TARGET_PLAIN,
    TARGET_STRING,
    TARGET_ROOTWIN,
    TARGETS
};


static GtkTargetEntry target_table[] = {
    {"text/uri-list", 0, TARGET_URI_LIST},
    {"text/plain", 0, TARGET_PLAIN},
    {"STRING", 0, TARGET_STRING}
};
#define NUM_TARGETS (sizeof(target_table)/sizeof(GtkTargetEntry))
static gboolean drop_cancelled = FALSE;

static GList *selection_list = NULL;
static int selection_len;
static char *dnd_data = NULL;
static int the_mode;
static int drag_type;

#define DRAG_TYPE_UNDEFINED	0
#define DRAG_TYPE_LOCAL		0x01
#define DRAG_TYPE_NET		0x02
#define DRAG_TYPE_INCONSISTENT	0x04

typedef struct selection_list_t
{
    tree_entry_t *en;
    GtkTreeRowReference *reference;
}
selection_list_t;

void cancel_drop(gboolean state)
{
    drop_cancelled = state;
}

void reselect_dnd_list(GtkTreeView *treeview){
    GList *tmp;
    GtkTreeSelection *selection = gtk_tree_view_get_selection(treeview);
    for (tmp=selection_list;tmp;tmp=tmp->next){
	selection_list_t *sl=tmp->data;
        if(sl && gtk_tree_row_reference_valid(sl->reference)){
           GtkTreePath *treepath = gtk_tree_row_reference_get_path(sl->reference);
	   /*printf("DBG: reselecting %s\n",sl->en->path);*/
	   gtk_tree_selection_select_path (selection,treepath);
	   gtk_tree_path_free(treepath);
	}
    }
}

void clear_dnd_selection_list(void)
{
    GList *tmp = selection_list;
    if (dragging) return;
    /*printf("DBG: clearing selection list!\n");*/
    while(tmp)
    {
	selection_list_t *sl;
	sl = (selection_list_t *) tmp->data;
	if(sl->reference)
	    gtk_tree_row_reference_free(sl->reference);
	g_free(sl);
	sl=NULL;
	tmp = tmp->next;
    }
    if(selection_list)
    {
	g_list_free(selection_list);
	selection_list = NULL;
	selection_len = 0;
    }
    selection_list = NULL;
}

void clear_path_from_selection_list(GtkTreeView *treeview,GtkTreePath *treepath)
{
    GtkTreePath *path;
    GtkTreeSelection *selection = gtk_tree_view_get_selection(treeview);
    GList *tmp = selection_list;
    if (!treepath || ! treeview) return;
    gtk_tree_selection_unselect_path (selection,treepath);
    for (tmp=selection_list;tmp;tmp=tmp->next)
    {
	selection_list_t *sl;
	sl = (selection_list_t *) tmp->data;
	path=gtk_tree_row_reference_get_path(sl->reference);
	if (gtk_tree_path_compare(path,treepath)==0){
	    selection_list = g_list_remove(selection_list,sl);
	    g_free(sl);
	    gtk_tree_path_free(path);
	    return;
	}
        gtk_tree_path_free(path);
    }
    return;
}

void   on_drag_begin(GtkWidget *widget,GdkDragContext *drag_context,gpointer user_data){
    	dragging = TRUE;	
	/* this works better with plain motion-event */
	/*reselect_dnd_list((GtkTreeView *)widget);   */
}

static gboolean valid_drop_site(GtkTreeView * treeview, gint x, gint y, tree_entry_t ** en, GtkTreeRowReference ** reference)
{
    tree_details_t *tree_details = get_tree_details(treeview);
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    GtkTreeIter iter;
    GtkTreePath *treepath;
    int title_offset=0;


    if (gtk_tree_view_get_headers_visible(treeview)){
	/* FIXME:    offset to title button assumed at fontheight+8 but should be
	 *           widget vertical ascent, queried from the gtkcontainer child 
	 *           (how do you do that?)
	 *           */
  	PangoRectangle logical_rect;
  	PangoLayout *layout = gtk_widget_create_pango_layout((GtkWidget *)treeview,"W");
  	pango_layout_get_pixel_extents(layout, NULL, &logical_rect);
  	title_offset= PANGO_ASCENT(logical_rect) + PANGO_DESCENT(logical_rect);
  	g_object_unref(layout);
	title_offset +=8;
    } 
    
    /*{
      GtkTreePath *tpath=gtk_tree_path_new_first        ();
      GdkRectangle rect;
      gtk_tree_view_get_cell_area     (treeview,
                                             tpath,
                                             NULL,
                                             &rect);
      gtk_tree_path_free(tpath);
      printf("DBG: title offset at %d y=%d wy=%d height=%d\n",title_offset,y,rect.y,rect.height);
    }*/

    if(!gtk_tree_view_get_path_at_pos(treeview, x, (tree_details->preferences & SHOW_TITLES) ? y - title_offset : y, &treepath, NULL, NULL, NULL))
    {
      nodrop:
	    print_status(treeview, "xf_ERROR_ICON",
#ifdef ENOTSUP
			    strerror(ENOTSUP),
#else 
#ifdef EOPNOTSUPP
			    strerror(EOPNOTSUPP),
#else
			    strerror(EINVAL),
#endif
#endif
			    ": ",_("Drop"), NULL);
	return FALSE;
    }
    if(*reference)
	gtk_tree_row_reference_free(*reference);
    *reference = gtk_tree_row_reference_new(treemodel, treepath);
    gtk_tree_model_get_iter(treemodel, &iter, treepath);
    gtk_tree_path_free(treepath);
    gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, en, -1);

    if(!en) goto nodrop;

    if(IS_BOOKMARK_TYPE((*en)->type) && IS_ROOT_TYPE((*en)->type))
	    goto drop_ok;	
    if(IS_APP_TYPE((*en)->type) && !IS_ROOT_TYPE((*en)->type))
	    goto drop_ok;
    
    if(IS_DIR((*en)->type)||
       IS_NETDIR((*en)->subtype) ||
       IS_XF_NETSHARE((*en)->subtype)
      ) goto drop_ok;
    
    if(IS_XF_RPM_CHILD((*en)->type) || 
       IS_XF_TAR_CHILD((*en)->type) ||
       IS_TRASH_TYPE((*en)->type)
      )	goto nodrop;
   
     
    if(IS_FIND_TYPE((*en)->type) &&
       (IS_ROOT_TYPE((*en)->type)||IS_XF_FND((*en)->type))
      )  goto nodrop;
	
    /* check for parent status */	
    {
	GtkTreeIter parent;
	if(!gtk_tree_model_iter_parent(treemodel, &parent, &iter))
	    goto nodrop;
	gtk_tree_model_get(treemodel, &parent, ENTRY_COLUMN, en, -1);
	if(!(*en)) goto nodrop;
	if(IS_BOOKMARK_TYPE((*en)->type) && IS_ROOT_TYPE((*en)->type))
	    goto drop_ok;
	if(!IS_DIR((*en)->type) && 
	   !IS_NETDIR((*en)->subtype) &&
	   !IS_XF_NETSHARE((*en)->subtype)
	  ) goto nodrop;
    }

drop_ok:
    print_status(treeview, "xf_INFO_ICON", _("Drop"), ": ", abreviate((*en)->path), NULL);
    return TRUE;
}

char  *CreateSMBTmpList(GtkTreeView *treeview,
		GList *in_list,char *target,gboolean samba_server){
    FILE *tmpfile;
    uri *u;
    char *w;
    static char *fname=NULL;
    int nitems=0;
    struct stat s;
    GList *list;
    
    nitems=0;
    if ((fname=randomTmpName(NULL))==NULL) return NULL;
    if ((tmpfile=fopen(fname,"w"))==NULL) return NULL;
    fprintf(tmpfile,"cd /;cd \"%s\";\n",target);
    for (list=in_list;list;list=list->next){
        u = list->data;
		 nitems++;
		 /*fprintf(tmpfile,"%d:%s:%s\n",u->type,u->url,target);*/ 
		 if (!strchr(u->url,'/')) {
			 fclose(tmpfile);
			 unlink (fname);
			 return NULL;
		 }
		 w=g_strdup(strrchr(u->url,'/')+1);
		 if (!samba_server) {
		 	/* FIXME: this should work but doesn't, it is 
			 *  a smbclient bug to watch  */
			 ascii_unreadable(w);
		 }
			 
		 if (lstat(u->url,&s)<0){
			 print_diagnostics(treeview,"xf_ERROR_ICON",
					 strerror(errno),":",
					 u->url,"\n",NULL);
			 fclose(tmpfile);
			 unlink (fname);
			 g_free(w);
			 w=NULL;
			 return NULL;
		 } 
		 if (S_ISREG(s.st_mode)) {
  		   /*fprintf(tmpfile,"put \"%s\" \"%s\\%s\";\n",u->url,selected.dirname+1,w);*/
  		   fprintf(tmpfile,"put \"%s\" \"%s\";\n",u->url,w);
		 }
		 else if (S_ISDIR(s.st_mode)) {
	  	   fprintf(tmpfile,"mkdir \"%s\";\n",w);
	  	   fprintf(tmpfile,"cd \"%s\";\n",w);
	  	   fprintf(tmpfile,"prompt;recurse;\n");
	  	   fprintf(tmpfile,"lcd \"%s\";\n",u->url);
	  	   fprintf(tmpfile,"mput *;\n");
	  	   fprintf(tmpfile,"prompt;recurse;\n");
    		   fprintf(tmpfile,"cd /;cd \"%s\";\n",target);
		 }
		 else { /*CHR, BLK, FIFO, LNK, SOCK */
 			 print_diagnostics(treeview,"xf_ERROR_ICON",
					 "cannot upload ",
					 u->url,"\n",NULL);
	         }
		 fflush(NULL);
		 g_free(w);
		 w=NULL;
    }
    fprintf(tmpfile,"ls;\n");
    fclose (tmpfile);
    /*fprintf(stderr,"dbg:nitems = %d\n",nitems);*/
    if (!nitems) {
       /*fprintf(stderr,"dbg:nitems = %d\n",nitems);*/
	    unlink(fname);
	    return NULL;
    }
    /*fprintf(stderr,"dbg: same device=%d\n",same_device);*/
    return fname;
}




void on_drag_data(GtkWidget * widget, GdkDragContext * context, gint x, gint y, GtkSelectionData * data, guint info, guint time, void *client)
{
    GtkTreeView *treeview = (GtkTreeView *) widget;
    tree_details_t *tree_details = get_tree_details(treeview);
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    tree_entry_t *en;
    GtkTreeIter iter;
    int nitems, action;
    uri *u;
    int mode = 0;
    GList *tmp, *list = NULL;
    char *tmpfile = NULL;
    Atom atomo;
    gboolean will_expand=FALSE;

    /*printf("DBG: on drag data\n"); */


    if(!set_load_wait(&tree_details))
    {
	gtk_drag_finish(context, FALSE, FALSE, time);
	return;
    }
    if(!widget || data->length < 0 || data->format != 8)
    {
	gtk_drag_finish(context, FALSE, FALSE, time);
	goto drag_over;
    }

    if(src_host){
	g_free(src_host);
	src_host=NULL;
    }
    atomo = XInternAtom(GDK_DISPLAY(), "WM_CLIENT_MACHINE", FALSE);
    {
	unsigned char *property_data;
	unsigned long items, remaining;
	int actual_format;
	Atom actual_atom;
	if(XGetWindowProperty(GDK_DISPLAY(), GDK_WINDOW_XID(context->source_window), atomo, 0, 255, FALSE, XA_STRING, &actual_atom, &actual_format, &items, &remaining, &property_data) == Success)
	{
	    src_host = g_strdup(property_data);
	    XFree(property_data);
	    /*printf("dbg: drop from host=%s\n",src_host); */
	}
	else
	{
	    src_host = g_strdup(our_host_name(treeview));
	    /*printf("dbg: failed property get\n"); */
	}

    }



    /*printf("dbg:on_drag_data\n"); */

    action = (context->action <= GDK_ACTION_DEFAULT) ? ((tree_details->preferences & DRAG_DO_COPY) ? GDK_ACTION_COPY : GDK_ACTION_MOVE) : context->action;


    if(!valid_drop_site(treeview, x, y, &en, &target_ref) || !en)
    {
	gtk_drag_finish(context, FALSE, FALSE, time);
	goto drag_over;
    }
    if(!gtk_tree_row_reference_valid(target_ref))
	goto drag_over;
    {
      GtkTreePath *treepath;
      treepath=gtk_tree_row_reference_get_path(target_ref);
      gtk_tree_model_get_iter (treemodel,&iter,treepath);
      gtk_tree_path_free(treepath);
    }
    
    /* entry comes from valid_drop_site(), this gives us the iter*/
    /*if(!get_entry_from_reference(treeview, target_ref, &iter, &en))
	goto drag_over;*/
	

    if(!(info == TARGET_STRING) && !(info == TARGET_URI_LIST))
    {
	gtk_drag_finish(context, FALSE, FALSE, time);
	goto drag_over;
    }


    if(action == GDK_ACTION_MOVE)
	the_mode = mode = TR_MOVE;
    else if(action == GDK_ACTION_COPY)
	the_mode = mode = TR_COPY;
    else if(action == GDK_ACTION_LINK)
	the_mode = mode = TR_LINK;
    else
    {
	print_status(treeview, "xf_ERROR_ICON",strerror(EINVAL), NULL);
	goto drag_over;
    }

    if (IS_EXPANDED(en->type)) will_expand=TRUE;


    cursor_wait(treeview);
    /*printf("DBG:drag data=%s\n",(const char *) data->data); */
    nitems = uri_parse_list((const char *)data->data, &list);
    /*printf ("dbg:nitems=%d\n",nitems); */
    if(!nitems)
	goto drag_over;		/* of course */
    u = list->data;
    
    if(u->type == URI_SMB)
    {	/* src_host may be remote or local, but target only local
	  or bookmark */		
      if (IS_DIR(en->type)) {    
	SMBGetFile(treeview, en->path, list);	
      } 
      if (!(IS_BOOKMARK_TYPE(en->type) && IS_ROOT_TYPE(en->type))){
        list = uri_free_list(list);
        goto drag_over;
      }
    }
    else
	uri_remove_file_prefix_from_list(list);

    if(IS_BOOKMARK_TYPE(en->type) && (IS_ROOT_TYPE(en->type)))
    {
	for(tmp = list; tmp != NULL; tmp = tmp->next)
	{
	    u = tmp->data;
	    add2bookmarks(treeview, u->url);
	    /*printf("DBG: adding %s to bookmarks\n",u->url);*/
	}
	list = uri_free_list(list);
	goto drag_over;
    }

    if(IS_NETDIR(en->subtype) || 
	 IS_XF_NETSHARE(en->subtype)|| 
	 IS_NETFILE(en->subtype) 
      ){
      gchar *target;
        
      if (IS_XF_NETSHARE(en->subtype)){
        target=g_strdup("/"); 	    
      } else {
	char *p;
	p=strstr(en->path+2,"/")+1;
	target=g_strdup(strstr(p,"/"));
        if (IS_NETFILE(en->subtype)) *strrchr(target,'/')=0;
      } 
      /*printf("DBG:target=%s\n",target);*/
      
      tmpfile=CreateSMBTmpList(treeview,list,target,
		      IS_SAMBA_SERVER(en->subtype));
      g_free(target);
      target=NULL;
      if (!tmpfile) {
         /*fprintf(stderr,"dbg:null tmpfile\n");*/
         list=uri_free_list (list);
         goto drag_over;
      }
      /*else fprintf(stderr,"dbg:tmpfile=%s\n",tmpfile);*/
      SMBDropFile (treeview,en,&iter,tmpfile);
      list = uri_free_list(list);
      goto drag_over;
    }


    if(IS_APP_TYPE(en->type) && !IS_ROOT_TYPE(en->type))
    {
	for(tmp = list; tmp != NULL; tmp = tmp->next)
	{  
	    gchar *in_cmd;
	    
	    u = tmp->data;
	    
	    if (en->filter) {
		in_cmd = g_strconcat(en->path," ",en->filter,NULL);
	    	print_status(treeview, "xf_INFO_ICON", 
			       _("Executing"), " : ", 
			        FILENAME(en)," ",en->filter,
			       NULL);
	    }
	    else {
		in_cmd = g_strdup(en->path);
	    	print_status(treeview, "xf_INFO_ICON", 
			       _("Executing"), " : ", 
			        FILENAME(en),
			       NULL);
	    }
	    
 	    on_run_path(treeview, in_cmd, u->url, IS_IN_TERM(en->subtype),
			   FALSE, FALSE);
	    g_free(in_cmd);
	}
	list = uri_free_list(list);
	goto drag_over;
    }

    if(strcmp(src_host, our_host_name(treeview)) != 0)
    {
	int l = 0;
	char **srcs = NULL;

	l=g_list_length(list);
	srcs = (char **)malloc((l+1) * sizeof(char *));
	if(!srcs) g_assert_not_reached();
	srcs[l] = NULL;
	for(l = 0, tmp = list; tmp != NULL; tmp = tmp->next)
	{
	    u = tmp->data;
	    srcs[l] = u->url;
	    srcs[l + 1] = NULL;
	    /* with rsync, one by one */
	    if(!(tree_details->preferences & RSYNC_X_SCP))
	    {
		rsync(treeview, srcs + l, en->path);
	    }
	    l++;
	}
	if(tree_details->preferences & RSYNC_X_SCP)
	    rsync(treeview, srcs, en->path);
	g_free(srcs);
	srcs=NULL;

#if 0
	for(tmp = list; tmp != NULL; tmp = tmp->next)
	{
	    u = tmp->data;
	    if(!rsync(treeview, u->url, en->path))
		break;
	}
#endif
	list = uri_free_list(list);
	goto drag_over;
    }

    /* above: u = list->data; */
    /*fprintf(stderr,"dbg:dnd, src=%s(tarchild=%d) tgt=%s\n",
     *  u->url,t_en->type & FT_TARCHILD,t_en->path);*/
    /*fprintf(stderr,"dbg:dnd, src=%s tgt=%s\n",u->url,en->path); */

    {
	/* nonsense check */
	struct stat st;
	lstat(u->url, &st);
	if(st.st_ino == en->st->st_ino)
	{
	    list = uri_free_list(list);
	    print_status(treeview, "xf_ERROR_ICON", _("Source and target are the same!"), NULL);
	    /*fprintf(stderr,"dbg:nonsense 1\n"); */
	    goto drag_over;
	}
	if(!S_ISDIR(st.st_mode) && strchr(u->url, '/'))
	{
	    char *p;
	    p = g_strdup(u->url);
	    *(strrchr(p, '/')) = 0;
	    if(strcmp(p, en->path) == 0)
	    {
		list = uri_free_list(list);
		g_free(p);
		p=NULL;
		print_status(treeview, "xf_ERROR_ICON", _("Source and target are the same!"), NULL);
		/*fprintf(stderr,"dbg:nonsense 2\n"); */
		goto drag_over;
	    }
	    g_free(p);
	    p=NULL;
	}
    }

    /* tmpfile ==NULL means drop cancelled */


    tmpfile = CreateTmpList(treeview, list, en);
    /*fprintf(stderr,"dbg:dnd, tmpfile=%s\n",tmpfile); */
    if(!tmpfile)
    {
	char *what;
	if(mode == TR_MOVE)
	    what = _("moved");
	else if(mode == TR_COPY)
	    what = _("copied");
	else
	    what = _("linked");
	/*printf("DBG:null tmpfile\n"); */
	print_status(treeview, "xf_INFO_ICON", _("Nothing"), " ", what, NULL);
	list = uri_free_list(list);
	goto drag_over;
    }
    /* acording to tmpfile name, do a direct move, here, and break.- */


    /*FIXME: uncomment when tar module is added 
       if (tar_extraction) DirectTransfer(ctree,mode,tmpfile); 
       else */
    {
	if((mode & TR_LINK) || (on_same_device() && (mode & TR_MOVE)))
	{
	    /*printf("dbg:going to Direct transfer\n");*/
	    DirectTransfer(treeview, mode, tmpfile);
	}
	else
	{
	    /*printf("dbg:going to Indirect transfer\n");*/
	    IndirectTransfer(treeview, mode, tmpfile);
	}
    }

    if(tmpfile)
	unlink(tmpfile);

    list = uri_free_list(list);


    /*fprintf(stderr,"dbg:parent:runOver\n"); */

  drag_over:
    update_dir(treeview, target_ref);
    /*tar_extraction=FALSE;*/
    unset_load_wait(&tree_details);
    if (will_expand && (IS_NETDIR(en->subtype) || 
			IS_XF_NETSHARE(en->subtype)||  
			IS_NETFILE(en->subtype)) )
    {
	GtkTreePath *treepath=gtk_tree_row_reference_get_path(target_ref);
    	gtk_tree_view_expand_row(treeview, treepath, FALSE);  
    	gtk_tree_path_free(treepath);
    }

    hide_stop(tree_details->window);
    cursor_reset(treeview);
/* some files may or may not succeed in move, but we'll let monitor fix up later */
    gtk_drag_finish(context, TRUE, (the_mode & TR_MOVE) ? TRUE : FALSE, time);
    if (!IS_BOOKMARK_TYPE(en->type)) local_monitor(treeview,TRUE);
}

gboolean is_in_dnd_selection(GtkTreePath * path){
	GList *tmp;
	if (!path) return TRUE;
	for (tmp=selection_list;tmp;tmp=tmp->next){
		selection_list_t *sl=tmp->data;
           	GtkTreePath *treepath = gtk_tree_row_reference_get_path(sl->reference);
		if(gtk_tree_path_compare (path,treepath)==0){
	           gtk_tree_path_free(treepath);
		   return TRUE;
		}
	        gtk_tree_path_free(treepath);
	}
	return FALSE;
}

void get_dnd_selection(GtkTreeModel * treemodel, GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
    GtkTreeView *treeview = (GtkTreeView *) data;
    GtkTreeSelection *selection = gtk_tree_view_get_selection(treeview);
    tree_entry_t *en;

   /* if (drag_type==DRAG_TYPE_INCONSISTENT) return;*/

    gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &en, -1);
    if(!IS_PATH(en->type) && !IS_NETTHING(en->subtype) )
    {
	gtk_tree_selection_unselect_iter(selection, iter);
	return;
    }
    if (!IS_LOCAL_TYPE(en->type) && IS_ROOT_TYPE(en->type)) return;
    /*if((IS_PATH(en->type) && drag_type==DRAG_TYPE_NET) ||
       (IS_NETTHING(en->subtype) && drag_type==DRAG_TYPE_LOCAL)){
        drag_type=DRAG_TYPE_INCONSISTENT;
	print_diagnostics(treeview,"xf_ERROR_ICON",strerror(E2BIG),"\n",NULL); 
	return;
    }*/
    if(IS_PATH(en->type)) drag_type=DRAG_TYPE_LOCAL;
    if(IS_NETTHING(en->subtype)) drag_type=DRAG_TYPE_NET;

    {
	selection_list_t *sl;
	GList *tmp;
	for (tmp=selection_list;tmp;tmp=tmp->next){
	    sl = (selection_list_t *)tmp->data;
	    if (strcmp(sl->en->path,en->path)==0) return;
	}
	sl = (selection_list_t *) malloc(sizeof(selection_list_t));
	if(!sl)
	    g_assert_not_reached();
	sl->en = en;
	sl->reference = gtk_tree_row_reference_new(treemodel, path);
	selection_list = g_list_append(selection_list, sl);
    }
    return;
}


/*
 * DND sender: prepare data for the remote
 * event: drag_data_get
 */
void on_drag_data_get(GtkWidget * widget, GdkDragContext * context, GtkSelectionData * selection_data, guint info, guint time, gpointer data)
{
    char *files;
    GList *tmp;

    GtkTreeView *treeview = (GtkTreeView *) widget;
    GtkTreeSelection *selection = gtk_tree_view_get_selection(treeview);
    
    if(!widget)	return;
    /*if(selection_list)	clear_dnd_selection_list();*/
    
    /*gtk_tree_selection_selected_foreach(selection, get_dnd_selection, (gpointer) treeview);*/
    /*drag_type=DRAG_TYPE_UNDEFINED;*/
    if(!selection_list) return;
   
    if(!(drag_type&(DRAG_TYPE_LOCAL|DRAG_TYPE_NET))) 
	    return;


    /* prepare data for the receiver */
    switch (info)
    {
	case TARGET_ROOTWIN:	/* not implemented (yet) */
	 return;
	default:
	 if(dnd_data){
		 g_free(dnd_data);
		 dnd_data=NULL;
	 }
	 switch (drag_type){
	  default:
		  g_assert_not_reached();
	  case DRAG_TYPE_LOCAL:	    
	    selection_len = 0;
	    for (tmp = selection_list;tmp;tmp = tmp->next) {
	        selection_list_t *sl;
	        sl = (selection_list_t *) tmp->data;
		if (!gtk_tree_row_reference_valid(sl->reference)) return;
	        selection_len += (strlen(sl->en->path) + 7);
	    }
	    dnd_data = files = g_malloc(selection_len + 1);
	    dnd_data[0] = '\0';
	    for (tmp = selection_list;tmp;tmp = tmp->next)
	    {
		selection_list_t *sl;
		sl = (selection_list_t *) tmp->data;
		if (!gtk_tree_row_reference_valid(sl->reference)) return;
		if(strncmp(sl->en->path, "tar:", strlen("tar:")) == 0)
		{
		    sprintf(files, "%s\r\n", sl->en->path);
		    files += (strlen(sl->en->path) + 2);
		}
		else
		{
		    sprintf(files, "file:%s\r\n", sl->en->path);
		    files += (strlen(sl->en->path) + strlen("file:") + 2);
		}
	    }
	    break;
	  case DRAG_TYPE_NET: 
	    selection_len = 0;
	    for (tmp = selection_list;tmp;tmp = tmp->next) {
	        int addlen;
		selection_list_t *sl;
		sl = (selection_list_t *) tmp->data;
		if (!gtk_tree_row_reference_valid(sl->reference)) return;
		addlen=strlen("smb://@://\r\n");
		addlen += (strlen(sl->en->path));
		addlen += ((sl->en->tag)?strlen(sl->en->tag):strlen("GUEST%%"));
		selection_len += addlen;
	    }
	    
    	    dnd_data = files = g_malloc (selection_len + 1);
    	    if (!files) g_assert_not_reached();
    	    files[0]=0;

	    for (tmp = selection_list;tmp;tmp = tmp->next) {
		selection_list_t *sl;
		char *server,*remote_file;
		sl = (selection_list_t *) tmp->data;
		if (!gtk_tree_row_reference_valid(sl->reference)) return;
		server=g_strdup(sl->en->path+2);
		strtok(server,"/");
		if (IS_XF_NETWS(sl->en->subtype)) {
		  	sprintf (files, "%s://%s@%s:\r\n",
			  IS_SAMBA_SERVER(sl->en->subtype)?"SMB":"smb",
			  (sl->en->tag)?sl->en->tag:"GUEST%%",
			  server);
		} else {
		  	remote_file=server+strlen(server)+1;
		  	sprintf (files, "%s://%s@%s:%s%s",
			  IS_SAMBA_SERVER(sl->en->subtype)?"SMB":"smb",
			  (sl->en->tag)?sl->en->tag:"GUEST%%",
			  server,remote_file,
			  (IS_NETDIR(sl->en->subtype))?"/\r\n":"\r\n");
		}
		g_free(server);
		server=NULL;
	        files = files + strlen(files);
	    }
	    break;
	 }
	 break;
    }
    /*printf("gdkatom=%lu(%s)\n",
       selection_data->target,
       gdk_atom_name(selection_data->target)); */

    gtk_selection_data_set(selection_data, selection_data->target, 
		    8, (const guchar *)dnd_data, selection_len); 
    dragging=FALSE;
    /*printf("DBG drag get done\n");    */
}

gboolean on_drag_motion(GtkWidget * widget, GdkDragContext * dc, gint x, gint y, guint t, gpointer data)
{
    GtkTreeView *treeview = (GtkTreeView *) widget;
    tree_details_t *tree_details = (tree_details_t *) get_tree_details(treeview);
    tree_entry_t *en;
    GdkDragAction action;

    /*printf("DBG:  on_drag_motion \n"); */



    /* Insert code to get our default action here. */
    if(tree_details->preferences & DRAG_DO_COPY)
	action = GDK_ACTION_COPY;
    else
	action = GDK_ACTION_MOVE;




    if(!valid_drop_site(treeview, x, y, &en, &target_ref) || !en)
    {				/*change icon */
	gdk_drag_status(dc, 0, t);

    }
    else if(dc->actions == GDK_ACTION_MOVE)
	gdk_drag_status(dc, GDK_ACTION_MOVE, t);
    else if(dc->actions == GDK_ACTION_COPY)
	gdk_drag_status(dc, GDK_ACTION_COPY, t);
    else if(dc->actions == GDK_ACTION_LINK)
	gdk_drag_status(dc, GDK_ACTION_LINK, t);
    else if(dc->actions & action)
	gdk_drag_status(dc, action, t);
    else
	gdk_drag_status(dc, 0, t);
    /*fprintf(stderr,"dbg: drag motion done...\n"); */



    return (TRUE);
}

#if 0
void on_drag_data_delete(GtkWidget * widget, GdkDragContext * context, gpointer data)
{
    printf("DBG: on_drag_data_delete!\n\n");


    return;
}
#endif

void on_drag_end(GtkWidget * widget, GdkDragContext * context, gpointer data)
{
    GList *tmp;
    GtkTreeView *treeview = (GtkTreeView *) widget;
    tree_details_t *tree_details = get_tree_details(treeview);

    /*printf("DBG drag end\n");*/
    dragging = FALSE;
    if(!widget)
	return;

    if(dnd_data)
    {
	g_free(dnd_data);
	dnd_data = NULL;
    }

    if(!set_load_wait(&tree_details))
    {
	printf("DBG: cannot set tree_details->loading! (on_drag_end())\n");
	return;
    }
    for (tmp=selection_list;tmp;tmp=tmp->next)
    {
	selection_list_t *sl;
	sl = (selection_list_t *) tmp->data;
	if(sl->reference && gtk_tree_row_reference_valid(sl->reference)){
		update_dir(treeview, sl->reference);
	}
    }

    /* this would unselect selection after drop: */
#if 0
    for (tmp=selection_list;tmp;tmp=tmp->next)
    {
	g_free(tmp->data);
    }
    if(selection_list)
    {
	g_list_free(selection_list);
	selection_len = 0;
    }
    selection_list = NULL;
#endif

    
    unset_load_wait(&tree_details);
    return;
}

void setup_drag_signal(GtkTreeView * treeview)
{
    g_signal_connect(G_OBJECT(treeview), "drag_data_received", G_CALLBACK(on_drag_data), NULL);
    g_signal_connect(G_OBJECT(treeview), "drag_data_get", G_CALLBACK(on_drag_data_get), NULL);
    g_signal_connect(G_OBJECT(treeview), "drag_motion", G_CALLBACK(on_drag_motion), NULL);
    g_signal_connect(G_OBJECT(treeview), "drag_end", G_CALLBACK(on_drag_end), NULL);
    g_signal_connect(G_OBJECT(treeview), "drag_begin", G_CALLBACK(on_drag_begin), NULL);
#if 0
    this signal looks bugged in gtk.only works well forsame widget.segv otherwise g_signal_connect(G_OBJECT(treeview), "drag_data_delete", G_CALLBACK(on_drag_data_delete), NULL);
#endif

    /*
     * DO NOT put gtk_drag_source_set here. It is very buggy if you do.
     * Instead set at first mouse click and unset on expand+contract
     * (inherent GtkCTree bug workaround) [This is a gtk1.2 note,
     * this may or may not apply in gtk2.]
     * gtk_drag_source_set (ctree, GDK_BUTTON1_MASK | GDK_BUTTON2_MASK, 
     target_table, NUM_TARGETS, 
     GDK_ACTION_MOVE | GDK_ACTION_COPY | GDK_ACTION_LINK);*/
    drag_source_set(treeview);	/*this location might need change */

    /*  this kind of dnd is useful to rearrange the order of the root nodes: */
    /*
       gtk_tree_view_enable_model_drag_source(treeview, 
       GTK_DEST_DEFAULT_DROP|GTK_DEST_DEFAULT_HIGHLIGHT, 
       target_table, 
       NUM_TARGETS, 
       GDK_ACTION_MOVE | GDK_ACTION_COPY | GDK_ACTION_LINK
       ); */


    gtk_drag_dest_set((GtkWidget *) treeview, GTK_DEST_DEFAULT_DROP | GTK_DEST_DEFAULT_HIGHLIGHT, target_table, NUM_TARGETS, GDK_ACTION_MOVE | GDK_ACTION_COPY | GDK_ACTION_LINK);

}


void drag_source_set(GtkTreeView * treeview)
{
    gtk_drag_source_set((GtkWidget *) treeview, GDK_BUTTON1_MASK | GDK_BUTTON2_MASK, target_table, NUM_TARGETS, GDK_ACTION_MOVE | GDK_ACTION_COPY | GDK_ACTION_LINK);
}

void drag_source_unset(GtkTreeView * treeview)
{
    gtk_drag_source_unset((GtkWidget *) treeview);
}
