#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include "interface.h"
#include "support.h"

#include "conf.h"
#include "main.h"
#include "trx.h"
#include "ptt.h"

static GConfClient *client = NULL;

/* ---------------------------------------------------------------------- */

static gchar *mkkey(const gchar *key)
{
	static gchar buf[256];

	g_snprintf(buf, sizeof(buf), "/apps/%s/%s", PACKAGE, key);
	buf[sizeof(buf) - 1] = 0;

	return buf;
}

/* ---------------------------------------------------------------------- */

void conf_set_string(const gchar *key, const gchar *val)
{
	g_return_if_fail(client);
	g_return_if_fail(key);

	gconf_client_set_string(client, mkkey(key), val, NULL);
}

void conf_set_bool(const gchar *key, gboolean val)
{
	g_return_if_fail(client);
	g_return_if_fail(key);

	gconf_client_set_bool(client, mkkey(key), val, NULL);
}

void conf_set_int(const gchar *key, gint val)
{
	g_return_if_fail(client);
	g_return_if_fail(key);

	gconf_client_set_int(client, mkkey(key), val, NULL);
}

void conf_set_float(const gchar *key, gdouble val)
{
	g_return_if_fail(client);
	g_return_if_fail(key);

	gconf_client_set_float(client, mkkey(key), val, NULL);
}

/* ---------------------------------------------------------------------- */

gchar *conf_get_string(const gchar *key, const gchar *def)
{
	gchar *s;

	g_return_val_if_fail(client, NULL);
	g_return_val_if_fail(key, NULL);

	s = gconf_client_get_string(client, mkkey(key), NULL);

	return s ? s : g_strdup(def);
}

gboolean conf_get_bool(const gchar *key, gboolean def)
{
	GConfValue *value;

	g_return_val_if_fail(client, FALSE);
	g_return_val_if_fail(key, FALSE);

	value = gconf_client_get(client, mkkey(key), NULL);

	if (!value || value->type != GCONF_VALUE_BOOL)
		return def;

	return gconf_value_get_bool(value);
}

gint conf_get_int(const gchar *key, gint def)
{
	GConfValue *value;

	g_return_val_if_fail(client, 0);
	g_return_val_if_fail(key, 0);

	value = gconf_client_get(client, mkkey(key), NULL);

	if (!value || value->type != GCONF_VALUE_INT)
		return def;

	return gconf_value_get_int(value);
}

gdouble conf_get_float(const gchar *key, gdouble def)
{
	GConfValue *value;

	g_return_val_if_fail(client, 0.0);
	g_return_val_if_fail(key, 0.0);

	value = gconf_client_get(client, mkkey(key), NULL);

	if (!value || value->type != GCONF_VALUE_FLOAT)
		return def;

	return gconf_value_get_float(value);
}

/* ---------------------------------------------------------------------- */

void conf_set_tag_color(GtkTextTag *tag, const gchar *color)
{
	g_return_if_fail(color);

	g_object_set(G_OBJECT(tag), "foreground", color, NULL);
}

void conf_set_bg_color(GtkWidget *widget, const gchar *color)
{
	GdkColormap *map;
	GdkColor *clr;

	g_return_if_fail(color);

	clr = g_malloc(sizeof(GdkColor));	// FIXME: leaking memory!!
	gdk_color_parse(color, clr);
	map = gdk_colormap_get_system();

	if (!gdk_colormap_alloc_color(map, clr, FALSE, TRUE)) {
		g_warning("Couldn't allocate color");
		return;
	}

	gtk_widget_modify_base(widget, GTK_STATE_NORMAL, clr);
}

void conf_set_font(GtkWidget *widget, const gchar *font)
{
	PangoFontDescription *font_desc;

	g_return_if_fail(font);

	font_desc = pango_font_description_from_string(font);
	gtk_widget_modify_font(widget, font_desc);
	pango_font_description_free(font_desc);
}

void conf_set_ptt(void)
{
	const gchar *dev;
	gboolean inv;
	gint mode;

	dev  = conf_get_string("ptt/dev", "none");
	inv  = conf_get_bool("ptt/inverted", FALSE);
	mode = conf_get_int("ptt/mode",	2);

	init_ptt(dev, inv, mode);
}

void conf_set_qso_bands(const gchar *bandlist)
{
	GtkCombo *qsobandcombo;
	GList *list = NULL;
	gchar *bands, **bandp, **p;

	bands = g_strdup(bandlist);

	g_strdelimit(bands, ", \t\r\n", ',');
	bandp = g_strsplit(bands, ",", 64);

	for (p = bandp; *p; p++)
		list = g_list_append(list, (gpointer) *p);

	qsobandcombo = GTK_COMBO(lookup_widget(appwindow, "qsobandcombo"));
	gtk_combo_set_popdown_strings(qsobandcombo, list);

	g_free(bands);
	g_strfreev(bandp);
	g_list_free(list);
}

void conf_set_rtty_config(void)
{
	gfloat shift, baud;
	gint bits, par, stop;
	gboolean rev, msb;

	shift = conf_get_float("rtty/shift", 170.0);
	baud = conf_get_float("rtty/baud", 45.45);
	bits = conf_get_int("rtty/bits", 0);
	par = conf_get_int("rtty/parity", 0);
	stop = conf_get_int("rtty/stop", 1);
	rev = conf_get_bool("rtty/reverse", TRUE);
	msb = conf_get_bool("rtty/msbfirst", FALSE);

	trx_set_rtty_parms(shift, baud, bits, par, stop, rev, msb);
}

void conf_set_mt63_config(void)
{
	gint bandwidth, interleave;
	gboolean cwid;

	bandwidth = conf_get_int("mt63/bandwidth", 1);
	interleave = conf_get_int("mt63/interleave", 1);

	cwid = conf_get_bool("mt63/cwid", FALSE);

	trx_set_mt63_parms(bandwidth, interleave, cwid);
}

/* ---------------------------------------------------------------------- */

static void tag_color_changed_notify(GConfClient *client,
				     guint id,
				     GConfEntry *entry,
				     gpointer data)
{
	GtkTextTag *tag = GTK_TEXT_TAG(data);
	GConfValue *value = gconf_entry_get_value(entry);

	if (value && value->type == GCONF_VALUE_STRING)
		conf_set_tag_color(tag, gconf_value_get_string(value));
}

static void bg_color_changed_notify(GConfClient *client,
				    guint id,
				    GConfEntry *entry,
				    gpointer data)
{
	GtkWidget *widget = GTK_WIDGET(data);
	GConfValue *value = gconf_entry_get_value(entry);

	if (value && value->type == GCONF_VALUE_STRING)
		conf_set_bg_color(widget, gconf_value_get_string(value));
}

static void font_changed_notify(GConfClient *client,
				guint id,
				GConfEntry *entry,
				gpointer data)
{
	GtkWidget *widget = GTK_WIDGET(data);
	GConfValue *value = gconf_entry_get_value(entry);

	if (value && value->type == GCONF_VALUE_STRING)
		conf_set_font(widget, gconf_value_get_string(value));
}

static void txoffset_changed_notify(GConfClient *client,
				    guint id,
				    GConfEntry *entry,
				    gpointer data)
{
	GConfValue *value = gconf_entry_get_value(entry);

	if (value && value->type == GCONF_VALUE_FLOAT)
		trx_set_txoffset(gconf_value_get_float(value));
}

static void sqval_changed_notify(GConfClient *client,
				 guint id,
				 GConfEntry *entry,
				 gpointer data)
{
	GConfValue *value = gconf_entry_get_value(entry);

	if (value && value->type == GCONF_VALUE_FLOAT)
		trx_set_sqval(gconf_value_get_float(value));
}

static void ptt_config_changed_notify(GConfClient *client,
				      guint id,
				      GConfEntry *entry,
				      gpointer data)
{
	conf_set_ptt();
}

static void qso_bands_changed_notify(GConfClient *client,
				     guint id,
				     GConfEntry *entry,
				     gpointer data)
{
	GConfValue *value = gconf_entry_get_value(entry);

	if (value && value->type == GCONF_VALUE_STRING)
		conf_set_qso_bands(gconf_value_get_string(value));
}

static void rtty_config_changed_notify(GConfClient *client,
				       guint id,
				       GConfEntry *entry,
				       gpointer data)
{
	conf_set_rtty_config();
}

static void mt63_config_changed_notify(GConfClient *client,
				       guint id,
				       GConfEntry *entry,
				       gpointer data)
{
	conf_set_mt63_config();
}

/* ---------------------------------------------------------------------- */

void conf_init(void)
{
	GtkWidget *w;
	const gchar *s;
	gboolean b;
	gfloat f;
	gint i;

	g_return_if_fail(!client);

	client = gconf_client_get_default();

	gconf_client_add_dir(client, "/apps/" PACKAGE,
			     GCONF_CLIENT_PRELOAD_RECURSIVE, NULL);


	s = conf_get_string("colors/hl", "blue");
	conf_set_tag_color(hltag, s);
	gconf_client_notify_add(client,
				mkkey("colors/hl"),
				tag_color_changed_notify,
				hltag,
				NULL, NULL);

	s = conf_get_string("colors/rx", "black");
	conf_set_tag_color(rxtag, s);
	gconf_client_notify_add(client,
				mkkey("colors/rx"),
				tag_color_changed_notify,
				rxtag,
				NULL, NULL);

	s = conf_get_string("colors/tx", "red");
	conf_set_tag_color(txtag, s);
	gconf_client_notify_add(client,
				mkkey("colors/tx"),
				tag_color_changed_notify,
				txtag,
				NULL, NULL);


	s = conf_get_string("colors/rxwin", "white");
	w = lookup_widget(appwindow, "rxtext");
	conf_set_bg_color(w, s);
	gconf_client_notify_add(client,
				mkkey("colors/rxwin"),
				bg_color_changed_notify,
				w,
				NULL, NULL);

	s = conf_get_string("colors/txwin", "white");
	w = lookup_widget(appwindow, "txtext");
	conf_set_bg_color(w, s);
	gconf_client_notify_add(client,
				mkkey("colors/txwin"),
				bg_color_changed_notify,
				w,
				NULL, NULL);


	s = conf_get_string("fonts/rxfont", "Sans 10");
	w = lookup_widget(appwindow, "rxtext");
	conf_set_font(w, s);
	gconf_client_notify_add(client,
				mkkey("fonts/rxfont"),
				font_changed_notify,
				w,
				NULL, NULL);

	s = conf_get_string("fonts/txfont", "Sans 10");
	w = lookup_widget(appwindow, "txtext");
	conf_set_font(w, s);
	gconf_client_notify_add(client,
				mkkey("fonts/txfont"),
				font_changed_notify,
				w,
				NULL, NULL);


	f = conf_get_float("misc/txoffset", 0.0);
	trx_set_txoffset(f);
	gconf_client_notify_add(client,
				mkkey("misc/txoffset"),
				txoffset_changed_notify,
				NULL,
				NULL, NULL);

	f = conf_get_float("misc/sqval", 15.0);
	trx_set_sqval(f);
	gconf_client_notify_add(client,
				mkkey("misc/sqval"),
				sqval_changed_notify,
				NULL,
				NULL, NULL);


	conf_set_ptt();
	gconf_client_notify_add(client,
				mkkey("ptt"),
				ptt_config_changed_notify,
				NULL,
				NULL, NULL);


	s = conf_get_qsobands();
	conf_set_qso_bands(s);
	gconf_client_notify_add(client,
				mkkey("misc/bands"),
				qso_bands_changed_notify,
				NULL,
				NULL, NULL);


	conf_set_rtty_config();
	gconf_client_notify_add(client,
				mkkey("rtty"),
				rtty_config_changed_notify,
				NULL,
				NULL, NULL);

	conf_set_mt63_config();
	gconf_client_notify_add(client,
				mkkey("mt63"),
				mt63_config_changed_notify,
				NULL,
				NULL, NULL);

	/*
	 * Load waterfall config
	 */
	f = conf_get_float("wf/dbdiv", 10.0);
	waterfall_set_dbdiv(waterfall, f);

	f = conf_get_float("wf/reflevel", 0.0);
	waterfall_set_reflevel(waterfall, f);

	i = conf_get_int("wf/mode", 0);
	switch (i) {
	default:
	case 0:
		w = lookup_widget(WFPopupMenu, "wfmode_normal");
		break;
	case 1:
		w = lookup_widget(WFPopupMenu, "wfmode_spectrum");
		break;
	case 2:
		w = lookup_widget(WFPopupMenu, "wfmode_scope");
		break;
	}
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(w), TRUE);

	i = conf_get_int("wf/magnification", 0);
	switch (i) {
	default:
	case 0:
		w = lookup_widget(WFPopupMenu, "wfmag_1");
		break;
	case 1:
		w = lookup_widget(WFPopupMenu, "wfmag_2");
		break;
	case 2:
		w = lookup_widget(WFPopupMenu, "wfmag_4");
		break;
	}
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(w), TRUE);

	i = conf_get_int("wf/speed", 1024);
	switch (i) {
	default:
	case 2048:
		w = lookup_widget(WFPopupMenu, "wfspeed_half");
		break;
	case 1024:
		w = lookup_widget(WFPopupMenu, "wfspeed_normal");
		break;
	case 512:
		w = lookup_widget(WFPopupMenu, "wfspeed_double");
		break;
	}
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(w), TRUE);

	i = conf_get_int("wf/window", 1);
	switch (i) {
	default:
	case 0:
		w = lookup_widget(WFPopupMenu, "wfwindow_rect");
		break;
	case 1:
		w = lookup_widget(WFPopupMenu, "wfwindow_tria");
		break;
	case 2:
		w = lookup_widget(WFPopupMenu, "wfwindow_hamm");
		break;
	}
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(w), TRUE);

	/*
	 * Load misc settings
	 */
	b = conf_get_bool("misc/afc", FALSE);
	w = lookup_widget(appwindow, "afcbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);

	b = conf_get_bool("misc/squelch", FALSE);
	w = lookup_widget(appwindow, "squelchbutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);

	b = conf_get_bool("misc/reverse", FALSE);
	w = lookup_widget(appwindow, "reversebutton");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), b);

}

void conf_clear(void)
{
	GtkToggleButton *button;
	gboolean b;
	wf_config_t config;

	g_return_if_fail(client);

	/*
	 * Save waterfall config
	 */
	waterfall_get_config(waterfall, &config);
	gconf_client_set_float(client,
			       mkkey("wf/dbdiv"),
			       config.dbdiv, NULL);
	gconf_client_set_float(client,
			       mkkey("wf/reflevel"),
			       config.reflevel, NULL);
	gconf_client_set_int(client,
			     mkkey("wf/mode"),
			       config.mode, NULL);
	gconf_client_set_int(client,
			     mkkey("wf/magnification"),
			     config.magnification, NULL);
	gconf_client_set_int(client,
			     mkkey("wf/speed"),
			     config.overlap, NULL);
	gconf_client_set_int(client,
			     mkkey("wf/window"),
			     config.window, NULL);

	/*
	 * Save misc settings
	 */
	button = GTK_TOGGLE_BUTTON(lookup_widget(appwindow, "afcbutton"));
	b = gtk_toggle_button_get_active(button);
	gconf_client_set_bool(client, mkkey("misc/afc"), b, NULL);

	button = GTK_TOGGLE_BUTTON(lookup_widget(appwindow, "squelchbutton"));
	b = gtk_toggle_button_get_active(button);
	gconf_client_set_bool(client, mkkey("misc/squelch"), b, NULL);

	button = GTK_TOGGLE_BUTTON(lookup_widget(appwindow, "reversebutton"));
	b = gtk_toggle_button_get_active(button);
	gconf_client_set_bool(client, mkkey("misc/reverse"), b, NULL);

	/*
	 * Clear GConfClient
	 */
	g_object_unref(G_OBJECT(client));
	client = NULL;
}

/* ---------------------------------------------------------------------- */
