/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 2001 Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*/

#include "config.h"

#include <parted/parted.h>
#include <parted/endian.h>

#if ENABLE_NLS
#  include <libintl.h>
#  define _(String) dgettext (PACKAGE, String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

#include <uuid/uuid.h>
#include "platform_defs.h"
#include "xfs_types.h"
#include "xfs_sb.h"

#include <string.h>

static PedGeometry*
xfs_probe (PedGeometry* geom)
{
	PedSector	block_size;
	PedSector	block_count;
	union {
		struct xfs_sb	sb;
		char		bytes [512];
	} buf;

	if (geom->length < XFS_SB_DADDR + 1)
		return NULL;
	if (!ped_geometry_read (geom, &buf, XFS_SB_DADDR, 1))
		return NULL;

	if (PED_LE32_TO_CPU (buf.sb.sb_magicnum) == XFS_SB_MAGIC) {
		block_size = PED_LE32_TO_CPU (buf.sb.sb_blocksize) / 512;
		block_count = PED_LE64_TO_CPU (buf.sb.sb_dblocks);

		return ped_geometry_new (geom->dev, geom->start,
					 block_size * block_count);
	}

	if (PED_BE32_TO_CPU (buf.sb.sb_magicnum) == XFS_SB_MAGIC) {
		block_size = PED_BE32_TO_CPU (buf.sb.sb_blocksize) / 512;
		block_count = PED_BE64_TO_CPU (buf.sb.sb_dblocks);

		return ped_geometry_new (geom->dev, geom->start,
					 block_size * block_count);
	}

	return NULL;
}

#ifndef DISCOVER_ONLY
static int
xfs_clobber (PedGeometry* geom)
{
	char	buf[512];

	memset (buf, 0, 512);
	return ped_geometry_write (geom, buf, XFS_SB_DADDR, 1);
}
#endif /* !DISCOVER_ONLY */

static PedFileSystemOps xfs_ops = {
	probe:		xfs_probe,
#ifndef DISCOVER_ONLY
	clobber:	xfs_clobber,
#else
	clobber:	NULL,
#endif
	open:		NULL,
	create:		NULL,
	close:		NULL,
	check:		NULL,
	copy:		NULL,
	resize:		NULL,
	get_create_constraint:	NULL,
	get_resize_constraint:	NULL,
	get_copy_constraint:	NULL
};

static PedFileSystemType xfs_type = {
	next:	NULL,
	ops:	&xfs_ops,
	name:	"xfs"
};

void
ped_file_system_xfs_init ()
{
	ped_file_system_type_register (&xfs_type);
}

void
ped_file_system_xfs_done ()
{
	ped_file_system_type_unregister (&xfs_type);
}

