/* -*-c-*- ---------------- mixgtk_device.c :
 * actual types for mixgtk devices
 * ------------------------------------------------------------------
 *  Last change: Time-stamp: <2001-05-10 23:42:26 jao>
 * ------------------------------------------------------------------
 * Copyright (C) 2001 Free Software Foundation, Inc.
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *  
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *  
 */


#include <stdio.h>
#include <mixlib/xmix_device.h>
#include "mixgtk.h"
#include "mixgtk_device.h"

#define BIN_DEV_COL_ 5

/* device container */
static GtkNotebook *container_ = NULL;
static gint last_pos_ = 0;
/* virtual machine */
static mix_vm_t *vm_ = NULL;

/* a mixgtk device */
struct mixgtk_device_t
{
  mix_device_t device;
  GtkWidget *widget;
  gint pos;
};

struct mixgtk_bin_device_t
{
  struct mixgtk_device_t gtk_device;
  guint last_insert;
  GtkWidget *scroll;
};

/* callbacks for output devices */
static void
write_char_ (struct mixgtk_device_t *dev, const mix_word_t *block)
{
  enum {MAX_BLOCK = 16, BUFF_SIZE = MAX_BLOCK * 5 + 2};
  static gchar BUFFER[BUFF_SIZE];
  guint k, j;

  for (k = 0; k < SIZES_[dev->device.type]; k++)
    for (j = 1; j < 6; j++)
      {
	mix_char_t ch = mix_word_get_byte (block[k], j);
	BUFFER[5 * k + j - 1] = mix_char_to_ascii (ch);
      }

  BUFFER[5 * k] = '\n';
  BUFFER[5 * k + 1] = '\0';
  
  gtk_text_insert (GTK_TEXT (dev->widget), NULL, NULL, NULL, BUFFER, -1);
  
}

static void
write_bin_  (struct mixgtk_bin_device_t *dev, const mix_word_t *block)
{
  enum {BUFF_SIZE = 17};
  static gchar BUFFER[BUFF_SIZE] = { 0 };
  static gchar *DEFTEXT[BIN_DEV_COL_] = { "0", "0", "0", "0", "0" };
    
  guint k, col, row;
  GtkCList *list = GTK_CLIST (dev->gtk_device.widget);
  
  gtk_clist_freeze (list);
  
  for (k = 0; k < SIZES_[dev->gtk_device.device.type]; k++)
    {
      mix_word_print_to_buffer (block[k], BUFFER);
      row = dev->last_insert / BIN_DEV_COL_;
      col = dev->last_insert % BIN_DEV_COL_;
      if (col == 0) gtk_clist_append (list, DEFTEXT);
      gtk_clist_set_text (list, row, col, BUFFER);
      dev->last_insert++;
    }
  
  gtk_clist_thaw (list);
}

static gboolean
write_ (mix_device_t *dev, const mix_word_t *block)
{
  struct mixgtk_device_t *gtkdev  = (struct mixgtk_device_t *) dev;
  
  if (dev->type != mix_dev_CONSOLE && !(DEF_DEV_VTABLE_->write)(dev, block))
    return FALSE;

  if (MODES_[dev->type] == mix_dev_CHAR) write_char_ (gtkdev, block);
  else write_bin_ ((struct mixgtk_bin_device_t *)gtkdev, block);
  
  gtk_notebook_set_page (container_, gtkdev->pos);

  return TRUE;
}


static gboolean 
read_ (mix_device_t *dev, mix_word_t *block)
{
  struct mixgtk_device_t *gtkdev  = (struct mixgtk_device_t *) dev;

  if (dev->type != mix_dev_CONSOLE && !(DEF_DEV_VTABLE_->read)(dev, block))
    return FALSE;

  if (MODES_[dev->type] == mix_dev_CHAR) write_char_ (gtkdev, block);
  else write_bin_ ((struct mixgtk_bin_device_t *)gtkdev, block);
  
  gtk_notebook_set_page (container_, gtkdev->pos);

  return TRUE;
}

static gboolean 
ioc_ (mix_device_t *dev, mix_short_t cmd)
{
  return (DEF_DEV_VTABLE_->ioc)(dev, cmd);
}

static gboolean
busy_ (const mix_device_t *dev) 
{
  return (DEF_DEV_VTABLE_->busy)(dev);
}

static void
destroy_ (mix_device_t *dev)
{
  struct mixgtk_device_t *gtkdev = (struct mixgtk_device_t *)dev;
  gtk_notebook_remove_page (GTK_NOTEBOOK (container_), gtkdev->pos);
  gtk_widget_destroy (gtkdev->widget);
  if (MODES_[dev->type] == mix_dev_BIN)
    gtk_widget_destroy (((struct mixgtk_bin_device_t *)dev)->scroll);
  (DEF_DEV_VTABLE_->destroy) (dev);
  --last_pos_;
}

static mix_device_vtable_t MIXGTK_VTABLE_ = {
  write_, read_, ioc_, busy_, destroy_
};

/* create the gui part of the device */
static void
mixgtk_device_construct_gui_ (struct mixgtk_device_t *dev)
{
  GtkWidget *label = gtk_label_new (DEF_NAMES_[dev->device.type]);

  g_assert (label);

  dev->pos = last_pos_++;
  if (MODES_[dev->device.type] == mix_dev_CHAR)
    {
      GtkWidget *box = gtk_hbox_new (0, 0);
      GtkWidget *scroll = NULL;
      GtkAdjustment *vadj = NULL;
      dev->widget = gtk_text_new (NULL, NULL);
      g_assert (box);
      g_assert (dev->widget);
      gtk_text_set_editable (GTK_TEXT (dev->widget), FALSE);
      vadj = GTK_TEXT (dev->widget)->vadj;
      scroll = gtk_vscrollbar_new (vadj);
      gtk_box_pack_start (GTK_BOX (box), dev->widget, TRUE, TRUE, 0);
      gtk_box_pack_start (GTK_BOX (box), scroll, FALSE, FALSE, 0);
      gtk_notebook_insert_page (container_, box, label, dev->pos);
      gtk_widget_show (box);
      gtk_widget_show (scroll);
    }
  else
    {
      gint k;
      struct mixgtk_bin_device_t *bindev =
	(struct mixgtk_bin_device_t *)dev;
      
      bindev->scroll = gtk_scrolled_window_new (NULL, NULL);
      g_assert (bindev->scroll);
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (bindev->scroll),
				      GTK_POLICY_AUTOMATIC,
				      GTK_POLICY_AUTOMATIC);
      dev->widget = gtk_clist_new (BIN_DEV_COL_);
      g_assert (dev->widget);
      for (k =0; k < BIN_DEV_COL_; ++k)
	{
	  gtk_clist_set_column_width (GTK_CLIST (dev->widget), k, 120);
	  gtk_clist_set_column_auto_resize (GTK_CLIST (dev->widget), k, TRUE);
	}
      gtk_container_add (GTK_CONTAINER (bindev->scroll), dev->widget);
      gtk_notebook_insert_page (container_, bindev->scroll, label, dev->pos);
      gtk_widget_show (bindev->scroll);
    }
  
  gtk_widget_show (label);
  gtk_widget_set_style (dev->widget,
			gtk_widget_get_style (GTK_WIDGET (container_)));
  gtk_widget_show (dev->widget);
  gtk_widget_draw (GTK_WIDGET (container_), NULL);
}

/* create a new mixgtk device */
static mix_device_t *
mixgtk_device_new_ (mix_device_type_t type)
{
  struct mixgtk_device_t *dev = NULL;
  gchar *name = NULL;
  
  g_return_val_if_fail (type < mix_dev_INVALID, NULL);

  if (MODES_[type] == mix_dev_CHAR)
    {
      dev = g_new (struct mixgtk_device_t, 1);
    }
  else
    {
      dev = (struct mixgtk_device_t *) g_new (struct mixgtk_bin_device_t, 1);
      ((struct mixgtk_bin_device_t *)dev)->last_insert = 0;
    }

  name = g_strconcat (g_get_home_dir(), "/",
		      MIXGTK_FILES_DIR, "/", DEF_NAMES_[type], NULL);
  construct_device_with_name_ (&dev->device, type, name);
  g_free (name);

  dev->device.vtable = &MIXGTK_VTABLE_;

  mixgtk_device_construct_gui_ (dev);

  return (mix_device_t *)dev;
}

/* init default devices */
gboolean
mixgtk_device_init (GtkNotebook *container, mix_vm_t *vm)
{
  static mix_device_type_t def_types[] = {
    /* mix_dev_CONSOLE,*/ mix_dev_INVALID
  };
  
  gint k = 0;
				     
  g_return_val_if_fail (container != NULL, FALSE);
  g_return_val_if_fail (vm != NULL, FALSE);
  container_ = container;
  vm_ = vm;

  /* remove dummy page from container */
  gtk_notebook_remove_page (container_, 0);

  mix_vm_set_device_factory (vm, mixgtk_device_new_);
  
  /* connect default devices */
  while (def_types[k] != mix_dev_INVALID)
    {
      struct mix_device_t *dev = mixgtk_device_new_ (def_types[k]);
      if (dev != NULL)	mix_vm_connect_device (vm, dev);
      ++k;
    }

  /* set to first page */
  gtk_notebook_set_page (container_, 0);
  
  return TRUE;
}

/* connect a new (file-based) device */
gboolean
mixgtk_device_connect (mix_device_type_t type, const gchar *name);
