/*  $Id: unit_test_agp_seq_entry.cpp 398248 2013-05-03 18:58:12Z rafanovi $
* ===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
* Author:  Michael Kornbluh, NCBI
*          (initial skeleton generated by script written by Pavel Ivanov)
*
* File Description:
*   Unit test class CAgpToSeqEntry
*
*
* ===========================================================================
*/

#include <ncbi_pch.hpp>

#include <corelib/ncbi_system.hpp>
#include <corelib/ncbiapp.hpp>
#include <corelib/ncbifile.hpp>

// This header must be included before all Boost.Test headers if there are any
#include <corelib/test_boost.hpp>

#include <objtools/readers/agp_seq_entry.hpp>

#include <objects/seqset/Seq_entry.hpp>

USING_NCBI_SCOPE;
USING_SCOPE(objects);

namespace {

    struct STestInfo {
        CFile m_AGPFile;
        vector<CFile> m_vecExpectedSeqEntryFiles;
        CFile m_FlagFile;
    };

    typedef string TTestName;

    typedef map<TTestName, STestInfo> TTestNameToInfoMap;

    class CTestNameToInfoMapLoader {
    public:
        CTestNameToInfoMapLoader(
            TTestNameToInfoMap * pTestNameToInfoMap )
            : m_pTestNameToInfoMap(pTestNameToInfoMap) { }

        void operator()( const CDirEntry & dirEntry ) {
            const static size_t kInvalidFileNumber = numeric_limits<size_t>::max();

            if( ! dirEntry.IsFile() ) {
                return;
            }

            CFile file(dirEntry);

            // skip the README file
            if( file.GetName() == "README.txt" ) {
                return;
            }

            // skip .svn files
            if( NStr::Find(file.GetPath(), ".svn") != NPOS )
            {
                return;
            }

            const string sFilePath = file.GetPath();
            cout << "Parsing file name: " << sFilePath << endl;

            // extract info from the file name
            const string sFileName = file.GetName();
            vector<CTempString> vecFileNamePieces;
            NStr::Tokenize( sFileName, ".", vecFileNamePieces );

            BOOST_REQUIRE( vecFileNamePieces.size() == 2 ||
                vecFileNamePieces.size() == 3 );

            CTempString tsTestName = vecFileNamePieces[0];
            BOOST_REQUIRE( ! tsTestName.empty() );
            CTempString tsFileType = vecFileNamePieces[1];
            size_t iFileNumber = kInvalidFileNumber;
            if( vecFileNamePieces.size() > 2 ) {
                iFileNumber = NStr::StringToUInt(vecFileNamePieces[2]);
            }
            
            STestInfo & test_info_to_load =
                (*m_pTestNameToInfoMap)[vecFileNamePieces[0]];

            // figure out what type of file we have and set appropriately
            if( tsFileType == "agp" && iFileNumber == kInvalidFileNumber ) {
                // handle agp file
                // (and make sure we don't have duplicates)
                BOOST_REQUIRE( test_info_to_load.m_AGPFile.GetPath().empty() );
                test_info_to_load.m_AGPFile = file;
            } else if( tsFileType == "expected_seq_entry" && iFileNumber != kInvalidFileNumber )
            {
                // handle expected seq-entry file
                // (Note that the files could come in in any order)

                vector<CFile> & vecExpectedSeqEntryFiles = 
                    test_info_to_load.m_vecExpectedSeqEntryFiles;

                if( vecExpectedSeqEntryFiles.size() <= iFileNumber ) {
                    // expand the vector to include this file number
                    vecExpectedSeqEntryFiles.resize( 1 + iFileNumber );
                }

                // make sure no duplicates
                BOOST_REQUIRE( vecExpectedSeqEntryFiles[iFileNumber].GetPath().empty() );
                vecExpectedSeqEntryFiles[iFileNumber] = file;
            } else if( tsFileType == "flags" && iFileNumber == kInvalidFileNumber ) {
                // handle flags file
                BOOST_REQUIRE( test_info_to_load.m_FlagFile.GetPath().empty() );
                test_info_to_load.m_FlagFile = file;
            } else {
                BOOST_FAIL("Unknown file type");
            }
        }

    private:
        // raw pointer because we do NOT own this
        TTestNameToInfoMap * m_pTestNameToInfoMap;
    };

    // run a test
    void s_RunTest(const string &sTestName, const STestInfo & testInfo)
    {
        CAgpToSeqEntry::TFlags fFlags = 0;

        // adjust flags based on whatever's in the flags file
        if( ! testInfo.m_FlagFile.GetPath().empty() ) {
            CNcbiIfstream flagfile( testInfo.m_FlagFile.GetPath().c_str() );
            BOOST_CHECK(flagfile);
            while( flagfile ) {
                string sLine;
                NcbiGetlineEOL(flagfile, sLine);
                NStr::TruncateSpacesInPlace(sLine);
                if( sLine.empty() ) {
                    // ignore blank lines
                } else if( sLine == "fForceLocalId" ) {
                    // make sure flag not already set
                    BOOST_CHECK(0 == (fFlags & CAgpToSeqEntry::fForceLocalId));
                    fFlags |= CAgpToSeqEntry::fForceLocalId;
                } else if( sLine == "fSetSeqGap" ) {
                    BOOST_CHECK(0 == (fFlags & CAgpToSeqEntry::fSetSeqGap));
                    fFlags |= CAgpToSeqEntry::fSetSeqGap;
                } else {
                    BOOST_ERROR("Unknown line in flags file (" 
                        << testInfo.m_FlagFile.GetPath() << "): " << sLine);
                }
            }
            BOOST_CHECK(flagfile.eof());
        }

        // create the object to do the actual reading
        CAgpToSeqEntry agpToSeqEntry(fFlags);

        // read the AGP
        CAgpToSeqEntry::TSeqEntryRefVec seqEntryRefVec;
        {
            CNcbiIfstream agpfile(testInfo.m_AGPFile.GetPath().c_str());
            BOOST_CHECK(agpfile);
            int iErrCode = agpToSeqEntry.ReadStream(agpfile);
            // in the future, we might have a file to allow us
            // to expect certain errors rather than assuming all errors are
            // bad.
            BOOST_CHECK_EQUAL( iErrCode, 0 );

            const string sErrorMsgs = 
                agpToSeqEntry.GetErrorHandler()->GetErrorMessage();
            if( ! sErrorMsgs.empty() ) {
                cout << "AGP errors/warnings: " << endl;
                cout << "(Ignore warnings about assuming AGP version)" << endl;
                cout << sErrorMsgs << endl;
            }

            // swap faster than assignment, and okay in this case
            seqEntryRefVec.swap( agpToSeqEntry.GetResult() );
        }

        const size_t uNumExpected = testInfo.m_vecExpectedSeqEntryFiles.size();
        const size_t uNumResulted = seqEntryRefVec.size();
        BOOST_CHECK_EQUAL( uNumResulted, uNumExpected );

        // check the results against what was expected
        for( size_t idx = 0; idx < uNumResulted; ++idx ) {

            // the SeqEntry that we got
            CConstRef<CSeq_entry> pResultingSeqEntry = seqEntryRefVec[idx];

            // determine what SeqEntry was expected
            CRef<CSeq_entry> pExpectedSeqEntry;
            if( idx < uNumExpected )
            {
                CNcbiIfstream expected_seq_entry_file( 
                    testInfo.m_vecExpectedSeqEntryFiles[idx].GetPath().c_str() );
                pExpectedSeqEntry.Reset( new CSeq_entry );
                BOOST_CHECK_NO_THROW(expected_seq_entry_file >> MSerial_AsnText >> *pExpectedSeqEntry);
            }

            // check if same (or if none was expected here)
            if( pExpectedSeqEntry.IsNull() ||
                ! pResultingSeqEntry->Equals(*pExpectedSeqEntry) ) 
            {
                BOOST_ERROR("For test " << sTestName 
                    << " on index " << idx 
                    << ", the resulting seq-entry does not match "
                    "what was expected");
                cerr << "##### Resulting Seq-entry: " << endl;
                cerr << MSerial_AsnText << *pResultingSeqEntry << endl;
            }
        }
    }
}

NCBITEST_INIT_CMDLINE(arg_descrs)
{
    arg_descrs->AddDefaultKey("test-dir", "DIRECTORY",
        "Set the root directory under which all test files can be found.",
        CArgDescriptions::eDirectory,
        "agp_seq_entry_test_cases" );
}

// This test case will automatically walk the test data directory to 
// find all the tests to do.  It is pretty much file-driven, so it
// should be quite easy to add new tests as necessary.
BOOST_AUTO_TEST_CASE(RunTests)
{
    const CArgs& args = CNcbiApplication::Instance()->GetArgs();

    const vector<string> kEmptyStringVec;

    TTestNameToInfoMap testNameToInfoMap;

    CTestNameToInfoMapLoader testInfoLoader(&testNameToInfoMap);

    CDir test_cases_dir( args["test-dir"].AsDirectory() );

    BOOST_REQUIRE_MESSAGE( test_cases_dir.IsDir(), 
        "Cannot find dir: " << test_cases_dir.GetPath() );

    cout << endl;
    cout << "#################### COLLECTING INFO ABOUT ALL TESTS" << endl;
    cout << endl;

    FindFilesInDir(
        test_cases_dir,
        kEmptyStringVec,
        kEmptyStringVec,
        testInfoLoader,
        fFF_Default | fFF_Recursive );

    cout << endl;
    cout << "#################### BASIC SANITY CHECKING OF TESTS" << endl;
    cout << endl;
    
    // validate every STestInfo
    ITERATE(TTestNameToInfoMap, name_to_info_it, testNameToInfoMap) {
        const string & sName = name_to_info_it->first;
        const STestInfo & testInfo = name_to_info_it->second;
        
        cout << "Verifying: " << sName << endl;

        BOOST_REQUIRE_MESSAGE( testInfo.m_AGPFile.Exists(),
            "AGP file does not exist: " << testInfo.m_AGPFile.GetPath() );
        for( size_t idx = 0; idx < testInfo.m_vecExpectedSeqEntryFiles.size(); ++idx ) {
            BOOST_REQUIRE_MESSAGE( 
                testInfo.m_vecExpectedSeqEntryFiles[idx].Exists(),
                "Seq-entry file with index " << idx << " does not exist");
        }
        if( ! testInfo.m_FlagFile.GetPath().empty() ) {
            BOOST_REQUIRE_MESSAGE( testInfo.m_FlagFile.Exists(),
                "Flag file specified but not found: " << testInfo.m_FlagFile.GetPath() );
        }
    }

    cout << endl;
    cout << "#################### RUNNING TESTS" << endl;
    cout << endl;

    // run the tests
    ITERATE(TTestNameToInfoMap, name_to_info_it, testNameToInfoMap) {
        const string & sName = name_to_info_it->first;
        const STestInfo & testInfo = name_to_info_it->second;
        
        cout << "Running test: " << sName << endl;

        BOOST_CHECK_NO_THROW(s_RunTest(sName, testInfo));
    }
}
