// Copyright (C) 2005 - 2021 Settlers Freaks (sf-team at siedler25.org)
//
// SPDX-License-Identifier: GPL-2.0-or-later

#include "GameWithLuaAccess.h"
#include <boost/test/unit_test.hpp>

struct LuaConstsTestsFixture : public LuaTestsFixture
{
    LuaConstsTestsFixture()
    {
        initWorld();
        // This adds an `elements` function which can iterate over a table generated by
        // table.pack (important as nil entries caused by missing constants abort regular iteration)
        executeLua(R"LUA(
            function elements(t)
                local f = function(s, i)
                    i = i + 1
                    if i <= s.n then
                        return i, s[i]
                    end
                end
                return f, t, 0
            end
        )LUA");
#if LUA_VERSION_NUM < 502
        executeLua(R"LUA(
            function table.pack(...)
                -- Returns a new table with parameters stored into an array, with field "n" being the total number of parameters
                local t = {...}
                t.n = #t
                return t
            end
        )LUA");
#endif
    }
};

/// Tests that documented constants from doc/lua/constants.md work
BOOST_FIXTURE_TEST_SUITE(LuaConstantsSuite, LuaConstsTestsFixture)

BOOST_AUTO_TEST_CASE(AllGoodTypesWork)
{
    executeLua("player = rttr:GetPlayer(0); assert(player)");
    executeLua(R"LUA(
        wares = table.pack(
            GD_BEER,
            GD_TONGS,
            GD_HAMMER,
            GD_AXE,
            GD_SAW,
            GD_PICKAXE,
            GD_SHOVEL,
            GD_CRUCIBLE,
            GD_RODANDLINE,
            GD_SCYTHE,
            GD_WATER,
            GD_CLEAVER,
            GD_ROLLINGPIN,
            GD_BOW,
            GD_BOAT,
            GD_SWORD,
            GD_IRON,
            GD_FLOUR,
            GD_FISH,
            GD_BREAD,
            GD_SHIELD,
            GD_WOOD,
            GD_BOARDS,
            GD_STONES,
            GD_GRAIN,
            GD_COINS,
            GD_GOLD,
            GD_IRONORE,
            GD_COAL,
            GD_MEAT,
            GD_HAM
        )
    )LUA");
    executeLua(R"LUA(
        for i, ware in elements(wares) do
            assert(ware ~= nil, "nil at " .. tostring(i))
            prior = player:GetNumWares(ware)
            assert(player:AddWares({[ware] = 1}))
            assert(player:GetNumWares(ware) == prior + 1)
        end
    )LUA");
}

BOOST_AUTO_TEST_CASE(AllBuildingTypesWork)
{
    executeLua("player = rttr:GetPlayer(0); assert(player)");
    executeLua(R"LUA(
        buildings = table.pack(
            BLD_HEADQUARTERS,
            BLD_BARRACKS,
            BLD_GUARDHOUSE,
            BLD_WATCHTOWER,
            BLD_FORTRESS,
            BLD_GRANITEMINE,
            BLD_COALMINE,
            BLD_IRONMINE,
            BLD_GOLDMINE,
            BLD_LOOKOUTTOWER,
            BLD_CATAPULT,
            BLD_WOODCUTTER,
            BLD_FISHERY,
            BLD_QUARRY,
            BLD_FORESTER,
            BLD_SLAUGHTERHOUSE,
            BLD_HUNTER,
            BLD_BREWERY,
            BLD_ARMORY,
            BLD_METALWORKS,
            BLD_IRONSMELTER,
            BLD_CHARBURNER,
            BLD_PIGFARM,
            BLD_STOREHOUSE,
            BLD_MILL,
            BLD_BAKERY,
            BLD_SAWMILL,
            BLD_MINT,
            BLD_WELL,
            BLD_SHIPYARD,
            BLD_FARM,
            BLD_DONKEYBREEDER,
            BLD_HARBORBUILDING
        )
    )LUA");
    executeLua(R"LUA(
        for i, bld in elements(buildings) do
            assert(bld ~= nil, "nil at " .. tostring(i))
            num = player:GetNumBuildings(bld)
            assert(num == 0 or num == 1)
        end
    )LUA");
}

BOOST_AUTO_TEST_CASE(AllJobTypesWork)
{
    executeLua("player = rttr:GetPlayer(0); assert(player)");
    executeLua(R"LUA(
        jobs = table.pack(
            JOB_HELPER,
            JOB_WOODCUTTER,
            JOB_FISHER,
            JOB_FORESTER,
            JOB_CARPENTER,
            JOB_STONEMASON,
            JOB_HUNTER,
            JOB_FARMER,
            JOB_MILLER,
            JOB_BAKER,
            JOB_BUTCHER,
            JOB_MINER,
            JOB_BREWER,
            JOB_PIGBREEDER,
            JOB_DONKEYBREEDER,
            JOB_IRONFOUNDER,
            JOB_MINTER,
            JOB_METALWORKER,
            JOB_ARMORER,
            JOB_BUILDER,
            JOB_PLANER,
            JOB_PRIVATE,
            JOB_PRIVATEFIRSTCLASS,
            JOB_SERGEANT,
            JOB_OFFICER,
            JOB_GENERAL,
            JOB_GEOLOGIST,
            JOB_SHIPWRIGHT,
            JOB_SCOUT,
            JOB_PACKDONKEY,
            JOB_BOATCARRIER,
            JOB_CHARBURNER
        )
    )LUA");
    executeLua(R"LUA(
        for i, job in elements(jobs) do
            assert(job ~= nil, "nil at " .. tostring(i))
            num = player:GetNumPeople(job)
            assert(num >= 0)
        end
    )LUA");
}

// Put the remaining ones together
BOOST_AUTO_TEST_CASE(AllOtherConstantsWork)
{
    executeLua(R"LUA(
        constants = table.pack(
            RES_IRON,
            RES_GOLD,
            RES_COAL,
            RES_GRANITE,
            RES_WATER,

            NON_AGGRESSION_PACT,
            TREATY_OF_ALLIANCE,
            DURATION_INFINITE
        )
        animals = table.pack(
            SPEC_POLARBEAR,
            SPEC_RABBITWHITE,
            SPEC_RABBITGREY,
            SPEC_FOX,
            SPEC_STAG,
            SPEC_DEER,
            SPEC_DUCK,
            SPEC_SHEEP
        )
    )LUA");
    executeLua(R"LUA(
        for i, constant in elements(constants) do
            assert(constant ~= nil, "nil at " .. tostring(i))
        end
    )LUA");
    executeLua(R"LUA(
        for i, animal in elements(animals) do
            assert(animal ~= nil, "nil at " .. tostring(i))
            rttr:GetWorld():AddAnimal(i, i, animal)
        end
    )LUA");
}

BOOST_AUTO_TEST_SUITE_END()
