/*
 *                           TERMS AND CONDITIONS
 *                                   FOR
 *                         OPEN SOURCE CODE LICENSE
 *                               Version 1.1
 * 
 * Japan Registry Services Co., Ltd. ("JPRS"), a Japanese corporation
 * having its head office at Chiyoda First Bldg. East 13F 3-8-1 Nishi-Kanda,
 * Chiyoda-ku, Tokyo 101-0065, Japan, grants you the license for open source
 * code specified in EXHIBIT A the "Code" subject to the following Terms and
 * Conditions ("OSCL").
 * 
 * 1. License Grant.
 *   JPRS hereby grants you a worldwide, royalty-free, non-exclusive
 *   license, subject to third party intellectual property claims:
 *   (a) under intellectual property rights (other than patent or
 *       trademark) licensable by JPRS to use, reproduce, modify, display,
 *       perform, sublicense and distribute the Code (or portions thereof)
 *       with or without modifications, and/or as part of a derivative work;
 *       or
 *   (b) under claims of the infringement through the making, using,
 *       offering to sell and/or otherwise disposing the JPRS Revised Code
 *       (or portions thereof);
 *   (c) the licenses granted in this Section 1(a) and (b) are effective on
 *       the date JPRS first distributes the Code to you under the terms of
 *       this OSCL;
 *   (d) Notwithstanding the above stated terms, no patent license is
 *       granted:
 *       1)  for a code that you delete from the Code;
 *       2)  separate from the Code; or
 *       3)  for infringements caused by:
 *            i) modification of the Code; or
 *           ii) combination of the Code with other software or devices.
 * 
 * 2. Consents.
 *   You agree that:
 *   (a) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of the Code you distribute;
 *   (b) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of binary form of the Code in the
 *       documentation and/or other materials provided with the distribution;
 *   (c) you may not offer or impose any terms on any source code version
 *       that alters or restricts the applicable version of this OSCL or
 *       the recipients' rights hereunder.
 *   (d) If the terms and conditions are set forth in EXHIBIT A, you must
 *       comply with those terms and conditions.
 * 
 * 3. Proprietary Information.
 *   All trademarks, service marks, patents, copyrights, trade secrets, and
 *   other proprietary rights in or related to the Code are and will remain
 *   the exclusive property of JPRS or its licensors, whether or not
 *   specifically recognized or perfected under local law except specified
 *   in this OSCL; provided however you agree and understand that the JPRS
 *   name may not be used to endorse or promote this Code without prior
 *   written approval of JPRS.
 * 
 * 4. WARRANTY DISCLAIMER.
 *   JPRS MAKES NO REPRESENTATIONS AND WARRANTIES REGARDING THE USE OF THE
 *   CODE, NOR DOES JPRS MAKE ANY REPRESENTATIONS THAT THE CODE WILL BECOME
 *   COMMERCIALLY AVAILABLE. JPRS, ITS AFFILIATES, AND ITS SUPPLIERS DO NOT
 *   WARRANT OR REPRESENT THAT THE CODE IS FREE OF ERRORS OR THAT THE CODE
 *   IS SUITABLE FOR TRANSLATION AND/OR LOCALIZATION. THE CODE IS PROVIDED
 *   ON AN "AS IS" BASIS AND JPRS AND ITS SUPPLIERS HAVE NO OBLIGATION TO
 *   CORRECT ERRORS OR TO SUPPORT THE CODE UNDER THIS OSCL FOR ANY REASON.
 *   TO THE FULL EXTENT PERMITTED BY LAW, ALL OBLIGATIONS ARE HEREBY
 *   EXCLUDED WHETHER EXPRESS, STATUTORY OR IMPLIED UNDER LAW, COURSE OF
 *   DEALING, CUSTOM, TRADE USAGE, ORAL OR WRITTEN STATEMENT OR OTHERWISE,
 *   INCLUDING BUT NOT LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
 *   OR FITNESS FOR A PARTICULAR PURPOSE CONCERNING THE CODE.
 * 
 * 5. NO LIABILITY.
 *   UNDER NO CIRCUMSTANCES SHALL JPRS AND/OR ITS AFFILIATES, LICENSORS, OR
 *   REPRESENTATIVES BE LIABLE FOR ANY DAMAGES INCLUDING BUT NOT LIMITED TO
 *   CONSEQUENTIAL, INDIRECT, SPECIAL, PUNITIVE OR INCIDENTAL DAMAGES,
 *   WHETHER FORESEEABLE OR UNFORESEEABLE, BASED ON YOUR CLAIMS, INCLUDING,
 *   BUT NOT LIMITED TO, CLAIMS FOR LOSS OF DATA, GOODWILL, PROFITS, USE OF
 *   MONEY, INTERRUPTION IN USE OR AVAILABILITY OF DATA, STOPPAGE, IMPLIED
 *   WARRANTY, BREACH OF CONTRACT, MISREPRESENTATION, NEGLIGENCE, STRICT
 *   LIABILITY IN TORT, OR OTHERWISE.
 * 
 * 6. Indemnification.
 *   You hereby agree to indemnify, defend, and hold harmless JPRS for any
 *   liability incurred by JRPS due to your terms of warranty, support,
 *   indemnity, or liability offered by you to any third party.
 * 
 * 7. Termination.
 * 7.1 This OSCL shall be automatically terminated in the events that:
 *   (a) You fail to comply with the terms herein and fail to cure such
 *       breach within 30 days of becoming aware of the breach;
 *   (b) You initiate patent or copyright infringement litigation against
 *       any party (including a cross-claim or counterclaim in a lawsuit)
 *       alleging that the Code constitutes a direct or indirect patent or
 *       copyright infringement, in such case, this OSCL to you shall
 *       terminate as of the date such litigation is filed;
 * 7.2 In the event of termination under Sections 7.1(a) or 7.1(b) above,
 *     all end user license agreements (excluding distributors and
 *     resellers) which have been validly granted by You or any distributor
 *     hereunder prior to termination shall survive termination.
 *
 * 
 * 8. General.
 *   This OSCL shall be governed by, and construed and enforced in
 *   accordance with, the laws of Japan. Any litigation or arbitration
 *   between the parties shall be conducted exclusively in Tokyo, Japan
 *   except written consent of JPRS provides other venue.
 * 
 * 
 *                                EXHIBIT A
 * 
 * The original open source code of idnkit-2 is idnkit-1.0 developed and
 * conceived by Japan Network Information Center ("JPNIC"), a Japanese
 * association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan, and JPRS modifies above original code
 * under following Terms and Conditions set forth by JPNIC.
 * 
 *                                  JPNIC
 * 
 * Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved.
 * 
 * By using this file, you agree to the terms and conditions set forth bellow.
 * 
 *                       LICENSE TERMS AND CONDITIONS
 * 
 * The following License Terms and Conditions apply, unless a different
 * license is obtained from Japan Network Information Center ("JPNIC"),
 * a Japanese association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan.
 * 
 * 1. Use, Modification and Redistribution (including distribution of any
 *    modified or derived work) in source and/or binary forms is permitted
 *    under this License Terms and Conditions.
 * 
 * 2. Redistribution of source code must retain the copyright notices as they
 *    appear in each source code file, this License Terms and Conditions.
 * 
 * 3. Redistribution in binary form must reproduce the Copyright Notice,
 *    this License Terms and Conditions, in the documentation and/or other
 *    materials provided with the distribution. For the purposes of binary
 *    distribution the "Copyright Notice" refers to the following language:
 *    "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * 
 * 4. The name of JPNIC may not be used to endorse or promote products
 *    derived from this Software without specific prior written approval of
 *    JPNIC.
 * 
 * 5. Disclaimer/Limitation of Liability: THIS SOFTWARE IS PROVIDED BY JPNIC
 *    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL JPNIC BE LIABLE
 *    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 *    OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *    ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * 
 *                        JPRS Public License Notice
 *                                   For
 *                                idnkit-2.
 * 
 * The contents of this file are subject to the Terms and Conditions for
 * the Open Source Code License (the "OSCL"). You may not use this file
 * except in compliance with above terms and conditions. A copy of the OSCL
 * is available at <http://jprs.co.jp/idn/>.
 * The JPRS Revised Code is idnkit-2.
 * The Initial Developer of the JPRS Revised Code is Japan Network
 * Information Center ("JPNIC"), a Japanese association,
 * Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda, Chiyoda-ku, Tokyo
 * 101-0047, Japan.
 * "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * "Copyright (c) 2010-2012 Japan Registry Services Co., Ltd.  All rights reserved."
 * Contributor(s): ______________________________________.
 * 
 * If you wish to allow use of your version of this file only under the
 * above License(s) and not to allow others to use your version of this
 * file, please indicate your decision by deleting the relevant provisions
 * above and replacing them with the notice and other provisions required
 * by the above License(s). If you do not delete the relevant provisions,
 * a recipient may use your version of this file under either the above
 * License(s).
 */

#include <config.h>

#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#include <idn/assert.h>
#include <idn/debug.h>
#include <idn/logmacro.h>
#include <idn/result.h>
#include <idn/strhash32.h>
#include <idn/utf32.h>

/*
 * Initially, the number of hash buckets is INITIAL_HASH_SIZE.
 * As the more elements are put in the hash, the number of elements
 * per bucket will exceed THRESHOLD eventually.  When it happens,
 * the number of buckets will be multiplied by FACTOR.
 */
#define INITIAL_HASH_SIZE	67
#define FACTOR			7
#define THRESHOLD		5

#define HASH_MULT		31

typedef struct strhash32_entry {
	struct strhash32_entry *next;
	unsigned long hash_value;
	unsigned long *key;
	void *value;
} strhash32_entry_t;

struct idn__strhash32 {
	int nbins;
	int nelements;
	strhash32_entry_t **bins;
};

static unsigned long	hash_value(const unsigned long *key);
static strhash32_entry_t
			*find_entry(strhash32_entry_t *entry,
				    const unsigned long *key,
				    unsigned long hash);
static strhash32_entry_t
			*new_entry(const unsigned long *key, void *value);
static idn_result_t	expand_bins(idn__strhash32_t hash, int new_size);

idn_result_t
idn__strhash32_create(idn__strhash32_t *hashp) {
	idn__strhash32_t hash;
	idn_result_t r = idn_success;

	TRACE(("idn__strhash32_create()\n"));

	assert(hashp != NULL);

	*hashp = NULL;

	if ((hash = malloc(sizeof(struct idn__strhash32))) == NULL) {
		WARNING(("idn__strhash32_create: malloc failed (hash)\n"));
		return (idn_nomemory);
	}
	hash->nbins = 0;
	hash->nelements = 0;
	hash->bins = NULL;
	if ((r = expand_bins(hash, INITIAL_HASH_SIZE)) != idn_success) {
		WARNING(("idn__strhash32_create: malloc failed (bins)\n"));
		free(hash);
		return (r);
	}

	*hashp = hash;

	return (idn_success);
}

void
idn__strhash32_destroy(idn__strhash32_t hash,
		       idn__strhash32_destroyproc_t proc) {
	int i;

	assert(hash != NULL && hash->bins != NULL);

	for (i = 0; i < hash->nbins; i++) {
		strhash32_entry_t *bin = hash->bins[i];
		strhash32_entry_t *next;

		while (bin != NULL) {
			next = bin->next;
			if (proc != NULL)
				(*proc)(bin->value);
			free(bin->key);
			free(bin);
			bin = next;
		}
	}
	free(hash->bins);
	free(hash);
}

idn_result_t
idn__strhash32_put(idn__strhash32_t hash, const unsigned long *key,
		   void *value) {
	unsigned long h, h_index;
	strhash32_entry_t *entry;

	assert(hash != NULL && key != NULL);

	h = hash_value(key);
	h_index = h % hash->nbins;

	if ((entry = find_entry(hash->bins[h_index], key, h)) != NULL) {
		/* Entry exists.  Replace the value. */
		entry->value = value;
	} else {
		/* Create new entry. */
		if ((entry = new_entry(key, value)) == NULL) {
			return (idn_nomemory);
		}
		/* Insert it to the list. */
		entry->next = hash->bins[h_index];
		hash->bins[h_index] = entry;
		hash->nelements++;

		if (hash->nelements > hash->nbins * THRESHOLD) {
			idn_result_t r = idn_success;
			r = expand_bins(hash, hash->nbins * FACTOR);
			if (r != idn_success) {
				TRACE(("idn__strhash32_put: hash table "
					"expansion failed\n"));
			}
		}
	}

	return (idn_success);
}

void *
idn__strhash32_get(idn__strhash32_t hash, const unsigned long *key) {
	unsigned long h;
	strhash32_entry_t *entry;

	assert(hash != NULL && key != NULL);

	h = hash_value(key);
	entry = find_entry(hash->bins[h % hash->nbins], key, h);
	if (entry == NULL)
		return (NULL);

	return (entry->value);
}

int
idn__strhash32_exists(idn__strhash32_t hash, const unsigned long *key) {
	unsigned long h;

	assert(hash != NULL && key != NULL);

	h = hash_value(key);
	return (find_entry(hash->bins[h % hash->nbins], key, h) != NULL);
}

static unsigned long
hash_value(const unsigned long *key) {
	unsigned long h = 0;
	const unsigned long *p = key;
	unsigned long c;
	unsigned long c0, c1, c2, c3;

	while ((c = *p++) != '\0') {
		c3 = (c >> 24) & 0xff;
		c2 = (c >> 16) & 0xff;
		c1 = (c >>  8) & 0xff;
		c0 = (c      ) & 0xff;
		h = h * HASH_MULT + c3;
		h = h * HASH_MULT + c2;
		h = h * HASH_MULT + c1;
		h = h * HASH_MULT + c0;
	}
	return (h);
}

static strhash32_entry_t *
find_entry(strhash32_entry_t *entry, const unsigned long *key,
	   unsigned long hash) {
	assert(key != NULL);

	while (entry != NULL) {
		if (entry->hash_value == hash &&
		    idn__utf32_strcmp(key, entry->key) == 0) {
			return (entry);
		}
		entry = entry->next;
	}
	return (NULL);
}

static strhash32_entry_t *
new_entry(const unsigned long *key, void *value) {
	strhash32_entry_t *entry;

	assert(key != NULL);

	entry = malloc(sizeof(strhash32_entry_t));
	if (entry == NULL)
		return (NULL);

	entry->key = idn__utf32_strdup(key);
	if (entry->key == NULL) {
		free(entry);
		return (NULL);
	}

	entry->next = NULL;
	entry->hash_value = hash_value(key);
	entry->value = value;

	return (entry);
}

static idn_result_t
expand_bins(idn__strhash32_t hash, int new_size) {
	strhash32_entry_t **old_bins, **new_bins;
	int old_size;
	int old_index, new_index;

	new_bins = malloc(sizeof(strhash32_entry_t *) * new_size);
	if (new_bins == NULL)
		return (idn_nomemory);

	memset(new_bins, 0, sizeof(strhash32_entry_t *) * new_size);

	old_bins = hash->bins;
	old_size = hash->nbins;
	for (old_index = 0; old_index < old_size; old_index++) {
		strhash32_entry_t *entries = old_bins[old_index];

		while (entries != NULL) {
			strhash32_entry_t *e = entries;

			/* Remove the top element from the linked list. */
			entries = entries->next;

			/* ..and move to the new hash. */
			new_index = e->hash_value % new_size;
			e->next = new_bins[new_index];
			new_bins[new_index] = e;
		}
	}

	hash->nbins = new_size;
	hash->bins = new_bins;

	if (old_bins != NULL)
		free(old_bins);

	return (idn_success);
}
