;;; Guile-Git --- GNU Guile bindings of libgit2
;;; Copyright © 2017, 2021 Ludovic Courtès <ludo@gnu.org>
;;;
;;; This file is part of Guile-Git.
;;;
;;; Guile-Git is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or
;;; (at your option) any later version.
;;;
;;; Guile-Git is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with Guile-Git.  If not, see <http://www.gnu.org/licenses/>.

(define-module (git errors)
  #:use-module (system foreign)
  #:use-module (git bindings)
  #:use-module (git structs)
  #:export (;; Error codes.
            GIT_OK
            GIT_ERROR
            GIT_ENOTFOUND
            GIT_EEXISTS
            GIT_EAMBIGUOUS
            GIT_EBUFS
            GIT_EUSER
            GIT_EBAREREPO
            GIT_EUNBORNBRANCH
            GIT_EUNMERGED
            GIT_ENONFASTFORWARD
            GIT_EINVALIDSPEC
            GIT_ECONFLICT
            GIT_ELOCKED
            GIT_EMODIFIED
            GIT_EAUTH
            GIT_ECERTIFICATE
            GIT_EAPPLIED
            GIT_EPEEL
            GIT_EEOF
            GIT_EINVALID
            GIT_EUNCOMMITTED
            GIT_EDIRECTORY
            GIT_EMERGECONFLICT
            GIT_PASSTHROUGH
            GIT_ITEROVER

            ;; Error classes.
            GITERR_NONE
            GITERR_NOMEMORY
            GITERR_OS
            GITERR_INVALID
            GITERR_REFERENCE
            GITERR_ZLIB
            GITERR_REPOSITORY
            GITERR_CONFIG
            GITERR_REGEX
            GITERR_ODB
            GITERR_INDEX
            GITERR_OBJECT
            GITERR_NET
            GITERR_TAG
            GITERR_TREE
            GITERR_INDEXER
            GITERR_SSL
            GITERR_SUBMODULE
            GITERR_THREAD
            GITERR_STASH
            GITERR_CHECKOUT
            GITERR_FETCHHEAD
            GITERR_MERGE
            GITERR_SSH
            GITERR_FILTER
            GITERR_REVERT
            GITERR_CALLBACK
            GITERR_CHERRYPICK
            GITERR_DESCRIBE
            GITERR_REBASE
            GITERR_FILESYSTEM
            GITERR_PATCH
            GITERR_WORKTREE
            GITERR_SHA1
            GITERR_HTTP
            GITERR_INTERNAL

            clear-git-error!)
  #:re-export (raise-git-error))

;; Error codes, classes, and functions defined in <git2/errors.h>.

(define GIT_OK          0)  ;No error

(define GIT_ERROR       -1)  ;Generic error
(define GIT_ENOTFOUND   -3)  ;Requested object could not be found
(define GIT_EEXISTS     -4)  ;Object exists preventing operation
(define GIT_EAMBIGUOUS  -5)  ;More than one object matches
(define GIT_EBUFS       -6)  ;Output buffer too short to hold data

;; GIT_EUSER is a special error that is never generated by libgit2 code.  You
;; can return it from a callback (e.g to stop an iteration) to know that it
;; was generated by the callback and not by libgit2.
(define GIT_EUSER      -7)

(define GIT_EBAREREPO        -8)  ;Operation not allowed on bare repository
(define GIT_EUNBORNBRANCH    -9)  ;HEAD refers to branch with no commits
(define GIT_EUNMERGED        -10)  ;Merge in progress prevented operation
(define GIT_ENONFASTFORWARD  -11)  ;Reference was not fast-forwardable
(define GIT_EINVALIDSPEC     -12)  ;Name/ref spec was not in a valid format
(define GIT_ECONFLICT        -13)  ;Checkout conflicts prevented operation
(define GIT_ELOCKED          -14)  ;Lock file prevented operation
(define GIT_EMODIFIED        -15)  ;Reference value does not match expected
(define GIT_EAUTH            -16)  ;Authentication error
(define GIT_ECERTIFICATE     -17)  ;Server certificate is invalid
(define GIT_EAPPLIED         -18)  ;Patch/merge has already been applied
(define GIT_EPEEL            -19)  ;The requested peel operation is not possible
(define GIT_EEOF             -20)  ;Unexpected EOF
(define GIT_EINVALID         -21)  ;Invalid operation or input
(define GIT_EUNCOMMITTED     -22)  ;Uncommitted changes in index prevented operation
(define GIT_EDIRECTORY       -23)  ;The operation is not valid for a directory
(define GIT_EMERGECONFLICT   -24)  ;A merge conflict exists and cannot continue

(define GIT_PASSTHROUGH      -30)  ;Internal only
(define GIT_ITEROVER         -31)  ;Signals end of iteration with iterator

;; Error classes.
(define GITERR_NONE 0)
(define GITERR_NOMEMORY 1)
(define GITERR_OS 2)
(define GITERR_INVALID 3)
(define GITERR_REFERENCE 4)
(define GITERR_ZLIB 5)
(define GITERR_REPOSITORY 6)
(define GITERR_CONFIG 7)
(define GITERR_REGEX 8)
(define GITERR_ODB 9)
(define GITERR_INDEX 10)
(define GITERR_OBJECT 11)
(define GITERR_NET 12)
(define GITERR_TAG 13)
(define GITERR_TREE 14)
(define GITERR_INDEXER 15)
(define GITERR_SSL 17)
(define GITERR_SUBMODULE 18)
(define GITERR_THREAD 19)
(define GITERR_STASH 20)
(define GITERR_CHECKOUT 21)
(define GITERR_FETCHHEAD 22)
(define GITERR_MERGE 23)
(define GITERR_SSH 24)
(define GITERR_FILTER 25)
(define GITERR_REVERT 26)
(define GITERR_CALLBACK 26)
(define GITERR_CHERRYPICK 27)
(define GITERR_DESCRIBE 28)
(define GITERR_REBASE 29)
(define GITERR_FILESYSTEM 30)
(define GITERR_PATCH 31)
(define GITERR_WORKTREE 32)
(define GITERR_SHA1 33)
(define GITERR_HTTP 34)
(define GITERR_INTERNAL 35)


(define clear-git-error!
  (let ((proc (libgit2->procedure void "giterr_clear" '())))
    (lambda ()
      "Clear the last Git error."
      (proc))))
