/* $Id: xkbseldb.c,v 1.4 1999/07/24 15:12:25 stano Exp $

   xkbsel database management

   (C) 1999 Stanislav Meduna <stano@eunet.sk>
*/

#include <config.h>

#include <xkbsel.h>
#include <dbaccess.h>
#include <compile.h>
#include <configuration.h>

#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <libintl.h>
#include <limits.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>
#include <locale.h>

#define _(s) gettext(s)

static void usage(void);
static void help(void);

static int rebuild(int user);
static int list(int user);
static int missing_defs(int user);
static int check_config();
static int compile_configured(int user);

static int test(const char *arg);

static int list_expl = 0;
static int list_brief = 0;

int main(int argc, char **argv)
{
	int c;

	int oper = 0;
	int res;

	char test_arg[PATH_MAX+1];

	setlocale(LC_ALL, "");
	textdomain(PACKAGE);

	while(1)
	{
		static struct option long_options[] =
		{
        	     { "rebuild-system", 0, 0, 'R' },
        	     { "rebuild-user", 0, 0, 'r' },
        	     { "missing-system", 0, 0, 'M' },
        	     { "missing-user", 0, 0, 'm' },
        	     { "list-system", 0, 0, 'L' },
        	     { "list-user", 0, 0, 'l' },
        	     { "check", 0, 0, 'c' },
        	     { "brief", 0, 0, 'b' },
        	     { "explicit", 0, 0, 'x' },
        	     { "verbose", 0, 0, 'v' },
        	     { "debug", 0, 0, 0 },
        	     { "help", 0, 0, 0 },
        	     { "version", 0, 0, 0 },
        	     { "test", 1, 0, 't' },
		     { 0, 0, 0, 0 }
		};

		int option_index = 0;

		c = getopt_long(argc, argv, "rRmMlLcvxbt:", long_options, &option_index);
		if (c == -1)
			break;

		switch(c)
		{
		case 0:
			/* Long option without corresponding short one */
			if (!strcmp("help", long_options[option_index].name))
			{
				help();
				exit(0);
			}
			else if (!strcmp("version", long_options[option_index].name))
			{
				printf("xkbseldb (xkbsel) %s\n", VERSION);
				exit(0);
			}
			if (!strcmp("debug", long_options[option_index].name))
			{
				flag_debug = 1;
			}
			else
			{
				usage();
				exit(0);
			}
			break;

		case 'r':
		case 'R':
		case 'm':
		case 'M':
		case 'l':
		case 'L':
		case 'c':
		case 't':
			if (oper)
			{
				fprintf(stderr, _("Options -r, -R, -m, -M, -l, -L and -c are mutually exclusive\n"));
				usage();
				exit(1);
			}
			oper = c;

			if (oper == 't')
				strcpy(test_arg, optarg);
			break;

		case 'x':
			list_expl = 1;
			break;

		case 'b':
			list_brief = 1;
			break;

		case 'v':
			flag_verbose = 1;
			break;

		case '?':
			{
				usage();
				exit(1);
			}
			break;

		default:
			break;
		}
	}

	if (! oper)
	{
		fprintf(stderr, _("One of -r, -R, -m, -M, -l, -L and -c options must be specified\n"));
		usage();
		exit(1);
	}

	read_config();

	switch(oper)
	{
	case 'R':
		res = rebuild(0);
		break;
	case 'r':
		res = rebuild(1);
		break;
	case 'M':
		res = missing_defs(0);
		break;
	case 'm':
		res = missing_defs(1);
		break;
	case 'l':
		res = list(1);
		break;
	case 'L':
		res = list(0);
		break;
	case 'c':
		res = check_config();
		break;
	case 't':
		res = test(test_arg);
		break;
	}

	if (res < 0)
		exit(2);

	exit(0);
}


static void usage(void)
{
	fprintf(stderr, _("Usage: xkbseldb -r|-R|-m|-M|-l|-L|-c [OPTION...]\n"));
}


static void help(void)
{
	usage();

	printf(_("Manage xkbsel databases\n"));

	help_line("r", "rebuild-user",   _("rebuild user database"));
	help_line("R", "rebuild-system", _("rebuild system database (must be root)"));
	help_line("m", "missing-user",   _("output redefinition template for user"));
	help_line("M", "missing-system", _("output redefinition template for system"));

	help_line("l", "list-user",      _("list the content of the database"));
	help_line("L", "list-system",    _("as --list, but ignore the user definitions"));

	help_line("c", "check",          _("check the configuration"));

	help_line("x", "explicit",       _("with -l/-L: list only data changed by xkbsel"));
	help_line("b", "brief",          _("with -l/-L: brief listing (names only)"));
	help_line("v", "verbose",        _("be verbose"));
	help_line("",  "debug",          _("enter debug mode"));

	help_line("",  "help",           _("display this help and exit"));
	help_line("",  "version",        _("output version information and exit"));
	printf(_("Report bugs to <stano@trillian.eunet.sk>\n"));
}


static int rebuild(int user)
{
	int err;

	if (user)
	{
		err = create_user_dirs();
		if (err < 0)
			return err;
	}

	err = remove_compiled_maps(user);
	if (err < 0)
		return err;

	err = rebuild_db(user);
	if (err < 0)
		return err;

	err = compile_configured(user);
	if (err < 0)
		return err;

	return err;
}

static int list(int user)
{
	int i;
	int err;
	int found;

	char **names=0;
        int nnames=0, allocnames=0;

	db_record_t rec;

	err = open_db();
	if (err < 0)
		return err;

	list_start(&names, &nnames, &allocnames);

	err = traverse_db(user, 1, &rec, &found);

	while (! err && found)
	{
		if (! list_expl || 
		    rec.descr_src > X11_DISTR ||
		    rec.keymap_src > X11_DISTR ||
		    rec.pixmap_src > X11_DISTR)
		{
			err = list_append(&names, &nnames, &allocnames, rec.map_name);
                	if (err < 0)
				return err;

		}

		err = traverse_db(user, 0, &rec, &found);
	}

	list_sort(&names, &nnames, &allocnames);

	for (i=0; i < nnames; i++)
	{
		if (list_brief)
			printf("%s", names[i]);
		else
		{
			err = read_db_name(names[i], &rec, &found);
			if (err)
				return err;

			if (! found)
			{
		                fprintf(stderr, _("map %s not found in the database\n"), names[i]);
				return -1;
			}

			print_record(stdout, &rec);
		}

		printf("\n");
	}

	list_free(&names, &nnames, &allocnames);

	close_db();

	return err;
}

static int missing_defs(int user)
{
	char **names=0;
        int nnames=0, allocnames=0;
        int i;
	int err;
	int found;
	db_record_t rec;
	char cur_name[MAX_DATA_PATH_LEN+1];

	err = open_db();
	if (err < 0)
		return err;

	/* Get a list of X11 maps not having pixmap or description */
	list_start(&names, &nnames, &allocnames);

	err = traverse_db(user, 1, &rec, &found);

	while (! err && found)
	{
		int userdef = (rec.descr_src >= USER ||
				rec.keymap_src >= USER ||
				rec.pixmap_src >= USER);

		int somedflt = (rec.descr_src == DEFAULT ||
				rec.pixmap_src == DEFAULT);

		if ((user && userdef && somedflt) ||
		    (! user && ! userdef && somedflt))
		{
			err = list_append(&names, &nnames, &allocnames, rec.map_name);
                	if (err < 0)
				return err;
		}

		err = traverse_db(user, 0, &rec, &found);
	}


	/* Sort it so we don't have to switch between files too often */
	list_sort(&names, &nnames, &allocnames);

	if (nnames > 0)
	{
		time_t ct = time(NULL);

		char *un = getenv("USER");

		printf("// Redefinition file automatically generated by %s %s\n", PACKAGE, VERSION);
		if (user && un != NULL)
			printf("// User: %s    Date: %s", un, ctime(&ct));
		else
			printf("// Date: %s", ctime(&ct));

		printf("\n");
	}

	/* Iterate through the list and write templates */
	*cur_name = 0;
	for (i=0; i < nnames; i++)
	{
		int found;
		char *p1, *p2;

		err = read_db_name(names[i], &rec, &found);
		if (err)
			return err;

		if (! found)
		{
	                fprintf(stderr, _("map %s not found in the database\n"), names[i]);
			return -1;
		}

		if (strcmp(cur_name, rec.keymap))
		{
			printf("//%%xkb_file%% \"%s\"\n", rec.keymap);
			strcpy(cur_name, rec.keymap);
		}

		p1 = strchr(names[i], '(');
		p2 = strchr(names[i], ')');
		if (p1 == NULL || p2 == NULL || p1 > p2 || p2 - p1 < 2)
		{
	                fprintf(stderr, _("invalid map name %s - doesn't contain ()\n"), names[i]);
			return -1;
		}

		*p2 = 0;
		printf("xkb_keymap \"%s\" {\n", p1+1);
		if (rec.descr_src == DEFAULT)
			printf("\t//%%xkb_description%% \"\"\n");
		if (rec.pixmap_src == DEFAULT)
			printf("\t//%%xkb_pixmap%% \"unknown.xpm\"\n");
		printf("}\n\n");

	}

	list_free(&names, &nnames, &allocnames);


	close_db();


	return -1;
}

static int check_config()
{
	return print_config();
}

static int compile_configured(int user)
{
	int i, err;
	char *src;

	err = open_db();
	if (err < 0)
		return err;

	for (i=0; i < n_cfg_maps; i++)
	{
		err = compile_map_if_needed(cfg_maps[i].map_name, &src);
		if (err < 0)
			break;

		if (src != NULL)
			free(src);
	}

	close_db();

	return 0;
}

static int test(const char *arg)
{
	open_db();
	install_map(arg);
	close_db();
	return 0;
}

