/* $Id: configuration.c,v 1.4 1999/07/25 10:57:22 stano Exp $

   Configuration

  (C) 1999 Stanislav Meduna <stano@eunet.sk>
*/

#include <xkbsel.h>
#include <dbaccess.h>
#include <configuration.h>
#include <utils.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <stdio.h>
#include <libintl.h>
#include <errno.h>
#include <locale.h>

#define _(s) dgettext(PACKAGE, s)

const char *xkbsel_package = PACKAGE;
const char *xkbsel_version = VERSION;

char *x11_xkb_dir     = X11_XKB_DIR;
char *x11_keymap_dir  = NULL;

char *sys_dir         = SYS_DIR;
char *sys_keymap_dir  = NULL;
char *sys_redef_dir   = SYS_REDEF_DIR;
char *sys_pixmap_dir  = SYS_PIXMAP_DIR;
char *sys_xkm_dir     = SYS_XKM_DIR;
char *sys_db          = SYS_DB;
char *sys_config      = SYS_CONFIG;

char *user_dir        = NULL;
char *user_keymap_dir = NULL;
char *user_redef_dir  = NULL;
char *user_pixmap_dir = NULL;
char *user_xkm_dir    = NULL;
char *user_db         = NULL;
char *user_config     = NULL;

char *display_name    = NULL;

char *locale_lang     = NULL;
char *locale_territ   = NULL;

/* A list of all configured maps */
sel_info_t *cfg_maps = NULL;
int        n_cfg_maps = 0;
static int a_cfg_maps = 0;

/* A list of maps valid for the specified locale (for GUI selectors) */
sel_info_t *locale_maps = NULL;
int        n_locale_maps = 0;
static int a_locale_maps = 0;

/* Data for our locale and the default one */
static config_entry_t *locale_mapset = NULL; 
static config_entry_t *dflt_mapset = NULL; 


int flag_verbose = 0;
int flag_debug = 0;
int xkbcomp_has_bug = -1;

extern FILE *configparserin;
extern int configparserlex();

extern int configparserlineno;
static const char *cur_cfg_file;


/* === Forward declarations === */

static void print_ex(const char *desc, const char *fn, int dir);

static int determine_locale();
static int set_paths();
static int parse_config();
static int set_x_data();

static int parse_shortcut_map(config_entry_t *entry);
static int parse_mapset_map(config_entry_t *entry);

static int add_maps(sel_info_t **info, int *n, int *an, const char *shortcut, const char *map_name);

/* === Public interface === */

/* Fill in the configuration */
int read_config(void)
{
	int r;

	/* Set paths */
	r = set_paths();
	if (r < 0)
		return r;

	/* Determine locale */
	r = determine_locale();
	if (r < 0)
		return r;

	/* Parse the configuration file */
	r = parse_config();
	if (r < 0)
		return r;

	/* Display etc. */
	r = set_x_data();
	if (r < 0)
		return r;

	return 0;
}

/* Print the actual configuration */
int print_config(void)
{
	int i;

	printf(_("Actual configuration:\n"));
	printf("\n");

	printf(_("System data:\n"));
	print_ex(_("X11 keymaps"),   x11_keymap_dir, 1);
	print_ex(_("Keymaps"),       sys_keymap_dir, 1);
	print_ex(_("Redefinitions"), sys_redef_dir,  1);
	print_ex(_("Pixmaps"),       sys_pixmap_dir, 1);
	print_ex(_("Compiled maps"), sys_xkm_dir,    1);
	print_ex(_("Database"),      sys_db,         2);
	print_ex(_("Config"),        sys_config,     0);
	printf("\n");

	printf(_("User data:\n"));
	print_ex(_("Keymaps"),       user_keymap_dir, 1);
	print_ex(_("Redefinitions"), user_redef_dir,  1);
	print_ex(_("Pixmaps"),       user_pixmap_dir, 1);
	print_ex(_("Compiled maps"), user_xkm_dir,    1);
	print_ex(_("Database"),      user_db,         2);
	print_ex(_("Config"),        user_config,     0);

	printf("\n");
	print_ex(_("Display"),       display_name,   -1);
	print_ex(_("Language"),      locale_lang,    -1);
	print_ex(_("Territory"),     locale_territ,  -1);
	printf("\n");

	check_xkbcomp_bug();

	if (xkbcomp_has_bug == 1)
		printf(_("xkbcomp has a bug - will work around it\n"));
	else if (xkbcomp_has_bug == 0)
		printf(_("xkbcomp is OK\n"));
	else
		printf(_("cannot check xkbcomp for bugs - expect problems\n"));

	printf("\n");
	printf(_("Configured maps for selectors:\n"));
	for (i=0; i < n_locale_maps; i++)
	{
		print_ex(locale_maps[i].shortcut, locale_maps[i].map_name, -1);
	}
	printf("\n");
	return 0;
}

/* Create user directories */
int create_user_dirs()
{
	int err;
	char buf[PATH_MAX+1];

	err = create_dir(user_dir);
	if (err)
		return err;

	err = create_dir(user_keymap_dir);
	if (err)
		return err;

	err = create_dir(user_redef_dir);
	if (err)
		return err;

	err = create_dir(user_pixmap_dir);
	if (err)
		return err;

	err = create_dir(user_xkm_dir);
	if (err)
		return err;


	sprintf(buf, "%s/compat", user_dir);
	err = create_dir(buf);
	if (err)
		return err;

	sprintf(buf, "%s/geometry", user_dir);
	err = create_dir(buf);
	if (err)
		return err;

	sprintf(buf, "%s/keycodes", user_dir);
	err = create_dir(buf);
	if (err)
		return err;

	sprintf(buf, "%s/rules", user_dir);
	err = create_dir(buf);
	if (err)
		return err;

	sprintf(buf, "%s/semantics", user_dir);
	err = create_dir(buf);
	if (err)
		return err;

	sprintf(buf, "%s/symbols", user_dir);
	err = create_dir(buf);
	if (err)
		return err;

	sprintf(buf, "%s/types", user_dir);
	err = create_dir(buf);
	if (err)
		return err;

	return 0;
}

/* Check for xkbcomp bug and save result in xkb_comp_has_bug */
void check_xkbcomp_bug()
{
	/* The stock version of xkbcomp has a bug, where the -I
	   option causes an immediate exit. Check whether this
	   bug exists here.

	   The check is simple - pick a command that gives
	   an exit status 0. xkbcomp with a bug will end
	   with 1.
	*/

	char buf[256];
	int r;
	FILE *fp;

	xkbcomp_has_bug = -1;

	sprintf(buf, "%s -I. -help >/dev/null 2>/dev/null", XKBCOMP);

	fp = popen(buf, "r");

	r = pclose(fp);

	if (WEXITSTATUS(r) == 1)
		xkbcomp_has_bug = 1;
	else if (WEXITSTATUS(r) == 0)
		xkbcomp_has_bug = 0;
}

/* Take a shortcut and fill a map name */
int get_locale_map(const char *shortcut, sel_info_t *info, int *idx, int *found)
{
	int i;

	*found = 0;

	for (i=0; i < n_locale_maps; i++)
		if (! strcmp(locale_maps[i].shortcut, shortcut))
		{
			if (info != NULL)
				*info = locale_maps[i];

			*found = 1;
			*idx = i;
			break;
		}

	return 0;
}

/* Take a shortcut and fill a map name */
int get_cfg_map(const char *shortcut, sel_info_t *info, int *idx, int *found)
{
	int i;

	*found = 0;

	for (i=0; i < n_cfg_maps; i++)
		if (! strcmp(cfg_maps[i].shortcut, shortcut))
		{
			if (info != NULL)
				*info = cfg_maps[i];

			*found = 1;
			*idx = i;
			break;
		}

	return 0;
}


/* lex callback, called once for found entry
   The entry was allocated via malloc and should
   be freed by the caller.
*/
void config_entry_cb(config_entry_t *entry)
{
	int sec_err = 0;
	int ent_err = 0;

	static int err_reported = 0;

	if (flag_debug)
	{
		int i;

		fprintf(stderr, _("Read configuration line:\n"));

		fprintf(stderr, _("Section:      %s\n"), entry->section);
		fprintf(stderr, _("Name:         %s\n"), entry->entry);
		fprintf(stderr, _("Qualifier:    %s\n"), entry->qualif);
		fprintf(stderr, _("Values:       "));

		for (i=0; i < entry->nvalues; i++)
			fprintf(stderr, "\"%s\" ", entry->values[i]);

		fprintf(stderr, "\n\n");
	}

	if (! strcmp(entry->section, SEC_SHORTCUT))
	{
		err_reported = 0;

		parse_shortcut_map(entry);
	}
	else if (! strcmp(entry->section, SEC_MAPSET))
	{
		err_reported = 0;

		if (! strcmp(entry->entry, ENT_MAP))
			parse_mapset_map(entry);
		else
			ent_err = 1;
	}
	else
		sec_err = 1;

	if (sec_err)
	{
		if (! err_reported)
		{
			fprintf(stderr, _("Invalid section name %s in %s near line %d\n"), entry->section, cur_cfg_file, configparserlineno-1);
			err_reported = 1;
		}
	}

	if (ent_err)
	{
		fprintf(stderr, _("Invalid entry name %s in %s at line %d\n"), entry->entry, cur_cfg_file, configparserlineno-1);
		err_reported = 1;
	}

	return;
}

/* free the memory used by a config entry */
void free_config_entry(config_entry_t *entry)
{
	int i;

	if (! entry)
		return;

	for (i=0; i < entry->nvalues; i++)
		if (entry->values[i] != NULL)
			free(entry->values[i]);

	free(entry);
}

/* Get a path name for a compiled map. Returned data
   was allocated via malloc and should be freed when
   no more needed.

   The databases must be open (calls read_db internally)
*/
int get_compiled_map_names(const db_record_t *rec, char **user, char **sys)
{
	char mappath[MAX_DATA_PATH_LEN+1];
	int lu, ls;
	char *p;

	*user = *sys = NULL;

	strcpy(mappath, rec->map_name);

	/* Convert ...foo(xyz) to ...foo/xyz */
	if ((p=strrchr(mappath, '(')) != NULL)
		*p = '/';

	if ((p=strrchr(mappath, ')')) != NULL)
		*p = 0;

	lu = strlen(user_xkm_dir) + strlen(mappath) + 8;
	ls = strlen(sys_xkm_dir)  + strlen(mappath) + 8;

	if (lu >= PATH_MAX)
	{
		fprintf(stderr, _("Total path name %s/%s too long\n"), user_xkm_dir, mappath);
		return -1;
	}
	else
		*user = (char *) malloc(lu);

	if (ls >= PATH_MAX)
	{
		fprintf(stderr, _("Total path name %s/%s too long\n"), sys_xkm_dir, mappath);
		return -1;
	}
	else
		*sys = (char *) malloc(ls);

	if (user == NULL || sys == NULL)
	{
		fprintf(stderr, _("Out of memory\n"));
		return -1;
	}

	sprintf(*user, "%s/%s.xkm", user_xkm_dir, mappath);
	sprintf(*sys, "%s/%s.xkm",  sys_xkm_dir,  mappath);

	return 0;
}



/* === Private functions === */

static void print_ex(const char *desc, const char *fn, int type)
{
	struct stat st;

	printf("%-14s: %-34s", desc, fn);

	if (type == -1)
	{
		printf("\n");
		return;
	}

	printf(": ");

	if (type == 2)
	{
		DBM *db;

		db = dbm_open(fn, O_RDONLY, 0);

		if (db)
		{
			printf(_("exists\n"));
			dbm_close(db);
		}
		else
			printf(_("nonexistent or not readable\n"));

		return;
	}

	if (access(fn, F_OK) < 0)
	{
		printf(_("nonexistent\n"));
		return;
	}

	if (access(fn, R_OK) < 0)
	{
		printf(_("not readable\n"));
		return;
	}

	if (stat(fn, &st) < 0)
	{
		printf(_("cannot stat?!\n"));
		return;
	}

	if ((type == 0 && S_ISREG(st.st_mode)) || (type == 1 && S_ISDIR(st.st_mode)))
	{
		printf(_("exists\n"));
		return;
	}

	if (type == 0)
		printf(_("exists, but not a file\n"));
	else if (type == 1)
		printf(_("exists, but not a directory\n"));

	return;
}

static int set_paths()
{
	char *home;
	char buf[PATH_MAX+1];

	/* Initialise user data */
	home = getenv("HOME");
	if (home == NULL)
	{
		fprintf(stderr, _("HOME variable not set\n"));
		return -1;
	}

	if (strlen(home) >= PATH_MAX - 128)
	{
		fprintf(stderr, _("HOME variable too long\n"));
		return -1;
	}

	sprintf(buf, "%s/keymap", X11_XKB_DIR);
	x11_keymap_dir = strdup(buf);

	sprintf(buf, "%s/keymap", SYS_DIR);
	sys_keymap_dir = strdup(buf);

	sprintf(buf, "%s/%s", home, USER_DIR);
	user_dir = strdup(buf);

	sprintf(buf, "%s/%s/keymap", home, USER_DIR);
	user_keymap_dir = strdup(buf);

	sprintf(buf, "%s/%s", home, USER_REDEF_DIR);
	user_redef_dir = strdup(buf);

	sprintf(buf, "%s/%s", home, USER_PIXMAP_DIR);
	user_pixmap_dir = strdup(buf);

	sprintf(buf, "%s/%s", home, USER_XKM_DIR);
	user_xkm_dir = strdup(buf);

	sprintf(buf, "%s/%s", home, USER_DB);
	user_db = strdup(buf);

	sprintf(buf, "%s/%s", home, USER_CONFIG);
	user_config = strdup(buf);

	return 0;
}

static int determine_locale()
{
	/* Caution: this is most probably system-specific, especially
	   when the returned value says the alias etc. I don't know
	   of any function, that reliably returns language and
	   region.
	*/

	char *loc;
	char *point, *under, *at;
	char buf[64];

	loc = getenv("XKBSEL_LOCALE");

	if (loc == NULL)
		loc = setlocale(LC_CTYPE, NULL);

	if (loc == NULL)
	{
		fprintf(stderr, _("Cannot get LC_CTYPE - fix your locale settings or use XKBSEL_LOCALE\n"));
		locale_lang = strdup("C");
		locale_territ = strdup("C");
		return 0;
	}

	if (strlen(loc) >= sizeof(buf))
	{
		fprintf(stderr, _("Locale name %s too long\n"), loc);
		locale_lang = strdup("C");
		locale_territ = strdup("C");
		return -1;
	}

	if (flag_debug)
		fprintf(stderr, _("Parsing locale name %s\n"), loc);

	/* The format should be ll_tt.charset@modifier */
	strcpy(buf, loc);

	under = strchr(buf, '_');
	point = strchr(buf, '.');
	at    = strchr(buf, '@');

	/* Ignore modifier, if any */
	if (at != NULL)
		*at = 0;

	/* We don't (yet) need the codeset */
	if (point != NULL)
		*point = 0;

	/* Mark the end of language part */
	if (under != NULL)
		*under = 0;

	if (! strcmp(buf, "C") || ! strcmp(buf, "POSIX"))
	{
		locale_lang = strdup("C");
		locale_territ = strdup("C");

		return 0;
	}

	if (strlen(buf) == 2)
	{
		locale_lang = strdup(buf);
	}
	else
	{
		fprintf(stderr, _("cannot determine language from %s - use XKBSEL_LOCALE\n"), buf);
		locale_lang = strdup("C");
		locale_territ = strdup("C");
		return -1;
	}

	if (under != NULL)
	{
		if (strlen(under+1) == 2)
		{
			locale_territ = strdup(under+1);
		}
		else
		{
			fprintf(stderr, _("cannot determine territory from %s - use XKBSEL_LOCALE\n"), under+1);
			locale_lang = strdup("C");
			locale_territ = strdup("C");
			return -1;
		}
	}

	return 0;
}

static int parse_config()
{
	const int FIRST_CHUNK = 16;
	int r;

	configparserin = NULL;

	/* Allocate the first chunk - parser callback assumes this */
	cfg_maps = (sel_info_t *) malloc(FIRST_CHUNK * sizeof(sel_info_t));
	locale_maps = (sel_info_t *) malloc(FIRST_CHUNK * sizeof(sel_info_t));

	if (cfg_maps == NULL || locale_maps == NULL)
	{
		fprintf(stderr, _("Out of memory\n"));
		return -1;
	}

	n_cfg_maps = n_locale_maps = 0;
	a_cfg_maps = a_locale_maps = FIRST_CHUNK;
	cur_cfg_file = NULL;

	/* Try user one first, if accessible */
	if (! access(user_config, R_OK))
	{
		configparserin = fopen(user_config, "r");

		if (configparserin == NULL)
		{
			fprintf(stderr, _("cannot open user configuration file %s: %s\n"), user_config, strerror(errno));
		}

		if (configparserin != NULL)
			cur_cfg_file = user_config;

		if (flag_debug && configparserin != NULL)
		{
			fprintf(stderr, _("Reading configuration from %s\n"), user_config);
			fprintf(stderr, "\n");
		}
	}

	if (configparserin == NULL)
	{
		configparserin = fopen(sys_config, "r");

		if (configparserin == NULL)
		{
			fprintf(stderr, _("cannot open system configuration file %s: %s\n"), sys_config, strerror(errno));
			return -1;
		}

		if (configparserin != NULL)
			cur_cfg_file = sys_config;

		if (flag_debug && configparserin != NULL)
		{
			fprintf(stderr, _("Reading configuration from %s\n"), sys_config);
			fprintf(stderr, "\n");
		}
	}

	/* Parse the file */
	configparserlex();

	fclose(configparserin);

	if (! n_cfg_maps)
	{
		fprintf(stderr, _("No maps were specified in %s\n"), cur_cfg_file);
		return -1;
	}

	/* Configure lists for GUI selectors */
	if (locale_mapset != NULL || dflt_mapset != NULL)
	{
		int i;

		char **vals = (locale_mapset != NULL) ?
			locale_mapset->values :
			dflt_mapset->values;

		int nvals = (locale_mapset != NULL) ?
			locale_mapset->nvalues :
			dflt_mapset->nvalues;

		for (i=0; i < nvals; i++)
		{
			/* Find in all maps (must find) */
			int j;

			for (j=0; j < n_cfg_maps; j++)
				if (! strcmp(vals[i], cfg_maps[j].shortcut))
					break;

			if (j == n_cfg_maps)
			{
				fprintf(stderr, _("Map name %s was not defined in [%s] section\n"), vals[i], SEC_SHORTCUT);
				return -1;
			}

			r = add_maps(&locale_maps, &n_locale_maps, &a_locale_maps, cfg_maps[j].shortcut, cfg_maps[j].map_name);
			if (r < 0)
				return -1;
		}
	}
	else
	{
		int i;

		fprintf(stderr, _("No default map set specified - using all configured maps\n"));

		for (i = 0; i < n_cfg_maps; i++)
		{
			r = add_maps(&locale_maps, &n_locale_maps, &a_locale_maps, cfg_maps[i].shortcut, cfg_maps[i].map_name);
			if (r < 0)
				return -1;
		}
	}

	return 0;
}

static int set_x_data()
{
	/* Handle display, if not already done */
	if (display_name == NULL)
	{
		char *p;

		p = getenv("DISPLAY");

		if (p != NULL)
			display_name = strdup(p);

		/* Fall back to :0.0 */
		if (! display_name)
			display_name = strdup(":0.0");
	}

	return 0;
}

static int parse_shortcut_map(config_entry_t *entry)
{
	int i, r;

	if (*entry->qualif)
		fprintf(stderr, _("Qualification ignored in %s at line %d\n"), cur_cfg_file, configparserlineno-1);

	if (entry->nvalues != 1)
	{
		fprintf(stderr, _("Exactly one value needed in %s at line %d\n"), cur_cfg_file, configparserlineno-1);
		return -1;
	}

	if (strlen(entry->entry) >= sizeof(cfg_maps[0].shortcut) ||
	    strlen(entry->values[0]) >= sizeof(cfg_maps[0].map_name))
	{
		fprintf(stderr, _("Entry or value too long in %s at line %d\n"), cur_cfg_file, configparserlineno-1);
		return -1;
	}

	/* Check if we have it already */
	for (i=0; i < n_cfg_maps; i++)
		if (! strcmp(cfg_maps[i].shortcut, entry->entry))
		{
			fprintf(stderr, _("Entry %s already exists in %s at line %d\n"), entry->entry, cur_cfg_file, configparserlineno-1);
			return -1;
		}

	/* OK - add */
	r = add_maps(&cfg_maps, &n_cfg_maps, &a_cfg_maps, entry->entry, entry->values[0]);
	if (r < 0)
		return r;

	return 0;
}

static int parse_mapset_map(config_entry_t *entry)
{
	int i;

	if (! entry->nvalues)
	{
		fprintf(stderr, _("One or more values needed in %s at line %d\n"), cur_cfg_file, configparserlineno-1);
		return -1;
	}

	/* Check for duplicity and whether we know about the shortcut */
	for (i=0; i < entry->nvalues; i++)
	{
		int j;

		for (j=0; j < n_cfg_maps; j++)
			if (! strcmp(entry->values[i], cfg_maps[j].shortcut))
				break;

		if (j == n_cfg_maps)
		{
			fprintf(stderr, _("Map name %s was not defined in [%s] section in %s at line %d\n"), entry->values[i], SEC_SHORTCUT, cur_cfg_file, configparserlineno-1);
			return -1;
		}

		for (j=0; j < i; j++)
			if (! strcmp(entry->values[j], entry->values[i]))
			{
				fprintf(stderr, _("Duplicate entry %s ignored in %s at line %d\n"), entry->values[i], cur_cfg_file, configparserlineno-1);
				return 0;
			}
	}

	if (! strcmp(entry->qualif, locale_lang) ||
	    ! strcmp(entry->qualif, locale_territ))
	{
		if (locale_mapset != NULL)
		{
			fprintf(stderr, _("Map set for language/territory %s already set in %s at line %d\n"), entry->qualif, cur_cfg_file, configparserlineno-1);
			return -1;
		}

		locale_mapset = entry;
	}
	else if (! *entry->qualif ||
	         ! strcmp(entry->qualif, "C") ||
	         ! strcmp(entry->qualif, "POSIX"))
	{
		if (dflt_mapset != NULL)
		{
			fprintf(stderr, _("Map set for default locale already set in %s at line %d\n"), cur_cfg_file, configparserlineno-1);
			return -1;
		}

		dflt_mapset = entry;
	}

	return 0;
}

static int add_maps(sel_info_t **info, int *n, int *an, const char *shortcut, const char *map_name)
{
	const int CHUNK = 16;

	if (*n > *an)
	{
		cfg_maps = realloc(*info, (*an + CHUNK) * sizeof(sel_info_t));
		if (*info == NULL)
		{
			fprintf(stderr, _("Out of memory\n"));
			return -1;
		}

		*an += CHUNK;
	}

	if (flag_debug)
		fprintf(stderr, _("Adding into maps: %d: %s %s\n"), *n, shortcut, map_name);

	strcpy((*info)[*n].shortcut, shortcut);
	strcpy((*info)[*n].map_name, map_name);
	(*n)++;

	return 0;
}
