package net.iharder.dnd;

import java.awt.datatransfer.DataFlavor;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.io.Reader;

/**
 * This class makes it easy to drag and drop files from the operating system
 * to a Java program. Any <tt>java.awt.Component</tt> can be dropped onto,
 * but only <tt>javax.swing.JComponent</tt>s will indicate the drop event
 * with a changed border. <p/> To use this class, construct a new
 * <tt>FileDrop</tt> by passing it the target component and a
 * <tt>Listener</tt> to receive notification when file(s) have been dropped.
 * Here is an example: <p/> <code><pre>
 *      JPanel myPanel = new JPanel();
 *      new FileDrop( myPanel, new FileDrop.Listener()
 *      {   public void filesDropped( java.io.File[] files )
 *          {   
 *              // handle file drop
 *              ...
 *          }   // end filesDropped
 *      }); // end FileDrop.Listener
 * </pre></code> <p/> You can specify the border that will appear when files are
 * being dragged by calling the constructor with a
 * <tt>javax.swing.border.Border</tt>. Only <tt>JComponent</tt>s will
 * show any indication with a border. <p/> You can turn on some debugging
 * features by passing a <tt>PrintStream</tt> object (such as
 * <tt>System.out</tt>) into the full constructor. A <tt>null</tt> value
 * will result in no extra debugging information being output. <p/>
 * 
 * <p>
 * I'm releasing this code into the Public Domain. Enjoy.
 * </p>
 * <p>
 * <em>Original author: Robert Harder, rharder@usa.net</em>
 * </p>
 * <p>
 * 2007-09-12 Nathan Blomquist -- Linux (KDE/Gnome) support added.
 * </p>
 * 
 * @author Robert Harder
 * @author rharder@users.sf.net
 * @version 1.0.1
 */
@SuppressWarnings("all")
public class FileDrop {
    private transient javax.swing.border.Border normalBorder;

    private transient java.awt.dnd.DropTargetListener dropListener;

    /** Discover if the running JVM is modern enough to have drag and drop. */
    private static Boolean supportsDnD;

    // Default border color
    private static java.awt.Color defaultBorderColor = new java.awt.Color(0f,
            0f, 1f, 0.25f);

    /** Class used to check if DND is supported. (Presence is checked). */
    private static final String DND_REFERENCE_CLASS = "java.awt.dnd.DnDConstants";

    /**
     * Constructs a {@link FileDrop} with a default light-blue border and, if
     * <var>c</var> is a {@link java.awt.Container}, recursively sets all
     * elements contained within as drop targets, though only the top level
     * container will change borders.
     * 
     * @param c
     *                Component on which files will be dropped.
     * @param listener
     *                Listens for <tt>filesDropped</tt>.
     * @since 1.0
     */
    public FileDrop(final java.awt.Component c, final DropListener listener) {
        this(null, // Logging stream
                c, // Drop target
                javax.swing.BorderFactory.createMatteBorder(2, 2, 2, 2,
                        defaultBorderColor), // Drag border
                true, // Recursive
                listener);
    } // end constructor

    /**
     * Constructor with a default border and the option to recursively set
     * drop targets. If your component is a <tt>java.awt.Container</tt>,
     * then each of its children components will also listen for drops, though
     * only the parent will change borders.
     * 
     * @param c
     *                Component on which files will be dropped.
     * @param recursive
     *                Recursively set children as drop targets.
     * @param listener
     *                Listens for <tt>filesDropped</tt>.
     * @since 1.0
     */
    public FileDrop(final java.awt.Component c, final boolean recursive,
            final DropListener listener) {
        this(null, // Logging stream
                c, // Drop target
                javax.swing.BorderFactory.createMatteBorder(2, 2, 2, 2,
                        defaultBorderColor), // Drag border
                recursive, // Recursive
                listener);
    } // end constructor

    /**
     * Constructor with a default border and debugging optionally turned on.
     * With Debugging turned on, more status messages will be displayed to
     * <tt>out</tt>. A common way to use this constructor is with
     * <tt>System.out</tt> or <tt>System.err</tt>. A <tt>null</tt>
     * value for the parameter <tt>out</tt> will result in no debugging
     * output.
     * 
     * @param out
     *                PrintStream to record debugging info or null for no
     *                debugging.
     * @param c
     *                Component on which files will be dropped.
     * @param listener
     *                Listens for <tt>filesDropped</tt>.
     * @since 1.0
     */
    public FileDrop(final java.io.PrintStream out,
            final java.awt.Component c, final DropListener listener) {
        this(out, // Logging stream
                c, // Drop target
                javax.swing.BorderFactory.createMatteBorder(2, 2, 2, 2,
                        defaultBorderColor), false, // Recursive
                listener);
    } // end constructor

    /**
     * Constructor with a default border, debugging optionally turned on and
     * the option to recursively set drop targets. If your component is a
     * <tt>java.awt.Container</tt>, then each of its children components
     * will also listen for drops, though only the parent will change borders.
     * With Debugging turned on, more status messages will be displayed to
     * <tt>out</tt>. A common way to use this constructor is with
     * <tt>System.out</tt> or <tt>System.err</tt>. A <tt>null</tt>
     * value for the parameter <tt>out</tt> will result in no debugging
     * output.
     * 
     * @param out
     *                PrintStream to record debugging info or null for no
     *                debugging.
     * @param c
     *                Component on which files will be dropped.
     * @param recursive
     *                Recursively set children as drop targets.
     * @param listener
     *                Listens for <tt>filesDropped</tt>.
     * @since 1.0
     */
    public FileDrop(final java.io.PrintStream out,
            final java.awt.Component c, final boolean recursive,
            final DropListener listener) {
        this(out, // Logging stream
                c, // Drop target
                javax.swing.BorderFactory.createMatteBorder(2, 2, 2, 2,
                        defaultBorderColor), // Drag border
                recursive, // Recursive
                listener);
    } // end constructor

    /**
     * Constructor with a specified border
     * 
     * @param c
     *                Component on which files will be dropped.
     * @param dragBorder
     *                Border to use on <tt>JComponent</tt> when dragging
     *                occurs.
     * @param listener
     *                Listens for <tt>filesDropped</tt>.
     * @since 1.0
     */
    public FileDrop(final java.awt.Component c,
            final javax.swing.border.Border dragBorder,
            final DropListener listener) {
        this(null, // Logging stream
                c, // Drop target
                dragBorder, // Drag border
                false, // Recursive
                listener);
    } // end constructor

    /**
     * Constructor with a specified border and the option to recursively set
     * drop targets. If your component is a <tt>java.awt.Container</tt>,
     * then each of its children components will also listen for drops, though
     * only the parent will change borders.
     * 
     * @param c
     *                Component on which files will be dropped.
     * @param dragBorder
     *                Border to use on <tt>JComponent</tt> when dragging
     *                occurs.
     * @param recursive
     *                Recursively set children as drop targets.
     * @param listener
     *                Listens for <tt>filesDropped</tt>.
     * @since 1.0
     */
    public FileDrop(final java.awt.Component c,
            final javax.swing.border.Border dragBorder,
            final boolean recursive, final DropListener listener) {
        this(null, c, dragBorder, recursive, listener);
    } // end constructor

    /**
     * Constructor with a specified border and debugging optionally turned on.
     * With Debugging turned on, more status messages will be displayed to
     * <tt>out</tt>. A common way to use this constructor is with
     * <tt>System.out</tt> or <tt>System.err</tt>. A <tt>null</tt>
     * value for the parameter <tt>out</tt> will result in no debugging
     * output.
     * 
     * @param out
     *                PrintStream to record debugging info or null for no
     *                debugging.
     * @param c
     *                Component on which files will be dropped.
     * @param dragBorder
     *                Border to use on <tt>JComponent</tt> when dragging
     *                occurs.
     * @param listener
     *                Listens for <tt>filesDropped</tt>.
     * @since 1.0
     */
    public FileDrop(final java.io.PrintStream out,
            final java.awt.Component c,
            final javax.swing.border.Border dragBorder,
            final DropListener listener) {
        this(out, // Logging stream
                c, // Drop target
                dragBorder, // Drag border
                false, // Recursive
                listener);
    } // end constructor

    /**
     * Full constructor with a specified border and debugging optionally
     * turned on. With Debugging turned on, more status messages will be
     * displayed to <tt>out</tt>. A common way to use this constructor is
     * with <tt>System.out</tt> or <tt>System.err</tt>. A <tt>null</tt>
     * value for the parameter <tt>out</tt> will result in no debugging
     * output.
     * 
     * @param out
     *                PrintStream to record debugging info or null for no
     *                debugging.
     * @param c
     *                Component on which files will be dropped.
     * @param dragBorder
     *                Border to use on <tt>JComponent</tt> when dragging
     *                occurs.
     * @param recursive
     *                Recursively set children as drop targets.
     * @param listener
     *                Listens for <tt>filesDropped</tt>.
     * @since 1.0
     */
    public FileDrop(final java.io.PrintStream out,
            final java.awt.Component c,
            final javax.swing.border.Border dragBorder,
            final boolean recursive, final DropListener listener) {

        if (supportsDnD()) { // Make a drop listener
            dropListener = new java.awt.dnd.DropTargetListener() {
                public final void dragEnter(
                        final java.awt.dnd.DropTargetDragEvent evt) {
                    log(out, "FileDrop: dragEnter event.");

                    // Is this an acceptable drag event?
                    if (isDragOk(out, evt)) {
                        // If it's a Swing component, set its border
                        if (c instanceof javax.swing.JComponent) {
                            final javax.swing.JComponent jc = (javax.swing.JComponent) c;
                            normalBorder = jc.getBorder();
                            log(out, "FileDrop: normal border saved.");
                            jc.setBorder(dragBorder);
                            log(out, "FileDrop: drag border set.");
                        } // end if: JComponent

                        // Acknowledge that it's okay to enter
                        // evt.acceptDrag(
                        // java.awt.dnd.DnDConstants.ACTION_COPY_OR_MOVE );
                        evt.acceptDrag(java.awt.dnd.DnDConstants.ACTION_COPY);
                        log(out, "FileDrop: event accepted.");
                    } // end if: drag ok
                    else { // Reject the drag event
                        evt.rejectDrag();
                        log(out, "FileDrop: event rejected.");
                    } // end else: drag not ok
                } // end dragEnter

                public final void dragOver(
                        final java.awt.dnd.DropTargetDragEvent evt) { // This
                    // is
                    // called
                    // continually
                    // as long
                    // as the
                    // mouse
                    // is
                    // over the drag target.
                } // end dragOver

                public final void drop(
                        final java.awt.dnd.DropTargetDropEvent evt) {
                    log(out, "FileDrop: drop event.");
                    try { // Get whatever was dropped
                        final java.awt.datatransfer.Transferable tr = evt
                                .getTransferable();

                        // Is it a file list?
                        if (tr
                                .isDataFlavorSupported(java.awt.datatransfer.DataFlavor.javaFileListFlavor)) {
                            // Say we'll take it.
                            // evt.acceptDrop (
                            // java.awt.dnd.DnDConstants.ACTION_COPY_OR_MOVE
                            // );
                            evt
                                    .acceptDrop(java.awt.dnd.DnDConstants.ACTION_COPY);
                            log(out, "FileDrop: file list accepted.");

                            // Get a useful list
                            final Object lst = tr
                                    .getTransferData(java.awt.datatransfer.DataFlavor.javaFileListFlavor);
                            if (lst instanceof java.util.List<?>) {
                                final java.util.List<File> fileList = (java.util.List<File>) lst;
                                // java.util.Iterator iterator =
                                // fileList.iterator();

                                // Convert list to array
                                final java.io.File[] filesTemp = new java.io.File[fileList
                                        .size()];
                                fileList.toArray(filesTemp);
                                final java.io.File[] files = filesTemp;

                                // Alert listener to drop.
                                if (listener != null) {
                                    listener.filesDropped(files);
                                }
                            } else {
                                log(out, "FileDrop: not a list of files");
                            }

                            // Mark that drop is completed.
                            evt.getDropTargetContext().dropComplete(true);
                            log(out, "FileDrop: drop complete.");
                        } // end if: file list
                        else // this section will check for a reader
                        // flavor.
                        {
                            // Thanks, Nathan!
                            // BEGIN 2007-09-12 Nathan Blomquist -- Linux
                            // (KDE/Gnome) support
                            // added.
                            final DataFlavor[] flavors = tr
                                    .getTransferDataFlavors();
                            boolean handled = false;
                            for (int zz = 0; zz < flavors.length; zz++) {
                                if (flavors[zz].isRepresentationClassReader()) {
                                    // Say we'll take it.
                                    // evt.acceptDrop (
                                    // java.awt.dnd.DnDConstants.ACTION_COPY_OR_MOVE
                                    // );
                                    evt
                                            .acceptDrop(java.awt.dnd.DnDConstants.ACTION_COPY);
                                    log(out, "FileDrop: reader accepted.");

                                    final Reader reader = flavors[zz]
                                            .getReaderForText(tr);

                                    final BufferedReader br = new BufferedReader(
                                            reader);

                                    if (listener != null) {
                                        listener
                                                .filesDropped(createFileArray(
                                                        br, out));
                                    }

                                    // Mark that drop is completed.
                                    evt.getDropTargetContext().dropComplete(
                                            true);
                                    log(out, "FileDrop: drop complete.");
                                    handled = true;
                                    break;
                                }
                            }
                            if (!handled) {
                                log(out,
                                        "FileDrop: not a file list or reader - abort.");
                                evt.rejectDrop();
                            }
                            // END 2007-09-12 Nathan Blomquist -- Linux
                            // (KDE/Gnome) support
                            // added.
                        } // end else: not a file list
                    } // end try
                    catch (final java.io.IOException io) {
                        log(out, "FileDrop: IOException - abort:");
                        io.printStackTrace(out);
                        evt.rejectDrop();
                    } // end catch IOException
                    catch (final java.awt.datatransfer.UnsupportedFlavorException ufe) {
                        log(out,
                                "FileDrop: UnsupportedFlavorException - abort:");
                        ufe.printStackTrace(out);
                        evt.rejectDrop();
                    } // end catch: UnsupportedFlavorException
                    finally {
                        // If it's a Swing component, reset its border
                        if (c instanceof javax.swing.JComponent) {
                            final javax.swing.JComponent jc = (javax.swing.JComponent) c;
                            jc.setBorder(normalBorder);
                            log(out, "FileDrop: normal border restored.");
                        } // end if: JComponent
                    } // end finally
                } // end drop

                public final void dragExit(
                        final java.awt.dnd.DropTargetEvent evt) {
                    log(out, "FileDrop: dragExit event.");
                    // If it's a Swing component, reset its border
                    if (c instanceof javax.swing.JComponent) {
                        final javax.swing.JComponent jc = (javax.swing.JComponent) c;
                        jc.setBorder(normalBorder);
                        log(out, "FileDrop: normal border restored.");
                    } // end if: JComponent
                } // end dragExit

                public void dropActionChanged(
                        final java.awt.dnd.DropTargetDragEvent evt) {
                    log(out, "FileDrop: dropActionChanged event.");
                    // Is this an acceptable drag event?
                    if (isDragOk(out, evt)) { // evt.acceptDrag(
                        // java.awt.dnd.DnDConstants.ACTION_COPY_OR_MOVE
                        // );
                        evt.acceptDrag(java.awt.dnd.DnDConstants.ACTION_COPY);
                        log(out, "FileDrop: event accepted.");
                    } // end if: drag ok
                    else {
                        evt.rejectDrag();
                        log(out, "FileDrop: event rejected.");
                    } // end else: drag not ok
                } // end dropActionChanged
            }; // end DropTargetListener

            // Make the component (and possibly children) drop targets
            makeDropTarget(out, c, recursive);
        } // end if: supports dnd
        else {
            log(out, "FileDrop: Drag and drop is not supported with this JVM");
        } // end else: does not support DnD
    } // end constructor

    private final static boolean supportsDnD() { // Static Boolean
        if (supportsDnD == null) {
            boolean support = false;
            try {
                support = Class.forName(DND_REFERENCE_CLASS).getName()
                        .equals(DND_REFERENCE_CLASS);
            } // end try
            catch (final Exception e) {
                support = false;
            } // end catch
            supportsDnD = Boolean.valueOf(support);
        } // end if: first time through
        return supportsDnD.booleanValue();
    } // end supportsDnD

    // BEGIN 2007-09-12 Nathan Blomquist -- Linux (KDE/Gnome) support added.
    private final static String ZERO_CHAR_STRING = "" + (char) 0;

    private final static File[] createFileArray(final BufferedReader bReader,
            final PrintStream out) {
        try {
            final java.util.List<File> list = new java.util.ArrayList<File>();
            java.lang.String line = null;
            while ((line = bReader.readLine()) != null) {
                try {
                    // kde seems to append a 0 char to the end of the reader
                    if (ZERO_CHAR_STRING.equals(line)) {
                        continue;
                    }

                    final java.io.File file = new java.io.File(
                            new java.net.URI(line));
                    list.add(file);
                } catch (final java.net.URISyntaxException ex) {
                    log(out, "FileDrop: URISyntaxException");
                }
            }

            return list.toArray(new File[list.size()]);
        } catch (final IOException ex) {
            log(out, "FileDrop: IOException");
        }
        return new File[0];
    }

    // END 2007-09-12 Nathan Blomquist -- Linux (KDE/Gnome) support added.

    private final void makeDropTarget(final java.io.PrintStream out,
            final java.awt.Component c, final boolean recursive) {
        // Make drop target
        final java.awt.dnd.DropTarget dt = new java.awt.dnd.DropTarget();
        try {
            dt.addDropTargetListener(dropListener);
        } // end try
        catch (final java.util.TooManyListenersException e) {
            e.printStackTrace();
            log(
                    out,
                    "FileDrop: Drop will not work due to previous error. Do you have another listener attached?");
        } // end catch

        // Listen for hierarchy changes and remove the drop target when the
        // parent
        // gets cleared out.
        c.addHierarchyListener(new java.awt.event.HierarchyListener() {
            public void hierarchyChanged(
                    final java.awt.event.HierarchyEvent evt) {
                log(out, "FileDrop: Hierarchy changed.");
                final java.awt.Component parent = c.getParent();
                if (parent == null) {
                    c.setDropTarget(null);
                    log(out, "FileDrop: Drop target cleared from component.");
                } // end if: null parent
                else {
                    new java.awt.dnd.DropTarget(c, dropListener);
                    log(out, "FileDrop: Drop target added to component.");
                } // end else: parent not null
            } // end hierarchyChanged
        }); // end hierarchy listener
        if (c.getParent() != null) {
            new java.awt.dnd.DropTarget(c, dropListener);
        }

        if (recursive && (c instanceof java.awt.Container)) {
            // Get the container
            final java.awt.Container cont = (java.awt.Container) c;

            // Get it's components
            final java.awt.Component[] comps = cont.getComponents();

            // Set it's components as listeners also
            for (int i = 0; i < comps.length; i++) {
                makeDropTarget(out, comps[i], recursive);
            }
        } // end if: recursively set components as listener
    } // end dropListener

    /** Determine if the dragged data is a file list. */
    private final boolean isDragOk(final java.io.PrintStream out,
            final java.awt.dnd.DropTargetDragEvent evt) {
        boolean ok = false;

        // Get data flavors being dragged
        final java.awt.datatransfer.DataFlavor[] flavors = evt
                .getCurrentDataFlavors();

        // See if any of the flavors are a file list
        int i = 0;
        while (!ok && (i < flavors.length)) {
            // BEGIN 2007-09-12 Nathan Blomquist -- Linux (KDE/Gnome) support
            // added.
            // Is the flavor a file list?
            final DataFlavor curFlavor = flavors[i];
            if (curFlavor
                    .equals(java.awt.datatransfer.DataFlavor.javaFileListFlavor)
                    || curFlavor.isRepresentationClassReader()) {
                ok = true;
            }
            // END 2007-09-12 Nathan Blomquist -- Linux (KDE/Gnome) support
            // added.
            i++;
        } // end while: through flavors

        // If logging is enabled, show data flavors
        if (out != null) {
            if (flavors.length == 0) {
                log(out, "FileDrop: no data flavors.");
            }
            for (i = 0; i < flavors.length; i++) {
                log(out, flavors[i].toString());
            }
        } // end if: logging enabled

        return ok;
    } // end isDragOk

    /** Outputs <tt>message</tt> to <tt>out</tt> if it's not null. */
    private static void log(final java.io.PrintStream out,
            final String message) { // Log
        // message if requested
        if (out != null) {
            out.println(message);
        }
    } // end log

    /**
     * Removes the drag-and-drop hooks from the component and optionally from
     * the all children. You should call this if you add and remove components
     * after you've set up the drag-and-drop. This will recursively unregister
     * all components contained within <var>c</var> if <var>c</var> is a
     * {@link java.awt.Container}.
     * 
     * @param c
     *                The component to unregister as a drop target
     * @since 1.0
     */
    public static boolean remove(final java.awt.Component c) {
        return remove(null, c, true);
    } // end remove

    /**
     * Removes the drag-and-drop hooks from the component and optionally from
     * the all children. You should call this if you add and remove components
     * after you've set up the drag-and-drop.
     * 
     * @param out
     *                Optional {@link java.io.PrintStream} for logging drag
     *                and drop messages
     * @param c
     *                The component to unregister
     * @param recursive
     *                Recursively unregister components within a container
     * @since 1.0
     */
    public static boolean remove(final java.io.PrintStream out,
            final java.awt.Component c, final boolean recursive) { // Make
        // sure we
        // support
        // dnd.
        if (supportsDnD()) {
            log(out, "FileDrop: Removing drag-and-drop hooks.");
            c.setDropTarget(null);
            if (recursive && (c instanceof java.awt.Container)) {
                final java.awt.Component[] comps = ((java.awt.Container) c)
                        .getComponents();
                for (int i = 0; i < comps.length; i++) {
                    remove(out, comps[i], recursive);
                }
                return true;
            } // end if: recursive
            else {
                return false;
            }
        } // end if: supports DnD
        else {
            return false;
        }
    } // end remove

    /** Runs a sample program that shows dropped files */
    public final static void main(final String[] args) {
        final javax.swing.JFrame frame = new javax.swing.JFrame("FileDrop");
        // javax.swing.border.TitledBorder dragBorder = new
        // javax.swing.border.TitledBorder( "Drop 'em" );
        final javax.swing.JTextArea text = new javax.swing.JTextArea();
        frame.getContentPane().add(new javax.swing.JScrollPane(text),
                java.awt.BorderLayout.CENTER);

        new FileDrop(System.out, text, /* dragBorder, */new DropListener() {
            public void filesDropped(final java.io.File[] files) {
                for (int i = 0; i < files.length; i++) {
                    try {
                        text.append(files[i].getCanonicalPath() + "\n");
                    } // end try
                    catch (final java.io.IOException e) {
                    }
                } // end for: through each dropped file
            } // end filesDropped
        }); // end FileDrop.Listener

        frame.setBounds(100, 100, 300, 400);
        frame.setDefaultCloseOperation(frame.EXIT_ON_CLOSE);
        frame.show();
    } // end main

} // end class FileDrop
