/*
  libreiserfs - a library for manipulating reiserfs partitions
  Copyright (C) 2001-2004 Yury Umanets <torque@ukrpost.net>.

  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.
                                                                                                 
  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  details.
                                                                                                 
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 Temple
  Place, Suite 330, Boston, MA 02111-1307 USA
*/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "getopt.h"

#include <unistd.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <dal/file_dal.h>
#include <reiserfs/reiserfs.h>

#include <progs/tools.h>
#include <progs/gauge.h>

#if ENABLE_NLS
# include <locale.h>
# include <libintl.h>
# define _(String) dgettext (PACKAGE, String)
#else
# define _(String) (String)
#endif

static void resizefs_print_usage(void) {
	fprintf(stderr,
		_("Usage: resizefs.reiserfs [ options ] device [+|-]size[K|M|G]\n"
		  "Options:\n"
		  "  -v | --version                  prints current version\n"
		  "  -u | --usage                    prints program usage\n"
		  "  -j FILE | --journal-device=FILE journal device for separated journal\n"
		  "  -n | --no-journal-available     no journal device available now\n"
		  "  -f | --force                    force resizer to resize partition anyway\n"
		  "  -q | --quiet                    non-interactive mode\n"));
}

int main(int argc, char *argv[]) {
	long fs_len, dev_len;
	int quiet = 0, force = 0, choice, journal = 1;
	
	dal_t *host_dal = NULL, *journal_dal = NULL;
	char *host_dev = NULL, *journal_dev = NULL, *fs_len_str = NULL;

	reiserfs_fs_t *fs;
	reiserfs_gauge_t *gauge = NULL;
    
	static struct option long_options[] = {
		{"version", no_argument, NULL, 'v'},
		{"usage", no_argument, NULL, 'u'},
		{"journal-device", required_argument, NULL, 'j'},
		{"no-journal-available", required_argument, NULL, 'n'},
		{"force", no_argument, NULL, 'f'},
		{"quiet", no_argument, NULL, 'q'},
		{0, 0, 0, 0}
	};
    
#ifdef ENABLE_NLS
	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
#endif
    
	while ((choice = getopt_long_only(argc, argv, "uvj:nqf0123456789KMG", 
					  long_options, (int *)0)) != EOF) 
	{
		switch (choice) {
		case 'u': {
			resizefs_print_usage();
			return 0;
		}
		case 'v': {
			printf("%s %s\n", argv[0], VERSION);
			return 0;
		}
		case 'n': {
			journal = 0;
			break;
		}
		case 'q': {
			quiet = 1;
			break;
		}
		case 'f': {
			force = 1;
			break;
		}
		case 'j': {
			if (!progs_check_dev((journal_dev = optarg))) {
				libreiserfs_exception_throw(EXCEPTION_ERROR,
							    EXCEPTION_CANCEL, 
							    _("Device %s doesn't "
							      "exists or invalid."),
							    journal_dev);
				return 0;
			}
		}
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
		case 'K':
		case 'M':
		case 'G': break;
		case '?': {
			resizefs_print_usage();
			return 0xfe;
		}
		}
	}
    
	host_dev = argv[optind];
    
	fs_len_str = argv[optind + (optind == argc - 1 ? -1 : 1)];
	choice = optind + (optind == argc - 1 ? -1 : 1);
    
	if (choice >= argc || choice == 0) {
		resizefs_print_usage();
		return 0xfe;
	}
    
	libreiserfs_exception_fetch_all();
	if (!progs_check_dev(host_dev)) {
		char *tmp = host_dev;
		host_dev = fs_len_str;
		fs_len_str = tmp;
	}
	libreiserfs_exception_leave_all();

	if (!progs_parse_size(fs_len_str, DEFAULT_BLOCK_SIZE)) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    _("Invalid filesystem size modificator "
					      "(%s)."), fs_len_str);
		return 0xfe;
	}
    
	/* Checking given device for validness */
	if (!progs_check_dev(host_dev)) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    _("Device %s doesn't exists or "
					      "invalid."), host_dev);
		return 0xfe;
	}
    
	/* Creating device abstraction layer */
	if (!(host_dal = file_dal_open(host_dev, DEFAULT_BLOCK_SIZE, O_RDWR))) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    _("Couldn't create device abstraction "
					      "handler for device %s."), host_dev);
		goto error;
	}

	if (journal_dev) {
		if (!strcmp(journal_dev, host_dev)) {
			journal_dal = NULL;
		} else {
			if (!(journal_dal = file_dal_open(journal_dev,
							  dal_block_size(host_dal),
							  O_RDONLY))) 
			{
				libreiserfs_exception_throw(EXCEPTION_ERROR,
							    EXCEPTION_CANCEL, 
							    _("Couldn't create "
							      "device abstraction "
							      "handler for device %s."), 
							    journal_dev);
				goto error_free_host_dal;
			}
		}    
	}

	if (!(fs = reiserfs_fs_open(host_dal, !journal ? NULL :
				    (journal_dal ? journal_dal : host_dal))))
	{
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    _("Couldn't open reiserfs on device "
					      "%s."), host_dev);
		goto error_free_journal_dal;
	}

	if (!(fs_len = progs_parse_size(fs_len_str, reiserfs_fs_block_size(fs)))) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    _("Invalid filesystem size "
					      "modificator (%s)."), fs_len_str);
		goto error_free_fs;
	}
    
	if (fs_len_str[0] == '-' || fs_len_str[0] == '+')
		fs_len += reiserfs_fs_size(fs);

	if (fs_len <= 0) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    _("Invalid filesystem size (%d)."), fs_len);
		goto error_free_fs;
	}

	dev_len = dal_len(host_dal);

	if (force) {
		if (fs_len > dev_len)
			fs_len = dev_len;
		
		if ((blk_t)fs_len < reiserfs_fs_min_size(fs))
			fs_len = reiserfs_fs_min_size(fs);
	} else {
		if (fs_len > dev_len) {
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
						    _("Can't resize filesystem outside "
						      "the device."));
			goto error_free_fs;
		}
	}
    
	choice = 'y';
	if (!quiet) {
		if (!(choice = progs_user_choose("ynYN", _("Please select (y/n) "), 
						 _("Are you ready (y/n) "))))
			goto error_free_fs;
	}
    
	if (choice == 'n' || choice == 'N')
		goto error_free_fs;

	fprintf(stderr, _("Resizing %s\n"), host_dev);
	fflush(stderr);

	gauge = progs_gauge_create();
	
	if (!(reiserfs_tree_check(fs->tree, gauge))) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    _("Filesystem %s seems to be not "
					      "consistent. Run reiserfsck first ."),
					    host_dev);
		goto error_free_fs;
	}
	
	if (!(reiserfs_fs_resize(fs, fs_len, gauge))) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
					    _("Couldn't resize filesystem on "
					      "%s to %lu blocks."), host_dev,
					    fs_len);
		goto error_free_fs;
	}
    
	progs_gauge_free(gauge);
	reiserfs_fs_close(fs);
    
	fprintf(stderr, _("syncing..."));
	fflush(stderr);
    
	if (journal_dal) {
		dal_sync(journal_dal);
		file_dal_close(journal_dal);
	}
    
	dal_sync(host_dal);
	file_dal_close(host_dal);
    
	fprintf(stderr, _("done\n"));
	fflush(stderr);
    
	return 0;
    
error_free_fs:
	if (gauge)
		progs_gauge_free(gauge);
	reiserfs_fs_close(fs);    
error_free_journal_dal:
	if (journal_dal)
		file_dal_close(journal_dal);
error_free_host_dal:
	file_dal_close(host_dal);
error:
	return 0xff;
}
